//-----------------------------------------------------------------------------
//  VinceFontXe.cpp : This is a version of 3d Fonts for use on Xenon versions
//  of VINCE using DX9. Adapted from sample code in Xenon XDK.
//
//	Created 2004/08/23 Rich Bonny <rbonny@microsoft.com>
//
// Texture-based font class. This class reads .abc font files that are generated by the
// FontMaker tool. These .abc files are used to create a texture with all the font's
// glyph, and also extract information on the dimensions of each glyph.
//
// Once created, this class is used to render text in a 3D scene with the following
// function:
//    DrawText( fScreenY, fScreenSpaceY, dwTextColor, strText,
//              dwJustificationFlags );
//
// For performance, you can batch multiple DrawText calls together between Begin() and
// End() calls, as in the following example:
//    pFont->Begin();
//    pFont->DrawText( ... );
//    pFont->DrawText( ... );
//    pFont->DrawText( ... );
//    pFont->End();
//
// The size (extent) of the text can be computed without rendering with the following
// function:
//    GetTextExtent( strText, pfReturnedWidth, pfReturnedHeight,
//                   bComputeExtentUsingFirstLineOnly );
//
// Finally, the font class can create a texture to hold rendered text, which is useful
// for static text that must be rendered for many frames, or can even be used within a
// 3D scene. (For instance, for a player's name on a jersey.) Use the following function
// for this:
//    CreateTexture( strText, d3dTextureFormat );
//
// Adapted from AtgFont class developed by Xbox Advanced Technology Group.
//
//	MICROSOFT CONFIDENTIAL.  DO NOT DISTRIBUTE.
//	Copyright (c) 2004 Microsoft Corp.  All rights reserved.
//-----------------------------------------------------------------------------

#include "VinceControl.h"

#ifdef _VINCE_
#ifndef NO_VINCE_SURVEYS

#include "VinceFontXe.h"
#include <xgraphics.h>

//--------------------------------------------------------------------------------------
// Vertex and pixel shaders for font rendering
//--------------------------------------------------------------------------------------
static const CHAR* g_strFontVertexShader =
" struct VS_IN                                  "
" {                                             "
"     float2   Pos          : POSITION;         "
"     float2   Tex          : TEXCOORD0;        "
" };                                            "
"                                               "
" struct VS_OUT                                 "
" {                                             "
"     float4 Position       : POSITION;         "
"     float4 Diffuse        : COLOR0_center;    "
"     float2 TexCoord0      : TEXCOORD0;        "
" };                                            "
"                                               "
" uniform float2   PosScale : register(c0);     "
" uniform float4   Color    : register(c1);     "
" uniform float2   TexScale : register(c2);     "
"                                               "
" VS_OUT main( VS_IN In )                       "
" {                                             "
"     VS_OUT Out;                               "
"     Out.Position.x  = ( In.Pos.x * PosScale.x - 1.0 ); "
"     Out.Position.y  =-( In.Pos.y * PosScale.y - 1.0 ); "
"     Out.Position.z  = ( 0.0 );                "
"     Out.Position.w  = ( 1.0 );                "
"     Out.Diffuse     = Color;                  "
"     Out.TexCoord0.x = In.Tex.x * TexScale.x;  "
"     Out.TexCoord0.y = In.Tex.y * TexScale.y;  "
"     return Out;                               "
" }                                             ";


static const CHAR* g_strFontPixelShader =
" struct VS_OUT                                 "
" {                                             "
"     float4 Position       : POSITION;         "
"     float4 Diffuse        : COLOR0;           "
"     float2 TexCoord0      : TEXCOORD0;        "
" };                                            "
"                                               "
" uniform float4 ChannelSelector : register(c0);"
" uniform float4 Mask            : register(c1);"
"                                               "
" sampler FontTexture : register(s0);           "
"                                               "
" float4 FontPixelShader( VS_OUT In ) : COLOR0  "
" {                                             "
    // Fetch a texel from the font texture
"   float4 FontTexel = tex2D( FontTexture, In.TexCoord0 ); "

    // Select the color from the channel
"   float value = dot( FontTexel, ChannelSelector ); "

    // For white pixels, the high bit is 1 and the low bits are luminance, so r0.a will be > 0.5.
    // For the RGB channel, we want to lop off the msb and shift the lower bits up one bit.
    // This is simple to do with the _bx2 modifier.
    // These pixels are opaque, so we emit a 1 for the alpha channel (which is 0.5 x2 ).

    // For black pixels, the high bit is 0 and the low bits are alpha, so r0.a will be < 0.5.
    // For the RGB channel, we emit zero.
    // For the alpha channel, we just use the x2 modifier to scale up the low bits of the alpha.
"   float4 Color;                               "
"   Color.rgb = ( value > 0.5f ? 2*value-1 : 0.0f ); "
"   Color.a   = 2 * ( value > 0.5f ? 1.0f : value ); "

    // Apply a mask value, which let's ignore the above and use the original texel
    // with all it's channels
"   Color = lerp( Color, FontTexel, Mask );     "

    // Return the texture color modulated with the vertex color
"   return Color * In.Diffuse;                  "
" }                                             ";


namespace Vince
{
	//-----------------------------------------------------------------------------
	// Static objects
	//-----------------------------------------------------------------------------
	static D3DVertexDeclaration* m_pFontVertexDecl   = NULL;
	static D3DVertexShader*      m_pFontVertexShader = NULL;
	static D3DPixelShader*       m_pFontPixelShader  = NULL;

	//--------------------------------------------------------------------------------------
	// Name: CVinceFont()
	// Desc: Constructor
	//--------------------------------------------------------------------------------------
	CVinceFont::CVinceFont()
	{
		m_pFontTexture       = NULL;

		m_dwNumGlyphs        = 0L;
		m_Glyphs             = NULL;

		m_fCursorX           = 0.0f;
		m_fCursorY           = 0.0f;

		m_fXScaleFactor      = 1.0f;
		m_fYScaleFactor      = 1.0f;
		m_fSlantFactor       = 0.0f;

		m_cMaxGlyph          = 0;
		m_TranslatorTable    = NULL;
	    
		m_dwNestedBeginCount = 0L;
	}


	//--------------------------------------------------------------------------------------
	// Name: ~CVinceFont()
	// Desc: Destructor
	//--------------------------------------------------------------------------------------
	CVinceFont::~CVinceFont()
	{
		Destroy();
	}

	//--------------------------------------------------------------------------------------
	// Name: CreateFontShaders()
	// Desc: Creates the global font shaders
	//--------------------------------------------------------------------------------------
	HRESULT CVinceFont::CreateFontShaders()
	{
		// Create vertex declaration
		if( NULL == m_pFontVertexDecl )
		{   
			D3DVERTEXELEMENT9 decl[] = 
			{
				{ 0, 0, D3DDECLTYPE_FLOAT2, D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_POSITION, 0 },
				{ 0, 8, D3DDECLTYPE_FLOAT2, D3DDECLMETHOD_DEFAULT, D3DDECLUSAGE_TEXCOORD, 0 },
				D3DDECL_END()
			};

			if( FAILED( m_pd3dDevice->CreateVertexDeclaration( decl, &m_pFontVertexDecl ) ) )
				return E_FAIL;
		}
		else
		{
			m_pFontVertexDecl->AddRef();
		}

		// Create vertex shader
		ID3DXBuffer* pShaderCode;
		if( NULL == m_pFontVertexShader )
		{
			if( FAILED( D3DXCompileShader( g_strFontVertexShader, strlen(g_strFontVertexShader),
										NULL, NULL, "main", "vs.2.0", 0,
										&pShaderCode, NULL, NULL ) ) )
				return E_FAIL;

			if( FAILED( m_pd3dDevice->CreateVertexShader( (DWORD*)pShaderCode->GetBufferPointer(),
														&m_pFontVertexShader ) ) )
				return E_FAIL;
	            
			pShaderCode->Release();
		}
		else
		{
			m_pFontVertexShader->AddRef();
		}
	    
		// Create pixel shader.
		if( NULL == m_pFontPixelShader )
		{
	//        if( FAILED( D3DXAssembleShader( g_strFontPixelShader, strlen(g_strFontPixelShader),
	//                                      NULL, NULL, 0, &pShaderCode, NULL ) ) )
			if( FAILED( D3DXCompileShader( g_strFontPixelShader, strlen(g_strFontPixelShader),
										NULL, NULL, "FontPixelShader", "ps.2.0", 0,
										&pShaderCode, NULL, NULL ) ) )
				return E_FAIL;

			if( FAILED( m_pd3dDevice->CreatePixelShader( (DWORD*)pShaderCode->GetBufferPointer(),
														&m_pFontPixelShader ) ) )
				return E_FAIL;
	            
			pShaderCode->Release();
		}
		else
		{
			m_pFontPixelShader->AddRef();
		}

		return S_OK;
	}

	//--------------------------------------------------------------------------------------
	// Name: Create()
	// Desc: Create the font's internal objects (texture and array of glyph info)
	//       using the XPR packed resource file
	//--------------------------------------------------------------------------------------
	HRESULT CVinceFont::Create( const CHAR* strFontFileName )
	{
		// Create the font
		if( FAILED( m_xprResource.Create( strFontFileName ) ) )
			return E_FAIL;

		return Create( m_xprResource.GetTexture("FontTexture"),
					m_xprResource.GetData("FontData") );
	}


	//--------------------------------------------------------------------------------------
	// Name: Create()
	// Desc: Create the font's internal objects (texture and array of glyph info)
	//--------------------------------------------------------------------------------------
	HRESULT CVinceFont::Create( IDirect3DTexture9* pFontTexture, const VOID* pFontData )
	{
		// Save a copy of the texture
		m_pFontTexture = pFontTexture;

		// Check version of file (to make sure it matches up with the FontMaker tool)
		const BYTE* pData = (BYTE*)pFontData;
		DWORD dwFileVersion = *((DWORD*)pData); pData += sizeof(DWORD);
	    
		if( dwFileVersion == 0x00000005 )
		{
			// Parse the font data
			m_fFontHeight        = *((FLOAT*)pData); pData += sizeof(FLOAT);
			m_fFontTopPadding    = *((FLOAT*)pData); pData += sizeof(FLOAT);
			m_fFontBottomPadding = *((FLOAT*)pData); pData += sizeof(FLOAT);
			m_fFontYAdvance      = *((FLOAT*)pData); pData += sizeof(FLOAT);

			// Point to the translator string
			m_cMaxGlyph       = ((WORD*)pData)[0];   pData += sizeof(WORD);
			m_TranslatorTable = (SHORT*)pData;       pData += sizeof(WCHAR)*(m_cMaxGlyph+1);

			// Read the glyph attributes from the file
			m_dwNumGlyphs = ((DWORD*)pFontData)[0];  pData += sizeof(DWORD);
			m_Glyphs      = (VINCE_GLYPH_ATTR*)pData;
		}
		else
		{
			//AtgUtil_PrintError( "Incorrect version number on font file!\n" );
			return E_FAIL;
		}

		// Create the vertex and pixel shaders for rendering the font
		if( FAILED( CreateFontShaders() ) )
		{
			//AtgUtil_PrintError( "Could not create font shaders!\n" );
			return E_FAIL;
		}

		return S_OK;
	}


	//--------------------------------------------------------------------------------------
	// Name: Destroy()
	// Desc: Destroy the font object
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::Destroy()
	{
		m_pFontTexture       = NULL;
		m_dwNumGlyphs        = 0L;
		m_Glyphs             = NULL;
		m_cMaxGlyph          = 0;
		m_TranslatorTable    = NULL;
		m_dwNestedBeginCount = 0L;
		m_xprResource.Destroy();

		// Safely release shaders
		if( ( m_pFontVertexDecl != NULL ) && ( m_pFontVertexDecl->Release() == 0 ) )
			m_pFontVertexDecl = NULL;
		if( ( m_pFontVertexShader != NULL ) && ( m_pFontVertexShader->Release() == 0 ) )
			m_pFontVertexShader = NULL;
		if( ( m_pFontPixelShader != NULL ) && ( m_pFontPixelShader->Release() == 0 ) )
			m_pFontPixelShader = NULL;
	}


	//--------------------------------------------------------------------------------------
	// Name: SetCursorPosition()
	// Desc: Sets the cursor position for drawing text
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::SetCursorPosition( FLOAT fCursorX, FLOAT fCursorY )
	{
		m_fCursorX = floorf( fCursorX );
		m_fCursorY = floorf( fCursorY );
	}


	//--------------------------------------------------------------------------------------
	// Name: SetScaleFactors()
	// Desc: Sets X and Y scale factor to make rendered text bigger or smaller.
	//       Note that since text is pre-anti-aliased and therefore point-filtered,
	//       any scale factors besides 1.0f will degrade the quality.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::SetScaleFactors( FLOAT fXScaleFactor, FLOAT fYScaleFactor )
	{
		m_fXScaleFactor = fXScaleFactor;
		m_fYScaleFactor = fYScaleFactor;
	}


	//--------------------------------------------------------------------------------------
	// Name: SetSlantFactor()
	// Desc: Sets the slant factor for rendering slanted text.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::SetSlantFactor( FLOAT fSlantFactor )
	{
		m_fSlantFactor = fSlantFactor;
	}


	//-----------------------------------------------------------------------------
	// Name: SetDevice()
	// Desc: Re-assigns pointer to d3d device in case it has changed since creation
	//-----------------------------------------------------------------------------
	void CVinceFont::SetDevice( LPDIRECT3DDEVICE9 pd3dDevice )
	{
		m_pd3dDevice = pd3dDevice;
	}

	//--------------------------------------------------------------------------------------
	// Name: GetTextExtent()
	// Desc: Get the dimensions of a text string
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::GetTextExtent( const WCHAR* strText, FLOAT* pWidth, 
								FLOAT* pHeight, BOOL bFirstLineOnly ) const
	{
		//assert( strText != NULL );
		//assert( pWidth != NULL );
		//assert( pHeight != NULL );

		// Set default text extent in output parameters
		*pWidth   = 0.0f;
		*pHeight  = 0.0f;

		// Initialize counters that keep track of text extent
		FLOAT sx = 0.0f;
		FLOAT sy = m_fFontHeight;

		// Loop through each character and update text extent
		while( *strText )
		{
			WCHAR letter = *strText++;
	    
			// Handle newline character
			if( letter == L'\n' )
			{
				if( bFirstLineOnly )
					break;
				sx  = 0.0f;
				sy += m_fFontYAdvance;
			}

			// Handle carriage return characters by ignoring them. This helps when
			// displaying text from a file.
			if( letter == L'\r' )
				continue;

			// Translate unprintable characters
			VINCE_GLYPH_ATTR* pGlyph;
			if( letter > m_cMaxGlyph || m_TranslatorTable[letter] == 0 )
				pGlyph = &m_Glyphs[0];
			else
				pGlyph = &m_Glyphs[m_TranslatorTable[letter]];

			// Get text extent for this character's glyph
			sx += pGlyph->wOffset;
			sx += pGlyph->wAdvance;

			// Store text extent of string in output parameters
			if( sx > (*pWidth) )
				*pWidth = sx;
			if( sy > (*pHeight) )
				*pHeight = sy;
		}

		// Apply the scale factor to the result
		(*pWidth)  *= m_fXScaleFactor;
		(*pHeight) *= m_fYScaleFactor;
	}


	//--------------------------------------------------------------------------------------
	// Name: GetTextWidth()
	// Desc: Returns the width in pixels of a text string
	//--------------------------------------------------------------------------------------
	FLOAT CVinceFont::GetTextWidth( const WCHAR* strText ) const
	{
		FLOAT fTextWidth  = 0.0f;
		FLOAT fTextHeight = 0.0f;
		GetTextExtent( strText, &fTextWidth, &fTextHeight );
		return fTextWidth;
	}


	//--------------------------------------------------------------------------------------
	// Name: Begin()
	// Desc: Prepares the font vertex buffers for rendering.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::Begin()
	{
		// Set state on the first call
		if( 0 == m_dwNestedBeginCount )
		{
			// Set render state
			m_pd3dDevice->SetTexture( 0, m_pFontTexture );
			m_pd3dDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
			m_pd3dDevice->SetRenderState( D3DRS_SRCBLEND,         D3DBLEND_SRCALPHA );
			m_pd3dDevice->SetRenderState( D3DRS_DESTBLEND,        D3DBLEND_INVSRCALPHA );
			m_pd3dDevice->SetRenderState( D3DRS_ALPHATESTENABLE,  TRUE );
			m_pd3dDevice->SetRenderState( D3DRS_ALPHAREF,         0x08 );
			m_pd3dDevice->SetRenderState( D3DRS_ALPHAFUNC,        D3DCMP_GREATEREQUAL );
			m_pd3dDevice->SetRenderState( D3DRS_FILLMODE,         D3DFILL_SOLID );
			m_pd3dDevice->SetRenderState( D3DRS_CULLMODE,         D3DCULL_CCW );
			m_pd3dDevice->SetRenderState( D3DRS_ZENABLE,          FALSE );
			m_pd3dDevice->SetRenderState( D3DRS_STENCILENABLE,    FALSE );
			m_pd3dDevice->SetRenderState( D3DRS_VIEWPORTENABLE,   TRUE );
			m_pd3dDevice->SetSamplerState( 0, D3DSAMP_MINFILTER, D3DTEXF_LINEAR );
			m_pd3dDevice->SetSamplerState( 0, D3DSAMP_MAGFILTER, D3DTEXF_LINEAR );
			m_pd3dDevice->SetSamplerState( 0, D3DSAMP_ADDRESSU,  D3DTADDRESS_CLAMP );
			m_pd3dDevice->SetSamplerState( 0, D3DSAMP_ADDRESSV,  D3DTADDRESS_CLAMP );

			m_pd3dDevice->SetVertexDeclaration( m_pFontVertexDecl );
			m_pd3dDevice->SetVertexShader( m_pFontVertexShader );
			m_pd3dDevice->SetPixelShader( m_pFontPixelShader );

			D3DSURFACE_DESC desc;
			LPDIRECT3DSURFACE9 pRenderTarget;
			m_pd3dDevice->GetRenderTarget( 0, &pRenderTarget );
			pRenderTarget->GetDesc( &desc );
			pRenderTarget->Release();

			// Set the position scale factor as a vertex shader constant
			FLOAT vPosScale[4];
			vPosScale[0] = 2.0f/desc.Width;
			vPosScale[1] = 2.0f/desc.Height;
			vPosScale[2] = 0.0f;
			vPosScale[3] = 0.0f;
			m_pd3dDevice->SetVertexShaderConstantF( 0, vPosScale, 1 );

			// Set the texture scaling factor as a vertex shader constant
			D3DSURFACE_DESC TextureDesc;
			m_pFontTexture->GetLevelDesc( 0, &TextureDesc );
			FLOAT vTexScale[4];
			vTexScale[0] = 1.0f/TextureDesc.Width;
			vTexScale[1] = 1.0f/TextureDesc.Height;
			vTexScale[2] = 0.0f;
			vTexScale[3] = 0.0f;
			m_pd3dDevice->SetVertexShaderConstantF( 2, vTexScale, 1 );
		}

		// Keep track of the nested begin/end calls.
		m_dwNestedBeginCount++;
	}


	//--------------------------------------------------------------------------------------
	// Name: DrawText()
	// Desc: Draws text as textured polygons
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::DrawText( DWORD dwColor, const WCHAR* strText, DWORD dwFlags,
							FLOAT fMaxPixelWidth )
	{
		DrawText( m_fCursorX, m_fCursorY, dwColor, strText, dwFlags, fMaxPixelWidth );
	}


	//--------------------------------------------------------------------------------------
	// Name: DrawText()
	// Desc: Draws text as textured polygons
	//       TODO: This function should use the Begin/SetVertexData/End() API when it
	//       becomes available.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::DrawText( FLOAT fOriginX, FLOAT fOriginY, DWORD dwColor,
							const WCHAR* strText, DWORD dwFlags,
							FLOAT fMaxPixelWidth )
	{
		// Set up stuff to prepare for drawing text
		Begin();

		// Set the color as a vertex shader constant
		FLOAT vColor[4];
		vColor[0] = ((dwColor&0x00ff0000)>>16L)/255.0f;
		vColor[1] = ((dwColor&0x0000ff00)>> 8L)/255.0f;
		vColor[2] = ((dwColor&0x000000ff)>> 0L)/255.0f;
		vColor[3] = ((dwColor&0xff000000)>>24L)/255.0f;
		m_pd3dDevice->SetVertexShaderConstantF( 1, vColor, 1 );

		// Set the starting screen position
		if( ( dwFlags & FONT_RIGHT ) && ( fOriginX < 0.0f ) )
		{
			D3DSURFACE_DESC desc;
			LPDIRECT3DSURFACE9 pRenderTarget;
			m_pd3dDevice->GetRenderTarget( 0, &pRenderTarget );
			pRenderTarget->GetDesc( &desc );
			pRenderTarget->Release();

			fOriginX += desc.Width;
		}

		m_fCursorX = floorf( fOriginX );
		m_fCursorY = floorf( fOriginY );

		// Adjust for padding
		fOriginY -= m_fFontTopPadding;

		FLOAT fEllipsesPixelWidth = m_fXScaleFactor * 3.0f * (m_Glyphs[m_TranslatorTable[L'.']].wOffset + m_Glyphs[m_TranslatorTable[L'.']].wAdvance);

		if( dwFlags & FONT_TRUNCATED )
		{
			// Check if we will really need to truncate the string
			if( fMaxPixelWidth <= 0.0f )
			{
				dwFlags &= (~FONT_TRUNCATED);
			}
			else
			{
				FLOAT w, h;
				GetTextExtent( strText, &w, &h, TRUE );
	    
				// If not, then clear the flag
				if( w <= fMaxPixelWidth )
					dwFlags &= (~FONT_TRUNCATED);
			}
		}

		// If vertically centered, offset the starting m_fCursorY value
		if( dwFlags & FONT_CENTER_Y )
		{
			FLOAT w, h;
			GetTextExtent( strText, &w, &h );
			m_fCursorY = floorf( m_fCursorY - h/2 );
		}

		// Set a flag so we can determine initial justification effects
		BOOL bStartingNewLine = TRUE;

		while( *strText )
		{
			// If starting text on a new line, determine justification effects
			if( bStartingNewLine )
			{
				if( dwFlags & (FONT_RIGHT|FONT_CENTER_X) )
				{
					// Get the extent of this line
					FLOAT w, h;
					GetTextExtent( strText, &w, &h, TRUE );

					// Offset this line's starting m_fCursorX value
					if( dwFlags & FONT_RIGHT )
						m_fCursorX = floorf( fOriginX - w );
					if( dwFlags & FONT_CENTER_X )
						m_fCursorX = floorf( fOriginX - w/2 );
				}
				bStartingNewLine = FALSE;
			}

			// Get the current letter in the string
			WCHAR letter = *strText++;

			// Handle the newline character
			if( letter == L'\n' )
			{
				m_fCursorX  = fOriginX;
				m_fCursorY += m_fFontYAdvance * m_fYScaleFactor;
				bStartingNewLine = TRUE;
				continue;
			}

			// Handle carriage return characters by ignoring them. This helps when
			// displaying text from a file.
			if( letter == L'\r' )
				continue;

			// Translate unprintable characters
			VINCE_GLYPH_ATTR* pGlyph = &m_Glyphs[ (letter<=m_cMaxGlyph) ? m_TranslatorTable[letter] : 0 ];

			FLOAT fOffset  = m_fXScaleFactor * (FLOAT)pGlyph->wOffset;
			FLOAT fAdvance = m_fXScaleFactor * (FLOAT)pGlyph->wAdvance;
			FLOAT fWidth   = m_fXScaleFactor * (FLOAT)pGlyph->wWidth;
			FLOAT fHeight  = m_fYScaleFactor * m_fFontHeight;

			if( dwFlags & FONT_TRUNCATED )
			{
				// Check if we will be exceeded the max allowed width
				if( m_fCursorX + fOffset + fWidth + fEllipsesPixelWidth + m_fSlantFactor > fOriginX + fMaxPixelWidth )
				{
					// Yup. Let's draw the ellipses, then bail
					DrawText( m_fCursorX, m_fCursorY, dwColor, L"..." );
					End();
					return;
				}
			}

			// Select the mask
			FLOAT pShaderConsts0[4] = { pGlyph->wMask&0x0f00?1.0f:0.0f,  pGlyph->wMask&0x00f0?1.0f:0.0f,  pGlyph->wMask&0x000f?1.0f:0.0f,  pGlyph->wMask&0xf000?1.0f:0.0f };
			FLOAT pShaderConsts1[4] = { pGlyph->wMask==0x0000?1.0f:0.0f, pGlyph->wMask==0x0000?1.0f:0.0f, pGlyph->wMask==0x0000?1.0f:0.0f, pGlyph->wMask==0x0000?1.0f:0.0f };
			m_pd3dDevice->SetPixelShaderConstantF( 0, pShaderConsts0, 1 );
			m_pd3dDevice->SetPixelShaderConstantF( 1, pShaderConsts1, 1 );

			// Setup the screen coordinates
			m_fCursorX  += fOffset;
			FLOAT left1  = m_fCursorX;
			FLOAT left2  = left1 + m_fSlantFactor;
			FLOAT right1 = left1 + fWidth;
			FLOAT right2 = left2 + fWidth;
			FLOAT top    = m_fCursorY;
			FLOAT bottom = top + fHeight;
			m_fCursorX  += fAdvance;

			FLOAT v[4][4] = 
			{
				{ left2,  top,    pGlyph->tu1, pGlyph->tv1 },
				{ right2, top,    pGlyph->tu2, pGlyph->tv1 },
				{ right1, bottom, pGlyph->tu2, pGlyph->tv2 },
				{ left1,  bottom, pGlyph->tu1, pGlyph->tv2 },
			};
			m_pd3dDevice->DrawPrimitiveUP( D3DPT_QUADLIST, 1, v, sizeof(v[0]) );
		}

		// Call End() to complete the begin/end pair for drawing text
		End();
	}


	//--------------------------------------------------------------------------------------
	// Name: DrawBox()
	// Desc: We implement the DrawBox routine as part of the font class to avoid having to
	//       create font and pixel shaders for just this simple function.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::DrawBox(float x1, float y1, float x2, float y2, DWORD dwOutlineColor, DWORD dwFillColor)
	{

		// Set up stuff to prepare for drawing box
		Begin();

		// Set up the box coordinates
		XMFLOAT4 v[5];
		v[0] = XMFLOAT4( x1-0.5f, y1-0.5f, 0, 1 );
		v[1] = XMFLOAT4( x2-0.5f, y1-0.5f, 0, 1 );
		v[2] = XMFLOAT4( x2-0.5f, y2-0.5f, 0, 1 );
		v[3] = XMFLOAT4( x1-0.5f, y2-0.5f, 0, 1 );
		v[4] = XMFLOAT4( x1-0.5f, y1-0.5f, 0, 1 );

		// Set the colors as vertex shader constants
		FLOAT vOutlineColor[4];
		vOutlineColor[0] = ((dwOutlineColor&0x00ff0000)>>16L)/255.0f;
		vOutlineColor[1] = ((dwOutlineColor&0x0000ff00)>> 8L)/255.0f;
		vOutlineColor[2] = ((dwOutlineColor&0x000000ff)>> 0L)/255.0f;
		vOutlineColor[3] = ((dwOutlineColor&0xff000000)>>24L)/255.0f;

		FLOAT vFillColor[4];
		vFillColor[0] = ((dwFillColor&0x00ff0000)>>16L)/255.0f;
		vFillColor[1] = ((dwFillColor&0x0000ff00)>> 8L)/255.0f;
		vFillColor[2] = ((dwFillColor&0x000000ff)>> 0L)/255.0f;
		vFillColor[3] = ((dwFillColor&0xff000000)>>24L)/255.0f;

		// Set Pixel shader and renderstate
		//m_pd3dDevice->SetPixelShader( m_pConstantColorPS );
		m_pd3dDevice->SetRenderState( D3DRS_ALPHABLENDENABLE, TRUE );
		m_pd3dDevice->SetRenderState( D3DRS_SRCBLEND,  D3DBLEND_SRCALPHA );
		m_pd3dDevice->SetRenderState( D3DRS_DESTBLEND, D3DBLEND_INVSRCALPHA );

		// Set Vertex shader and declaration
		//m_pd3dDevice->SetVertexShader( m_pScreenspaceVS );
		//m_pd3dDevice->SetVertexDeclaration( m_pScreenspaceDecl );
		m_pd3dDevice->SetRenderState( D3DRS_VIEWPORTENABLE, FALSE );

		// Render the box
		m_pd3dDevice->SetPixelShaderConstantF( 0, (FLOAT*)&vFillColor, 1 );
		m_pd3dDevice->DrawPrimitiveUP( D3DPT_QUADLIST, 1, v, sizeof(v[0]) );

		// Render the lines
		m_pd3dDevice->SetPixelShaderConstantF( 0, (FLOAT*)&vOutlineColor, 1 );
		m_pd3dDevice->DrawPrimitiveUP( D3DPT_LINESTRIP, 4, v, sizeof(v[0]) );

		m_pd3dDevice->SetRenderState( D3DRS_VIEWPORTENABLE, TRUE );

		// Call End() to complete the begin/end pair for drawing box
		End();
	}

	//--------------------------------------------------------------------------------------
	// Name: End()
	// Desc: Paired call that restores state set in the Begin() call.
	//--------------------------------------------------------------------------------------
	VOID CVinceFont::End()
	{
		//assert( m_dwNestedBeginCount > 0 );
		if( --m_dwNestedBeginCount > 0 )
			return;
	}


	//--------------------------------------------------------------------------------------
	// Name: CreateTexture()
	// Desc: Creates a texture and renders a text string into it.
	//--------------------------------------------------------------------------------------
	LPDIRECT3DTEXTURE9 CVinceFont::CreateTexture( const WCHAR* strText, D3DCOLOR dwBackgroundColor, 
												D3DCOLOR dwTextColor, D3DFORMAT d3dFormat )
	{
		// Make sure the format is tiled (otherwise the Resolve will fail)
		if( FALSE == XGIsTiledFormat( d3dFormat ) )
		{
			//AtgUtil_PrintError( "Format must be tiled!\n" );
			return NULL;
		}

		// Calculate texture dimensions
		FLOAT fTexWidth;
		FLOAT fTexHeight;
		GetTextExtent( strText, &fTexWidth, &fTexHeight );
		DWORD dwWidth  = (DWORD)fTexWidth;
		DWORD dwHeight = (DWORD)fTexHeight;

		// Create a render target
		LPDIRECT3DSURFACE9 pNewRenderTarget;
		if( FAILED( m_pd3dDevice->CreateRenderTarget( dwWidth, dwHeight, d3dFormat, D3DMULTISAMPLE_NONE,
													0L, FALSE, &pNewRenderTarget, NULL ) ) )
		{
			//AtgUtil_PrintError( "Could not create a render target for the font texture!\n" );
			return NULL;
		}
	    
		// Create the texture
		LPDIRECT3DTEXTURE9 pNewTexture;
		if( FAILED( m_pd3dDevice->CreateTexture( dwWidth, dwHeight, 1, 0L, d3dFormat, 
												D3DPOOL_DEFAULT, &pNewTexture, NULL ) ) )
		{
			//AtgUtil_PrintError( "Could not create a font texture!\n" );
			pNewRenderTarget->Release();
			return NULL;
		}
	    
		// Get the current backbuffer and zbuffer
		LPDIRECT3DSURFACE9 pOldRenderTarget;
		m_pd3dDevice->GetRenderTarget( 0, &pOldRenderTarget );

		// Set the new texture as the render target
		m_pd3dDevice->SetRenderTarget( 0, pNewRenderTarget );
		m_pd3dDevice->Clear( 0L, NULL, D3DCLEAR_TARGET, dwBackgroundColor, 1.0f, 0L );

		// Render the text
		DrawText( 0, 0, dwTextColor, strText, 0L );

		// Resolve to the texture
		m_pd3dDevice->Resolve( 0L, NULL, pNewTexture,  NULL, 0, 0, NULL, 0, 0, NULL );

		// Restore the render target
		m_pd3dDevice->SetRenderTarget( 0, pOldRenderTarget );
		pOldRenderTarget->Release();
		pNewRenderTarget->Release();

		// Return the new texture
		return pNewTexture;
	}

	
}	// namespace

#endif // !NO_VINCE_SURVEYS
#endif // _VINCE_

