/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    driver.c

Abstract:

    This module implements the basic WDF entry routines for the GPIO class
    extension driver.


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"

#if defined(EVENT_TRACING)
#include "driver.tmh"               // auto-generated by WPP
#endif

//
// ----------------------------------------------------------------- Prototypes
//

DRIVER_INITIALIZE DriverEntry;

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(INIT, DriverEntry)
#pragma alloc_text(PAGE, GpioClxEvtDriverUnload)

//
// ------------------------------------------------------------------ Functions
//

NTSTATUS
DriverEntry (
    __in PDRIVER_OBJECT  DriverObject,
    __in PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine is the installable driver initialization entry point. This
    routine will create the driver object and register the GPIO class extension
    as a class library with WDF. This will allow GPIO client driver drivers to
    bind with the GPIO class extension.

Arguments:

    DriverObject - Supplies a pointer to the driver object created by the
        I/O manager.

    RegistryPath - Supplies a pointer to the driver specific registry key.

Return Value:

    NTSTATUS code.

--*/

{

    WDF_DRIVER_CONFIG DriverConfig;
    NTSTATUS Status;
    WDFDRIVER WdfDriver;
    BOOLEAN WdfDriverCreated;

#if defined(EVENT_TRACING)

    RECORDER_CONFIGURE_PARAMS RecorderConfigureParams;
    RECORDER_LOG_CREATE_PARAMS RecorderParams;

#endif

    //
    // Register with ETW.
    //

    EventRegisterMicrosoft_Windows_GPIO_ClassExtension();

    //
    // Initialize WPP tracing.
    //

    WdfDriverCreated = FALSE;
    WPP_INIT_TRACING(DriverObject, RegistryPath);

#if defined(EVENT_TRACING)

    //
    // Create an "in-flight recorder log"  If there is an issue, the tokens that
    // are recorded in the log will be in the minidump file and can help
    // track the issue.
    //
    // Processing of those tokens back into text depends on having the
    // "traceformat" files, which are generated during the build, and also
    // derivable after the fact from the .pdb file. Dumping of this log depends
    // on a debugger extension, rcdrkd.dll. The trace entries that go into this
    // log can also be consumed through any WPP tracing consumer, like
    // tracelog.exe or traceview.exe.
    //
    //
    // Before that, disable the default log created by WPP as that log is
    // not available until driver object is created.
    //

    RECORDER_CONFIGURE_PARAMS_INIT(&RecorderConfigureParams);
    RecorderConfigureParams.CreateDefaultLog = FALSE;
    WppRecorderConfigure(&RecorderConfigureParams);

    RECORDER_LOG_CREATE_PARAMS_INIT(&RecorderParams, NULL);
    RecorderParams.TotalBufferSize = GpioLogBufferSize;
    RtlStringCbPrintfA(RecorderParams.LogIdentifier,
                       RECORDER_LOG_IDENTIFIER_MAX_CHARS,
                       "GPIO_Default");

    Status = WppRecorderLogCreate(&RecorderParams, &GpioLogHandle);
    if (!NT_SUCCESS(Status)) {
        goto DriverEntryEnd;
    }

#endif

    //
    // Perform Phase 0 GPIO class extension initialization.
    //

    Status = GpiopInitialize(0, DriverObject, NULL, RegistryPath);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "DriverEntry: GPIO initialization failed! Status = %#x\n",
                    Status);

        goto DriverEntryEnd;
    }

    //
    // Initialize the configuration object with necessary callbacks. In this
    // case, the only the unload callback is set to allow cleanup of the
    // class library object and any other global data.
    //
    // The driver object is marked as non-PNP to get GPIO class extension to
    // be unloaded correctly.
    //

    WDF_DRIVER_CONFIG_INIT(&DriverConfig, NULL);
    DriverConfig.DriverInitFlags = WdfDriverInitNonPnpDriver;
    DriverConfig.EvtDriverUnload = GpioClxEvtDriverUnload;

    //
    // Create the WDF driver object for the GPIO class extension.
    //

    Status = WdfDriverCreate(DriverObject,
                             RegistryPath,
                             WDF_NO_OBJECT_ATTRIBUTES,
                             &DriverConfig,
                             &WdfDriver);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "DriverEntry: WdfDriverCreate() failed! Status = %#x\n",
                    Status);

        goto DriverEntryEnd;
    }

    WdfDriverCreated = TRUE;

    //
    // Perform Phase 1 GPIO class extension initialization.
    //

    Status = GpiopInitialize(1, DriverObject, WdfDriver, RegistryPath);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "DriverEntry: GPIO initialization failed! Status = %#x\n",
                    Status);

        goto DriverEntryEnd;
    }

    //
    // Create the WDF class library object for the class extension.
    //
    // N.B. Per WDF team, it is a requirement that class extension registration
    //      to be done as a last step (when it is guaranteed that nothing else
    //      can fail.)
    //

    Status = GpiopClassLibraryCreate(DriverObject, RegistryPath);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "DriverEntry: Failed to create class library object! "
                    "Status = %#x\n",
                    Status);

        goto DriverEntryEnd;
    }

    //
    // If the driver object was created, then clean up will be done in the
    // unload routine. Otherwise, things need to be cleaned up here.
    //

DriverEntryEnd:
    if ((!NT_SUCCESS(Status)) && (WdfDriverCreated == FALSE)) {
        GpiopUninitialize(DriverObject);

#if defined(EVENT_TRACING)

        //
        // Delete the default IFR log.
        //

        if (GpioLogHandle != NULL) {
            WppRecorderLogDelete(GpioLogHandle);
            GpioLogHandle = NULL;
        }

#endif

        WPP_CLEANUP(DriverObject);
        EventUnregisterMicrosoft_Windows_GPIO_ClassExtension();
    }

    return Status;
}

VOID
GpioClxEvtDriverUnload (
    __in WDFDRIVER Driver
    )

/*++

Routine Description:

    This routine is called by WDF to allow final cleanup prior to unloading
    the GPIO class extension. This routine destroys the class library and stops
    tracing.

Arguments:

    Driver - Supplies a handle to a framework driver object.

Return Value:

    None.

--*/

{

    PDRIVER_OBJECT DriverObject;

    PAGED_CODE();

    DriverObject = WdfDriverWdmGetDriverObject(Driver);

    //
    // Destroy the class library.
    //

    GpiopUninitialize(DriverObject);
    GpiopClassLibraryDestroy(DriverObject);
    TraceEvents(GpioLogHandle,
                Info,
                DBG_INIT,
                "GpioClxEvtDriverUnload: Driver unloaded!\n");

#if defined(EVENT_TRACING)

    //
    // Delete the default IFR log.
    //

    if (GpioLogHandle != NULL) {
        WppRecorderLogDelete(GpioLogHandle);
        GpioLogHandle = NULL;
    }

#endif

    WPP_CLEANUP(DriverObject);
    EventUnregisterMicrosoft_Windows_GPIO_ClassExtension();
    return;
}


