/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    isrdpc.c

Abstract:

    This module contains routines for enabling and disabling of interrupts as
    well as handling of interrupts generated by Express bridge.









--*/

// -------------------------------------------------------------------- Includes

#include "pcip.h"
#include "isrdpc.tmh"

// --------------------------------------------------------------------- Defines
// ----------------------------------------------------------------------- Types

typedef struct _PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT {

    __deref_volatile PLONG DelegatedWorkItems;
    PPCI_HOTPLUG_CONTROLLER HotPlugController;
    PCI_HOTPLUG_INTERRUPTS HotPlugInterrupts;

} PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT, *PPCI_INTERRUPT_SYNCHRONIZATION_CONTEXT;

typedef struct _PCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT {

    PEXPRESS_BRIDGE ExpressBridge;
    PPCI_HOTPLUG_CONTROLLER HotPlugController;
    __deref_volatile PLONG DelegatedWorkItems;

} PCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT, *PPCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT;

// ------------------------------------------------------------------ Prototypes

__drv_functionClass(KMESSAGE_SERVICE_ROUTINE)
_IRQL_requires_same_
KMESSAGE_SERVICE_ROUTINE ExpressRootPortMessageRoutine;

_Function_class_(KSERVICE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
KSERVICE_ROUTINE ExpressRootPortInterruptRoutine;

VOID
ExpressRootPortAerInterruptRoutine (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __out PBOOLEAN AssertInterrupt,
    __out PBOOLEAN QueueDpc
    );

VOID
ExpressRootPortPmeInterruptRoutine (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __out PBOOLEAN AssertInterrupt,
    __out PBOOLEAN QueueDpc
    );

VOID
ExpressRootPortPmeEventWorker (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortEnableAerInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortEnablePmeInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortEnableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortDisableAerInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortDisablePmeInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressRootPortDisableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

__drv_functionClass(KSERVICE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL)
_IRQL_requires_same_
KSERVICE_ROUTINE ExpressDownstreamSwitchPortInterruptRoutine;

VOID
ExpressDownstreamSwitchPortEnableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressDownstreamSwitchPortDisableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    );

NTSTATUS
ExpressBridgeConnectInterrupt (
    __in PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList,
    __in ULONG MessageNumber,
    __in PKSERVICE_ROUTINE InterruptRoutine,
    __out PKINTERRUPT *InterruptObject,
    __out PULONG InterruptConnectVersion
    );

NTSTATUS
ExpressBridgeConnectMessageInterrupt (
    __in PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList,
    __in PKSERVICE_ROUTINE FallbackInterruptRoutine,
    __in PKMESSAGE_SERVICE_ROUTINE MessageServiceRoutine,
    __in_opt PKSPIN_LOCK MessageRoutineSpinLock,
    __out PVOID *ConnectionContext,
    __out PULONG InterruptConnectVersion
    );

_Function_class_(KSYNCHRONIZE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
KSYNCHRONIZE_ROUTINE ExpressBridgeGetHotPlugInterrupts;

ULONG
ExpressBridgeGetHotPlugMessageNumber (
    __in PEXPRESS_BRIDGE ExpressBridge
    );

VOID
ExpressBridgeProcessAsyncHotPlugEvents (
    __in PEXPRESS_BRIDGE ExpressBridge
    );

_Function_class_(KSYNCHRONIZE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
KSYNCHRONIZE_ROUTINE ExpressBridgeSimulateHotPlugInterrupt;

VOID
ExpressBridgeClearSpuriousHotPlugStatus (
    __in PEXPRESS_BRIDGE ExpressBridge
    );

// --------------------------------------------------------------------- Pragmas

#ifdef ALLOC_PRAGMA
    #pragma alloc_text(PAGE, ExpressRootPortConnectInterrupt)
    #pragma alloc_text(PAGE, ExpressRootPortDisconnectInterrupt)
    #pragma alloc_text(PAGE, ExpressDownstreamSwitchPortConnectInterrupt)
    #pragma alloc_text(PAGE, ExpressDownstreamSwitchPortDisconnectInterrupt)
    #pragma alloc_text(PAGE, ExpressBridgeConnectInterrupt)
    #pragma alloc_text(PAGE, ExpressBridgeConnectMessageInterrupt)
    #pragma alloc_text(PAGE, ExpressBridgeGetHotPlugMessageNumber)
#endif

// --------------------------------------------------------------------- Globals
// ------------------------------------------------------------------- Functions
// ------------------------------------------------------ Routines for root port


NTSTATUS
ExpressRootPortConnectInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList
    )

/*++

Routine Description:

    This routine connects and registers an ISR for an interrupt resource
    assigned to a root port bridge.

Arguments:

    ExpressBridge - Supplies a pointer to an Express Bridge.

    RawList - Supplies a list of resources assigned to a bridge.

    TranslatedList - Supplies a list of resources assigned to a bridge
        in a translated form.

Return Value:

    Status code as returned from a function call in this routine.

--*/

{

    NTSTATUS Status;

    PAGED_CODE();

    //
    // Invoke a helper function to register a message service routine.
    //

    KeInitializeSpinLock(&ExpressBridge->Downstream.u.RootPort.MessageRoutineSpinLock);
    Status = ExpressBridgeConnectMessageInterrupt(
                ExpressBridge,
                RawList,
                TranslatedList,
                ExpressRootPortInterruptRoutine,
                ExpressRootPortMessageRoutine,
                &ExpressBridge->Downstream.u.RootPort.MessageRoutineSpinLock,
                &ExpressBridge->Downstream.u.RootPort.ConnectionContext,
                &ExpressBridge->Downstream.InterruptConnectVersion);

    if (NT_SUCCESS(Status)) {
        ExpressBridge->Downstream.InterruptConnected = TRUE;
    }

    return Status;
}

__drv_functionClass(KMESSAGE_SERVICE_ROUTINE)
_IRQL_requires_same_
BOOLEAN
ExpressRootPortMessageRoutine (
    _In_ struct _KINTERRUPT *Interrupt,
    _In_ PVOID ServiceContext,
    _In_ ULONG MessageID
    )

/*++

Routine Description:

    This is a message service routine registered for handling interrupts
    on a root port.

Arguments:

    Interrupt - Supplies a pointer to an interrupt object.

    ServiceContext - Supplies a pointer to an express bridge for a root port.

    MessageID - Supplies the message number that caused this invocation.

Return Value:

    TRUE if this interrupt belongs to the root port.

    FALSE otherwise.

--*/

{

    NON_PAGED_CODE();

    UNREFERENCED_PARAMETER(MessageID);

    return ExpressRootPortInterruptRoutine(Interrupt, ServiceContext);
}

_Function_class_(KSERVICE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
BOOLEAN
ExpressRootPortInterruptRoutine (
    _In_ struct _KINTERRUPT *Interrupt,
    _In_opt_ PVOID ServiceContext
    )

/*++

Routine Description:

    This is an interrupt service routine registered for handling interrupts
    on a root port. It invokes specific service routines to handle its
    interrupts.

Arguments:

    Interrupt - Supplies a pointer to an interrupt object.

    ServiceContext - Supplies a pointer to an express bridge for a root port.

Return Value:

    TRUE if this interrupt belongs to the root port.

    FALSE otherwise.

--*/

{

    BOOLEAN AssertAerInterrupt;
    BOOLEAN AssertHotPlugInterrupt;
    BOOLEAN AssertPmeInterrupt;
    PPCI_BUS Bus;
    PEXPRESS_BRIDGE ExpressBridge;
    BOOLEAN QueueDpcForAer;
    BOOLEAN QueueDpcForHotPlug;
    BOOLEAN QueueDpcForPme;

    NON_PAGED_CODE();

    UNREFERENCED_PARAMETER(Interrupt);

    ExpressBridge = (PEXPRESS_BRIDGE)ServiceContext;
    _Analysis_assume_(ExpressBridge != NULL);

    //
    // Invoke the AER interrupt service routine.
    //

    ExpressRootPortAerInterruptRoutine(ExpressBridge,
                                       &AssertAerInterrupt,
                                       &QueueDpcForAer);

    //
    // Invoke the PME interrupt service routine.
    //

    ExpressRootPortPmeInterruptRoutine(ExpressBridge,
                                       &AssertPmeInterrupt,
                                       &QueueDpcForPme);

    //
    // Invoke the HotPlug interrupt routine.
    //

    AssertHotPlugInterrupt = FALSE;
    QueueDpcForHotPlug = FALSE;
    if (ExpressBridge->Downstream.HotPlugCapable != FALSE) {
        ExpressBridgeHotPlugInterruptRoutine(
            ExpressBridge,
            ExpressBridge->Downstream.HotPlugController,
            &ExpressBridge->Downstream.DelegatedWorkItems,
            &AssertHotPlugInterrupt,
            &QueueDpcForHotPlug);
    }

    if ((QueueDpcForPme != FALSE) ||
        (QueueDpcForHotPlug != FALSE ||
         QueueDpcForAer != FALSE)) {

        Bus = ExpressBridge->PciBridge->Bus;
        IoRequestDpc(Bus->DeviceObject, NULL, ExpressBridge);
    }

    return (AssertPmeInterrupt || AssertHotPlugInterrupt || AssertAerInterrupt);
}

PPCI_DEVICE
ExpressRootPortFindErrorDevice (
    __in PPCI_BUS RootBus,
    __in ULONG BusNumber,
    __in PCI_SLOT_NUMBER Slot
    )

/*++

Routine Description:

    This routine scans the pcitree looking for the specified device.

Arguments:

    RootBus - Supplies the root bus of the hierarchy to search.

    BusNumber - Supplies the bus number of the device we're searching for.

    Slot - Supplies the PCI slot number of the device we're searching for.

Return Value:

    If successful, the address of a the specified device.

    If unsuccessful, NULL is returned.

--*/

{

    PPCI_DEVICE Device;
    PPCI_BUS Bus;

    Device = NULL;

    //
    // Find the bus FDO.  The root we search from is the root port on which an
    // error was reported.
    //

    Bus = RootBus;
    while (Bus != NULL) {
        if (Bus->SecondaryBusNumber == BusNumber) {
            break;
        }

        Bus = PciNextBusInSegment(Bus);
    }

    if (Bus == NULL) {
        goto end;
    }

    Device = Bus->ChildDevices;
    while (Device != NULL) {

        //
        // People probably don't clear the unused bits in the PCI_SLOT_NUMBER,
        // so ignore them in the main build but assert checked so we can get
        // them fixed.
        //

        if ((Device->Slot.u.bits.DeviceNumber == Slot.u.bits.DeviceNumber) &&
            (Device->Slot.u.bits.FunctionNumber == Slot.u.bits.FunctionNumber)) {

            PCI_ASSERT(Device->Slot.u.AsULONG == Slot.u.AsULONG);

            //
            // This is the device we're searching for.
            //

            break;
        }

        Device = Device->Sibling;
    }

    if (Device == NULL) {
        ERROR("PciFindErrorDevice: Could not find PDO for device @ %x.%x.%x\n",
              BusNumber,
              Slot.u.bits.DeviceNumber,
              Slot.u.bits.FunctionNumber);

    } else if ((Device->ExpressPort == NULL) ||
               (Device->ExpressPort->AerCapability == 0)) {

        ERROR("PciFindErrorDevice: PDO has no AER capability for device @ %x.%x.%x\n",
              BusNumber,
              Slot.u.bits.DeviceNumber,
              Slot.u.bits.FunctionNumber);

        Device = NULL;
    }

end:
    return Device;
}

VOID
ExpressRootPortInitWheaErrorPacket (
    __in PPCI_DEVICE Device,
    __inout PEXPRESS_ERROR_PACKET Packet,
    __in PPCI_EXPRESS_AER_CAPABILITY AerCaps,
    __in WHEA_ERROR_SEVERITY Severity,
    __in PPCI_EXPRESS_ERROR_SOURCE_ID ErrSourceId,
    __in ULONG ErrorSourceId
    )

/*++

Routine Description:

    This routine initilializes a WHEA_ERROR_PACKET using the information
    supplied in the supplied AER capabilities structure.

Arguments:

    Device - Supplies a pointer to the device that took the error.

    Packet - Supplies the address of a WHEA_ERROR_PACKET.

    AerCaps - Supplies the address of a PCI Express AER capabilities structure.

    Severity - Supplies the severity level of the error condition.

    ErrSourceId - Supplies a pointer to the PCI Express device identifier.

    ErrorSourceId - Supplies a value that uniquely identifies the PCI Express
        error source.

Return Value:

    None.

--*/

{

    ULONG AerInfoSize;
    USHORT Command;
    USHORT Status;

    UNREFERENCED_PARAMETER(ErrSourceId);

    //
    // Zero the packet.
    //

    RtlZeroMemory(Packet, sizeof(EXPRESS_ERROR_PACKET));

    //
    // Fill in packet header information.
    //

    Packet->Header.Signature = WHEA_ERROR_PACKET_SIGNATURE;
    Packet->Header.Version = WHEA_ERROR_PACKET_VERSION;
    Packet->Header.Length = sizeof(EXPRESS_ERROR_PACKET);
    Packet->Header.ErrorType = WheaErrTypePCIExpress;
    Packet->Header.ErrorSeverity = Severity;
    Packet->Header.ErrorSourceId = ErrorSourceId;
    Packet->Header.ErrorSourceType = WheaErrSrcTypePCIe;
    Packet->Header.NotifyType = PCIe_NOTIFY_TYPE_GUID;
    Packet->Header.DataFormat = WheaDataFormatPCIExpress;
    Packet->Header.DataOffset = FIELD_OFFSET(EXPRESS_ERROR_PACKET,
                                             PciExpressError);

    Packet->Header.DataLength = sizeof(WHEA_PCIEXPRESS_ERROR_SECTION);

    //
    // Fill in the Express error information.
    //

    Packet->PciExpressError.ValidBits.PortType = 1;
    Packet->PciExpressError.ValidBits.Version = 1;
    Packet->PciExpressError.ValidBits.CommandStatus = 1;
    Packet->PciExpressError.ValidBits.DeviceId = 1;
    Packet->PciExpressError.ValidBits.DeviceSerialNumber = 1;
    Packet->PciExpressError.ValidBits.ExpressCapability = 1;
    Packet->PciExpressError.ValidBits.AerInfo = 1;

    //
    // Fill in the device type and PCIe version implemented by the OS.
    //

    Packet->PciExpressError.PortType = Device->ExpressPort->DeviceType;
    Packet->PciExpressError.Version.AsULONG = 0x0101;

    //
    // Read the Command and Status information from the device.
    //

    PciReadConfigRegister(Device, Command, &Command);
    PciReadConfigRegister(Device, Status, &Status);
    Packet->PciExpressError.CommandStatus.AsULONG = (Command << 16) | Status;

    //
    // Fill in the device identifier.
    //

    Packet->PciExpressError.DeviceId.VendorID = Device->VendorID;
    Packet->PciExpressError.DeviceId.DeviceID = Device->DeviceID;
    Packet->PciExpressError.DeviceId.PrimaryBusNumber = Device->BusNumber;
    Packet->PciExpressError.DeviceId.ClassCode =
        (Device->BaseClass << 15) | (Device->SubClass << 8) | (Device->ProgIf);

    Packet->PciExpressError.DeviceId.FunctionNumber =
        Device->Slot.u.bits.FunctionNumber;

    Packet->PciExpressError.DeviceId.DeviceNumber =
        Device->Slot.u.bits.DeviceNumber;

    Packet->PciExpressError.DeviceId.Segment =
        Device->Root->Segment->SegmentNumber;

    if (Device->HotPlugSlot != NULL) {
        Packet->PciExpressError.DeviceId.SlotNumber =
            Device->HotPlugSlot->PhysicalSlotNumber;
    }

    if (Device->ExpressPort->DeviceType == PciExpressRootPort) {
        Packet->PciExpressError.DeviceId.SecondaryBusNumber =
            Device->BusNumber;
    }

    //
    // Fill in the device serial number.
    //

    Packet->PciExpressError.DeviceSerialNumber =
        Device->ExpressPort->SerialNumber;

    //
    // Read the device's Express Capability information into the error packet.
    // The amount of information retrieved depends on the device type. For
    // endpoints, we read through the link capabilities.  For root ports, we
    // read the entire express capability.  For all other device types, we just
    // read the information up to the link capabilities.
    //

    if (Device->ExpressPort->DeviceType == PciExpressEndpoint) {
        PciReadDeviceConfig(Device,
                            &Packet->PciExpressError.ExpressCapability[0],
                            Device->ExpressCapability,
                            FIELD_OFFSET(PCI_EXPRESS_CAPABILITY,
                                         SlotCapabilities));

    } else if (Device->ExpressPort->DeviceType == PciExpressRootPort) {
        PciReadDeviceConfig(Device,
                            &Packet->PciExpressError.ExpressCapability[0],
                            Device->ExpressCapability,
                            sizeof(PCI_EXPRESS_CAPABILITY));

    } else {
        PciReadDeviceConfig(Device,
                            &Packet->PciExpressError.ExpressCapability[0],
                            Device->ExpressCapability,
                            FIELD_OFFSET(PCI_EXPRESS_CAPABILITY,
                                         LinkCapabilities));
    }

    //
    // Copy the entire AER capability structure into the packet.
    //

    AerInfoSize = min(RTL_FIELD_SIZE(WHEA_PCIEXPRESS_ERROR_SECTION, AerInfo),
                      sizeof(PCI_EXPRESS_AER_CAPABILITY));

    RtlCopyMemory(&Packet->PciExpressError.AerInfo[0], AerCaps, AerInfoSize);
    return;
}

VOID
ExpressRootPortAerInterruptRoutine (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __out PBOOLEAN AssertInterrupt,
    __out PBOOLEAN QueueDpc
    )

/*++

Routine Description:

    This is an interrupt service handler for AER events on a root port. This
    routine examines the root status register to assert a AER event and saves
    the AER requestor Id in the root port extension for any AER related work
    delegated to the DPC or work item. If the root port triggered a AER event
    then this routine will disable the AER interrupt until this event is
    handled to completion subject to other conditions being favorable - like
    the root port is not stopped/removed or powered down.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

    AssertInterrupt - Supplies a place holder to assert if this interrupt was
        generated at the root port because of a PME event.

    QueueDpc - Supplies a place holder to indicate a need for queuing a DPC.

Return Value:

    None.

--*/

{

    PEXPRESS_ERROR_PACKET AerPacket;
    USHORT Bus;
    PPCI_DEVICE Device;
    PPCI_EXPRESS_AER_CAPABILITY DeviceAerCaps;
    ULONG ErrorSeverity;
    PPCI_EXPRESS_ERROR_SOURCE_ID ErrSourceId;
    PPCI_EXPRESS_ROOT_ERROR_COMMAND RootErrCommand;
    PPCI_EXPRESS_ROOT_ERROR_STATUS RootErrStatus;
    PCI_EXPRESS_ROOT_ERROR_STATUS RootStatus;
    PCI_SLOT_NUMBER Slot;

    *QueueDpc = FALSE;
    *AssertInterrupt = FALSE;

    //
    // In order to read slot config info on the bridge, the parent bus
    // must support extended config access.
    //

    if (ExpressBridge->PciBridge->Bus->ParentBus->ExtendedConfigAvailable
            != TRUE) {

        return;
    }

    //
    // If this bridge does not support AER, we can terminate here.
    //

    if (ExpressBridge->AerCapability == 0 ||
        !PciIsAerGranted(ExpressBridge->PciBridge->Bus->Root)) {

        return;
    }

    //
    // Read the root port's AER extended capability structure.
    //

    RootErrCommand = &ExpressBridge->Downstream.u.RootPort.RootErrorCommand;
    PciReadExpressRootPortAerRegister(ExpressBridge,
                                      RootErrorCommand,
                                      RootErrCommand);

    RootErrStatus = &ExpressBridge->Downstream.u.RootPort.RootErrorStatus;
    PciReadExpressRootPortAerRegister(ExpressBridge,
                                      RootErrorStatus,
                                      RootErrStatus);

    //
    // If the root port is not setup to interrupt when it receives an error
    // message or if the RootErrorStatus register is zero, no errors have been
    // signaled.
    //

    if (((RootErrCommand->AsULONG &
          ROOT_CMD_ERROR_REPORTING_ENABLE_MASK) == 0) ||
        ((RootErrStatus->AsULONG & 0x07ffffff) == 0)) {

        return;
    }

    TRACE(AER_TRACE_LEVEL, "ExpressBridge %p RootErrorCommand %x "
          "RootErrorStatus %x\n",
          ExpressBridge,
          RootErrCommand->AsULONG,
          RootErrStatus->AsULONG);

    //
    // Signal the caller that an AER root port interrupt was present.
    //

    *AssertInterrupt = TRUE;

    //
    // Save a local copy of the RootErrorStatus.
    //

    RootStatus = *RootErrStatus;

    //
    // Read the error source identifier from the root port's AER capabilities.
    //

    ErrSourceId = &ExpressBridge->Downstream.u.RootPort.ErrorSourceId;
    PciReadExpressRootPortAerRegister(ExpressBridge,
                                      ErrorSourceId,
                                      ErrSourceId);

    TRACE(AER_TRACE_LEVEL, "ExpressBridge %p ErrorSourceId %x\n",
          ExpressBridge,
          ErrSourceId->AsULONG);

    //
    // Depending on what kind of error(s) has been reported, set the severity
    // level and extract the information from the root port's AER capabilities
    // structure that identifies the device that sent the error message.
    //

    Slot.u.AsULONG = 0;
    if (RootStatus.MultipleUncorrectableErrorsReceived != 0 ||
        RootStatus.UncorrectableErrorReceived != 0) {

        //
        // One or more uncorrectable errors have been reported.  If the fatal
        // error messages received bit is set, we will consider the error fatal
        // even though the device address logged may be that associated with a
        // non-fatal condition.  Note that priority is given to uncorrected
        // errors, so if both uncorrected and corrected errors are reported,
        // we will only report the uncorrected conditon.
        //
        //

        if (RootStatus.FatalErrorMessagesReceived != 0) {
            ErrorSeverity = WheaErrSevFatal;

        } else {
            ErrorSeverity = WheaErrSevRecoverable;
        }

        //
        // Extract the device address.
        //

        Bus = ErrSourceId->UncorrectableSourceIdBus;
        Slot.u.bits.DeviceNumber = ErrSourceId->UncorrectableSourceIdDev;
        Slot.u.bits.FunctionNumber = ErrSourceId->UncorrectableSourceIdFun;
        TRACE(AER_TRACE_LEVEL, "Uncorrectable error\n");

    } else if (RootStatus.MultipleCorrectableErrorsReceived != 0 ||
               RootStatus.CorrectableErrorReceived != 0) {

        //
        // If one or more correctable errors has been signaled and no
        // uncorrectable errors have been signaled, we set the severity level
        // to indicate that this is a corrected error condition.
        //

        ErrorSeverity = WheaErrSevCorrected;

        //
        // Extract the device address.
        //

        Bus = ErrSourceId->CorrectableSourceIdBus;
        Slot.u.bits.DeviceNumber = ErrSourceId->CorrectableSourceIdDev;
        Slot.u.bits.FunctionNumber = ErrSourceId->CorrectableSourceIdFun;
        TRACE(AER_TRACE_LEVEL, "Correctable error\n");

    } else {
        ERROR("ExpressRootPortAerInterruptRoutine: Invalid error severity!\n");
        return;
    }

    //
    // Find the device that sent the error message.
    //

    TRACE(AER_TRACE_LEVEL, "Device Bus %x Dev %x Fun %x\n",
          Bus,
          Slot.u.bits.DeviceNumber,
          Slot.u.bits.FunctionNumber);

    DeviceAerCaps = NULL;
    Device = ExpressRootPortFindErrorDevice(
                 ExpressBridge->PciBridge->Bus->ParentBus,
                 Bus,
                 Slot);

    if (Device == NULL) {
        ERROR("ExpressRootPortAerInterruptRoutine: Unable to locate device that signaled error!\n");
        goto ExitIsr;
    }

    //
    // Read the AER capabilities information from the device that signaled the
    // error condition.
    //

    DeviceAerCaps = &Device->ExpressPort->AerCapabilityBlock;
    RtlZeroMemory(DeviceAerCaps, sizeof(PCI_EXPRESS_AER_CAPABILITY));
    switch (Device->ExpressPort->DeviceType) {
        case PciExpressEndpoint:                        // 0
        case PciExpressRootPort:                        // 4
        case PciExpressLegacyEndpoint:                  // 1
        case PciExpressUpstreamSwitchPort:              // 5
        case PciExpressDownstreamSwitchPort:            // 6
        case PciXToExpressBridge:                       // 8
        case PciExpressRootComplexIntegratedEndpoint:   // 9
        case PciExpressRootComplexEventCollector:       // 10
            if (Device->ExpressPort->AerCapability != 0) {
                PciReadExpressAerCapabilities(Device->ExpressPort,
                                              DeviceAerCaps);
            }

            break;

        case PciExpressToPciXBridge:                    // 7
            if (Device->ExpressPort->AerCapability != 0) {
                PciReadExpressBridgeAerCapabilities(Device->ExpressPort,
                                                    DeviceAerCaps);
            }

            break;

        default:
            break;
    }

    //
    // Take a look at the error and decide if the severity needs to be adjusted
    // based on the who reported the error and what the condition was.
    // Strictly speaking, this should not be done here, but until the internal
    // WHEA error handling flow is more AER aware, we need to be careful what
    // types of errors we report as uncorrected and/or fatal.
    //

    if (Device->ExpressPort->AerCapability != 0 &&
        ErrorSeverity == WheaErrSevRecoverable) {

        if (DeviceAerCaps->UncorrectableErrorStatus.AsULONG == 0x00004000) {
            ErrorSeverity = WheaErrSevCorrected;
        }
    }

    //
    // Report this PCI Express error to WHEA.  Note that WHEA may bugcheck the
    // system.  For now, if WHEA is not initialized and ready to receive error
    // notifcations, we won't do anything with this error.  In the near future
    // I'll change this so we'll hold on to the error data until WHEA wakes up
    // and then we'll forward it on.
    //

    if (ExpressBridge->WheaInitialized == 1) {
        AerPacket =
            (PEXPRESS_ERROR_PACKET)&Device->ExpressPort->WheaErrorPacket;

        if (AerPacket != NULL) {
            AerPacket->PciExpressError.PortType =
                Device->ExpressPort->DeviceType;

            ExpressRootPortInitWheaErrorPacket(
                Device,
                AerPacket,
                DeviceAerCaps,
                ErrorSeverity,
                ErrSourceId,
                ExpressBridge->ErrorSource.ErrorSourceId);

            PshedRetrieveErrorInfo(
                (PWHEA_ERROR_PACKET)AerPacket,
                WheaGetErrorSource(AerPacket->Header.ErrorSourceId));

            WheaReportHwError((PWHEA_ERROR_PACKET)AerPacket);
        }
    }

ExitIsr:

    RootErrStatus->AsULONG = 0x7fffffff;
    PciWriteExpressRootPortAerRegister(ExpressBridge,
                                       RootErrorStatus,
                                       RootErrStatus);

    PciReadExpressRootPortAerRegister(ExpressBridge,
                                      RootErrorStatus,
                                      RootErrStatus);

    if (Device != NULL &&  Device->ExpressPort->AerCapability != 0) {
        DeviceAerCaps->UncorrectableErrorStatus.AsULONG = (ULONG)-1;
        PciWriteExpressAerRegister(Device->ExpressPort,
                                   UncorrectableErrorStatus,
                                   &DeviceAerCaps->UncorrectableErrorStatus);
        DeviceAerCaps->UncorrectableErrorStatus.AsULONG = 0;

        DeviceAerCaps->CorrectableErrorStatus.AsULONG = (ULONG)-1;
        PciWriteExpressAerRegister(Device->ExpressPort,
                                   CorrectableErrorStatus,
                                   &DeviceAerCaps->CorrectableErrorStatus);
        DeviceAerCaps->CorrectableErrorStatus.AsULONG = 0;
    }

    return;
}

VOID
ExpressRootPortPmeInterruptRoutine (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __out PBOOLEAN AssertInterrupt,
    __out PBOOLEAN QueueDpc
    )

/*++

Routine Description:

    This is an interrupt service handler for PME events on a root port. This
    routine examines the root status register to assert a PME event and saves
    the PME requestor Id in the root port extension for any PME related work
    delegated to the DPC or work item. If the root port triggered a PME event
    then this routine will disable the PME interrupt until this event is
    handled to completion subject to other conditions being favorable - like
    the root port is not stopped/removed or powered down.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

    AssertInterrupt - Supplies a place holder to assert if this interrupt was
        generated at the root port because of a PME event.

    QueueDpc - Supplies a place holder to indicate a need for queuing a DPC.

Return Value:

    None.

--*/

{

    LONG DelegatedWorkItems;
    PCI_EXPRESS_ROOT_CONTROL_REGISTER ExpressRootControlRegister;
    PCI_EXPRESS_ROOT_STATUS_REGISTER ExpressRootStatusRegister;

    NON_PAGED_CODE();

    *AssertInterrupt = FALSE;
    *QueueDpc = FALSE;

    //
    // The root port has control and status registers in the capability
    // structure. The RootControl register has a PME Enable bit. With this bit
    // enabled, a PME wake event will trigger an interrupt at the root port.
    // The root status register has a PME Status bit and a PME Requestor ID
    // field. The Status bit indicates that PME was asserted by the requestor
    // ID. This routine will examine both the PME Enable and PME Status bits
    // and saves the requestor ID before asserting an interrupt. In addition, it
    // delegates PME processing to the DPC/worker routine by updating a field
    // in the root port extension using atomic operation and also disables PME
    // interrupt on exit. If a PME wake event handling is in progress at some
    // lower IRQL, then this routine will not assert its interrupts. This
    // condition should never occur though.
    //

    PciReadExpressBridgeRegister(ExpressBridge,
                                 RootControl,
                                 &ExpressRootControlRegister);

    PciReadExpressBridgeRegister(ExpressBridge,
                                 RootStatus,
                                 &ExpressRootStatusRegister);

    if ((ExpressRootControlRegister.PMEInterruptEnable == 0) ||
        (ExpressRootStatusRegister.PMEStatus == 0)) {

        return;
    }

    DelegatedWorkItems =
        InterlockedCompareExchange(&ExpressBridge->Downstream.DelegatedWorkItems,
                                   0,
                                   0);

    //
    // If we've already taken a PME interrupt but not finished processing it,
    // then ack it at the hardware and exit.  This shouldn't actually ever be
    // able to happen, as the PME event should be masked at this point in the
    // root port.  This check will cover slightly busted hardware.
    //

    if (DelegatedWorkItems & EXPRESS_PME_WAKE_EVENT_HANDLING) {
        ExpressRootPortAckPmeEvent(ExpressBridge);
        return;
    }

    //
    // The root port PME wake event is the source for interrupt. Indicate so.
    //

    *AssertInterrupt = TRUE;

    //
    // Update the new requestor Id in the root port extension.
    //

    ExpressBridge->Downstream.u.RootPort.PmeRequestorId.AsUSHORT =
        (USHORT)ExpressRootStatusRegister.PMERequestorId;

    //
    // Update the flag for delegated work items.
    //

    InterlockedOr(&ExpressBridge->Downstream.DelegatedWorkItems,
                  EXPRESS_PME_WAKE_EVENT_HANDLING);

    //
    // Indicate to the caller a need for scheduling a DPC.
    //

    *QueueDpc = TRUE;

    //
    // Mask the interrupt until we are ready to handle a new one. This routine
    // must not update the interrupt state in the root port extension because
    // this is only a transient state.
    //

    ExpressRootPortDisablePmeInterrupt(ExpressBridge);

    //
    // Acknowledge the PME status bit in the root port.
    //

    ExpressRootPortAckPmeEvent(ExpressBridge);
    return;
}

__drv_functionClass(IO_DPC_ROUTINE)
_IRQL_always_function_min_(DISPATCH_LEVEL)
_IRQL_requires_(DISPATCH_LEVEL)
_IRQL_requires_same_
VOID
ExpressRootPortInterruptDpc (
    _In_ PKDPC Dpc,
    _In_ struct _DEVICE_OBJECT *DeviceObject,
    _Inout_ struct _IRP *Irp,
    _In_opt_ PVOID Context
    )

/*++

Routine Description:

    This is a deferred procedure routine queued by the ISR to process and
    execute delegated tasks at the DISPATCH_LEVEL.

Arguments:

    Dpc - Supplies a DPC object resident in the device object and initialized
        for the deferred routine to execute at DISPATCH_LEVEL.

    DeviceObject - Supplies the device object representing the physical device
        that generates interrupts.

    Irp - Supplies the current Irp for the device.

    Context - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    PPCI_BUS Bus;
    LONG DelegatedWorkItems;
    BOOLEAN EnqueuePmeEvent = FALSE;
    PEXPRESS_BRIDGE ExpressBridge;
    PKINTERRUPT InterruptObject;
    KIRQL OldIrql;
    PPCI_ROOT Root;
    NTSTATUS Status;
    PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT SynchronizationContext;

    NON_PAGED_CODE();

    UNREFERENCED_PARAMETER(Dpc);
    UNREFERENCED_PARAMETER(DeviceObject);
    UNREFERENCED_PARAMETER(Irp);

    //
    // Acquire a remove lock for the bus. If this fails then the bus is in the
    // process of being removed. In that case return immediately.
    //

    ExpressBridge = (PEXPRESS_BRIDGE)Context;
    Bus = ExpressBridge->PciBridge->Bus;
    Status = IoAcquireRemoveLock(&Bus->RemoveLock, PCI_PORT_REMOVE_LOCK_TAG);
    if (!NT_SUCCESS(Status)) {
        return;
    }

    DelegatedWorkItems =
        InterlockedCompareExchange(&ExpressBridge->Downstream.DelegatedWorkItems,
                                   0,
                                   0);

    //
    // Service the PME wake event to completion. Since some of the tasks
    // have to execute at PASSIVE level, queue a work item for the same.
    //

    if ((DelegatedWorkItems & EXPRESS_PME_WAKE_EVENT_HANDLING) != 0) {

        PCI_ASSERT(ExpressBridge->Downstream.u.RootPort.PmeRequestorId.AsUSHORT != 0);

        //
        // Acquire a remove lock and signal PME thread to dispatch a routine
        // for handling PME wake event. The remove lock is released after PME
        // processing.
        //

        Status = IoAcquireRemoveLock(&Bus->RemoveLock, PCI_PME_REMOVE_LOCK_TAG);
        if (NT_SUCCESS(Status)) {

            Root = Bus->Root;
            KeAcquireSpinLock(&Root->Bus->BusPowerLock, &OldIrql);
            if (!ExpressBridge->Downstream.u.RootPort.OnPmeList) {
                EnqueuePmeEvent = TRUE;
                ExpressBridge->Downstream.u.RootPort.OnPmeList = TRUE;
            }
            KeReleaseSpinLock(&Root->Bus->BusPowerLock, OldIrql);

            if (EnqueuePmeEvent) {

                ExpressBridge->Downstream.u.RootPort.PmeEventType = ExpressPmeRuntimeEvent;
                ExInterlockedInsertTailList(&Root->PmeRootPortList,
                                            &ExpressBridge->Downstream.u.RootPort.PmeListEntry,
                                            &Root->PmeListLock);
    
                KeSetEvent(&Root->PmeEvent[PciRootPmeEventRootPortAttention],
                           IO_NO_INCREMENT, 
                           FALSE);
            }
        }
    }

    //
    // Service hot plug events delegated by the ISR.
    //

    if ((DelegatedWorkItems & EXPRESS_HOT_PLUG_EVENT_HANDLING) != 0) {

        PCI_ASSERT(ExpressBridge->Downstream.HotPlugController != NULL);

        RtlZeroMemory(&SynchronizationContext,
                      sizeof(PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT));

        SynchronizationContext.DelegatedWorkItems =
            &ExpressBridge->Downstream.DelegatedWorkItems;

        SynchronizationContext.HotPlugController =
            ExpressBridge->Downstream.HotPlugController;

        if (ExpressBridge->Downstream.InterruptConnectVersion ==
            CONNECT_MESSAGE_BASED) {

            InterruptObject =
                ((PIO_INTERRUPT_MESSAGE_INFO)
                 ExpressBridge->Downstream.u.RootPort.ConnectionContext)->
                 MessageInfo[0].InterruptObject;

        } else {

            PCI_ASSERT(ExpressBridge->Downstream.InterruptConnectVersion ==
                       CONNECT_LINE_BASED);

            InterruptObject = 
                (PKINTERRUPT)ExpressBridge->Downstream.u.RootPort.ConnectionContext;
        }

        PCI_ASSERT(InterruptObject != NULL);

        KeSynchronizeExecution(InterruptObject,
                               ExpressBridgeGetHotPlugInterrupts,
                               &SynchronizationContext);

        PciProcessHotPlugInterrupts(ExpressBridge->Downstream.HotPlugController,
                                    &SynchronizationContext.HotPlugInterrupts);

    }

    IoReleaseRemoveLock(&Bus->RemoveLock, PCI_PORT_REMOVE_LOCK_TAG);
    return;
}

_IRQL_requires_same_
__drv_functionClass(KSTART_ROUTINE)
VOID
RootPmeEventDispatcher (
    _In_ PVOID Context
    )

/*++

Routine Description:

    This is a dedicated system thread routine created for a root BUS to
    process PME events for root ports and other bridges. This non-paged routine
    waits on multiple events and executes tasks based on the event that is
    signaled.
 
    Even though this routine runs at PASSIVE_LEVEL, it must not page since it
    runs at times when the paging device is spun down.

Arguments:

    Context - Supplies a pointer to a root PCI bus.

Return Value:

    None.

--*/

{

    PPCI_BUS Bus;
    PLIST_ENTRY BusListEntry;
    PEXPRESS_BRIDGE ExpressBridge;
    ULONG Index;
    KIRQL OldIrql;
    PPCI_ROOT Root;
    PLIST_ENTRY RootPortListEntry;
    NTSTATUS WaitStatus;
    KWAIT_BLOCK WaitBlocks[PciRootPmeEventMaximum];
    PKEVENT Events[PciRootPmeEventMaximum];

    NON_PAGED_CODE();

    Root = (PPCI_ROOT)Context;

    for (Index = 0; Index < PciRootPmeEventMaximum; Index++) {
        Events[Index] = &(Root->PmeEvent[Index]);
    }

    while (TRUE) {
        WaitStatus = KeWaitForMultipleObjects(PciRootPmeEventMaximum,
                                              &(Events[0]),
                                              WaitAny,
                                              Executive,
                                              KernelMode,
                                              FALSE,
                                              NULL,
                                              &WaitBlocks[0]);

        TraceEvents(WppRecorderLogGetDefault(),
                    TRACE_LEVEL_VERBOSE,
                    DBG_PPO,
                    "%!ROOT_PME_EVENTS!",
                    WaitStatus - STATUS_WAIT_0); 

        switch (WaitStatus) {

        //
        // This represents a PME wake event handling.
        //

        case STATUS_WAIT_0 + PciRootPmeEventRootPortAttention:

            RootPortListEntry =
                ExInterlockedRemoveHeadList(&Root->PmeRootPortList,
                                            &Root->PmeListLock);

            while (RootPortListEntry != NULL) {

                //
                // The ExpressDestroyVirtualBridge routine unconditionally
                // unlinks the root port from the PME list during remove Irp
                // processiong, so always keep the root port PmeListEntry in
                // an initialized state.  (This should probably be fixed, as
                // there's obviously a race here.)
                //

                InitializeListHead(RootPortListEntry);
                ExpressBridge = CONTAINING_RECORD(RootPortListEntry,
                                                  EXPRESS_BRIDGE,
                                                  Downstream.u.RootPort.PmeListEntry);

                Bus = ExpressBridge->PciBridge->Bus;

                KeAcquireSpinLock(&Bus->BusPowerLock, &OldIrql);
                PCI_ASSERT(ExpressBridge->Downstream.u.RootPort.OnPmeList);
                ExpressBridge->Downstream.u.RootPort.OnPmeList = FALSE;
                KeReleaseSpinLock(&Bus->BusPowerLock, OldIrql);

                if (ExpressBridge->Downstream.u.RootPort.PmeEventType ==
                    ExpressPmeRuntimeEvent) {

                    //
                    // This represents a regular PME wake event handling.
                    //
                    ExpressRootPortPmeEventWorker(ExpressBridge);
                    IoReleaseRemoveLock(&Bus->RemoveLock, 
                                        PCI_PME_REMOVE_LOCK_TAG);

                } else {

                    //
                    // This represents special case handling of PME event
                    // during power up.
                    //

                    PCI_ASSERT(ExpressBridge->Downstream.u.RootPort.PmeEventType ==
                               ExpressPmePowerUpEvent);

                    ExpressRootPortSingletonPmeEventWorker(ExpressBridge);
                }

                RootPortListEntry =
                    ExInterlockedRemoveHeadList(&Root->PmeRootPortList,
                                                &Root->PmeListLock);
            }

            break;

        //
        // This event signals that some stack needs to send a Wait/Wake IRP
        // and it wasn't running at PASSIVE_LEVEL.
        //

        case STATUS_WAIT_0 + PciRootPmeEventSendWaitWake:

            do {
                BusListEntry = 
                    ExInterlockedRemoveHeadList(&Root->PmeWaitWakeSendList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForWaitWakeSend);
                    PciBus_SendWWPassive(Bus);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This event signals that some stack needs to kick off processing after
        // completion of a wait/wake IRP.
        //

        case STATUS_WAIT_0 + PciRootPmeEventCompleteWaitWake:

            do {
                BusListEntry = 
                    ExInterlockedRemoveHeadList(&Root->PmeWaitWakeCompleteList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForWaitWakeComplete);
                    PciBus_CompleteWWPassive(Bus);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This event signals that some stack needs to cancel a Wait/Wake IRP.
        // While this could be done at DISPATCH_LEVEL, it keeps the W/W code
        // more synchronous through the stack if it happens at PASSIVE_LEVEL.
        // In addition to that, kicking this off to a worker thread ensures that
        // the IRP completion doesn't come back on the same thread, which is
        // important because the power policy state machine runs with a spinlock
        // held.
        //

        case STATUS_WAIT_0 + PciRootPmeEventCancelWaitWake:

            do {
                BusListEntry =
                    ExInterlockedRemoveHeadList(&Root->PmeWaitWakeCancelList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForWaitWakeCancel);
                    PciBus_CancelWWPassive(Bus);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This event signals that some stack needs a D0 IRP sent to itself.
        // this is done here so that we don't send the IRP with the power
        // policy owner state machine lock held.
        //

        case STATUS_WAIT_0 + PciRootPmeEventSendD0:

            do {
                BusListEntry =
                    ExInterlockedRemoveHeadList(&Root->PmeD0SendList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForD0Send);
                    PciBus_SendD0Unlocked(Bus);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This event signals that all the D0 IRPs waiting on a bus to
        // transition itself to D0 should be released.
        //

        case STATUS_WAIT_0 + PciRootPmeEventRetireD0:

            do {
                BusListEntry =
                    ExInterlockedRemoveHeadList(&Root->PmeD0RetireList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForChildD0Retirement);
                    PciBus_RetireChildPowerIrps(Bus);
                    IoReleaseRemoveLock(&Bus->RemoveLock, 
                                        PCI_D0RETIRE_LOCK_TAG);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This event signals that the bus power state machine needs to send
        // and event to the wake state machine.  This is done through the worker
        // thread so that locks are dropped and the stack is more or less empty.
        //

        case STATUS_WAIT_0 + PciRootPmeEventQueueWakeStateEvent:

            do {
                BusListEntry =
                    ExInterlockedRemoveHeadList(&Root->PmeQueueWakeEventList,
                                                &Root->PmeListLock);
                if (BusListEntry != NULL) {
                    Bus = CONTAINING_RECORD(BusListEntry,
                                            PCI_BUS,
                                            WaitingForWaitStateMachine);
                    PciBus_PassWakeEvent(Bus);
                } else {
                    break;
                }
            } while (TRUE);
            break;

        //
        // This represent a thread termination event.
        //

        case STATUS_WAIT_0 + PciRootPmeEventExitThread:

            PsTerminateSystemThread(STATUS_SUCCESS);
            return;

        default:

            PCI_ASSERTMSG("Impossible wait reason", FALSE);
        }
    }

    return;
}

VOID
ExpressRootPortPmeEventWorker (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine is executed as a result of an event signaled from a DPC for
    ISR for processing at PASSIVE level. This routine powers up the hierarchy
    for the device and acknowledges PME_Status bit in the PMCSR register for
    the device.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    KIRQL OldIrql;
    BOOLEAN SpuriousPmeEvent;
    NTSTATUS Status;

    NON_PAGED_CODE();

    SpuriousPmeEvent = ExpressRootPortCheckForSpuriousPmeEvent(ExpressBridge);
    if (SpuriousPmeEvent != FALSE) {
        goto Exit;
    }

    Status = ExpressRootPortAckPmeEventAtDeviceSource(ExpressBridge);
    if (!NT_SUCCESS(Status) ||
        (Status == STATUS_PENDING)) {
        goto Exit;
    }

Exit:

    //
    // Update the field that represents outstanding delegated work items.
    //

    InterlockedAnd(&ExpressBridge->Downstream.DelegatedWorkItems,
                   ~EXPRESS_PME_WAKE_EVENT_HANDLING);

    //
    // Enable PME interrupts as we are ready to service a new one. Do this
    // only when the root port is in a working state (not stopped, removed
    // or powered down) as indicated by the InterruptEnable field.
    //

    KeAcquireSpinLock(&ExpressBridge->Downstream.InterruptStateLock, &OldIrql);
    if (ExpressBridge->Downstream.InterruptsEnabled &
        INTERRUPT_SOURCE_EXPRESS_PME) {

        ExpressRootPortEnablePmeInterrupt(ExpressBridge);
    }

    KeReleaseSpinLock(&ExpressBridge->Downstream.InterruptStateLock, OldIrql);
    return;
}

BOOLEAN
ExpressRootPortCheckForSpuriousPmeEvent (
    __in PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine makes a best effort in determining if this is a spurious
    wake event without powering up the hierarchy for the device.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    TRUE if this is spurious PME event.

    FALSE otherwise.

--*/

{

    PPCI_DEVICE Device;
    PPCI_BUS ParentBus;
    PCI_EXPRESS_PME_REQUESTOR_ID PmeRequestorId;
    LONG WakeIrpState;

    NON_PAGED_CODE();

    Device = NULL;
    ParentBus = NULL;
    PmeRequestorId = ExpressBridge->Downstream.u.RootPort.PmeRequestorId;

    //
    // Traverse the hierarchy and locate the device identified in the requestor
    // ID. If no such device is found then claim this is a spurious wake event.
    //

    PciAcquirePassiveLock(&PciGlobalLock);
    ParentBus = ExpressBridge->PciBridge->Bus;
    while (ParentBus != NULL) {
        if (LOCAL_BUS_NUMBER(ParentBus->SecondaryBusNumber) ==
            PmeRequestorId.BusNumber) {

            break;
        }

        if ((LOCAL_BUS_NUMBER(ParentBus->SecondaryBusNumber) <
            PmeRequestorId.BusNumber) &&
            (LOCAL_BUS_NUMBER(ParentBus->SubordinateBusNumber) >=
            PmeRequestorId.BusNumber)) {

            ParentBus = ParentBus->ChildBuses;
            continue;
        }

        ParentBus = ParentBus->SiblingBus;
    }

    PciReleasePassiveLock(&PciGlobalLock);
    if (ParentBus == NULL) {
        return TRUE;
    }

    //
    // Once we have found the parent, locate the device identified by the
    // Device and Function number in the requestor ID. Again, if no such
    // device is found then claim this as a spurious wake event.
    //

    PciAcquirePassiveLock(&ParentBus->ChildDeviceLock);
    for (Device = ParentBus->ChildDevices;
         Device != NULL;
         Device = Device->Sibling) {

        if ((LOCAL_BUS_NUMBER(Device->BusNumber) ==
            PmeRequestorId.BusNumber) &&
            (Device->Slot.u.bits.DeviceNumber ==
            PmeRequestorId.DeviceNumber) &&
            (Device->Slot.u.bits.FunctionNumber ==
            PmeRequestorId.FunctionNumber)) {

            break;
        }
    }

    PciReleasePassiveLock(&ParentBus->ChildDeviceLock);

    if (Device == NULL) {
        return TRUE;
    }

    PCI_ASSERT((Device->Present != FALSE) &&
               (Device->ReportedMissing == FALSE));

    //
    // Ensure that there is a Wait Wake Irp pending for the device, otherwise
    // claim this as a spurious wake event. A definitive way of saying if this
    // is spurious or not is by examining the PME_Status bit in the PMCSR
    // register in the power management well of the device. This register can't
    // be accessed unless the entire hierarchy for the device is powered up.
    // So defer this decision for now.
    //

    WakeIrpState = InterlockedCompareExchange((volatile LONG*)&Device->WakeIrpState,
                                              PciWakeIrpQueued,
                                              PciWakeIrpQueued);

    if (WakeIrpState != PciWakeIrpQueued) {
        return TRUE;
    }

    return FALSE;
}

_IRQL_requires_(PASSIVE_LEVEL)
NTSTATUS
ExpressRootPortAckPmeEventAtDeviceSource (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine powers up the hierarchy for the device and acknowledges the
    PME source by completing its Wait Wake Irp.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    STATUS_UNSUCCESSFUL when a device identified in the requestor ID cannot
    be reached. Otherwise, a status code returned from one of the functions
    invoked in this routine.

--*/

{

    BOOLEAN ChildBusPowered;
    PPCI_DEVICE Device, ChildDevice;
    PPCI_BUS Bus, ChildBus;
    PCI_EXPRESS_PME_REQUESTOR_ID PmeRequestorId;
    KIRQL OldIrql;
    NTSTATUS Status = STATUS_UNSUCCESSFUL;

    NON_PAGED_CODE();

    Bus = ExpressBridge->PciBridge->Bus;
    PmeRequestorId = ExpressBridge->Downstream.u.RootPort.PmeRequestorId;

    //
    // Locate the device identified in the requestor ID on the bus.  If we
    // don't find it, assume it was on a subordinate bus.
    //

    PciAcquirePassiveLock(&Bus->ChildDeviceLock);
    for (Device = Bus->ChildDevices;
         Device != NULL;
         Device = Device->Sibling) {

        //
        // Now look to see if the device is a match.
        //

        if ((LOCAL_BUS_NUMBER(Device->BusNumber) ==
            PmeRequestorId.BusNumber) &&
            (Device->Slot.u.bits.DeviceNumber ==
            PmeRequestorId.DeviceNumber) &&
            (Device->Slot.u.bits.FunctionNumber ==
            PmeRequestorId.FunctionNumber)) {

            break;
        }
    }
    PciReleasePassiveLock(&Bus->ChildDeviceLock);

    //
    // If we found a device, wake it up.  If we found a bus, hunt down through
    // the bridges until we find the device and wake it up.
    //

    if (Device != NULL) {

        PCI_ASSERT((Device->Present != FALSE) &&
                   (Device->ReportedMissing == FALSE));

        //
        // Flag this device as the one which triggered the PME so that 
        // when we power on the parent bus of this device we don't notify
        // the runtime power framework that the device needs to be surprise
        // powered on.
        //

        PciDevice_WakeTriggered(Device, FALSE, STATUS_SUCCESS);
        Status = STATUS_SUCCESS;
        goto Exit;

    } else {

        Status = STATUS_NOT_FOUND;
        for (;;) {

UglyLabelThatAvoidsRecursion:
            PciAcquirePassiveLock(&Bus->ChildDeviceLock);

            for (ChildBus = Bus->ChildBuses; 
                 ChildBus != NULL; 
                 ChildBus = ChildBus->SiblingBus) {

                //
                // If this device is a bridge to the bridge we're looking
                // for, start over with this bridge.
                //

                if ((PmeRequestorId.BusNumber > ChildBus->SecondaryBusNumber) &&
                    (PmeRequestorId.BusNumber <= ChildBus->SubordinateBusNumber)) {

                    PciReleasePassiveLock(&Bus->ChildDeviceLock);
                    Bus = ChildBus;
                    goto UglyLabelThatAvoidsRecursion;
                }

                //
                // If this device is a bridge to the bus we're looking for,
                // start searching through the child devices.
                //

                if (PmeRequestorId.BusNumber == ChildBus->SecondaryBusNumber) {
                    
                    PciReleasePassiveLock(&Bus->ChildDeviceLock);

                    //
                    // Assert that this bus is turned on, possibly turning it on
                    // as a side effect.  Since this action might queue work to
                    // the very worker thread that is running this function,
                    // we can't wait synchronously here.  The raise to DISPATCH
                    // changes the semantics of PciBus_PowerReference, causing
                    // it to queue work and return.
                    //

                    KeRaiseIrql(DISPATCH_LEVEL, &OldIrql);
                    ChildBusPowered = PciBus_PowerReference(ChildBus, 
                                                            PciBusPowerReasonPmeTriggered);
                    KeLowerIrql(OldIrql);

                    //
                    // If we previously deferred processing, remove the extra
                    // reference from that pass through this code.  (See the
                    // code directly below this which can exit with the
                    // reference still on the bus, intentionally.  This function
                    // may be called multiple times, attempting to handle this
                    // event.)
                    //

                    if (ChildBus->PmeEventDeferred == 1) {
                        PCI_BUS_CLEAR_FLAG(ChildBus, PmeEventDeferred);
                        PciBus_PowerDereference(ChildBus, 
                                                PciBusPowerReasonPmeTriggered);
                    }

                    //
                    // If the child bus wasn't powered on, then we need to come
                    // back to this later.
                    //

                    if (!ChildBusPowered) {
                        PCI_BUS_SET_FLAG(ChildBus, PmeEventDeferred);
                        Status = STATUS_PENDING;
                        goto Exit;
                    }

                    PciAcquirePassiveLock(&ChildBus->ChildDeviceLock);

                    for (ChildDevice = ChildBus->ChildDevices;
                         ChildDevice != NULL;
                         ChildDevice = ChildDevice->Sibling) {

                        //
                        // Complete any wait/wake IRP that might be present.
                        //

                        if ((ChildDevice->Slot.u.bits.DeviceNumber == PmeRequestorId.DeviceNumber) &&
                            (ChildDevice->Slot.u.bits.FunctionNumber == PmeRequestorId.FunctionNumber)) {

                            PciDevice_WakeTriggered(ChildDevice, FALSE, STATUS_SUCCESS);
                            Status = STATUS_SUCCESS;
                            break;
                        }
                    }

                    PciReleasePassiveLock(&ChildBus->ChildDeviceLock);

                    //
                    // Let the bus fall back asleep if all the devices go idle
                    // again in the future.
                    //

                    PciBus_PowerDereference(ChildBus, PciBusPowerReasonPmeTriggered);
                    goto Exit;
                }
            }

            PciReleasePassiveLock(&Bus->ChildDeviceLock);
            break;
        }
    }

Exit:

    return Status;
}

VOID
ExpressRootPortEnableInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine enables interrupts on a root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    KIRQL OldIrql;

    NON_PAGED_CODE();

    if (ExpressBridge->Downstream.InterruptConnected == FALSE) {
        return;
    }

    KeAcquireSpinLock(&ExpressBridge->Downstream.InterruptStateLock, &OldIrql);

    //
    // Enable AER interrupt on the root port.
    //

    if (ExpressBridge->AerCapability != 0 &&
        PciIsAerGranted(ExpressBridge->PciBridge->Bus->Root)) {

        ExpressRootPortEnableAerInterrupt(ExpressBridge);
        ExpressBridge->Downstream.InterruptsEnabled |=
            INTERRUPT_SOURCE_EXPRESS_AER;
    }

    //
    // Enable PME interrupt on a root port.
    //

    ExpressRootPortEnablePmeInterrupt(ExpressBridge);
    ExpressBridge->Downstream.InterruptsEnabled |= INTERRUPT_SOURCE_EXPRESS_PME;

    //
    // Check if the root port slot is capable of any hot plug operations. In
    // that case enable hot plug interrupts on a root port.
    //
    // N.B. If hot plug interrupts are already enabled, then it is possible to
    //      return immediately (unlike PME) since enabling or disabling of hot
    //      plug interrupts is gated by a spin lock.
    //

    if (ExpressBridge->Downstream.HotPlugCapable != FALSE) {
        if ((ExpressBridge->Downstream.InterruptsEnabled &
             INTERRUPT_SOURCE_EXPRESS_HOTPLUG) == 0) {

            //
            // Clear any spurious hotplug status bits so that a root port can
            // receive notification for the corresponding events.
            //

            ExpressBridgeClearSpuriousHotPlugStatus(ExpressBridge);
            ExpressRootPortEnableHotPlugInterrupts(ExpressBridge);
            ExpressBridge->Downstream.InterruptsEnabled |=
                INTERRUPT_SOURCE_EXPRESS_HOTPLUG;

            //
            // Process hotplug status bits in case some hotplug events arrived
            // after the clear operation but before the interrupts were enabled.
            //

            ExpressBridgeProcessAsyncHotPlugEvents(ExpressBridge);
        }
    }

    KeReleaseSpinLock(&ExpressBridge->Downstream.InterruptStateLock, OldIrql);
    return;
}

VOID
ExpressRootPortEnableAerInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine enables AER interrupts on the root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NON_PAGED_CODE();

    ExpressBridge->Downstream.u.RootPort.RootErrorCommand.AsULONG =
        ExpressBridge->ErrorSource.Info.AerRootportDescriptor.RootErrorCommand;

    PciWriteExpressRootPortAerRegister(
            ExpressBridge,
            RootErrorCommand,
            &ExpressBridge->Downstream.u.RootPort.RootErrorCommand.AsULONG);

    return;
}

VOID
ExpressRootPortEnablePmeInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine sets PME Enable bit in the RootControl register in the root
    port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NON_PAGED_CODE();

    ExpressBridge->Downstream.u.RootPort.RootControl.PMEInterruptEnable = 1;
    PciWriteExpressBridgeRegister(ExpressBridge,
                                  RootControl,
                                  &ExpressBridge->Downstream.u.RootPort.RootControl);

    return;
}

VOID
ExpressRootPortEnableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine enables hot plug interrupts for a root port depending on the
    type of hot plug elements present for a controller or a slot.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NTSTATUS Status;

    NON_PAGED_CODE();

    PCI_ASSERT(ExpressBridge->Downstream.HotPlugCapable != FALSE);

    Status = PciEnableDisableHotPlugInterrupts(
                ExpressBridge->Downstream.HotPlugController,
                TRUE);

    PCI_ASSERT(NT_SUCCESS(Status));
    UNREFERENCED_PARAMETER(Status);

    return;
}

VOID
ExpressRootPortDisableInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disables interrupts on a root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    KIRQL OldIrql;

    NON_PAGED_CODE();

    KeAcquireSpinLock(&ExpressBridge->Downstream.InterruptStateLock, &OldIrql);

    //
    // Disable AER interrupt on a root port.
    //

    if (ExpressBridge->AerCapability != 0 &&
        PciIsAerGranted(ExpressBridge->PciBridge->Bus->Root)) {

        ExpressBridge->Downstream.InterruptsEnabled &=
            ~INTERRUPT_SOURCE_EXPRESS_AER;
        ExpressRootPortDisableAerInterrupt(ExpressBridge);
    }

    //
    // When the system is in S0, we may (new in Win8) put the root port in a
    // low-power D-state.  When we do that, we need to leave the internal
    // event enabled so that a chipset can use that signal to trigger a wake
    // event.
    // 
    // Transitions back to D0 from D3 cause an internal reset of the function
    // (for any PCI device) so the interrupts will be implicitly disabled.  
    // Clearing the interrupt flags even when we aren't going to touch the
    // hardware will cause the "enable" function to reenable them after this
    // reset.
    //

    ExpressBridge->Downstream.InterruptsEnabled &= ~INTERRUPT_SOURCE_EXPRESS_PME;
    if (ExpressBridge->Downstream.HotPlugCapable != FALSE) {
        ExpressBridge->Downstream.InterruptsEnabled &=
            ~INTERRUPT_SOURCE_EXPRESS_HOTPLUG;
    }

    if (ExpressBridge->PciBridge->Bus->SIrp != NULL) {

        //
        // Disable PME interrupt on a root port.
        //
        ExpressRootPortDisablePmeInterrupt(ExpressBridge);

        //
        // Disable hot plug interrupts on a root port.
        //
        // N.B. If the hot plug interrupts are already disabled then it is possible
        //      to return immediately (unlike PME) since enabling or disabling of
        //      hot plug interrupts is gated by a spin lock.
        //

        if ((ExpressBridge->Downstream.HotPlugCapable != FALSE) &&
            ((ExpressBridge->Downstream.InterruptsEnabled &
              INTERRUPT_SOURCE_EXPRESS_HOTPLUG) != 0)) {

                ExpressRootPortDisableHotPlugInterrupts(ExpressBridge);
        }
    }

    KeReleaseSpinLock(&ExpressBridge->Downstream.InterruptStateLock, OldIrql);
    return;
}

VOID
ExpressRootPortDisableAerInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine clears the bits in the root error command register which
    enable the generation of interrupts when an error is reported by any of
    the devices in the hierarchy associated with this root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NON_PAGED_CODE();

    ExpressBridge->Downstream.u.RootPort.RootErrorCommand.AsULONG = 0;
    PciWriteExpressRootPortAerRegister(
            ExpressBridge,
            RootErrorCommand,
            &ExpressBridge->Downstream.u.RootPort.RootErrorCommand.AsULONG);

    return;
}

VOID
ExpressRootPortDisablePmeInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine clears the PME Enable bit in the RootControl register in the
    root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NON_PAGED_CODE();

    ExpressBridge->Downstream.u.RootPort.RootControl.PMEInterruptEnable = 0;
    PciWriteExpressBridgeRegister(ExpressBridge,
                                  RootControl,
                                  &ExpressBridge->Downstream.u.RootPort.RootControl);

    return;
}

VOID
ExpressRootPortDisableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disables hot plug interrupts for a root port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port.

Return Value:

    None.

--*/

{

    NTSTATUS Status;

    NON_PAGED_CODE();

    PCI_ASSERT(ExpressBridge->Downstream.HotPlugCapable != FALSE);

    Status = PciEnableDisableHotPlugInterrupts(
                ExpressBridge->Downstream.HotPlugController,
                FALSE);

    PCI_ASSERT(NT_SUCCESS(Status));
    UNREFERENCED_PARAMETER(Status);

    return;
}

VOID
ExpressRootPortDisconnectInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disconnects an interrupt service routine for the interrupts
    in the root port.

Arguments:

    ExpressBridge - Supplies a pointer to an Express Bridge for a root port.

Return Value:

    None.

--*/

{

    IO_DISCONNECT_INTERRUPT_PARAMETERS InterruptParameters;

    PAGED_CODE();

    if (ExpressBridge->Downstream.InterruptConnected == FALSE) {
        return;
    }

    RtlZeroMemory(&InterruptParameters,
                  sizeof(IO_DISCONNECT_INTERRUPT_PARAMETERS));

    InterruptParameters.Version =
        ExpressBridge->Downstream.InterruptConnectVersion;

    InterruptParameters.ConnectionContext.Generic =
        ExpressBridge->Downstream.u.RootPort.ConnectionContext;

    IoDisconnectInterruptEx(&InterruptParameters);
    ExpressBridge->Downstream.InterruptConnected = FALSE;
    ExpressBridge->Downstream.InterruptConnectVersion = 0;
    ExpressBridge->Downstream.u.RootPort.ConnectionContext = NULL;
    return;
}

// ----------------------------------------- Routines for downstream switch port

NTSTATUS
ExpressDownstreamSwitchPortConnectInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList
    )

/*++

Routine Description:

    This routine connects and registers an ISR for an interrupt resource
    assigned to a downstream switch port bridge.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge.

    RawList - Supplies a list of resources assigned to a bridge.

    TranslatedList - Supplies a list of resources assigned to a bridge
        in a translated form.

Return Value:

    Status code as returned from a function call in this routine.

--*/

{

    ULONG MessageNumber;
    NTSTATUS Status;

    PAGED_CODE();

    PCI_ASSERT(ExpressBridge->Downstream.HotPlugCapable != FALSE);

    //
    // Obtain an interrupt message number for a hot plug event. This is the
    // message that the downstream switch port ISR will connect with.
    //

    MessageNumber = ExpressBridgeGetHotPlugMessageNumber(ExpressBridge);

    //
    // Invoke a helper function to connect to an appropriate interrupt resource.
    //

    Status = ExpressBridgeConnectInterrupt(
                ExpressBridge,
                RawList,
                TranslatedList,
                MessageNumber,
                ExpressDownstreamSwitchPortInterruptRoutine,
                &ExpressBridge->Downstream.u.SwitchPort.InterruptObject,
                &ExpressBridge->Downstream.InterruptConnectVersion);

    if (NT_SUCCESS(Status)) {
        ExpressBridge->Downstream.InterruptConnected = TRUE;
    }

    return Status;
}

_Function_class_(KSERVICE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
BOOLEAN
ExpressDownstreamSwitchPortInterruptRoutine (
    _In_ struct _KINTERRUPT *Interrupt,
    _In_opt_ PVOID ServiceContext
    )

/*++

Routine Description:

    This is an interrupt service routine registered for handling hot plug
    related interrupts on a downstream switch port.

Arguments:

    Interrupt - Supplies a pointer to an interrupt object.

    ServiceContext - Supplies a pointer to an express bridge for a downstream
        switch port.

Return Value:

    TRUE if this interrupt belongs to the downstream switch port.

    FALSE otherwise.

--*/

{

    BOOLEAN AssertHotPlugInterrupt;
    PPCI_BUS Bus;
    PEXPRESS_BRIDGE ExpressBridge;
    BOOLEAN QueueDpcForHotPlug;

    NON_PAGED_CODE();

    UNREFERENCED_PARAMETER(Interrupt);

    ExpressBridge = (PEXPRESS_BRIDGE)ServiceContext;
    _Analysis_assume_(ExpressBridge != NULL);

    //
    // Invoke the HotPlug interrupt service routine.
    //

    AssertHotPlugInterrupt = FALSE;
    if (ExpressBridge->Downstream.HotPlugCapable != FALSE) {
        ExpressBridgeHotPlugInterruptRoutine(
            ExpressBridge,
            ExpressBridge->Downstream.HotPlugController,
            &ExpressBridge->Downstream.DelegatedWorkItems,
            &AssertHotPlugInterrupt,
            &QueueDpcForHotPlug);

        if (QueueDpcForHotPlug != FALSE) {
            Bus = ExpressBridge->PciBridge->Bus;
            IoRequestDpc(Bus->DeviceObject, NULL, ExpressBridge);
        }
    }

    return AssertHotPlugInterrupt;
}

__drv_functionClass(IO_DPC_ROUTINE)
_IRQL_always_function_min_(DISPATCH_LEVEL)
_IRQL_requires_(DISPATCH_LEVEL)
_IRQL_requires_same_
VOID
ExpressDownstreamSwitchPortInterruptDpc (
    _In_ PKDPC Dpc,
    _In_ struct _DEVICE_OBJECT *DeviceObject,
    _Inout_ struct _IRP *Irp,
    _In_opt_ PVOID Context
    )

/*++

Routine Description:

    This is a deferred procedure routine queued by the ISR to process and
    execute delegated tasks at the DISPATCH_LEVEL.

Arguments:

    Dpc - Supplies a DPC object resident in the device object and initialized
        for the deferred routine to execute at DISPATCH_LEVEL.

    DeviceObject - Supplies the device object representing the physical device
        that has generated interrupts.

    Irp - Supplies the current Irp for the device.

    Context - Supplies a pointer to an express bridge for a downstream switch
        port.

Return Value:

    None.

--*/

{

    PPCI_BUS Bus;
    LONG DelegatedWorkItems;
    PEXPRESS_BRIDGE ExpressBridge;
    NTSTATUS Status;
    PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT SynchronizationContext;

    NON_PAGED_CODE();

    UNREFERENCED_PARAMETER(Dpc);
    UNREFERENCED_PARAMETER(DeviceObject);
    UNREFERENCED_PARAMETER(Irp);

    //
    // Acquire a remove lock for the bus. If this fails then the bus is in the
    // process of being removed. In that case return immediately.
    //

    ExpressBridge = (PEXPRESS_BRIDGE)Context;
    Bus = ExpressBridge->PciBridge->Bus;
    Status = IoAcquireRemoveLock(&Bus->RemoveLock, PCI_PORT_REMOVE_LOCK_TAG);
    if (!NT_SUCCESS(Status)) {
        return;
    }

    DelegatedWorkItems =
        InterlockedCompareExchange(
            &ExpressBridge->Downstream.DelegatedWorkItems,
            0,
            0);

    //
    // Service hot plug events delegated by the ISR.
    //

    if ((DelegatedWorkItems & EXPRESS_HOT_PLUG_EVENT_HANDLING) != 0) {

        PCI_ASSERT(ExpressBridge->Downstream.HotPlugController !=
                   NULL);

        RtlZeroMemory(&SynchronizationContext,
                      sizeof(PCI_INTERRUPT_SYNCHRONIZATION_CONTEXT));

        SynchronizationContext.DelegatedWorkItems =
            &ExpressBridge->Downstream.DelegatedWorkItems;

        SynchronizationContext.HotPlugController =
            ExpressBridge->Downstream.HotPlugController;

        KeSynchronizeExecution(
            ExpressBridge->Downstream.u.SwitchPort.InterruptObject,
            ExpressBridgeGetHotPlugInterrupts,
            &SynchronizationContext);

        PciProcessHotPlugInterrupts(
            ExpressBridge->Downstream.HotPlugController,
            &SynchronizationContext.HotPlugInterrupts);

    }

    IoReleaseRemoveLock(&Bus->RemoveLock, PCI_PORT_REMOVE_LOCK_TAG);
    return;
}

_IRQL_requires_max_(DISPATCH_LEVEL)
VOID
ExpressDownstreamSwitchPortEnableInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine enables interrupts on a downstream switch port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a downstream
        switch port.

Return Value:

    None.

--*/

{

    KIRQL OldIrql;

    NON_PAGED_CODE();

    if (ExpressBridge->Downstream.HotPlugCapable == FALSE) {
        return;
    }

    if (ExpressBridge->Downstream.InterruptConnected == FALSE) {
        return;
    }

    KeAcquireSpinLock(&ExpressBridge->Downstream.InterruptStateLock,
                      &OldIrql);

    //
    // Enable hot plug interrupts on a downstream switch port.
    //
    // N.B. If hot plug interrupts are already enabled, then it is possible to
    //      return immediately since enabling or disabling of hot plug
    //      interrupts is gated by a spin lock.
    //

    if ((ExpressBridge->Downstream.InterruptsEnabled &
        INTERRUPT_SOURCE_EXPRESS_HOTPLUG) == 0) {

        //
        // Clear any spurious hotplug status bits so that a root port can
        // receive notification for the corresponding events.
        //

        ExpressBridgeClearSpuriousHotPlugStatus(ExpressBridge);
        ExpressDownstreamSwitchPortEnableHotPlugInterrupts(ExpressBridge);
        ExpressBridge->Downstream.InterruptsEnabled |=
            INTERRUPT_SOURCE_EXPRESS_HOTPLUG;

        //
        // Process hotplug status bits in case some hotplug events arrived
        // after the clear operation but before the interrupts were enabled.
        //

        ExpressBridgeProcessAsyncHotPlugEvents(ExpressBridge);
    }

    KeReleaseSpinLock(&ExpressBridge->Downstream.InterruptStateLock,
                      OldIrql);

    return;
}

VOID
ExpressDownstreamSwitchPortEnableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine enables hot plug interrupts for a downstream switch port
    depending on the type of hot plug elements present for a controller or a
    slot.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a downstream
    switch port.

Return Value:

    None.

--*/

{

    NTSTATUS Status;

    NON_PAGED_CODE();

    if (ExpressBridge->Downstream.HotPlugCapable == FALSE) {
        return;
    }

    Status = PciEnableDisableHotPlugInterrupts(
                ExpressBridge->Downstream.HotPlugController,
                TRUE);

    PCI_ASSERT(NT_SUCCESS(Status));
    UNREFERENCED_PARAMETER(Status);

    return;
}

VOID
ExpressDownstreamSwitchPortDisableInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disables interrupts on a downstream switch port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a downstream
        switch port.

Return Value:

    None.

--*/

{

    KIRQL OldIrql;

    NON_PAGED_CODE();

    KeAcquireSpinLock(&ExpressBridge->Downstream.InterruptStateLock, &OldIrql);

    //
    // Disable hot plug interrupts on a downstream switch port.
    //
    // N.B. If the hot plug interrupts are already disabled then it is possible
    //      to return immediately since enabling or disabling of hot plug
    //      interrupts is gated by a spin lock.
    //
    // Skip actually disabling interrupts unless the system is leaving S0.  (See
    // comment in ExpressRootPortDisableInterrupts.)
    //

    ExpressBridge->Downstream.InterruptsEnabled &=
        ~INTERRUPT_SOURCE_EXPRESS_HOTPLUG;

    if (((ExpressBridge->Downstream.InterruptsEnabled &
          INTERRUPT_SOURCE_EXPRESS_HOTPLUG) != 0) &&
        (ExpressBridge->PciBridge->Bus->SIrp != NULL)) {

        ExpressDownstreamSwitchPortDisableHotPlugInterrupts(
            ExpressBridge);
    }

    KeReleaseSpinLock(&ExpressBridge->Downstream.InterruptStateLock, OldIrql);

    return;
}

VOID
ExpressDownstreamSwitchPortDisableHotPlugInterrupts (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disables hot plug interrupts for a downstream switch port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a downstream
        switch port.

    Return Value:

    None.

--*/

{

    NTSTATUS Status;

    NON_PAGED_CODE();

    PCI_ASSERT(ExpressBridge->Downstream.HotPlugCapable != FALSE);

    Status = PciEnableDisableHotPlugInterrupts(
                ExpressBridge->Downstream.HotPlugController,
                FALSE);

    PCI_ASSERT(NT_SUCCESS(Status));
    UNREFERENCED_PARAMETER(Status);

    return;
}

VOID
ExpressDownstreamSwitchPortDisconnectInterrupt (
    __inout PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine disconnects an interrupt service routine for the interrupts
    in the downstream switch port.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a downstream
    switch port.

Return Value:

    None.

--*/

{

    IO_DISCONNECT_INTERRUPT_PARAMETERS InterruptParameters;

    PAGED_CODE();

    if (ExpressBridge->Downstream.InterruptConnected == FALSE) {
        return;
    }

    RtlZeroMemory(&InterruptParameters,
                  sizeof(IO_DISCONNECT_INTERRUPT_PARAMETERS));

    InterruptParameters.Version =
        ExpressBridge->Downstream.InterruptConnectVersion;

    InterruptParameters.ConnectionContext.InterruptObject =
        ExpressBridge->Downstream.u.SwitchPort.InterruptObject;

    IoDisconnectInterruptEx(&InterruptParameters);
    ExpressBridge->Downstream.InterruptConnected = FALSE;
    ExpressBridge->Downstream.InterruptConnectVersion = 0;
    ExpressBridge->Downstream.u.SwitchPort.InterruptObject = NULL;
    return;
}

// ------------------------------------------------- Routines for express bridge

NTSTATUS
ExpressBridgeConnectInterrupt (
    __in PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList,
    __in ULONG MessageNumber,
    __in PKSERVICE_ROUTINE InterruptRoutine,
    __out PKINTERRUPT *InterruptObject,
    __out PULONG InterruptConnectVersion
    )

/*++

Routine Description:

    This routine connects and registers an ISR for an interrupt resource
    assigned to a bridge.

Arguments:

    ExpressBridge - Supplies a pointer to an Express Bridge.

    RawList - Supplies a list of resources assigned to a bridge.

    TranslatedList - Supplies a list of resources assigned to a bridge
        in a translated form.

    MessageNumber - Supplies a message number to connect with and register
        an ISR for in a set of message resources for a bridge.

    InterruptRoutine - Supplies an interrupt service routine to register for
        an interrupt resource.

    InterruptObject - Supplies a place holder for an interrupt object to be
        filled on successful connection.

    InterruptConnectVersion - Supplies a place holder for an interrupt type on
        successful connection.

Return Value:

    STATUS_UNSUCCESSFUL when no interrupt resource is assigned to a bridge.

    STATUS_INVALID_PARAMETER if an interrupt resource assigned to a bridge
        is not compatible with the interrupt message number that the bridge
        wants to connect with.

    STATUS_SUCCESS on success.

--*/

{

    ULONG DescriptorIndex;
    BOOLEAN FoundResource;
    IO_CONNECT_INTERRUPT_PARAMETERS InterruptParameters;
    ULONG MessageCount;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR RawDescriptor;
    NTSTATUS Status;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR TranslatedDescriptor;

    PAGED_CODE();

    //
    // Sanity check for resource lists.
    //

    if ((RawList == NULL) || (TranslatedList == NULL)) {
        return STATUS_UNSUCCESSFUL;
    }

    RtlZeroMemory(&InterruptParameters,
                  sizeof(IO_CONNECT_INTERRUPT_PARAMETERS));

    //
    // The interrupt connection parameter is initialized differently based
    // on the type of interrupt resource assigned to a bridge. So, first
    // initialize the common portions, then parse thru the raw and translated
    // resource list to determine the type and populate the remaining portions
    // followed by a call to IoConnectInterruptEx.
    //

    InterruptParameters.Version = CONNECT_FULLY_SPECIFIED_GROUP;
    InterruptParameters.FullySpecified.PhysicalDeviceObject =
        ExpressBridge->PciBridge->Bus->PhysicalDeviceObject;

    InterruptParameters.FullySpecified.InterruptObject = InterruptObject;
    InterruptParameters.FullySpecified.ServiceRoutine = InterruptRoutine;
    InterruptParameters.FullySpecified.ServiceContext = ExpressBridge;
    InterruptParameters.FullySpecified.SpinLock = NULL;
    InterruptParameters.FullySpecified.FloatingSave = FALSE;
    InterruptParameters.FullySpecified.ShareVector = TRUE;

    //
    // Parse thru the raw and translated resource list for a bridge to locate
    // the first interrupt resource descriptor.
    //

    DescriptorIndex = 0;
    FoundResource = FALSE;
    FOR_ALL_IN_PARALLEL_RESOURCE_LISTS(RawList,
                                       TranslatedList,
                                       RawDescriptor,
                                       TranslatedDescriptor) {

        DescriptorIndex += 1;
        if (TranslatedDescriptor->Type == CmResourceTypeInterrupt) {
            FoundResource = TRUE;
            break;
        }
    }

    //
    // Sanity check. This will fail the Start Irp for a bridge.
    //

    if (FoundResource == FALSE) {
        return STATUS_UNSUCCESSFUL;
    }

    if ((TranslatedDescriptor->Flags &
         CM_RESOURCE_INTERRUPT_LEVEL_LATCHED_BITS) ==
        CM_RESOURCE_INTERRUPT_LEVEL_SENSITIVE) {

        //
        // The bridge is assigned a line based interrupt. Initialize the
        // remaining fields in the interrupt connection parameter.
        //

        InterruptParameters.FullySpecified.SynchronizeIrql =
            (KIRQL)TranslatedDescriptor->u.Interrupt.Level;

        InterruptParameters.FullySpecified.Vector =
            TranslatedDescriptor->u.Interrupt.Vector;

        InterruptParameters.FullySpecified.Irql =
            (KIRQL)TranslatedDescriptor->u.Interrupt.Level;

        InterruptParameters.FullySpecified.InterruptMode = LevelSensitive;
        InterruptParameters.FullySpecified.Group =
            TranslatedDescriptor->u.Interrupt.Group;

        InterruptParameters.FullySpecified.ProcessorEnableMask =
            TranslatedDescriptor->u.Interrupt.Affinity;

    } else if ((TranslatedDescriptor->Flags &
                CM_RESOURCE_INTERRUPT_MESSAGE) ==
               CM_RESOURCE_INTERRUPT_MESSAGE) {

        PCI_ASSERT((TranslatedDescriptor->Flags &
                   CM_RESOURCE_INTERRUPT_LATCHED) ==
                   CM_RESOURCE_INTERRUPT_LATCHED);

        MessageCount = RawDescriptor->u.MessageInterrupt.Raw.MessageCount;

        //
        // A multiple-message interrupt descriptor corresponds to the PCI 2.2
        // MSI resource. For such descriptor, a contiguous array of Vectors
        // of 'MessageCount' length is allocated. The base Vector corresponds
        // to message 0. So, for 2.2 MSI resource descriptor, the bridge must
        // connect to the base Vector + MessageNumber. If this is a single
        // message interrupt descriptor, then either it corresponds to a single
        // message 2.2 MSI resource or is the first descriptor in the set of
        // MSI-X resources. For MSI-X, the resource list is guaranteed to have
        // as many interrupt descriptors as the number of MSI-X grants for a
        // bridge.
        //

        if ((MessageCount >= 1) || (MessageNumber == 0)) {
            if (MessageNumber >= MessageCount) {
                return STATUS_INVALID_PARAMETER;
            }

            InterruptParameters.FullySpecified.Vector =
                TranslatedDescriptor->u.MessageInterrupt.Translated.Vector +
                MessageNumber;

        } else {

            //
            // This is a MSI-X resource descriptor. Skip to the descriptor that
            // corresponds to a message number that the bridge wants to connect
            // with.
            //

            if ((DescriptorIndex + MessageNumber) >
                TranslatedList->List[0].PartialResourceList.Count) {

                return STATUS_INVALID_PARAMETER;
            }

            RawDescriptor += MessageNumber;
            TranslatedDescriptor += MessageNumber;

            //
            // Some sanity checks.
            //

            if ((TranslatedDescriptor->Type != CmResourceTypeInterrupt) ||
                !(TranslatedDescriptor->Flags &
                  CM_RESOURCE_INTERRUPT_LATCHED) ||
                !(TranslatedDescriptor->Flags &
                  CM_RESOURCE_INTERRUPT_MESSAGE) ||
                (RawDescriptor->u.MessageInterrupt.Raw.MessageCount != 1)) {

                return STATUS_INVALID_PARAMETER;
            }

            InterruptParameters.FullySpecified.Vector =
                TranslatedDescriptor->u.MessageInterrupt.Translated.Vector;
        }

        InterruptParameters.FullySpecified.SynchronizeIrql =
            (KIRQL)TranslatedDescriptor->u.MessageInterrupt.Translated.Level;

        InterruptParameters.FullySpecified.Irql =
            (KIRQL)TranslatedDescriptor->u.MessageInterrupt.Translated.Level;

        InterruptParameters.FullySpecified.InterruptMode = Latched;
        InterruptParameters.FullySpecified.Group =
            TranslatedDescriptor->u.MessageInterrupt.Translated.Group;

        InterruptParameters.FullySpecified.ProcessorEnableMask =
            TranslatedDescriptor->u.MessageInterrupt.Translated.Affinity;
    }

    Status = IoConnectInterruptEx(&InterruptParameters);
    if (NT_SUCCESS(Status)) {
        *InterruptConnectVersion = InterruptParameters.Version;
    }

    return Status;
}

NTSTATUS
ExpressBridgeConnectMessageInterrupt (
    __in PEXPRESS_BRIDGE ExpressBridge,
    __in PCM_RESOURCE_LIST RawList,
    __in PCM_RESOURCE_LIST TranslatedList,
    __in PKSERVICE_ROUTINE FallbackInterruptRoutine,
    __in PKMESSAGE_SERVICE_ROUTINE MessageServiceRoutine,
    __in_opt PKSPIN_LOCK MessageRoutineSpinLock,
    __out PVOID *ConnectionContext,
    __out PULONG InterruptConnectVersion
    )

/*++

Routine Description:

    This routine connects and registers a message service routine for interrupt
    message resources assigned to a bridge.

Arguments:

    ExpressBridge - Supplies a pointer to an Express Bridge.

    RawList - Supplies a list of resources assigned to a bridge.

    TranslatedList - Supplies a list of resources assigned to a bridge
        in a translated form.

    FallbackInterruptRoutine - Supplies an interrupt service routine for the
        line based interrupt resource assigned to a bridge.

    MessageServiceRoutine - Supplies a message service routine for message
        interrupt resources assigned to a bridge.

    ConnectionContext - Supplies a place holder for an interrupt message table
        or an interrupt object to be filled on successful connection.

    InterruptConnectVersion - Supplies a place holder for an interrupt type on
        successful connection.

Return Value:

    STATUS_UNSUCCESSFUL when no interrupt resource is assigned to a bridge.

    STATUS_INVALID_PARAMETER if an interrupt resource assigned to a bridge
        is not compatible with the interrupt message number that the bridge
        wants to connect with.

    STATUS_SUCCESS on success.

--*/

{

    IO_CONNECT_INTERRUPT_PARAMETERS InterruptParameters;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Sanity check for resource lists.
    //

    if ((RawList == NULL) || (TranslatedList == NULL)) {
        return STATUS_UNSUCCESSFUL;
    }

    RtlZeroMemory(&InterruptParameters,
                  sizeof(IO_CONNECT_INTERRUPT_PARAMETERS));

    //
    // Initialize the interrupt connection parameter block for a call to
    // IoConnectInterruptEx.
    //

    InterruptParameters.Version = CONNECT_MESSAGE_BASED;
    InterruptParameters.MessageBased.PhysicalDeviceObject =
        ExpressBridge->PciBridge->Bus->PhysicalDeviceObject;

    InterruptParameters.MessageBased.ConnectionContext.Generic =
        ConnectionContext;

    InterruptParameters.MessageBased.MessageServiceRoutine =
        MessageServiceRoutine;

    InterruptParameters.MessageBased.ServiceContext = ExpressBridge;
    if (ARGUMENT_PRESENT(MessageRoutineSpinLock) != FALSE) {
        InterruptParameters.MessageBased.SpinLock = MessageRoutineSpinLock;

    } else {
        InterruptParameters.MessageBased.SpinLock = NULL;
    }

    InterruptParameters.MessageBased.SynchronizeIrql = 0;
    InterruptParameters.MessageBased.FloatingSave = FALSE;
    InterruptParameters.MessageBased.FallBackServiceRoutine =
        FallbackInterruptRoutine;

    Status = IoConnectInterruptEx(&InterruptParameters);
    if (NT_SUCCESS(Status)) {
        *InterruptConnectVersion = InterruptParameters.Version;
    }

    return Status;
}

VOID
ExpressBridgeHotPlugInterruptRoutine (
    __inout PEXPRESS_BRIDGE ExpressBridge,
    __inout PPCI_HOTPLUG_CONTROLLER HotPlugController,
    __deref_volatile __inout PLONG DelegatedWorkItems,
    __out PBOOLEAN AssertInterrupt,
    __out PBOOLEAN QueueDpc
    )

/*++

Routine Description:

    This is an interrupt service handler for hot plug events on an express
    bridge. This routine examines the slot control and status registers to
    determine the hot plug interrupts that have fired and acknowledges them.
    It will also update the field to indicate that the processing of these
    hot plug events are delegated to the DPC.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port
        or a downstream switch port.

    HotPlugController - Supplies a pointer to a hot plug controller for an
        express bridge.

    DelegatedWorkItems - Supplies a field to be updated if any work is delegated
        for execution at DISPATCH_LEVEL.

    AssertInterrupt - Supplies a pointer to a boolean that is updated in this
        routine to indicate that this interrupt was generated on the express
        bridge because of a hot plug event.

    QueueDpc - Supplies a pointer to a boolean that is updated in this routine
        to indicate a need for queuing a DPC.

Return Value:

    None.

--*/

{

    PCI_EXPRESS_SLOT_CONTROL_REGISTER ExpressSlotControlRegister;
    PCI_EXPRESS_SLOT_STATUS_REGISTER ExpressSlotStatusRegister;
    PCI_EXPRESS_SLOT_STATUS_REGISTER ExpressSlotStatusAckRegister;
    PPCI_HOTPLUG_SLOT Slot;

    NON_PAGED_CODE();

    *AssertInterrupt = FALSE;
    *QueueDpc = FALSE;
    Slot = &HotPlugController->Slots[0];
    RtlZeroMemory(&ExpressSlotStatusAckRegister,
                  sizeof(PCI_EXPRESS_SLOT_STATUS_REGISTER));

    //
    // Read the slot control and status registers.
    //

    PciReadExpressBridgeRegister(ExpressBridge,
                                 SlotControl,
                                 &ExpressSlotControlRegister);

    PciReadExpressBridgeRegister(ExpressBridge,
                                 SlotStatus,
                                 &ExpressSlotStatusRegister);

    if (ExpressSlotControlRegister.HotPlugInterruptEnable == 0) {
        return;
    }

    //
    // Check for command complete interrupt.
    //

    if ((ExpressSlotControlRegister.CommandCompletedEnable != 0) &&
        (ExpressSlotStatusRegister.CommandCompleted != 0)) {

        if (HotPlugController->CommandCompleteInterruptPending != FALSE) {
            HotPlugController->UnprocessedCommandCompleteInterrupts += 1;
        }

        HotPlugController->CommandCompleteInterruptPending = TRUE;
        ExpressSlotStatusAckRegister.CommandCompleted = 1;
    }

    //
    // Check for slot interrupts.
    //

    if ((ExpressSlotControlRegister.AttentionButtonEnable != 0) &&
        (ExpressSlotStatusRegister.AttentionButtonPressed != 0)) {

        Slot->SlotInterrupts |= (1 << SlotInterruptAttentionButton);
        ExpressSlotStatusAckRegister.AttentionButtonPressed = 1;
    }

    if ((ExpressSlotControlRegister.PowerFaultDetectEnable != 0) &&
        (ExpressSlotStatusRegister.PowerFaultDetected != 0)) {

        Slot->SlotInterrupts |= (1 << SlotInterruptPowerFault);
        ExpressSlotStatusAckRegister.PowerFaultDetected = 1;
    }

    if ((ExpressSlotControlRegister.MRLSensorEnable != 0) &&
        (ExpressSlotStatusRegister.MRLSensorChanged != 0)) {

        Slot->SlotInterrupts |= (1 << SlotInterruptLatchChange);
        ExpressSlotStatusAckRegister.MRLSensorChanged = 1;
    }

    if ((ExpressSlotControlRegister.PresenceDetectEnable != 0) &&
        (ExpressSlotStatusRegister.PresenceDetectChanged != 0)) {

        Slot->SlotInterrupts |= (1 << SlotInterruptPresenceDetect);
        ExpressSlotStatusAckRegister.PresenceDetectChanged = 1;
    }

    if ((ExpressSlotControlRegister.DataLinkStateChangeEnable != 0) &&
        (ExpressSlotStatusRegister.DataLinkStateChanged != 0)) {

        Slot->SlotInterrupts |= (1 << SlotInterruptLinkStateChange);
        ExpressSlotStatusAckRegister.DataLinkStateChanged = 1;
    }

    if (ExpressSlotStatusAckRegister.AsUSHORT != 0) {
        *AssertInterrupt = TRUE;
        *QueueDpc = TRUE;

        //
        // Update the flag for delegated work items.
        //

        InterlockedOr(DelegatedWorkItems, EXPRESS_HOT_PLUG_EVENT_HANDLING);

        //
        // Dismiss pending hot plug interrupts.
        //

        PciWriteExpressBridgeRegister(ExpressBridge,
                                      SlotStatus,
                                      &ExpressSlotStatusAckRegister);

    }

    return;
}

_Function_class_(KSYNCHRONIZE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL) // HIGH_LEVEL is best approximation of DIRQL
_IRQL_requires_same_
BOOLEAN
ExpressBridgeGetHotPlugInterrupts (
    _In_opt_ PVOID Context
    )

/*++

Routine Description:

    This routine gets outstanding hot plug interrupts for an express bridge so
    that they can be handled in a DPC. This must be called in the context of
    message routine spin lock so that the ISR can't modify these interrupt
    sources while they're being examined here.

Arguments:

    SynchronizationContext - Supplies a place holder to save information on the
    outstanding hot plug interrupts for a controller and hot plug slots.

Return Value:

    TRUE to indicate that this routine executed successfully at the SYNCH
        level.

--*/

{

    PPCI_HOTPLUG_CONTROLLER Controller;
    PPCI_HOTPLUG_INTERRUPTS HotPlugInterrupts;
    PPCI_INTERRUPT_SYNCHRONIZATION_CONTEXT SynchronizationContext;

    NON_PAGED_CODE();

    SynchronizationContext = (PPCI_INTERRUPT_SYNCHRONIZATION_CONTEXT)Context;
    _Analysis_assume_(SynchronizationContext != NULL);

    Controller = SynchronizationContext->HotPlugController;
    HotPlugInterrupts = &SynchronizationContext->HotPlugInterrupts;

    //
    // Save the interrupts pending information in the synchronization context
    // field.
    //

    HotPlugInterrupts->CommandCompleteInterrupt =
        Controller->CommandCompleteInterruptPending;

    if (Controller->Slots->SlotInterrupts != 0) {
        HotPlugInterrupts->SlotMask = 1;
        HotPlugInterrupts->SlotInterrupts[0] =
            Controller->Slots->SlotInterrupts;

    }

    //
    // Clear the interrupts pending information in the controller field.
    //

    Controller->CommandCompleteInterruptPending = FALSE;
    Controller->Slots->SlotInterrupts = 0;

    //
    // Clear the delegated work items field.
    //

    InterlockedAnd(SynchronizationContext->DelegatedWorkItems,
                   ~EXPRESS_HOT_PLUG_EVENT_HANDLING);

    return TRUE;
}

ULONG
ExpressBridgeGetHotPlugMessageNumber (
    __in PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine returns an interrupt message number for HotPlug
    events in an express bridge.

Arguments:

    ExpressBridge - Supplies a pointer to an Express Bridge for a root port or
        a downstream switch port.

Return Value:

    The zero-indexed message number for features supported by a root port or a
        downstream switch port.

--*/

{

    PAGED_CODE();

    PCI_ASSERT((ExpressBridge->BridgeType == PciExpressRootPort) ||
               (ExpressBridge->BridgeType == PciExpressDownstreamSwitchPort));

    return ExpressBridge->ExpressCapabilities.InterruptMessageNumber;
}

VOID
ExpressBridgeProcessAsyncHotPlugEvents (
    __in PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine is invoked to process hotplug status bits on an express
    bridge that were set before interrupts were enabled.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge for a root port or
        downstream switch port.

Return Value:

    None.

--*/

{

    PKINTERRUPT InterruptObject;
    BOOLEAN Result;
    PCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT SimulateHotPlugInterruptContext;

    NON_PAGED_CODE();

    //
    // Prepare a context for a routine that synchronizes its execution with an
    // ISR.
    //

    InterruptObject = NULL;
    RtlZeroMemory(&SimulateHotPlugInterruptContext,
                  sizeof(PCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT));

    SimulateHotPlugInterruptContext.ExpressBridge = ExpressBridge;
    switch(ExpressBridge->BridgeType) {
    case PciExpressRootPort:
        SimulateHotPlugInterruptContext.HotPlugController =
            ExpressBridge->Downstream.HotPlugController;

        SimulateHotPlugInterruptContext.DelegatedWorkItems =
            &ExpressBridge->Downstream.DelegatedWorkItems;

        //
        // Find the interrupt object.
        //

        if (ExpressBridge->Downstream.InterruptConnectVersion ==
            CONNECT_MESSAGE_BASED) {

            InterruptObject =
                ((PIO_INTERRUPT_MESSAGE_INFO)
                 ExpressBridge->Downstream.u.RootPort.ConnectionContext)->
                 MessageInfo[0].InterruptObject;

        } else {

            PCI_ASSERT(ExpressBridge->Downstream.InterruptConnectVersion ==
                       CONNECT_LINE_BASED);

            InterruptObject = 
                (PKINTERRUPT)ExpressBridge->Downstream.u.RootPort.ConnectionContext;
        }

        break;

    case PciExpressDownstreamSwitchPort:
        SimulateHotPlugInterruptContext.HotPlugController =
            ExpressBridge->Downstream.HotPlugController;

        SimulateHotPlugInterruptContext.DelegatedWorkItems =
            &ExpressBridge->Downstream.DelegatedWorkItems;

        //
        // Find the interrupt object.
        //

        InterruptObject = ExpressBridge->Downstream.u.SwitchPort.InterruptObject;
        break;

    default:

        PCI_ASSERT(FALSE);

        break;
    }

    Result = KeSynchronizeExecution(InterruptObject,
                                    ExpressBridgeSimulateHotPlugInterrupt,
                                    &SimulateHotPlugInterruptContext);

    PCI_ASSERT(Result != FALSE);
    UNREFERENCED_PARAMETER(Result);

    return;
}

_Function_class_(KSYNCHRONIZE_ROUTINE)
_IRQL_requires_(HIGH_LEVEL)
_IRQL_requires_same_
BOOLEAN
ExpressBridgeSimulateHotPlugInterrupt (
    _In_opt_ PVOID SynchronizeContext
    )

/*++

Routine Description:

    This routine is invoked to process the hotplug status bits on an express
    bridge, similar to an ISR. After acknowledging the status bits, a DPC is
    queued to complete processing of hotplug events.

Arguments:

    SynchronizeContext - Supplies a pointer to a context that contains
        elements for processing hotplug events.

Return Value:

    TRUE.

--*/

{

    PPCI_BUS Bus;
    BOOLEAN Dummy;
    PEXPRESS_BRIDGE ExpressBridge;
    BOOLEAN QueueDpcForHotPlug;
    PPCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT SimulateHotPlugInterruptContext;

    NON_PAGED_CODE();

    SimulateHotPlugInterruptContext =
        (PPCI_SIMULATE_HOTPLUG_INTERRUPT_SYNC_CONTEXT)SynchronizeContext;
    _Analysis_assume_(SimulateHotPlugInterruptContext != NULL);

    ExpressBridge = SimulateHotPlugInterruptContext->ExpressBridge;
    ExpressBridgeHotPlugInterruptRoutine(
        ExpressBridge,
        SimulateHotPlugInterruptContext->HotPlugController,
        SimulateHotPlugInterruptContext->DelegatedWorkItems,
        &Dummy,
        &QueueDpcForHotPlug);

    if (QueueDpcForHotPlug != FALSE) {
        Bus = ExpressBridge->PciBridge->Bus;
        IoRequestDpc(Bus->DeviceObject, NULL, ExpressBridge);
    }

    return TRUE;
}

VOID
ExpressBridgeClearSpuriousHotPlugStatus (
    __in PEXPRESS_BRIDGE ExpressBridge
    )

/*++

Routine Description:

    This routine is invoked to clear the spurious bits in the slot status
    register.

Arguments:

    ExpressBridge - Supplies a pointer to an express bridge.

Return Value:

    None.

--*/

{

    PCI_EXPRESS_SLOT_STATUS_REGISTER ExpressSlotStatusRegister;

    NON_PAGED_CODE();

    PCI_ASSERT((ExpressBridge->BridgeType == PciExpressRootPort) ||
               (ExpressBridge->BridgeType == PciExpressDownstreamSwitchPort));

    //
    // Read the slot status register and write back the same value.
    // This will clear the bits that are designated as W1C.
    //

    PciReadExpressBridgeRegister(ExpressBridge,
                                 SlotStatus,
                                 &ExpressSlotStatusRegister);

    PciWriteExpressBridgeRegister(ExpressBridge,
                                  SlotStatus,
                                  &ExpressSlotStatusRegister);

    return;
}
