/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    callbacks.c

Abstract:

    This file implements all the WDF callbacks registered by the GPIO class extension
    driver on behalf of the GPIO client driver with the driver framework. These
    routines operate on the GPIO client driver device.


Environment:

    Kernel mode

--*/

#pragma once

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"

#if defined(EVENT_TRACING)
#include "callbacks.tmh"         // auto-generated by WPP
#endif

#include "clientinvk.h"
#include "hub.h"

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(PAGE, GpioClxEvtDevicePrepareHardware)
#pragma alloc_text(PAGE, GpioClxEvtDeviceContextCleanup)
#pragma alloc_text(PAGE, GpioClxEvtDeviceFileCreate)
#pragma alloc_text(PAGE, GpioClxEvtFileClose)
#pragma alloc_text(PAGE, GpioClxEvtDeviceSelfManagedIoFlush)

//
// IO and IOCTL handlers.
//

#pragma alloc_text(PAGE, GpioClxEvtProcessReadRequest)
#pragma alloc_text(PAGE, GpioClxEvtProcessWriteRequest)
#pragma alloc_text(PAGE, GpioClxEvtProcessInternalDeviceIoControl)

//
// Bank-based handlers.
//

#pragma alloc_text(PAGE, GpioClxBankEvtProcessIoDefault)
#pragma alloc_text(PAGE, GpioClxBankEvtProcessInternalIoControl)
#pragma alloc_text(PAGE, GpioClxBankEvtProcessIoCanceledOnQueue)

//
// -------------------------------------------------------------- WDF callbacks
//

NTSTATUS
GpioClxEvtDeviceQueryStop (
    __in WDFDEVICE Device
    )

/*++

Routine Description:

    This routine is invoked by the framework to determine whether the device
    can be stopped for rebalancing resources.


        The routine always returns "no" since rebalance is not yet supported.

    N.B. 1. The device may be in D0 or D3.

         2. This routine is not marked PAGED as it may be called before/after
            the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    Device - Supplies a handle to the framework device object.

Return Value:

    Failing NTSTATUS code.

--*/

{

    UNREFERENCED_PARAMETER(Device);

    //
    // MSDN says to return any failure code except STATUS_NOT_SUPPORTED.
    //

    return STATUS_UNSUCCESSFUL;
}

NTSTATUS
GpioClxEvtDevicePrepareHardware (
    __in WDFDEVICE Device,
    __in WDFCMRESLIST ResourcesRaw,
    __in WDFCMRESLIST ResourcesTranslated
    )

/*++

Routine Description:

    This routine is called by the framework when the PnP manager sends an
    IRP_MN_START_DEVICE request to the driver stack. This routine is
    responsible for performing operations that are necessary to make the
    driver's device operational (for e.g. mapping the hardware resources
    into memory).

    The GPIO class extension forwards this request to the GPIO client driver.
    The client driver will look at the raw & translated resources and discovers
    information like where different registers are located. Note the device
    is not yet considered as started (or in DO). That happens after D0Entry
    callback is invoked.


Arguments:

    Device - Supplies a handle to a framework device object.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

Return Value:

    NT status code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    PAGED_CODE();

    Status = STATUS_SUCCESS;

    //
    // The GPIO class extension itself doesn't use any hardware resources and
    // thus simply calls into the GPIO client driver to prepare the device.
    // The GPIO will map the resources allocated to it and take care of any
    // initialization.
    //

    GpioExtension = GpioClxGetDeviceExtension(Device);
    GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_INITIALIZING);
    Status = GpioClnInvokePrepareController(Device,
                                            GpioExtension,
                                            ResourcesRaw,
                                            ResourcesTranslated);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioEvtDevicePrepareHardware: Client failed initialization"
                    "! Status = %#x, Extension = %p\n",
                    Status,
                    GpioExtension);

        goto DevicePrepareHardwareEnd;
    }

    //
    // Query the GPIO controller for its attributes.
    //

    Status = GpiopQueryControllerInformation(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDevicePrepareHardware: Client failed controller "
                    "attributes query! Status = %#x\n",
                    Status);

        goto DevicePrepareHardwareEnd;
    }

    //
    // Set the idle power policy (i.e. D-state power management).
    //

    Status = GpiopInitializeDxIdlePowerPolicy(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDevicePrepareHardware: Failed to "
                    "set Dx Idle power policy! Status = %#x\n",
                    Status);

        goto DevicePrepareHardwareEnd;
    }

    //
    // Now that the client driver got its resources and performed some basic
    // device initialization, query it for the interrupt and IO ranges that
    // it supports.
    //

    Status = GpiopInitializeBanksAndInterrupts(GpioExtension,
                                               ResourcesTranslated,
                                               ResourcesRaw);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDevicePrepareHardware: Client failed to return "
                    " interrupt and IO information! Status = %#x\n",
                    Status);

        goto DevicePrepareHardwareEnd;
    }

    //
    // Now that the GPIO controller information has been successfully
    // retrieved and interrupt resources allocated, allocate the resources
    // to manage ACPI events on the controller.
    //

    GpiopManageAcpiEventing(GpioExtension,
                            AcpiEventStateAllocateResources,
                            FALSE);

    //
    // Update the device state.
    //

DevicePrepareHardwareEnd:
    if (!NT_SUCCESS(Status)) {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_INITIALIZE_FAILED);
        GpioExtension->ClientErrorStatus = Status;

    } else {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_INITIALIZED);
    }

    return Status;
}

NTSTATUS
GpioClxEvtDeviceReleaseHardware (
    __in WDFDEVICE Device,
    __in WDFCMRESLIST ResourcesTranslated
    )

/*++

Routine Description:

    This routine is called by the framework when the PnP manager is revoking
    ownership of our resources. This may be in response to either
    IRP_MN_STOP_DEVICE or IRP_MN_REMOVE_DEVICE. This routine is responsible for
    performing cleanup of resources allocated in PrepareHardware callback.
    This callback is invoked before passing  the request down to the lower
    driver.

    This routine will also be invoked by the framework if the prepare hardware
    callback returns a failure.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    Device - Supplies a handle to a framework device object.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

Return Value:

    NT status code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(ResourcesTranslated);

    Status = STATUS_SUCCESS;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // Cancel any pending and future operations that were sent to the "self
    // target". These operations may include enable/disable of ACPI eventing
    // pins. Since the device is stopped, these operations may be pending in
    // the WDF queue.
    //
    //

    WdfIoTargetPurge(GpioExtension->SelfIoTarget, WdfIoTargetPurgeIoAndWait);

    //
    // Release the resources that were allocated to manage ACPI events on
    // the controller.
    //

    GpiopManageAcpiEventing(GpioExtension,
                            AcpiEventStateReleaseResources,
                            TRUE);

    //
    // The uninitialize routine should only be invoked if the client driver was
    // successfully initialized earlier.
    //

    if (GpioExtension->DeviceState >= DEVICE_STATE_INITIALIZED) {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_UNINITIALIZING);
        Status = GpioClnInvokeReleaseController(Device, GpioExtension);
    }

    if (!NT_SUCCESS(Status)) {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_UNINITIALIZE_FAILED);
        GpioExtension->ClientErrorStatus = Status;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioEvtDeviceReleaseHardware: "
                    "GpioClnInvokeUninitializeDevice() failed! "
                    "Status = %#x\n", Status);

        goto DeviceReleaseHardwareEnd;
    }

    //
    // Update the device state.
    //

    GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_UNINITIALIZED);

DeviceReleaseHardwareEnd:
    return Status;
}

NTSTATUS
GpioClxEvtDeviceD0Entry (
    __in WDFDEVICE Device,
    __in WDF_POWER_DEVICE_STATE PreviousPowerState
    )

/*++

Routine Description:

    This routine is invoked by the framework to program the device to goto
    D0, which is the working state. The framework invokes callback every
    time the hardware needs to be (re-)initialized.  This includes after
    IRP_MN_START_DEVICE, IRP_MN_CANCEL_STOP_DEVICE, IRP_MN_CANCEL_REMOVE_DEVICE,
    and IRP_MN_SET_POWER-D0.

    This routine calls the GPIO client driver's StartDevice callback. A GPIO
    client driver is expected to program the device to its initialization state.

    N.B. 1. Registration with the power framework is deferred until the
            EvtDeviceSelfManagedIoInit callback for reasons mentioned in
            that routine.

         2. This function is not marked pageable because this function is in
            the device power up path. When a function is marked pagable and the
            code section is paged out, it will generate a page fault which could
            impact the fast resume behavior because the client driver will have
            to wait until the system drivers can service this page fault.

Arguments:

    Device - Supplies a handle to the framework device object.

    PreviousPowerState - Supplies the device power state that the device was in
        before this transition to D0.

Return Value:

    NTSTATUS code. A failure here will indicate a fatal error and cause the
    framework to tear down the stack.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN RestoreContext;
    NTSTATUS Status;

    Status = STATUS_SUCCESS;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // If the device is transitioning into D0 as part of a Dx IRP, then
    // request the client driver to restore the saved context. Otherwise,
    // request it to start the device.
    //
    // The should be in a non-D0 state when this callback is invoked.
    //

    GPIO_ASSERT(PreviousPowerState != WdfPowerDeviceD0);

    if (PreviousPowerState == WdfPowerDeviceD3Final) {
        RestoreContext = FALSE;

    } else {
        GPIO_ASSERT(PreviousPowerState == WdfPowerDeviceD3);

        RestoreContext = TRUE;
    }

    //
    // Invoke the GPIO client driver's start device callback to get the device
    // started.
    //

    GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_STARTING);
    Status = GpioClnInvokeStartController(GpioExtension,
                                          RestoreContext,
                                          PreviousPowerState);

    if (!NT_SUCCESS(Status)) {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_START_FAILED);
        GpioExtension->ClientErrorStatus = Status;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDeviceD0Entry: Client failed to start the "
                    "device! Status = %#x\n",
                    Status);

        goto DeviceD0EntryEnd;
    }

    //
    // Update the device state and device power state.
    //

    GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_STARTED);
    GPIO_SET_DEVICE_POWER_STATE(GpioExtension, PowerDeviceD0);

DeviceD0EntryEnd:
    if (!NT_SUCCESS(Status)) {
        GPIO_SET_IGNORE_PO_FX_CALLBACKS(GpioExtension);
    }

    return Status;
}

NTSTATUS
GpioClxEvtDeviceD0Exit (
    __in WDFDEVICE Device,
    __in WDF_POWER_DEVICE_STATE TargetState
    )

/*++

Routine Description:

    This routine is invoked by the framework to program the device to go into
    a certain Dx state. The framework invokes this callback every
    time the device is leaving the D0 state, which happens when the device is
    stopped, when it is removed, and when it is powered off.

    This routine calls the GPIO client driver's StopDevice callback. A GPIO
    client driver is expected to program the device to its initialization state.

    N.B. 1. The device is still in D0 state when this callback is invoked and
            thus can touch hardware state. If the device has been removed or
            surprise-removed, then this callback is called after the device
            has really gone away. The target state would be
            WdfPowerDeviceD3Final in which the client driver wouldn't really
            need to touch the hardware.

         2. Interrupts have already been disabled by the time that this callback
            is invoked.

         3. This routine is not marked PAGED as it may be called before/after
            the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    Device - Supplies a handle to the framework device object.

    TargetState - Supplies the device power state which the device will be put
        in once the callback is complete.

Return Value:

    NTSTATUS code. A failure here will indicate a fatal error and cause the
    framework to tear down the stack.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    GPIO_DEVICE_STATE OldState;
    WDF_DEVICE_POWER_STATE PowerState;
    BOOLEAN SaveContext;
    NTSTATUS Status;

    Status = STATUS_SUCCESS;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // The device should be in D0 state when this callback is invoked.
    //

    GPIO_ASSERT(GpioExtension->DevicePowerState == PowerDeviceD0);

    //
    // Currently only D3 (off) is supported. If the device is going into
    // D3 as part of a Dx IRP, then request the client driver to save its
    // context.
    //

    if (TargetState == WdfPowerDeviceD3Final) {

        //
        // D0->D3->D0:
        //
        // If EvtDeviceD0Entry succeeded but WDF failed to connect a bank's
        // WDF interrupt object afterwards, EvtDeviceD0Exit will be called
        // pretty much immediately with this failed power state.
        //

        PowerState = WdfDeviceGetDevicePowerState(Device);
        if ((PowerState == WdfDevStatePowerWakingConnectInterruptFailed) ||
            (PowerState == WdfDevStatePowerWakingConnectInterruptFailedNP)) {

            GPIO_SET_IGNORE_PO_FX_CALLBACKS(GpioExtension);
        }

        SaveContext = FALSE;

    } else {
        GPIO_ASSERT(TargetState == WdfPowerDeviceD3);

        SaveContext = TRUE;
    }

    //
    // Invoke the GPIO client driver's start device callback to get the device
    // started. If the device was never fully started, then skip invoking the
    // callback.
    //

    OldState = GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_STOPPING);
    Status = STATUS_SUCCESS;
    if (OldState == DEVICE_STATE_STARTED) {
        Status = GpioClnInvokeStopController(GpioExtension,
                                             SaveContext,
                                             TargetState);
    }

    if (!NT_SUCCESS(Status)) {
        GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_STOP_FAILED);
        GpioExtension->ClientErrorStatus = Status;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDeviceD0Exit: Client failed to stop the "
                    "device! Status = %#x\n",
                    Status);


        goto DeviceD0ExitEnd;
    }

    //
    // Update the device state and device power state.
    //

    GPIO_SET_DEVICE_STATE(GpioExtension, DEVICE_STATE_STOPPED);
    GPIO_SET_DEVICE_POWER_STATE(GpioExtension, TargetState);

DeviceD0ExitEnd:
    if (!NT_SUCCESS(Status)) {
        GPIO_SET_IGNORE_PO_FX_CALLBACKS(GpioExtension);
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpioClxEvtDevicePreInterruptsDisabled (
    __in WDFDEVICE Device,
    __in WDF_POWER_DEVICE_STATE TargetState
    )

/*++

Routine Description:

    This routine is invoked by the framework prior to disabling interrupts
    on the device. This routine disconnects all the ACPI event pins and then
    forwards the call to the client driver.

    This routine is also called in cases where the system or device is entering
    a new power state (e.g S5 -> D3) or the GPIO controller is going away or
    already gone away (e.g. surprise-remove). In such cases, all connected
    pins (either for interrupt or IO) have to cleaned up.

Arguments:

    Device - Supplies a handle to the framework device object.

    TargetState - Supplies the device power state which the device will be put
        in once the callback is complete.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Return Value:

    NTSTATUS code. A failure here will indicate a fatal error and cause the
    framework to tear down the stack.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN InSurpriseRemovalContext;
    WDF_POWER_DEVICE_STATE NewPowerState;
    WDF_POWER_DEVICE_STATE OldPowerState;
    POWER_ACTION PowerAction;
    NTSTATUS Status;

    GpioExtension = GpioClxGetDeviceExtension(Device);
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_PNP,
                "%s: Extn = %p\n",
                __FUNCTION__,
                GpioExtension);

    //
    // Perform power-related pre-processing before interrupts are disabled.
    //

    GpiopHandlePowerPreInterruptsDisabled(Device);

    //
    // If the device is entering its "final D3" state, then this GPIO controller
    // is going away (S5, removal, surprise-removal or stop). In such case,
    // perform all the cleanup of the pending interrupt and IO.
    //
    // 1. This operation has to be done synchronously to prevent the controller
    //    from being stopped while there are events to disconnect.
    //
    // 2. This operation is done here rather than in the D0Exit callback
    //    because WDF requires WdfInterruptReleaseLock() not be called once
    //    the interrupts have been disconnected. Disconnecting interrupts
    //    generally requires the interrupt lock to be acquired/released in the
    //    client driver.
    //

    if (TargetState == WdfPowerDeviceD3Final) {
        PowerAction = WdfDeviceGetSystemPowerAction(Device);

        //
        // Temporarily set a new device power state. The state is "D3Final"
        // for cases where the device is still present (e.g. shutdown) and
        // "invalid" for cases where the device is going away or already gone
        // away (surprise-removed).
        //
        // This is to force all requests to be directly dispatched to the driver
        // rather than being put on the queue during cleanup.
        //
        // GpioClxWdmPreprocessPowerIrp() won't ever be called simultaneously
        // so it won't be confused by this temporary set.
        //

        if (PowerAction == PowerActionNone) {
            InSurpriseRemovalContext = TRUE;
            NewPowerState = WdfPowerDeviceInvalid;

        } else {

            //
            // e.g. shutdown
            //

            InSurpriseRemovalContext = FALSE;
            NewPowerState = WdfPowerDeviceD3Final;
        }

        OldPowerState = GPIO_SET_DEVICE_POWER_STATE(GpioExtension,
                                                    NewPowerState);

        //
        // If the system is shutting down or rebooting, disable ACPI events
        // (note that they have already been deep-masked by
        //  GpioClxWdmPreprocessPowerIrp()).
        //
        // Do nothing for:
        //
        // 1. Surprise Removal: The hardware is already gone.
        //
        // 2. Removal & Stop: These IRPs can only be received if no ACPI
        //    events are enabled anyway, as they are blocked by
        //    WdfDeviceSetStaticStopRemove() if any events are enabled.
        //
        //

        if (InSurpriseRemovalContext == FALSE) {
            GpiopManageAcpiEventing(GpioExtension, AcpiEventStateDisable, TRUE);
        }

        GpiopCleanupAllBanksInterruptIo(GpioExtension, InSurpriseRemovalContext);

        //
        // Restore the previous state. It will be set to D3 final in the D0 exit
        // callback.
        //

        GPIO_SET_DEVICE_POWER_STATE(GpioExtension, OldPowerState);

    } else {

        //
        // Do nothing for D3 that is in response to idle, S3-sleep or hibernate.
        //
        //

    }

    //
    // Invoke the (non-public) client driver supplied pre-interrupt disable
    // callback, if any.
    //

    Status = GpioClnInvokePreControllerInterruptDisabledCallback(GpioExtension);
    return Status;
}

NTSTATUS
GpioClxEvtDevicePostInterruptsEnabled (
    __in WDFDEVICE Device,
    __in WDF_POWER_DEVICE_STATE PreviousState
    )

/*++

Routine Description:

    This routine is invoked by the framework post enabling the interrupts
    on the device. This routine connects all the ACPI event pins and then
    forwards the call to the client driver.

    This routine is not marked pageable since it is in the power-up path.

    N.B. If this is the first D0 transition, ACPI events will not be enabled
         until the first IRP_MN_QUERY_DEVICE_RELATIONS is sent by PnP manager.

         As for subsequent D0 transitions, consider the following cases:

         1. Rebalance: This needs to be considered once it's implemented.

         2. Other WdfDevicePowerD3Final cases (Shutdown, Surprise Removal,
            Removal): There cannot be a subsequent D0 transition.

         3. Non-WDFDevicePowerD3Final cases (D0 -> D1/D2/D3 -> D0):

            a) Device Idle: The D0 exit will not disable the ACPI events that
               were enabled by IRP_MN_QUERY_DEVICE_RELATIONS.

               Note that any associated F0 exit and F0 re-entry will ensure that
               any hardware context required for ACPI event interrupts will be
               transparently saved and restored (using
               CLIENT_SaveBankHardwareContext & CLIENT_RestoreBankHardwareContext).

            b) S3 & S4: The D0 exit masked the ACPI event pins, so we unmask
               them here.

         Therefore, there is no need to enable or re-enable ACPI events in this
         routine.

Arguments:

    Device - Supplies a handle to the framework device object.

    PreviousState - Supplies the previous device power state.

Return Value:

    NTSTATUS code. A failure here will indicate a fatal error and cause the
    framework to tear down the stack.

--*/

{

    BOOLEAN WokeFromSx;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    ULONG Index;
    BOOLEAN InitialD0Entry;
    NTSTATUS Status;

    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // If D0 transition completion needs to be reported to power framework,
    // then do so now. This flag is set if this callback is being invoked
    // post Sx -> S0 system state transition. In such cases, the D0
    // transition completion needs to be notified to the power framework.
    //

    WokeFromSx = GpioExtension->WokeFromSx;
    GpioExtension->WokeFromSx = FALSE;
    if (WokeFromSx != FALSE) {

        GPIO_ASSERT(GpioExtension->PowerHandle != NULL);

        PoFxReportDevicePoweredOn(GpioExtension->PowerHandle);

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_POWER,
                    "%s - PoFxReportDevicePoweredOn() called! Extn = %p\n",
                    __FUNCTION__,
                    GpioExtension);
    }

    //
    // If this is the very first D0 transition, then walk through all the
    // banks and mark the bank interrupt (DIRQL interrupt) as inactive if it
    // is being actively managed.
    //
    // N.B. No interrupts (from peripheral drivers) should have been connected
    //      until this point.
    //

    InitialD0Entry = GpioExtension->InitialD0Entry;
    if ((InitialD0Entry != FALSE) &&
        (GpioExtension->ActivelyManageBankInterrupt != FALSE)) {

        for (Index = 0; Index < GpioExtension->TotalBanks; Index += 1) {
            GpioBank = &GpioExtension->Banks[Index];

            GPIO_ASSERT(GpioBank->DebounceInterruptCount == 0);

            GpiopSetInterruptState(GpioBank, GpioBank->BankInterrupt, FALSE);
        }
    }

    GpiopHandlePowerPostInterruptsEnabled(Device, PreviousState);

    //
    // If waking from S3/S4, unmask ACPI events.
    //

    if (WokeFromSx != FALSE) {
        GpiopManageAcpiEventing(GpioExtension,
                                AcpiEventStateDeepUnmaskForSx,
                                TRUE);
    }

    //
    // Invoke the (non-public) client driver supplied post-interrupt disable
    // callback, if any.
    //

    Status = GpioClnInvokePostControllerInterruptEnabledCallback(GpioExtension);
    return Status;
}

VOID
GpioClxEvtDeviceFileCreate (
    __in WDFDEVICE Device,
    __in WDFREQUEST Request,
    __in WDFFILEOBJECT FileObject
    )

/*++

Routine Description:

    This routine is invoked by the framework when it receives an IRP_MJ_CREATE
    request (when a kernel-mode or user-mode driver tries to open handle to
    the device).

    This callback is called in the context of the thread that created the
    IRP_MJ_CREATE request.

    N.B. Only requests generated by a kernel mode driver or a UMDF driver
         would be allowed due to ACL restrictions on the device object
         [refer to GpioClxProcessAddDevicePreDeviceCreate()].

Arguments:

    Device - Supplies a handler to the framework device object.

    Request - Supplies the request object associated with this create operation.

    FileObject - Supplies a pointer to fileobject that represents the open
        handle.

Return Value:

   None.

--*/

{

    BOOLEAN CompleteRequest;
    GPIO_CONNECT_IO_PINS_MODE ConnectMode;
    LARGE_INTEGER ConnectionId;
    PGPIO_FILE_OBJECT_CONTEXT FileContext;
    PUNICODE_STRING FileName;
    BOOLEAN FunctionConfig;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    UCHAR IoRestriction;
    BOOLEAN ReadWriteMode;
    BOOLEAN ReleasePowerReference;
    BOOLEAN ReleaseBankPowerReference;
    WDF_REQUEST_PARAMETERS RequestParameters;
    NTSTATUS Status;

    PAGED_CODE();

    CompleteRequest = TRUE;
    GpioBank = NULL;
    ReleasePowerReference = FALSE;
    ReleaseBankPowerReference = FALSE;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // Even if D0 entry had previously failed, it is still possible to receive
    // a create request.
    //

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "GpioClxEvtDeviceFileCreate: D0 had previously failed\n");

        Status = STATUS_UNSUCCESSFUL;
        goto EvtDeviceFileCreateEnd;
    }

    //
    // Request power manager to make the component active. Since there is an
    // explicit EvtFileCreate() handler registered, WDF dispatches it to the
    // driver directly. The request does not come from the power managed queue.
    // The request may arrive while the device has been put into an idle
    // D-state.  Request the WDF framework to synchronously bring the device back
    // into D0 prior to activating the component.
    //
    // If the D0 entry request succeeds, the device will remain in D0 until
    // the create request control flow calls WdfDeviceResumeIdle. No unrelated
    // code on another thread will attempt to call WdfDeviceResumeIdle (without
    // first calling WdfDeviceStopIdle), since that would result in
    // WdfDeviceResumeIdle being called one time too many.
    //

    Status = WdfDeviceStopIdle(Device, TRUE);

    //
    // If the D0 entry request failed, WDF may have started tearing down the
    // device asynchronously (on a thread other than the current one).
    //

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "GpioClxEvtDeviceFileCreate: D0 entry failed! "
                    "Status=%#x\n",
                    Status);

        Status = STATUS_UNSUCCESSFUL;
        goto EvtDeviceFileCreateEnd;
    }

    GPIO_ASSERT(GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) == FALSE);

    ReleasePowerReference = TRUE;

    //
    // Get the parameters associated with the create request.
    //

    WDF_REQUEST_PARAMETERS_INIT(&RequestParameters);
    WdfRequestGetParameters(Request, &RequestParameters);

    GPIO_ASSERT(RequestParameters.Type == WdfRequestTypeCreate);

    //
    // Check if the request is for opening pins or some other type (e.g. query
    // interface). If it is not for opening pins (indicated by no trailing
    // name), then simply succeed the request.
    //

    FileName = WdfFileObjectGetFileName(FileObject);
    if ((FileName == NULL) || (FileName->Length == 0)) {
        Status = STATUS_SUCCESS;
        goto EvtDeviceFileCreateEnd;
    }

    Status = GpioUtilGetIdFromFileName(FileName, &ConnectionId);
    if (!NT_SUCCESS(Status)) {
        goto EvtDeviceFileCreateEnd;
    }

    Status = GpiopQueryBankIdFromGsivOrConnectionId(
                 GpioExtension,
                 FALSE,
                 ConnectionId.QuadPart,
                 &GpioBank,
                 &IoRestriction,
                 &FunctionConfig);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "%!FUNC!: Failed to map request to bank! Status = %#x.\n",
                    Status);

        goto EvtDeviceFileCreateEnd;
    }

    //
    // Determine whether the request is for read (input), write (output) or
    // read-write (input/output).
    //
    // For pins opened in read-write mode, the initial mode will be mapped
    // to output unless the descriptor specifies an input-only restriction.
    // Output is chosen because the connect operation is delayed until an
    // actual write is issued (delayed write). Thus it is a non-destructive
    // operation from the controller perspective.
    //

    ConnectMode = GpioUtilGetOpenModeFromRequest(&RequestParameters,
                                                 IoRestriction,
                                                 &ReadWriteMode);

    if (ConnectMode == ConnectModeInvalid) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "GpioClxEvtDeviceFileCreate: Invalid connect mode = %#x!\n",
                    ConnectMode);

        Status = STATUS_INVALID_PARAMETER;
        goto EvtDeviceFileCreateEnd;
    }

    PoFxActivateComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0);
    ReleaseBankPowerReference = TRUE;

    //
    // Initialize the file context.
    //

    FileContext = GpioClxGetFileObjectContext(FileObject);
    RtlZeroMemory(FileContext, sizeof(GPIO_FILE_OBJECT_CONTEXT));
    FileContext->Flags.ReadWriteSupported = ReadWriteMode;
    FileContext->BankId = GpioBank->BankId;
    FileContext->GpioExtension = GpioExtension;
    FileContext->ConnectMode = ConnectMode;
    FileContext->ConnectionId = ConnectionId;
    FileContext->Flags.FunctionConfig = FunctionConfig;
    InitializeListHead(&FileContext->ListEntry);

    //
    // Connect the pins in the specified mode. Queue the request to the
    // secondary queue for the respective bank. The request will be dispatched
    // from that queue when the component becomes active.
    //
    // Note if the bank is already active, then request would be dispatched
    // immediately.
    //

    Status = WdfRequestForwardToIoQueue(Request, GpioBank->IoQueue);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "%!FUNC!: Failed to forward request to bank queue! "
                    "Bank ID = 0x%x, Status = %#x.\n",
                    GpioBank->BankId,
                    Status);

        goto EvtDeviceFileCreateEnd;
    }

    //
    // Hold onto the request as it has not yet been processed. The
    // power reference also needs to be held until the request has been
    // processed.
    //

    CompleteRequest = FALSE;
    ReleasePowerReference = FALSE;
    ReleaseBankPowerReference = FALSE;

EvtDeviceFileCreateEnd:
    if (CompleteRequest != FALSE) {
        WdfRequestComplete(Request, Status);
    }

    //
    // Release the power reference that was previously taken on the
    // component, if requested.
    //

    if (ReleaseBankPowerReference != FALSE) {

        GPIO_ASSERT(ReleaseBankPowerReference != FALSE);

        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);
    }

    if (ReleasePowerReference != FALSE) {
        WdfDeviceResumeIdle(Device);
    }

    return;
}

VOID
GpioClxEvtFileClose (
    __in WDFFILEOBJECT FileObject
   )

/*++

Routine Description:

    This routine is invoked by the framework when it receives an IRP_MJ_CLOSE
    request. This is called just before the file object is about to be
    destroyed. The I/O Manager sends an IRP_MJ_CLOSE request for a file object
    when both of the following are true:

     - All handles to the file object are closed.
     - No outstanding references to the object, such as those caused by a
       pending IRP, remain.

    The file object is destroyed after the close. CLOSE follows (sometime after)
    CLEANUP.

Arguments:

    FileObject - Supplies a pointer to fileobject that represents the open
        handle.

Return Value:

   None.

--*/

{

    WDFDEVICE Device;
    PGPIO_FILE_OBJECT_CONTEXT FileContext;
    PUNICODE_STRING FileName;
    PDEVICE_EXTENSION GpioExtension;
    LONG ReferenceCount;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Check if the request was associated with some pins or a general request
    // (like querying interface etc.). If it is not for opening pins (indicated
    // by no trailing name), then simply succeed the request.
    //

    Device = WdfFileObjectGetDevice(FileObject);
    FileName = WdfFileObjectGetFileName(FileObject);
    GpioExtension = GpioClxGetDeviceExtension(Device);
    if ((FileName == NULL) || (FileName->Length == 0)) {
        goto EvtFileCloseEnd;
    }

    FileContext = GpioClxGetFileObjectContext(FileObject);
    if (FileContext == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "GpioClxEvtFileClose: Failed to get file context!\n");

        goto EvtFileCloseEnd;
    }

    //
    // Check the type of pin being disconnected and call the appropriate
    // disconnect function. There are two types of pins managed by GPIOCLX:
    //
    // 1. GPIO IO pins
    // 2. Function config pins
    //

    if (FileContext->Flags.FunctionConfig != 0) {
        Status = GpiopDisconnectFunctionConfigPins(GpioExtension,
                                                   FileContext,
                                                   FALSE);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_CREATE_CLOSE,
                        "GpioClxEvtFileClose: Failed to disconnect function "
                        "config pins! Status = %#x\n",
                        Status);
        }

    } else {
        Status = GpiopDisconnectGpioPins(GpioExtension,
                                         FileContext,
                                         FALSE);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_CREATE_CLOSE,
                        "GpioClxEvtFileClose: Failed to disconnect pins! Status"
                        " = %#x\n",
                        Status);
        }

        //
        // If the reference count on the Fast I/O interface is not zero, then
        // someone can still try to call the functions in the interface. Fail here.
        //

        ReferenceCount = InterlockedCompareExchange(
                            &FileContext->FastIoInterfaceReferenceCount,
                            0,
                            0);

        if (ReferenceCount != 0) {
            KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                         FAST_IO_INTERFACE_REFERENCE_COUNT_NOT_ZERO,
                         (ULONG_PTR)GpioExtension,
                         (ULONG_PTR)FileContext,
                         0);
        }
    }

EvtFileCloseEnd:
    return;
}

VOID
GpioClxEvtProcessReadRequest (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t Length
    )

/*++

Routine Description:

    This routine handles read on the GPIO client driver device. This callback is
    invoked by WDF when the framework receives IRP_MJ_READ requests. Since
    read on GPIO pins is handled via the IOCTL_GPIO_READ_PINS, this routine
    is not supported.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    Length - Supplies the length of the data buffer associated with the request.

    N.B. The default property of the queue is to not dispatch zero length read
         and write requests to the driver and complete is with status success.
         So this routine will never get a zero length request.

Return Value:

  None.

--*/

{

    UNREFERENCED_PARAMETER(Queue);
    UNREFERENCED_PARAMETER(Request);
    UNREFERENCED_PARAMETER(Length);

    PAGED_CODE();

    WdfRequestComplete(Request, STATUS_NOT_SUPPORTED);
    return;
}

VOID
GpioClxEvtProcessWriteRequest (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t Length
    )

/*++

Routine Description:

    This routine handles write on the GPIO client driver device. This callback is
    invoked by WDF when the framework receives IRP_MJ_WRITE requests. Since
    write on GPIO pins is handled via the IOCTL_GPIO_WRITE_PINS, this routine
    is not supported.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    Length - Supplies the length of the data buffer associated with the request.

    N.B. The default property of the queue is to not dispatch zero length read
         and write requests to the driver and complete is with status success.
         So this routine will never get a zero length request.

Return Value:

  None.

--*/

{

    UNREFERENCED_PARAMETER(Queue);
    UNREFERENCED_PARAMETER(Request);
    UNREFERENCED_PARAMETER(Length);

    PAGED_CODE();

    WdfRequestComplete(Request, STATUS_NOT_SUPPORTED);
    return;
}

VOID
GpioClxEvtProcessDeviceIoControl (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t OutputBufferLength,
    __in size_t InputBufferLength,
    __in ULONG IoControlCode
    )

/*++

Routine Description:

    This routine handles IOCTL requests sent to the GPIO client driver device. This
    callback is invoked by WDF when the framework receives IRP_MJ_DEVICE_CONTROL
    requests.

    N.B. 1. Since the device's WDF_OBJECT_ATTRIBUTES structure is set to
            WdfExecutionLevelPassive, the framework will always send this
            callback at IRQL < DISPATCH_LEVEL.

         2. Only requests generated by a kernel mode driver or a UMDF driver
            would be allowed due to ACL restrictions on the device object
            [refer to GpioClxProcessAddDevicePreDeviceCreate()].

         3. Note this routine is not marked PAGED as READ/WRITE IOCTLs may be
            issued before/after the boot device is in D0/D3 if boot device has
            GPIO dependencies.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    OutputBufferLength - Supplies the length of the request's output buffer, if
        an output buffer is available.

    InputBufferLength - Supplies the length of the request's input buffer, if an
        input buffer is available.

    IoControlCode - Supplies the driver-defined or system-defined I/O control
        code (IOCTL) that is associated with the request.

Return Value:

  None.

--*/

{

    BOOLEAN CompleteRequest;
    ULONG BytesReadWritten;
    PGPIO_FILE_OBJECT_CONTEXT FileContext;
    WDFFILEOBJECT FileObject;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(OutputBufferLength);
    UNREFERENCED_PARAMETER(InputBufferLength);

    //
    // Get the GPIO device extension from the queue handle.
    //

    GpioExtension = GpioClxGetDeviceExtension(WdfIoQueueGetDevice(Queue));

    GPIO_CLX_VALIDATE_SIGNATURE(GpioExtension);

    BytesReadWritten = 0;
    CompleteRequest = TRUE;

    //
    // Examine the IOCTL code and dispatch the request to the appropriate
    // handler.
    //

    FileContext = NULL;
    switch(IoControlCode) {

    case IOCTL_GPIO_READ_PINS:
    case IOCTL_GPIO_WRITE_PINS:
    case IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS:

        FileObject = WdfRequestGetFileObject(Request);
        if (FileObject != NULL) {
            FileContext = GpioClxGetFileObjectContext(FileObject);
        }

        if (FileContext == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%!FUNC!: Failed to get file "
                        "context from request! Request = %p\n",
                        Request);

            Status = STATUS_NOT_SUPPORTED;
            goto ProcessDeviceIoControlEnd;
        }

        //
        // Check IOCTL is sent on the correct type of connection
        //

        if (IoControlCode == IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS) {
            if (FileContext->Flags.FunctionConfig != 1) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_IO,
                            "%!FUNC!: IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS"
                            " sent to a GPIO resource descriptor!"
                            " Request = %p\n",
                            Request);

                Status = STATUS_NOT_SUPPORTED;
                goto ProcessDeviceIoControlEnd;
            }

        } else {
            if (FileContext->Flags.FunctionConfig == 1) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_IO,
                            "%!FUNC!: IOCTL_GPIO_READ_PINS/IOCTL_GPIO_WRITE_PINS sent"
                            " to a function config resource descriptor!"
                            " Request = %p\n",
                            Request);

                Status = STATUS_NOT_SUPPORTED;
                goto ProcessDeviceIoControlEnd;
            }
        }

        GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

        GPIO_ASSERT(GpioBank != NULL);

        Status = WdfRequestForwardToIoQueue(Request, GpioBank->IoQueue);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_CREATE_CLOSE,
                        "%!FUNC!: Failed to forward request to bank queue! "
                        "Bank ID = 0x%x, Status = %#x.\n",
                        GpioBank->BankId,
                        Status);

            goto ProcessDeviceIoControlEnd;
        }

        CompleteRequest = FALSE;
        break;

    //
    // N.B. This routine must forward private IOCTLs meant for the GPIO
    //      client driver to the client driver.
    //

    case IOCTL_GPIO_CONTROLLER_SPECIFIC_FUNCTION:
        Status = GpiopProcessControllerSpecificFunctionIoctl(GpioExtension,
                                                             Queue,
                                                             Request,
                                                             &BytesReadWritten);

        break;

    case IOCTL_GPIO_QUERY_FAST_IO_INTERFACE:
        FileObject = WdfRequestGetFileObject(Request);
        if (FileObject != NULL) {
            FileContext = GpioClxGetFileObjectContext(FileObject);
        }

        if (FileContext == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpioClxEvtProcessDeviceIoControl: Failed to get file "
                        "context from request! Request = %p\n",
                        Request);

            Status = STATUS_INVALID_HANDLE;
            goto ProcessDeviceIoControlEnd;
        }

        GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

        GPIO_ASSERT(GpioBank != NULL);

        Status = WdfRequestForwardToIoQueue(Request, GpioBank->IoQueue);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_CREATE_CLOSE,
                        "%!FUNC!: Failed to forward request to bank queue! "
                        "Bank ID = 0x%x, Status = %#x.\n",
                        GpioBank->BankId,
                        Status);

            goto ProcessDeviceIoControlEnd;
        }

        CompleteRequest = FALSE;
        break;

    default:
        Status = STATUS_INVALID_DEVICE_REQUEST;
        break;
    }

ProcessDeviceIoControlEnd:
    if (CompleteRequest != FALSE) {
        if (NT_SUCCESS(Status)) {
            WdfRequestCompleteWithInformation(Request, Status, BytesReadWritten);

        } else {
            WdfRequestComplete(Request, Status);
        }
    }

    return;
}

VOID
GpioClxEvtProcessInternalDeviceIoControl (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t OutputBufferLength,
    __in size_t InputBufferLength,
    __in ULONG IoControlCode
    )

/*++

Routine Description:

    This routine handles internal IOCTL requests sent to the GPIO client driver
    device. This callback is invoked by WDF when the framework receives
    IRP_MJ_INTERNAL_DEVICE_CONTROL requests.

    N.B. Since the device's WDF_OBJECT_ATTRIBUTES structure is set to
         WdfExecutionLevelPassive, the framework will always send this callback
         at IRQL < DISPATCH_LEVEL.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    OutputBufferLength - Supplies the length of the request's output buffer, if
        an output buffer is available.

    InputBufferLength - Supplies the length of the request's input buffer, if an
        input buffer is available.

    IoControlCode - Supplies the driver-defined or system-defined I/O control
        code (IOCTL) that is associated with the request.

Return Value:

  None.

--*/

{

    ULONG BytesWritten;
    BOOLEAN CompleteRequest;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    size_t InputLength;
    ULONG MinimumSize;
    BOOLEAN ReleasePowerReference;
    PGPIO_INTERRUPT_REQUEST_PARAMETERS RequestParameters;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(OutputBufferLength);
    UNREFERENCED_PARAMETER(InputBufferLength);

    PAGED_CODE();

    //
    // Get the GPIO device extension from the queue handle.
    //

    CompleteRequest = TRUE;
    GpioBank = NULL;
    ReleasePowerReference = FALSE;
    GpioExtension = GpioClxGetDeviceExtension(WdfIoQueueGetDevice(Queue));

    //
    // Examine the IOCTL code and dispatch the request to the appropriate
    // handler.
    //

    BytesWritten = 0;
    if (IoControlCode != IOCTL_GPIO_INTERRUPT_REQUEST) {
        Status = STATUS_INVALID_DEVICE_REQUEST;

        GPIO_ASSERT(FALSE);

        goto ProcessInternalDeviceIoControlEnd;
    }

    MinimumSize = sizeof(GPIO_INTERRUPT_REQUEST_PARAMETERS);
    Status = WdfRequestRetrieveInputBuffer(Request,
                                           MinimumSize,
                                           &RequestParameters,
                                           &InputLength);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopProcessInterruptRequestIoctl: "
                    "WdfRequestRetrieveInputBuffer() failed! Status = %#x\n",
                    Status);

        goto ProcessInternalDeviceIoControlEnd;
    }

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to enable an
    // interrupt if it manages the supplied VIRQ.
    //

    if (RequestParameters->Type != EnableInterruptRequest) {
        GpioBank = GpiopGetBankEntryFromVirq(GpioExtension,
                                             RequestParameters->Virq);

        if (GpioBank == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: Gsiv (%#x) is invalid!\n",
                        __FUNCTION__,
                        RequestParameters->Virq);

            Status = STATUS_INVALID_PARAMETER;
            goto ProcessInternalDeviceIoControlEnd;
        }

    } else {
        Status = GpiopQueryBankIdFromGsivOrConnectionId(GpioExtension,
                                                        TRUE,
                                                        RequestParameters->Virq,
                                                        &GpioBank,
                                                        NULL,
                                                        NULL);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: Failed to map request to bank! Status = %#x.\n",
                        __FUNCTION__,
                        Status);

            goto ProcessInternalDeviceIoControlEnd;
        }

        //
        // Request power manager to make the component active. The request
        // arrives from the primary queue, which is a power-managed queue.
        // Hence WDF will ensure that the device is put into D0, before the
        // request is dispatched. Thus it is not required to call
        // WdfDeviceStopIdle() prior to activating the component.
        //

        PoFxActivateComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0);
        ReleasePowerReference = TRUE;
    }

    Status = WdfRequestForwardToIoQueue(Request, GpioBank->IoQueue);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_CREATE_CLOSE,
                    "%s: Failed to forward request to bank queue! "
                    "Bank ID = 0x%x, Status = %#x.\n",
                    __FUNCTION__,
                    GpioBank->BankId,
                    Status);

        goto ProcessInternalDeviceIoControlEnd;
    }

    CompleteRequest = FALSE;
    ReleasePowerReference = FALSE;

ProcessInternalDeviceIoControlEnd:
    if (CompleteRequest != FALSE) {
        if (NT_SUCCESS(Status)) {
            WdfRequestCompleteWithInformation(Request, Status, BytesWritten);

        } else {
            WdfRequestComplete(Request, Status);
        }
    }

    //
    // Release the power reference that was previously taken on the
    // component, if requested.
    //

    if (ReleasePowerReference != FALSE) {
        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);
    }

    return;
}

VOID
GpioClxEvtDeviceContextCleanup (
    IN WDFOBJECT Object
    )

/*++

Routine Description:

    This routine is invoked by the framework when the device is being deleted
    in response to IRP_MN_REMOVE_DEVICE request. This callback must perform any
    cleanup operations that are necessary before the specified device is
    removed.

Arguments:

    Object - Supplies a handle to the framework device object.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    GPIO_HUB_REGISTRATION_DATA GpioData;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    PAGED_CODE();

    Device = (WDFDEVICE)Object;

    //
    // Guaranteed for clean-up of a WDFDEVICE.
    //

    GPIO_ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

    //
    // If the device represents the GPIO hub, then perform GPIO hub cleanup.
    // Otherwise perform device extension cleanup. Request the hub to unregister
    // the device. This will prevent any future requests from being dispatched
    // to the client driver.
    //
    // Note the hub cleanup should be the last (device) cleanup that should
    // happen as it happens as part of GPIO class extension unloading.
    //

    if (GPIO_IS_HUB_DEVICE(Device) != FALSE) {
        GpioHubpUninitialize();

    } else {
        GpioExtension = GpioClxGetDeviceExtension(Device);
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_PNP,
                    "GpioClxEvtDeviceContextCleanup: Cleanup called!\n");

        RtlZeroMemory(&GpioData, sizeof(GPIO_HUB_REGISTRATION_DATA));
        GpioData.BiosName = &GpioExtension->BiosName;
        Status = GpioHubpUnregisterGpioDevice(&GpioData);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_PNP,
                        "%s: Failed to unregister client from the hub! "
                        "Status = %#x\n",
                        __FUNCTION__,
                        Status);
        }

        GpiopDeleteDeviceExtension(GpioExtension);
        GpioClnInvokeControllerCleanupCallback(GpioExtension);
    }

    return;
}

NTSTATUS
GpioClxEvtDeviceSelfManagedIoInit (
    __in WDFDEVICE Device
    )

/*++

Routine Description:

    This routine is invoked by the framework to initialize and start the
    device's self-managed I/O operations. This callback is invoked after
    EvtDeviceD0Entry callback has been invoked.

    Note the framework calls EvtDeviceSelfManagedIoInit callback after the
    first D0 but not after subsequent D0 entries. This means that it is not
    called after D0 entries that occur due to resume from Sx system state
    transitions or resource rebalance (stop -> start). Hence the GPIO class
    extension registers with the power framework to enable runtime
    power management in this callback.

    N.B. This routine is called at PASSIVE_LEVEL but it is not marked as
         pageable because this function is in the device power up path.

Arguments:

    Device - Supplies a handle to the framework device object.

Return Value:

    None.

--*/

{

    ULONG BankId;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    //
    // At initialization, all GPIO banks are marked as being in F0 and in the
    // active state.
    //

    GpioExtension = GpioClxGetDeviceExtension(Device);
    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        GpioBank->PowerData.FState = FStateF0;
        GpioBank->PowerData.IsActive = TRUE;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_PNP,
                "%s: GPIO banks implicitly moved into F0 state!\n",
                __FUNCTION__);

    //
    // Register with the power framework.
    //

    Status = GpiopRegisterWithPowerFramework(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        goto DeviceSelfManagedIoInitEnd;
    }

    GpiopStartRuntimePowerManagement(GpioExtension);

    //
    // Defer enabling ACPI events until IRP_MN_QUERY_DEVICE_RELATIONS.
    //

    GpioExtension->NeedToEnableAcpiEvents = TRUE;

DeviceSelfManagedIoInitEnd:
    return Status;
}

VOID
GpioClxEvtDeviceSelfManagedIoFlush (
    __in WDFDEVICE Device
    )

/*++

Routine Description:

    This routine is invoked by the framework after the device has been removed
    to flush (that is, remove) I/O requests in self-managed I/O queues. In
    addition, this routine also unregisters with the power framework, since
    this routine should only be called once per device.

    The WDF framework would have already purged the primary queue before
    this callback is invoked. Thus only the the secondary (non-power-managed)
    queues need to be purged.

Arguments:

    Device - Supplies a handle to the framework device object.

Return Value:

    None.

--*/

{

    ULONG BankId;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;

    PAGED_CODE();

    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // Purge the secondary queues.
    //

    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        WdfIoQueuePurgeSynchronously(GpioBank->IoQueue);
    }

    //
    // Unregister with the power framework.
    //

    GpiopUnregisterWithPowerFramework(GpioExtension);
    return;
}

NTSTATUS
GpioClxWdmPreprocessPowerIrp (
    __in WDFDEVICE Device,
    __inout PIRP Irp
    )

/*++

Routine Description:

    This is the preprocess routine for IRP_MJ_POWER. Power manager system sends
    such IRPs when the system is transitioning into some Sx state (S0, S1 - S5).
    This routine checks if the transition is from Sx -> S0.

    WDF invokes this with the IO_REMOVE_LOCK acquired, but no other lock.

Arguments:

    Device - Pointer to the framework device object for this device.

    Irp - Pointer to the IRP for the current request.

Return Value:

    NTSTATUS code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    PIO_STACK_LOCATION IrpStack;
    NTSTATUS Status;

    //
    // Check if this is a Sx -> S0 system state transition request. If so,
    // the mark the flag that would cause GPIO class extension to both report
    // D0 completion to the power framework and unmask ACPI eventing pins (when
    // the next WDF D0Entry callback is called).
    //

    IrpStack = IoGetCurrentIrpStackLocation(Irp);
    if ((IrpStack->Parameters.Power.Type == SystemPowerState) &&
        (IrpStack->Parameters.Power.State.SystemState == PowerSystemWorking)) {

        GpioExtension = GpioClxGetDeviceExtension(Device);
        GpioExtension->WokeFromSx = TRUE;

    //
    // Check if this is a Dx, x > 0 in response to a Sx, x > 0 state. If so,
    // the system is S3-sleeping, hibernating, shutting down or rebooting.
    // If we are moving out of D0, the ACPI eventing pins should be deep-masked.
    //
    //
    // To avoid deadlock, deep-masking has to be done here -- where the WDF
    // default queue has not yet been stopped -- instead of in
    // GpioClxEvtDevicePreInterruptsDisabled().  This is because deep-masking
    // includes waiting for all ACPI firmware event methods to complete and
    // such methods could synchronously access opregions served by this GPIO
    // controller and these accesses would block if the default queue has been
    // stopped.
    //

    } else if ((IrpStack->Parameters.Power.Type == DevicePowerState) &&
               ((IrpStack->Parameters.Power.State.DeviceState == PowerDeviceD1) ||
                (IrpStack->Parameters.Power.State.DeviceState == PowerDeviceD2) ||
                (IrpStack->Parameters.Power.State.DeviceState == PowerDeviceD3))) {

        GpioExtension = GpioClxGetDeviceExtension(Device);
        if (GPIO_GET_DEVICE_POWER_STATE(GpioExtension) == WdfPowerDeviceD0) {
            switch (IrpStack->Parameters.Power.ShutdownType) {
            case PowerActionSleep:
            case PowerActionHibernate:
            case PowerActionShutdown:
            case PowerActionShutdownReset:
            case PowerActionShutdownOff:

                //
                // The use of GpiopManageAcpiEventing() ensures that:
                //
                //   1. We will wait until GpioClxWdmPreprocessQueryDeviceRelationsIrp() ->
                //      GpiopManageAcpiEventing(..., AcpiEventStateEnable, ...) has
                //      completed enabling all ACPI pins before attempting to mask
                //      them.
                //
                //   2. We will not be trying to mask pins while they are
                //      simultaneously being disabled.
                //

                GpiopManageAcpiEventing(GpioExtension,
                                        AcpiEventStateDeepMaskForSx,
                                        TRUE);

                break;
            }
        }
    }

    //
    // Deliver the IRP back to the framework.
    //

    IoSkipCurrentIrpStackLocation(Irp);
    Status = WdfDeviceWdmDispatchPreprocessedIrp(Device, Irp);
    return Status;
}

NTSTATUS
GpioClxWdmPreprocessQueryDeviceRelationsIrp (
    __in WDFDEVICE Device,
    __inout PIRP Irp
    )

/*++

Routine Description:

    This is the preprocess routine for IRP_MN_QUERY_DEVICE_RELATIONS.

    If this IRP came immediately after IRP_MN_START_DEVICE and from PnP
    manager, the routine will enable ACPI events defined in the ACPI namespace.
    It is safe to do so now because:

    1. The device has been registered with the power framework, so the enable
       path's calls to PoFxActivateComponent() will now work.

    2. IRP_MN_START_DEVICE has fully completed from PnP's point of view, so
       calls to WdfIoTargetOpen() will now work. Such calls can come from this
       control flow as soon as an event is enabled:

           Event interrupt
             -> firmware method
                -> operation region access
                   -> WdfIoTargetOpen()

    If this IRP did not come immediately after IRP_MN_START_DEVICE or did not
    come from PnP manager, this routine will perform no preprocessing.

Arguments:

    Device - Pointer to the framework device object for this device.

    Irp - Pointer to the IRP for the current request.

Return Value:

    NTSTATUS code.

--*/

{
    PDEVICE_EXTENSION GpioExtension;
    PIO_STACK_LOCATION IrpStack;
    NTSTATUS Status;

    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // PnP manager sends a IRP_MN_QUERY_DEVICE_RELATIONS (QDR) of type
    // BusRelations, immediately after IRP_MN_START_DEVICE (StartDevice). This
    // type of QDR can only come from PnP manager.
    //
    // Ignore all other types of QDRs, one of which (TargetDeviceRelation) is
    // not even synchronized with StartDevice.
    //

    IrpStack = IoGetCurrentIrpStackLocation(Irp);
    if (IrpStack->Parameters.QueryDeviceRelations.Type == BusRelations) {
        if (GpioExtension->NeedToEnableAcpiEvents != FALSE) {
            GpioExtension->NeedToEnableAcpiEvents = FALSE;
            GpiopManageAcpiEventing(GpioExtension, AcpiEventStateEnable, FALSE);
        }
    }

    //
    // Deliver the IRP back to the framework.
    //

    IoSkipCurrentIrpStackLocation(Irp);
    Status = WdfDeviceWdmDispatchPreprocessedIrp(Device, Irp);
    return Status;
}

//
// ------------------------------------------------ Bank related WDF callbacks
//

VOID
GpioClxBankEvtProcessIoDefault (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request
    )

/*++

Routine Description:

    This routine is invoked by the framework to allow a GPIO bank to handle an
    IRP_MJ_CREATE request that was initially processed at the
    class-extension-level by GpioClxEvtDeviceFileCreate.

    N.B. It is assumed that, on entry, a power reference has been taken on both
         the bank and the class extension. This function will release both.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

Return Value:

    None.

--*/

{

    PGPIO_FILE_OBJECT_CONTEXT FileContext;
    WDFFILEOBJECT FileObject;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_BANK_QUEUE_CONTEXT QueueContext;
    WDF_REQUEST_PARAMETERS RequestParameters;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Get the device extension and the GPIO bank.
    //

    FileContext = NULL;
    GpioExtension = GpioClxGetDeviceExtension(WdfIoQueueGetDevice(Queue));
    QueueContext = GpiopBankQueueGetContext(Queue);
    GpioBank = QueueContext->Bank;

    //
    // Get the parameters associated with the request.
    //

    WDF_REQUEST_PARAMETERS_INIT(&RequestParameters);
    WdfRequestGetParameters(Request, &RequestParameters);

    //
    // Only create requests are expected to be presented to this handler.
    //

    GPIO_ASSERT(RequestParameters.Type == WdfRequestTypeCreate);

    //
    // The file object must exist otherwise the request would not have been
    // queued to the bank queue in the first place.
    //

    FileObject = WdfRequestGetFileObject(Request);
    if (FileObject != NULL) {
        FileContext = GpioClxGetFileObjectContext(FileObject);
    }

    //
    // Determine which resource descriptor type the connection ID is
    // pointing to. There are two types:
    //
    // 1. GPIO IO/interrupt resource
    // 2. Function config resource
    //
    // To do this, we will query the resource hub and check the descriptor 
    // tag in the descriptor header common to the two resource types:
    // 
    // N.b. Exhaustive validation of the resource descriptor is deferred in
    // the corresponding GPIO/function config pin connection routines.
    //

    if (FileContext != NULL) {
        if (FileContext->Flags.FunctionConfig == 1) {

            //
            // Reserve first. Commit will be handled through a seperate IOCTL.
            //
            
            Status = GpiopConnectFunctionConfigPinsReserve(GpioExtension,
                                                           FileContext,
                                                           &FileContext->ConnectionId);

        } else {
            Status = GpiopConnectGpioPins(GpioExtension,
                                          FileContext,
                                          FileContext->ConnectMode,
                                          &FileContext->ConnectionId);
        }

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: Failed to get file context from request! "
                    "Request = %p\n",
                    __FUNCTION__,
                    Request);

        Status = STATUS_NOT_SUPPORTED;
    }

    WdfRequestComplete(Request, Status);

    //
    // Release the power reference that was previously taken on the component
    // when the request was queued. Drop the WDF power reference which was
    // taken when the IRP was originally received (GpioClxEvtDeviceFileCreate).
    //

    PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);

    GPIO_ASSERT(GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) == FALSE);

    WdfDeviceResumeIdle(GpioExtension->Device);
    return;
}

VOID
GpioClxBankEvtProcessDeviceIoControl (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t OutputBufferLength,
    __in size_t InputBufferLength,
    __in ULONG IoControlCode
    )

/*++

Routine Description:

    This routine handles IOCTL requests sent to the GPIO client driver device.
    This callback is invoked by WDF when the framework receives
    IRP_MJ_DEVICE_CONTROL requests.

    N.B. 1. Since the device's WDF_OBJECT_ATTRIBUTES structure is set to
            WdfExecutionLevelPassive, the framework will always send this
            callback at IRQL < DISPATCH_LEVEL.

         2. This routines that the request has already been validated to
            have originated from a valid caller (kernel-mode or UMDF-driver).

         3. Note this routine is not marked PAGED as it may be called
            before/after the boot device is in D0/D3 if boot device has GPIO
            dependencies.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    OutputBufferLength - Supplies the length of the request's output buffer, if
        an output buffer is available.

    InputBufferLength - Supplies the length of the request's input buffer, if an
        input buffer is available.

    IoControlCode - Supplies the driver-defined or system-defined I/O control
        code (IOCTL) that is associated with the request.

Return Value:

  None.

--*/

{

    ULONG BytesReadWritten;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(OutputBufferLength);
    UNREFERENCED_PARAMETER(InputBufferLength);

    //
    // Get the GPIO device extension from the queue handle.
    //

    GpioExtension = GpioClxGetDeviceExtension(WdfIoQueueGetDevice(Queue));

    GPIO_CLX_VALIDATE_SIGNATURE(GpioExtension);

    //
    // Examine the IOCTL code and dispatch the request to the appropriate
    // handler.
    //

    BytesReadWritten = 0;
    switch(IoControlCode) {

    case IOCTL_GPIO_READ_PINS:
        Status = GpiopProcessReadPinsRequest(GpioExtension,
                                             Queue,
                                             Request,
                                             &BytesReadWritten);

        break;

    case IOCTL_GPIO_WRITE_PINS:
        Status = GpiopProcessWritePinsRequest(GpioExtension,
                                              Queue,
                                              Request,
                                              &BytesReadWritten);

        break;

    case IOCTL_GPIO_QUERY_FAST_IO_INTERFACE:
        Status = GpiopProcessFastIoInterfaceRequest(GpioExtension,
                                                    Queue,
                                                    Request,
                                                    &BytesReadWritten);

        break;

    case IOCTL_GPIO_COMMIT_FUNCTION_CONFIG_PINS:
        Status = GpiopProcessFunctionConfigPinsCommitRequest(GpioExtension,
                                                             Queue,
                                                             Request,
                                                             &BytesReadWritten);

        break;

    default:
        Status = STATUS_INVALID_DEVICE_REQUEST;

        GPIO_ASSERT(FALSE);

        break;
    }

    if (NT_SUCCESS(Status)) {
        WdfRequestCompleteWithInformation(Request, Status, BytesReadWritten);

    } else {
        WdfRequestComplete(Request, Status);
    }

    return;
}

VOID
GpioClxBankEvtProcessInternalIoControl (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request,
    __in size_t OutputBufferLength,
    __in size_t InputBufferLength,
    __in ULONG IoControlCode
    )

/*++

Routine Description:

    This routine handles internal IOCTL requests sent to a GPIO bank. This
    callback is invoked once the secondary I/O queue is restarted post the
    bank becoming active.

    At present the only internal IOCTLs requests that are queued on the per-bank
    secondary I/O queue are interrupt enable requests.

    N.B. Since the device's WDF_OBJECT_ATTRIBUTES structure is set to
         WdfExecutionLevelPassive, the framework will always send this callback
         at IRQL < DISPATCH_LEVEL.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

    OutputBufferLength - Supplies the length of the request's output buffer, if
        an output buffer is available.

    InputBufferLength - Supplies the length of the request's input buffer, if an
        input buffer is available.

    IoControlCode - Supplies the driver-defined or system-defined I/O control
        code (IOCTL) that is associated with the request.

Return Value:

  None.

--*/

{

    ULONG BytesWritten;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    size_t InputLength;
    ULONG MinimumSize;
    PGPIO_BANK_QUEUE_CONTEXT QueueContext;
    PGPIO_INTERRUPT_REQUEST_PARAMETERS RequestParameters;
    GPIO_INTERRUPT_REQUEST_TYPE RequestType;
    NTSTATUS Status;

    UNREFERENCED_PARAMETER(OutputBufferLength);
    UNREFERENCED_PARAMETER(InputBufferLength);
    UNREFERENCED_PARAMETER(IoControlCode);

    PAGED_CODE();

    //
    // Get the device extension and the GPIO bank.
    //

    GpioExtension = GpioClxGetDeviceExtension(WdfIoQueueGetDevice(Queue));
    QueueContext = GpiopBankQueueGetContext(Queue);
    GpioBank = QueueContext->Bank;

    //
    // Get the input buffer. The buffer was validated before being put onto
    // the secondary queue.
    //

    MinimumSize = sizeof(GPIO_INTERRUPT_REQUEST_PARAMETERS);
    Status = WdfRequestRetrieveInputBuffer(Request,
                                           MinimumSize,
                                           &RequestParameters,
                                           &InputLength);

    GPIO_ASSERT(NT_SUCCESS(Status) != FALSE);

    Status = GpiopProcessInterruptRequestIoctl(GpioExtension,
                                               Request,
                                               RequestParameters,
                                               &BytesWritten);

    //
    // N.B. The RequestParameters is not accessible once the IRP is completed.
    //      Hence snapshot the request type prior to completing the WDF request.
    //

    RequestType = RequestParameters->Type;

    //
    // Complete the request.
    //

    if (NT_SUCCESS(Status)) {
        WdfRequestCompleteWithInformation(Request, Status, BytesWritten);

    } else {
        WdfRequestComplete(Request, Status);
    }

    //
    // Release the power reference that was previously taken on the component
    // when the request was queued.
    //
    // N.B. The reference is only taken for interrupt enable requests and
    //      not for other types of requests (e.g. mask/unmask etc.)
    //

    if (RequestType == EnableInterruptRequest) {
        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);
    }

    return;
}

VOID
GpioClxBankEvtProcessIoCanceledOnQueue (
    __in WDFQUEUE Queue,
    __in WDFREQUEST Request
    )

/*++

Routine Description:

    This routine is invoked by the framework to inform the class extension
    that an I/O request was canceled while it was in an I/O queue. It is
    mainly responsible for releasing the reference to power framework
    taken when the request arrived.

    Since WdfExecutionLevelPassive is specified, this callback would be called
    at PASSIVE_LEVEL.

Arguments:

    Queue -  Supplies a handle to the framework queue object that is associated
        with the I/O request.

    Request - Supplies a handle to a framework request object.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    size_t InputLength;
    PGPIO_INTERRUPT_REQUEST_PARAMETERS InputParameters;
    ULONG Ioctl;
    ULONG MinimumSize;
    PGPIO_BANK_QUEUE_CONTEXT QueueContext;
    BOOLEAN ReleasePepPowerReference;
    BOOLEAN ReleaseWdfPowerReference;
    WDF_REQUEST_PARAMETERS RequestParameters;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Get the device extension and the queue extension.
    //

    Device = WdfIoQueueGetDevice(Queue);
    GpioExtension = GpioClxGetDeviceExtension(Device);
    QueueContext = GpiopBankQueueGetContext(Queue);

    //
    // Get the parameters associated with the request.
    //

    WDF_REQUEST_PARAMETERS_INIT(&RequestParameters);
    WdfRequestGetParameters(Request, &RequestParameters);

    ReleasePepPowerReference = FALSE;
    ReleaseWdfPowerReference = FALSE;
    switch (RequestParameters.Type) {

    case WdfRequestTypeCreate:
        ReleasePepPowerReference = TRUE;
        ReleaseWdfPowerReference = TRUE;
        break;

    case WdfRequestTypeDeviceControl:
        break;

    case WdfRequestTypeDeviceControlInternal:
        Ioctl = RequestParameters.Parameters.DeviceIoControl.IoControlCode;
        if (Ioctl == IOCTL_GPIO_INTERRUPT_REQUEST) {
            MinimumSize = sizeof(GPIO_INTERRUPT_REQUEST_PARAMETERS);
            Status = WdfRequestRetrieveInputBuffer(Request,
                                                   MinimumSize,
                                                   &InputParameters,
                                                   &InputLength);

            if (!NT_SUCCESS(Status)) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_INTERRUPT,
                            "%s: WdfRequestRetrieveInputBuffer() failed! "
                            "Status = %#x\n",
                            __FUNCTION__,
                            Status);

                //
                // This call is not expected to fail as the parameter size was
                // validated before putting on the secondary queue.
                //

                GPIO_ASSERT(FALSE);

                //
                // Assume the power reference does not need to be released
                // by default.
                //

                break;
            }

            //
            // N.B. WDF power reference was not taken in this case (i.e.
            //      WdfDeviceStopIdle() was not called), thus that reference
            //      does not need to be dropped.
            //


            if (InputParameters->Type == EnableInterruptRequest) {
                ReleasePepPowerReference = TRUE;
            }
        }

        break;

    default:
        break;
    }

    //
    // Determine the bank/component this queue is associated with.
    //

    GpioBank = QueueContext->Bank;

    //
    // Complete the request with STATUS_CANCELLED
    //

    WdfRequestComplete(Request, STATUS_CANCELLED);

    //
    // Release the power reference that was previously taken on the component
    //

    if (ReleasePepPowerReference == TRUE) {
        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);
    }

    //
    // Drop the WDF power reference if it was taken when the IRP was originally
    // received (GpioClxEvtDeviceFileCreate).
    //
    // If D0 entry had previously failed, do not do this since the device is
    // being torn down by WDF and WdfDeviceResumeIdle() is not allowed to be
    // called. In such a case, the current code is probably executing in the
    // context of GpioClxEvtDeviceSelfManagedIoFlush.
    //

    if (ReleaseWdfPowerReference == TRUE) {
        if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) == FALSE) {
            WdfDeviceResumeIdle(GpioExtension->Device);
        }
    }

    return;
}


