/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    gpioclx.c

Abstract:

    This file implements GPIO class extension driver's core functionality
    including most of the interrupt, IO and other support routines.


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"

#if defined(EVENT_TRACING)
#include "gpioclx.tmh"         // auto-generated by WPP
#endif

#include <acpi\aml.h>
#include "hub.h"
#include "privdefs.h"


//
// -------------------------------------------------------------------- Globals
//

//
// Define a global list of all clients.
//

LIST_ENTRY GpioClientDriverList;

//
// Define a lock that is used to guard access to the global client driver list.
//

FAST_MUTEX GpioClientGlobalListLock;

BOOLEAN GpioForceEmergencyWorker = FALSE;
BOOLEAN GpioAllowInterruptOutput = FALSE;

//
// Whether the GPIO class extension should check that GPIO hardware agrees with
// it on which interrupts are enabled, every time an ISR fires or an interrupt
// is enabled/disabled. This triggers bugchecks as soon as a disagreement
// occurs, preventing one source of a potential interrupt storm.
//
// This hurts performance.
//

BOOLEAN GpioCheckEnabledInterrupts = FALSE;

//
// This is set to noise filtering interval override if specified in the
// registry.
//

ULONG GpioGlobalNoiseIntervalOverride = 0;

//
// Handle to the global/default recorder and the default log buffer size.
//

RECORDER_LOG GpioLogHandle = NULL;

volatile LONG GpioLogHandleInstance = 0;

//
// Note: The size of the log is the lower of the variable below, 64KB, and this
//       registry value:
//
//         HKLM\SYSTEM\CONTROLSET001\SERVICES\GPIOCLX0101\PARAMETERS:
//           WppRecorder_PerBufferMaxBytes (REG_DWORD):
//             Bytes per buffer (default: 8KB)
//
//       WppRecorder.sys will cap any size larger than 64KB.
//

#if DBG

ULONG GpioLogBufferSize = 64 * 1024;

#else

ULONG GpioLogBufferSize = 8 * 1024;

#endif


//
// ----------------------------------------------------------------- Prototypes
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildGpioBanks (
    __in PDEVICE_EXTENSION GpioExtension
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildBitmapBuffers (
    __in PDEVICE_EXTENSION GpioExtension
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildPinInformationTable (
    __in PDEVICE_EXTENSION GpioExtension
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
WDF_TRI_STATE
GpiopCheckLevelReconfigurationSupported (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in KINTERRUPT_MODE InterruptMode,
    __in KINTERRUPT_POLARITY Polarity,
    __in BOOLEAN EmulatedActiveBoth
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopCleanupBankInterruptsIo (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in BOOLEAN InSurpriseRemovalContext
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopAllocateDebounceResources (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber,
    __out PGPIO_DEBOUNCE_TIMER_CONTEXT *DebounceTimerContextOut,
    __out PGPIO_NOISE_FILTER_TIMER_CONTEXT *NoiseFilterTimerContextOut,
    __out WDFWORKITEM *DebounceNoiseTimerWorkerOut
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDetermineInterruptCountAndIrql (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw,
    __out PULONG InterruptCount,
    __out KIRQL *SynchronizationIrql,
    __out ULONG *BaseInterruptGsiv
   );

_Must_inspect_result_
__drv_maxIRQL(APC_LEVEL)
PGPIO_CLIENT_DRIVER
GpiopFindClientNoLock (
    __in WDFDRIVER DriverObject
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFormatRequestForRead (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in ULONG64 PinValues,
    __out_bcount((PinCount + 7) / 8) PUCHAR Buffer
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFormatRequestForWrite (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in_bcount((PinCount + 7) / 8) PUCHAR Buffer,
    __out PULONG64 SetMaskOutput,
    __out PULONG64 ClearMaskOutput
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitializeBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank
   );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopInitializeRegistryOptions (
    __in PDRIVER_OBJECT WdmDriverObject,
    __in WDFDRIVER WdfDriver,
    __in PUNICODE_STRING RegistryPath
    );

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank,
    __out PCLIENT_QUERY_BANK_POWER_INFORMATION_OUTPUT BankInformation
   );

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryEmulatedActiveBothInitialPolarities (
    __in PDEVICE_EXTENSION GpioExtension
    );

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFlushPendingPinInterruptActivities (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber
    );


__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopResolveInterruptBinding (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG InterruptCount,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw
   );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetupBankInterrupts (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetInterruptInformationFromResources (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_opt PCM_PARTIAL_RESOURCE_DESCRIPTOR InterruptDescriptor,
    __in_opt PCM_PARTIAL_RESOURCE_DESCRIPTOR RawInterruptDescriptor
   );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetupBankSecondaryIoQueue (
    __in PDEVICE_EXTENSION GpioExtension
);

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateClientBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT Information
    );

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateClientControllerBasicInformation (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PCLIENT_CONTROLLER_BASIC_INFORMATION Information
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateInterruptBindings (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw,
    __in_bcount(BindingBufferSize)
        PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT BindingOutput,

    __in ULONG BindingBufferSize
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateRegistrationPacketPhase1 (
    __in_xcount(RegistrationPacket->Size)
        PGPIO_CLIENT_REGISTRATION_PACKET RegistrationPacket
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateRegistrationPacketPhase2 (
    __in PDEVICE_EXTENSION GpioExtension
    );

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryEmulatedActiveBothInitialPolarity (
    __in WDFDEVICE Device,
    __inout __deref WDFIOTARGET *ResourceHubTarget,
    __in ULONG Virq,
    __out PKINTERRUPT_POLARITY InitialPolarity
    );

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text(PAGE, GpiopBuildGpioBanks)
#pragma alloc_text(PAGE, GpiopBuildBitmapBuffers)
#pragma alloc_text(PAGE, GpiopBuildDeviceExtension)
#pragma alloc_text(PAGE, GpiopBuildPinInformationTable)
#pragma alloc_text(PAGE, GpiopConnectGpioPins)
#pragma alloc_text(PAGE, GpiopCreateClient)
#pragma alloc_text(PAGE, GpiopAllocateDebounceResources)
#pragma alloc_text(PAGE, GpiopDeleteClient)
#pragma alloc_text(PAGE, GpiopDeleteDeviceExtension)
#pragma alloc_text(PAGE, GpiopDetermineInterruptCountAndIrql)
#pragma alloc_text(PAGE, GpiopFindClient)
#pragma alloc_text(PAGE, GpiopFindClientNoLock)
#pragma alloc_text(PAGE, GpiopInitialize)
#pragma alloc_text(PAGE, GpiopInitializeBankPowerInformation)
#pragma alloc_text(PAGE, GpiopInitializeBanksAndInterrupts)
#pragma alloc_text(PAGE, GpiopInitializeRegistryOptions)
#pragma alloc_text(PAGE, GpiopQueryBankIdFromGsivOrConnectionId)
#pragma alloc_text(PAGE, GpiopQueryControllerInformation)
#pragma alloc_text(PAGE, GpiopQueryEmulatedActiveBothInitialPolarities)
#pragma alloc_text(PAGE, GpiopQueryResourceHubBiosDescriptor)
#pragma alloc_text(PAGE, GpiopQueryEmulatedActiveBothInitialPolarity)
#pragma alloc_text(PAGE, GpiopUpdateEmulatedActiveBothInitialPolarity)
#pragma alloc_text(PAGE, GpiopResolveInterruptBinding)
#pragma alloc_text(PAGE, GpiopSetupBankInterrupts)
#pragma alloc_text(PAGE, GpiopSetInterruptInformationFromResources)
#pragma alloc_text(PAGE, GpiopSetupBankSecondaryIoQueue)
#pragma alloc_text(PAGE, GpiopValidateControllerFunctionBankMappings)
#pragma alloc_text(PAGE, GpiopValidateClientBankPowerInformation)
#pragma alloc_text(PAGE, GpiopValidateClientControllerBasicInformation)
#pragma alloc_text(PAGE, GpiopValidateInterruptBindings)
#pragma alloc_text(PAGE, GpiopValidateRegistrationPacketPhase1)
#pragma alloc_text(PAGE, GpiopValidateRegistrationPacketPhase2)
#pragma alloc_text(PAGE, GpiopUninitialize)
#pragma alloc_text(PAGE, GpiopConnectFunctionConfigPinsReserve)
#pragma alloc_text(PAGE, GpiopConnectFunctionConfigPinsCommit)
#pragma alloc_text(PAGE, GpiopDisconnectFunctionConfigPins)

//
// --------------------------------------------------------- Interrupt routines
//

_Must_inspect_result_
__drv_sameIRQL
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopEnableInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Virq,
    __in KINTERRUPT_MODE InterruptMode,
    __in KINTERRUPT_POLARITY Polarity,
    __in ULONG_PTR CallbackContext,
    __inout_opt volatile ULONG_PTR *HubVirqEntryCache
    )

/*++

Routine Description:

    This routine enables the specified interrupt line behind a GPIO controller.
    It is assumed that for GPIO lines used for regular device interrupts, the
    HAL will only invoke it when the first device connects the interrupt
    (for shared interrupt cases).

    Note this routine checks for the shared/exclusive flag explicitly
    as interrupt connect may happen post-IO connect for input. ACPI already
    takes care of shared/exclusive for interrupt resources during interrupt
    arbitration.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Virq - Supplies the VIRQ for interrupt line that should be enabled.

    InterruptMode - Supplies the trigger mode (edge or level) associated with
        this interrupt.

    Polarity - Supplies the polarity (active low or active high) associated with
        this interrupt.

    CallbackContext - Supplies a context that the caller of this routine should
        be passed with the interrupt line fires.

    HubVirqEntryCache - Supplies a variable that receives the
        controller-relative pin number.

        Chronologically, this is set after GPIO_PIN_INFORMATION_ENTRY::Virq is
        set but before the interrupt is enabled in hardware.

Return Value:

    NTSTATUS code.

--*/

{

    PIN_NUMBER AbsolutePinNumber;
    BOOLEAN AcpiEventPowerReferenceTaken;
    BOOLEAN AcpiEventInterrupt;
    KINTERRUPT_POLARITY InitialPolarity;
    USHORT BankId;
    GPIO_DEBOUNCE_MODEL DebounceModelOverride;
    WDFWORKITEM DebounceNoiseTimerWorker;
    WDF_TRI_STATE LevelReconfiguration;
    ULONG DebounceTimeout;
    PGPIO_DEBOUNCE_TIMER_CONTEXT DebounceTimerContext;
    UCHAR DescriptorOpenMode;
    BOOLEAN EmulateActiveBoth;
    BOOLEAN EmulateDebounce;
    PGPIO_BANK_ENTRY GpioBank;
    PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR GpioDescriptor;
    BOOLEAN MemoryMapped;
    size_t MinimumLength;
    GPIO_DEBOUNCE_PIN_CONTEXT NewDebounceContext;
    ULONG64 NoiseFilterTimeout;
    PGPIO_NOISE_FILTER_TIMER_CONTEXT NoiseFilterTimerContext;
    ULONG_PTR OldHubVirqEntryCacheValue;
    USHORT PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER RelativePinNumber;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER RhBuffer;
    BOOLEAN RunEVTMethod;
    NTSTATUS Status;
    BOOLEAN Valid;
    USHORT Value;

    AbsolutePinNumber = 0;
    AcpiEventPowerReferenceTaken = FALSE;
    BankId = 0;
    DebounceModelOverride = DebounceModelNone;
    DebounceNoiseTimerWorker = NULL;
    LevelReconfiguration = WdfFalse;
    DebounceTimeout = 0;
    DebounceTimerContext = NULL;
    EmulateActiveBoth = FALSE;
    EmulateDebounce = FALSE;
    GpioBank = NULL;
    GpioDescriptor = NULL;
    InitialPolarity = InterruptPolarityUnknown;
    NoiseFilterTimerContext = NULL;
    NoiseFilterTimeout = 0;
    RhBuffer = NULL;
    RunEVTMethod = FALSE;

    //
    // Silence compiler.
    //

    OldHubVirqEntryCacheValue = INVALID_PIN_NUMBER;

    //
    // Validate polarity supplied. Unknown polarity is not supported.
    //

    if (Polarity == InterruptPolarityUnknown) {
        Status = STATUS_NOT_SUPPORTED;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Invalid interrupt polarity specified"
                    "! Failing enable interrupt request! Gsiv = %#x, Mode = %d,"
                    " Polarity = %d\n",
                    Virq,
                    InterruptMode,
                    Polarity);

        goto EnableInterruptEnd;
    }

    //
    // ActiveBoth polarity is not supported for level-triggered interrupts.
    //

    if ((InterruptMode == LevelSensitive) &&
        (Polarity == InterruptActiveBoth)) {

        Status = STATUS_NOT_SUPPORTED;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Invalid interrupt polarity specified"
                    "! Failing enable interrupt request! Gsiv = %#x, Mode = %d,"
                    " Polarity = %d\n",
                    Virq,
                    InterruptMode,
                    Polarity);

        goto EnableInterruptEnd;
    }

    //
    // Query the actual BIOS descriptor mapping the supplied VIRQ from the
    // Resource hub. The actual descriptor will contain the pin number that
    // maps to this VIRQ.
    //

    Status = GpiopQueryResourceHubBiosDescriptor(
                 GpioExtension->Device,
                 &GpioExtension->ResourceHubTarget,
                 TRUE,
                 Virq,
                 &RhBuffer);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Failed to query BIOS descriptor "
                    "from the Resource Hub! Gsiv = %#x, Status = %#x\n",
                    Virq,
                    Status);

        goto EnableInterruptEnd;
    }

    //
    // The descriptor should at least be of the minimum required length.
    //

    MinimumLength = sizeof(PNP_GPIO_INTERRUPT_IO_DESCRIPTOR);
    if (RhBuffer->PropertiesLength < MinimumLength) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: BIOS GPIO descriptor length is "
                    "invalid! Gsiv = %#x, Length = %d, Reqd = %d\n",
                    Virq,
                    RhBuffer->PropertiesLength,
                    (ULONG)MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto EnableInterruptEnd;
    }

    GpioDescriptor =
        (PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR)RhBuffer->ConnectionProperties;

    Valid = GpioUtilIsBiosGpioDescriptorValid(GpioDescriptor,
                                              RhBuffer->PropertiesLength);

    if (Valid == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: BIOS GPIO descriptor length is "
                    "invalid! Gsiv = %#x, Length = %d, Reqd = %d\n",
                    Virq,
                    RhBuffer->PropertiesLength,
                    (ULONG)MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto EnableInterruptEnd;
    }

    GPIO_ASSERT(GpioDescriptor->DescriptorType ==
                    PNP_GPIO_IRQ_DESCRIPTOR_TYPE_INTERRUPT);

    Value = GpioDescriptor->ResourceSourceOffset;

    NT_ASSERT(Value >= GpioDescriptor->PinTableOffset);

    PinCount = (Value - GpioDescriptor->PinTableOffset)/sizeof(USHORT);
    AbsolutePinNumber = *((PUSHORT)Add2Ptr(GpioDescriptor,
                                           GpioDescriptor->PinTableOffset));

    DebounceTimeout = GpioDescriptor->DebounceTimeout;
    DescriptorOpenMode = (GpioDescriptor->InterruptIoFlags & GPIO_SHARED);

    //
    // Each interrupt descriptor should request only one pin to be connected.
    //

    if (PinCount != 1) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: BIOS GPIO descriptor pin count is "
                    "not 1! Gsiv = %#x, PinCount = %#x\n",
                    Virq,
                    PinCount);

        Status = STATUS_UNSUCCESSFUL;
        goto EnableInterruptEnd;
    }

    //
    // Differentiate between request for enabling ACPI event interrupts and
    // regular device interrupts.
    //

    AcpiEventInterrupt = GpiopIsAcpiEventInterrupt(GpioExtension, Virq);

    //
    // For ACPI event pins, validate that there is a handler associated with
    // this particular pin.
    //

    if (AcpiEventInterrupt != FALSE) {
        Status = GpiopValidateEventMethodForPin(GpioExtension,
                                                InterruptMode,
                                                AbsolutePinNumber,
                                                &RunEVTMethod);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "GpiopEnableInterrupt: No ACPI event handler associated"
                        " with ACPI event pin! Extn = %p, PinNumber = %#x\n",
                        GpioExtension,
                        AbsolutePinNumber);

            goto EnableInterruptEnd;
        }
    }

    //
    // Get the entry for the bank associated with this request.
    //

    BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension, AbsolutePinNumber);
    RelativePinNumber =
        GPIO_ABSOLUTE_TO_RELATIVE_PIN(GpioExtension, AbsolutePinNumber);

    GpioBank = GpiopGetBankEntry(GpioExtension, BankId);
    if (GpioBank == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: No bank associated with pin number!"
                    " Pin number is invalid! Gsiv = %#x, PinNumber = %#x\n",
                    Virq,
                    AbsolutePinNumber);

        Status = STATUS_INVALID_PARAMETER;
        goto EnableInterruptEnd;
    }

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to enable an
    // interrupt if it manages the supplied VIRQ.
    //

    PinInformation = GpiopCreatePinEntry(GpioBank, RelativePinNumber);
    if (PinInformation == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: BIOS GPIO descriptor pin number is "
                    "invalid, or out of memory! Gsiv = %#x, PinNumber = %#x\n",
                    Virq,
                    AbsolutePinNumber);

        Status = STATUS_INVALID_PARAMETER;
        goto EnableInterruptEnd;
    }

    //
    // If the interrupt polarity is ActiveBoth, and ActiveBoth is being
    // emulated, then query the initial polarity from the resource hub. The
    // initial polarity is used to determine the first edge for the pair of
    // edges for an emulated active both interrupt.

    // If the Resource Hub doesn't provide a valid value for the initial
    // polarity use the information provided by the _DSM method for the pin.
    //

    Status = GpiopQueryEmulatedActiveBothInitialPolarity(
                GpioExtension->Device,
                &GpioExtension->ResourceHubTarget,
                Virq,
                &InitialPolarity);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Failed to query initial polarity "
                    "for emulated active both interrupt using the Resource "
                    "Hub! Gsiv = %#x, Status = %#x\n",
                    Virq,
                    Status);

        goto EnableInterruptEnd;
    }


    if ((Polarity == InterruptActiveBoth) &&
        (GPIO_IS_ACTIVE_BOTH_EMULATED(GpioExtension) != FALSE)) {

        if (InitialPolarity == InterruptPolarityUnknown) {
            if (PinInformation->EmulatedActiveBothInitialPolarityIsHigh !=
                FALSE) {

                InitialPolarity = InterruptActiveHigh;

            } else {
                InitialPolarity = InterruptActiveLow;
            }
        }

        EmulateActiveBoth = TRUE;
    }

    //
    // If debouncing needs to be emulated, then allocate resources needed for
    // debouncing including debounce and noise filter timers. Also initialize
    // the debounce state machine context.
    //

    if ((GPIO_IS_DEBOUNCING_EMULATED(GpioExtension) != FALSE) &&
        (DebounceTimeout > 0x0)) {

        Status = GpiopAllocateDebounceResources(GpioBank,
                                                PinInformation,
                                                RelativePinNumber,
                                                &DebounceTimerContext,
                                                &NoiseFilterTimerContext,
                                                &DebounceNoiseTimerWorker);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "GpiopEnableInterrupt: Failed to allocate debounce "
                        "resources! Gsiv = %#x, PinNumber = %#x, Timeout = %#x,"
                        " Status = %d \n",
                        Virq,
                        AbsolutePinNumber,
                        DebounceTimeout,
                        Status);

            goto EnableInterruptEnd;
        }

        //
        // Initialize the default noise filter interval. If an override was
        // specified via the registry, then just use that value.
        //

        if (GpioGlobalNoiseIntervalOverride > 0x0) {
            NoiseFilterTimeout = GpioGlobalNoiseIntervalOverride;

        } else if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
            NoiseFilterTimeout =
                GPIO_DEFAULT_NOISE_FILTER_INTERVAL_MEMORY_MAPPED;

        } else {
            NoiseFilterTimeout = GPIO_DEFAULT_NOISE_FILTER_INTERVAL_PASSIVE;
        }

        //
        // Query for debounce and noise interval overrides for internal GPIO
        // clients.
        //

        GpioClnInvokeQueryDebounceNoiseIntervals(GpioExtension,
                                                 AbsolutePinNumber,
                                                 &DebounceTimeout,
                                                 &NoiseFilterTimeout,
                                                 &DebounceModelOverride);

        LevelReconfiguration = GpiopCheckLevelReconfigurationSupported(
                                   GpioBank,
                                   InterruptMode,
                                   Polarity,
                                   EmulateActiveBoth);

        EmulateDebounce = TRUE;
    }

    if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
        MemoryMapped = TRUE;

    } else {
        MemoryMapped = FALSE;
    }

    GpiopDebounceInitializeContext(MemoryMapped,
                                   EmulateDebounce,
                                   EmulateActiveBoth,
                                   LevelReconfiguration,
                                   DebounceModelOverride,
                                   &NewDebounceContext);

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopEnableInterrupt: Attempting to connect GPIO interrupt! "
                "Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, Mode = %#x, "
                "Polarity = %#x, Debounce interval (ms) = %#x, "
                "Noise filter interval (us) = 0x%I64x\n",
                GpioExtension,
                Virq,
                AbsolutePinNumber,
                InterruptMode,
                Polarity,
                (ULONG)(DebounceTimeout / 100),
                NoiseFilterTimeout);

    //
    // Take a reference on the component in a blocking manner when enabling
    // ACPI event pins. Enable requests for such pins arrive through a
    // direct path (and not via the secondary queues) and thus power reference
    // would not have been taken before the request got to this point.
    //

    if (AcpiEventInterrupt != FALSE) {
        PoFxActivateComponent(GpioExtension->PowerHandle,
                              GpioBank->BankId,
                              PO_FX_FLAG_BLOCKING);

        AcpiEventPowerReferenceTaken = TRUE;
    }

    //
    // Before enabling regular interrupts, mark the GPIO controller as not
    // stoppable or removable until the peripheral driver disconnects the
    // interrupt.
    //
    //
    // N.B. WDF ref-counts the StaticStopRemove enables and disables. So only
    //      last enable would actually enable stop/remove.
    //

    WdfDeviceSetStaticStopRemove(GpioExtension->Device, FALSE);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken before the request reaches to this point. Also, all operations
    // should have ceased by the time the device left the STARTED state.
    //

    GPIO_ASSERT((GpioBank->PowerData.IsActive != FALSE) &&
                (GPIO_GET_DEVICE_STATE(GpioExtension) == DEVICE_STATE_STARTED));

    //
    // Serialize operations on the GPIO bank.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // A pin configured for output cannot be configured for interrupt
    // simultaneously unless a registry override is present to allow this
    // configuration (for test purposes).
    //

    if ((PinInformation->Mode.Output != 0) &&
        (GpioAllowInterruptOutput == FALSE)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Pin already configured for output! "
                    "Gsiv = %#x, Pin = %#x, PinInformation = 0x%p\n",
                    Virq,
                    AbsolutePinNumber,
                    PinInformation);

        Status = STATUS_INVALID_PARAMETER;
        __fallthrough;
    }

    //
    // Fail the request if the pin is already configured for interrupt.
    //
    // Note this failure can happen if the firmware tries to share a pin for
    // ACPI events and regular device interrupts. This is not supported
    // currently.
    //
    // This can also happen if the firmware incorrectly declares multiple GPIO
    // interrupt descriptors referencing the same GPIO pin but the descriptors
    // are not identical (either in attributes or vendor data). Such pins
    // can be connected exclusively but not at the same time.
    //
    // N.B. The HAL will request an interrupt to be connected only if it is
    //      not already connected. For chained interrupts, connect requests
    //      after the first one only result in the IDT being updated.
    //

    if (PinInformation->Mode.Interrupt == 1) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Pin already configured for "
                    "interrupt! Gsiv = %#x, Pin = %#x, PinInformation = 0x%p\n",
                    Virq,
                    AbsolutePinNumber,
                    PinInformation);

        //
        // Add an assert to catch potentially bad firmware.
        //

        GPIO_ASSERT(AcpiEventInterrupt != FALSE);

        Status = STATUS_INVALID_PARAMETER;
        __fallthrough;
    }

    //
    // If the pin is already opened for IO as exclusive or if an opened pin is
    // being configured for interrupt in exclusive mode, then fail the request.
    //

    if (((PinInformation->Mode.Input != 0) ||
        (PinInformation->Mode.Output != 0)) &&
         ((PinInformation->Mode.ExclusiveMode != 0) ||
         (DescriptorOpenMode == GPIO_EXCLUSIVE))) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Pin already configured for IO in"
                    "exclusive mode or being opened for interrupt in "
                    "exclusive mode! Gsiv = %#x, Pin = %#x, "
                    "PinInformation = 0x%p\n",
                    Virq,
                    AbsolutePinNumber,
                    PinInformation);

        Status = STATUS_INVALID_PARAMETER;
        __fallthrough;
    }

    //
    // If pin is already opened/reserved for function config, it
    // cannot be used for interrupt.
    //
    
    if (PinInformation->Mode.FunctionConfig != 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Pin already reserved or configured"
                    " for function config! Gsiv = %#x, Pin = %#x,"
                    " PinInformation = 0x%p\n",
                    Virq,
                    AbsolutePinNumber,
                    PinInformation);

        Status = STATUS_INVALID_PARAMETER;
        __fallthrough;
    }

    //
    // Set the attributes for the pin. This is intentionally done prior to
    // calling the client driver's enable interrupt callback (and thus before
    // it is known whether the request will succeed or not). This is required
    // because the interrupt can fire the moment the client driver enables the
    // line for interrupt. The pin state must be accurate when the class
    // extension's ISR runs, which can race with setting of pin state.
    //

    if (NT_SUCCESS(Status)) {
        PinInformation->DebounceContext = NewDebounceContext;

        //
        // Erase any pending clears or mask/unmask values for this pin. This
        // will ensure that pending requests for those operations do not take
        // effect.
        //
        // This needs to be done under the protection of the interrupt lock (for
        // on-SoC GPIO) to ensure that in progress operations are completed
        // before the connect is attempted. Otherwise a mask or status clear
        // may be issued after a connect operation.
        //
        // N.B. The pending status clear only needs to be protected by the
        //      passive-level bank lock for off-SoC GPIOs. It needs to be
        //      protected by the interrupt for on-SoC GPIOs. However, taking the
        //      bank interrupt lock in both cases should not cause any issues.
        //

        GPIO_CLEAR_PIN_PENDING_DISCONNECT(GpioBank, RelativePinNumber);

        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);

        GPIO_CLEAR_PIN_DELAY_PENDING_STATUS_CLEAR(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_PENDING_STATUS_CLEAR(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_PENDING_INTERRUPT_MASK(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_PENDING_INTERRUPT_UNMASK(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_INTERRUPT_MASK(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_PENDING_RECONFIGURE(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_PENDING_ACTIVE_BOTH(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_DEBOUNCE_TIMER_MASK(GpioBank, RelativePinNumber);
        GPIO_CLEAR_PIN_NOISE_FILTER_TIMER_MASK(GpioBank, RelativePinNumber);

        //
        // Clear the last interrupt valid field.
        //

        PinInformation->LastInterruptWasValid = FALSE;

        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);

        //
        // Erase old failure counts associated with interrupts.
        //

        PinInformation->MaskUnmaskFailureCount = 0;
        PinInformation->StatusClearFailureCount = 0;
        PinInformation->DisconnectFailureCount = 0;

        //
        // Set the mode of the pin as interrupt. Also mark it as ACPI eventing
        // pin if it is configured as such.
        //

        PinInformation->Mode.Interrupt = 1;
        PinInformation->Mode.AcpiEvent = AcpiEventInterrupt;
        PinInformation->Mode.RunEVTMethod = RunEVTMethod;
        PinInformation->Mode.EmulateDebounce = EmulateDebounce;
        PinInformation->Mode.EmulateActiveBoth = EmulateActiveBoth;

        //
        // An interrupt pin can also be read from and thus can also be used as
        // input.
        //

        PinInformation->Mode.Input = 1;
        PinInformation->InputEnableCount += 1;

        //
        // Set the shared/exclusive mode for the pin.
        //

        if (DescriptorOpenMode == GPIO_EXCLUSIVE) {
            PinInformation->Mode.ExclusiveMode = 1;
        } else {
            PinInformation->Mode.ExclusiveMode = 0;
        }

        //
        // Set the interrupt mode, polarity and the interrupt callback context.
        //

        PinInformation->Polarity = Polarity;
        PinInformation->InterruptMode = InterruptMode;

        //
        // If the interrupt polarity is ActiveBoth, then program the correct
        // initial polarity at the controller.
        //

        if (EmulateActiveBoth != FALSE) {
            Polarity = InitialPolarity;

            //
            // To avoid race conditions with edges (missed edges), switch
            // the interrupt mode to be level.
            //

            InterruptMode = LevelSensitive;
        }

        PinInformation->CurrentInterruptPolarity = Polarity;
        PinInformation->CurrentInterruptMode = InterruptMode;
        PinInformation->InterruptCallbackContext = CallbackContext;

        //
        // Set the VIRQ mapping for the pin.
        //

        InterlockedExchange((PLONG)&PinInformation->Virq, Virq);

        //
        // The write to the cache must be ordered between PinInformation->Virq
        // and GpioClnInvokeEnableInterrupt().  It must also be ordered after
        // GpiopCreatePinEntry().
        //

        if (ARGUMENT_PRESENT(HubVirqEntryCache)) {
            OldHubVirqEntryCacheValue = (ULONG_PTR)InterlockedExchangePointer(
                                            (volatile PVOID *)HubVirqEntryCache,
                                            (PVOID)(ULONG_PTR)AbsolutePinNumber);
        }

        if (GpioDescriptor != NULL) {
            PinInformation->PullConfiguration =
                GPIO_BIOS_DESCRIPTOR_PULL_CONFIGURATION(GpioDescriptor);

        } else {
            PinInformation->PullConfiguration =
                GPIO_PIN_PULL_CONFIGURATION_DEFAULT;
        }

        PinInformation->DebounceTimeout = DebounceTimeout;
        PinInformation->DebounceTimerContext = DebounceTimerContext;
        PinInformation->NoiseFilterTimerContext = NoiseFilterTimerContext;
        PinInformation->NoiseFilterTimeout = NoiseFilterTimeout;
        PinInformation->DebounceNoiseTimerWorker = DebounceNoiseTimerWorker;

        if (EmulateDebounce != FALSE) {
            GpioBank->DebounceInterruptCount += 1;
        }

        //
        // Enabling an interrupt implies the pin is also now unmasked.
        //
        // N.B. The interrupt lock does not need to be acquired because:
        //      1. This field is updated atomically.
        //
        //      2. This cannot race with any other thread (e.g. masking the
        //         interrupt) since the interrupt has not been enabled by the
        //         GPIO client driver yet.
        //

        InterlockedExchange(&PinInformation->InterruptMaskCount, 0);

        //
        // Set the bit in enable register to mark the interrupt as enabled.
        //

        GPIO_SET_PIN_INTERRUPT_ENABLED(GpioBank, RelativePinNumber);

        //
        // Initialize the interrupt state.
        //

        GpiopInitializeInterruptState(PinInformation);

        //
        // Call into the GPIO client driver to enable the interrupt.
        //

        Status = GpioClnInvokeEnableInterrupt(GpioBank,
                                              RelativePinNumber,
                                              InterruptMode,
                                              Polarity,
                                              GpioDescriptor);

        //
        // Check even on failure to enforce an invariant.
        //

        GpiopCheckEnabledInterrupts(GpioBank);

        //
        // If the interrupt was not successfully enabled, then reset the pin
        // information.
        //

        if (!NT_SUCCESS(Status)) {
            PinInformation->Mode.Interrupt = 0;
            PinInformation->Mode.AcpiEvent = 0;
            PinInformation->Mode.RunEVTMethod = 0;
            PinInformation->Mode.EmulateDebounce = 0;
            PinInformation->Mode.EmulateActiveBoth = 0;

            //
            // Set the mode to shared as default. This is OK even if this isn't
            // the first open on the pin. The pin should already have been marked
            // in shared mode to get here.
            //

            PinInformation->Mode.ExclusiveMode = 0;
            if (ARGUMENT_PRESENT(HubVirqEntryCache)) {
                (VOID)InterlockedExchangePointer((volatile PVOID *)HubVirqEntryCache,
                                                 (PVOID)OldHubVirqEntryCacheValue);
            }

            InterlockedExchange((PLONG)&PinInformation->Virq, 0);
            PinInformation->InterruptMode = LevelSensitive;
            PinInformation->Polarity = InterruptPolarityUnknown;
            PinInformation->CurrentInterruptPolarity = InterruptPolarityUnknown;
            PinInformation->InterruptCallbackContext = 0;

            PinInformation->InputEnableCount -= 1;
            if (PinInformation->InputEnableCount == 0) {
                PinInformation->Mode.Input = 0;
            }

            PinInformation->PullConfiguration = GPIO_PIN_PULL_CONFIGURATION_DEFAULT;

            GPIO_CLEAR_PIN_INTERRUPT_ENABLED(GpioBank, RelativePinNumber);

            PinInformation->DebounceTimeout = 0;
            PinInformation->NoiseFilterTimeout = 0;
            if (EmulateDebounce != FALSE) {
                PinInformation->DebounceTimerContext = NULL;
                PinInformation->NoiseFilterTimerContext = NULL;
                PinInformation->DebounceNoiseTimerWorker = NULL;
                GpioBank->DebounceInterruptCount -= 1;
            }
        }
    }

    //
    // Activate the controller interrupt if it is not activated. This is
    // only needed if debounce time adjustment support is enabled.
    //

#ifdef DEBOUNCE_ELAPSED_TIME_ACCOUNTING

    if (NT_SUCCESS(Status) && (EmulateDebounce != FALSE)) {
        if ((GpioExtension->ActivelyManageBankInterrupt != FALSE) &&
            (GpioBank->DebounceInterruptCount == 1)) {

            GpiopSetInterruptState(GpioBank,
                                   GpioBank->BankInterrupt,
                                   TRUE);
        }
    }

#endif

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // If the interrupt was successfully connected, take a power reference on
    // the bank/component. This will prevent the PEP from transitioning the
    // bank to F1 while there is a pin that is actively connected for
    // interrupts.
    //
    // On failure, mark the driver as stoppable and removeable.
    //

    if (NT_SUCCESS(Status)) {
        PoFxActivateComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0);

    } else {
        WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);
    }

    //
    // If the request was for an ACPI event, and a reference was taken on the
    // component, then release it now. Additional reference was taken above
    // if interrupt was successfully connected.
    //

EnableInterruptEnd:
    if (AcpiEventPowerReferenceTaken != FALSE) {
        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);
    }

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Client driver failed enable "
                    "interrupt request! Gsiv = %#x, Status = %#x\n",
                    Virq,
                    Status);

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopEnableInterrupt: Successfully connected GPIO "
                    "interrupt! Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, "
                    "Mode = %#x, Polarity = %#x\n",
                    GpioExtension,
                    Virq,
                    AbsolutePinNumber,
                    InterruptMode,
                    Polarity);
    }

    if (!NT_SUCCESS(Status)) {
        if (DebounceTimerContext != NULL) {
            if (DebounceTimerContext->DebounceTimer != NULL) {
                ExDeleteTimer(DebounceTimerContext->DebounceTimer,
                              TRUE,
                              FALSE,
                              NULL);
            }

            GPIO_FREE_POOL(DebounceTimerContext);
        }

        if (NoiseFilterTimerContext != NULL) {
            if (NoiseFilterTimerContext->NoiseFilterTimer != NULL) {
                ExDeleteTimer(NoiseFilterTimerContext->NoiseFilterTimer,
                              TRUE,
                              FALSE,
                              NULL);
            }

            GPIO_FREE_POOL(NoiseFilterTimerContext);
        }

        if (DebounceNoiseTimerWorker != NULL) {
            WdfObjectDelete(DebounceNoiseTimerWorker);
        }
    }

    if (RhBuffer != NULL) {
        GPIO_FREE_POOL(RhBuffer);
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopDisableInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Virq,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine disables the specified interrupt line behind a GPIO controller.

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

    N.B. See warning in the routine description for GpiopUnmaskInterrupt.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Virq - Supplies the VIRQ for interrupt line that should be disconnected.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Surprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    BOOLEAN Debounced;
    WDFWORKITEM DebounceNoiseTimerWorker;
    PGPIO_DEBOUNCE_TIMER_CONTEXT DebounceTimerContext;
    PGPIO_NOISE_FILTER_TIMER_CONTEXT NoiseFilterTimerContext;
    CONTROLLER_ERROR_FLAGS Errors;
    PGPIO_BANK_ENTRY GpioBank;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER RelativePin;
    NTSTATUS Status;

    BankId = 0;
    DebounceNoiseTimerWorker = NULL;
    DebounceTimerContext = NULL;
    GpioBank = NULL;
    NoiseFilterTimerContext = NULL;
    RelativePin = 0;

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to enable an
    // interrupt if it manages the supplied VIRQ.
    //

    PinInformation = GpiopGetPinEntryFromVirq(GpioExtension,
                                              Virq,
                                              &GpioBank,
                                              &BankId,
                                              &RelativePin);

    if (PinInformation == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopDisableInterrupt: Gsiv (%#x) is invalid!\n",
                    Virq);

        Status = STATUS_INVALID_PARAMETER;
        goto DisableInterruptEnd;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopDisableInterrupt: Attempting to disconnect GPIO "
                "interrupt! Extn = 0x%p, Gsiv = %#x, Bank ID = 0x%x, "
                "RelativePinNumber = %#x\n",
                GpioExtension,
                Virq,
                BankId,
                RelativePin);

    //
    // N.B. The HAL will request an interrupt to be disabled only when
    //      the last connection to it goes away. For chained interrupts,
    //      disconnect requests prior to the last one only result in the IDT
    //      table being updated.
    //

    GPIO_ASSERT(PinInformation->Mode.Interrupt == 1);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken when the interrupt was connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    //
    // Luckily, GpiopDebounceWaitForStateMachineCompletionOfMaskedPin() cannot
    // be called while the pin is being simultaneously disabled.
    //

    //
    // If the request arrives in the Surprise Removal context, then GPIO client
    // driver is no longer allowed to touch hardware. Thus only state clean up
    // is performed and GPIO client driver not called.
    //

    if (InSurpriseRemovalContext == FALSE) {
        GpiopFlushPendingPinInterruptActivities(GpioBank,
                                                PinInformation,
                                                RelativePin);
    }

    //
    // Serialize operations on the GPIO bank.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Erase any pending status clear or mask requests for this pin.
    //
    // This needs to be done under the protection of the interrupt lock (for
    // on-SoC GPIO) to ensure that in-progress operations are completed
    // before the disconnect is attempted. Otherwise a mask or status clear may
    // be issued after a disconnect operation.
    //
    // N.B. The pending status clear only needs to be protected by the
    //      passive-level bank lock for off-SoC GPIOs. It needs to be protected
    //      by the interrupt for on-SoC GPIOs. However, taking the
    //      bank interrupt lock in both cases should not cause any issues.
    //

    GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);

    //
    // Clear the interrupt mask count to prevent further mask and unmask
    // requests.
    //

    InterlockedExchange(&PinInformation->InterruptMaskCount, 0x0);

    GPIO_CLEAR_PIN_DEBOUNCE_TIMER_MASK(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_DELAY_PENDING_STATUS_CLEAR(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_PENDING_STATUS_CLEAR(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_PENDING_INTERRUPT_MASK(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_PENDING_INTERRUPT_UNMASK(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_PENDING_RECONFIGURE(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_PENDING_ACTIVE_BOTH(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_DEBOUNCE_TIMER_MASK(GpioBank, RelativePin);
    GPIO_CLEAR_PIN_NOISE_FILTER_TIMER_MASK(GpioBank, RelativePin);

    //
    // Clear the last interrupt valid field.
    //

    PinInformation->LastInterruptWasValid = FALSE;

    GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);

    //
    // Call into the GPIO client driver to disable the interrupt.
    //
    // If the request arrives in the Surprise Removal context, then GPIO client
    // driver is no longer allowed to touch hardware. Thus only state clean up
    // is performed and GPIO client driver not called.
    //

    if (InSurpriseRemovalContext == FALSE) {
        GPIO_SET_PIN_PENDING_DISCONNECT(GpioBank, RelativePin);
        Status = GpiopDeferredDisableInterruptHandler(GpioBank, TRUE);

    } else {
        Status = STATUS_SUCCESS;
    }

    //
    // Clear the bit in enable register to mark the interrupt as disabled.
    //
    // Off-SoC:
    //   If the above disable request failed, there will be no interrupt storm
    //   before the retry since if the ISR fires, the retry will be done in
    //   there early.
    //
    // On-SoC:
    //   If the above disable request failed, the class extension would have
    //   bugchecked.
    //

    GPIO_CLEAR_PIN_INTERRUPT_ENABLED(GpioBank, RelativePin);

    //
    // Update the pin information to reflect the disable operation.
    //
    // N.B. The pin is assumed disabled irrespective of whether the
    //      client driver could successfully disable it or not. This is
    //      because IoDisconnectInterruptEx() does not return a NTSTATUS to
    //      the caller, so client would not be able to retry disable on failure.
    //
    //      It is assumed that the caller has requested its device to stop
    //      generating further interrupts prior to disconnecting. If an
    //      interrupt fires, that would imply an error on part of the device
    //      or its driver.
    //

    //
    // Clear the interrupt mode and associated flags.
    //

    PinInformation->Mode.Interrupt = 0;
    PinInformation->Mode.AcpiEvent = 0;
    PinInformation->Mode.RunEVTMethod = 0;
    PinInformation->Mode.EmulateDebounce = 0;
    PinInformation->Mode.EmulateActiveBoth = 0;

    //
    // Set the mode to shared (default). This is OK even if this isn't the
    // last disconnect on the pin. The pin should already be marked as shared
    // in that case.
    //

    PinInformation->Mode.ExclusiveMode = 0;

    //
    // Clear the VIRQ.
    //

    InterlockedExchange((PLONG)&PinInformation->Virq, 0);

    //
    // Clear the input flag if this was the last open on it.
    //

    PinInformation->InputEnableCount -= 1;
    if (PinInformation->InputEnableCount == 0) {
        PinInformation->Mode.Input = 0;
    }

    Debounced = (BOOLEAN)PinInformation->Mode.EmulateDebounce;
    PinInformation->Mode.EmulateDebounce = FALSE;

    //
    // Reset the interrupt state.
    //

    GpiopInitializeInterruptState(PinInformation);

    //
    // Check if the pin is being debounced. Pending debounce or noise would have
    // been cancelled as part of flushing pending interrupt activities on the
    // pin.
    //

    if (Debounced != FALSE) {
        DebounceTimerContext = PinInformation->DebounceTimerContext;
        PinInformation->DebounceTimerContext = NULL;

        NoiseFilterTimerContext = PinInformation->NoiseFilterTimerContext;
        PinInformation->NoiseFilterTimerContext = NULL;

        DebounceNoiseTimerWorker = PinInformation->DebounceNoiseTimerWorker;

        //
        // Drop the debounce timer count. If this was the last such interrupt,
        // then inactivate the DIRQL ISR. The DIRQL ISR is required for
        // memory-mapped GPIOs or off-SoC GPIOs that have a pre-process
        // callback.
        //

        GpioBank->DebounceInterruptCount -= 1;

#ifdef DEBOUNCE_ELAPSED_TIME_ACCOUNTING

        if ((GpioExtension->ActivelyManageBankInterrupt != FALSE) &&
            (GpioBank->DebounceInterruptCount == 0)) {

            GpiopSetInterruptState(GpioBank, GpioBank->BankInterrupt, FALSE);
        }

#endif

    }

    //
    // Release the passive-level bank lock.
    //

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // Drop the power reference that was previously taken on the bank/component
    // when the pin was connected. This can only be done if the interrupt
    // was successfully disconnected. Otherwise, the interrupt pin is still
    // live and can fire at any time.
    //

    if (NT_SUCCESS(Status)) {
        PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);

        //
        // As the interrupt is no longer connected, mark the driver as stoppable
        // and removable.
        //
        // N.B. 1. WDF ref-counts the StaticStopRemove enables and disables.
        //         So only last enable would actually enable stop/remove.
        //

        WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

    } else {

        //
        // Record the fact that a power reference had to be leaked due to a
        // disconnect failure.
        //

        Errors.AsULONG = 0;
        Errors.PowerReferenceLeaked = 1;
        InterlockedOr(
            (volatile LONG *)&GpioExtension->Errors.AsULONG,
            Errors.AsULONG);
    }

    //
    // Delete the debounce timer if one was created.
    //

DisableInterruptEnd:
    if (DebounceTimerContext != NULL) {
        if (DebounceTimerContext->DebounceTimer != NULL) {
            ExDeleteTimer(DebounceTimerContext->DebounceTimer,
                          TRUE,
                          TRUE,
                          NULL);
        }

        GPIO_FREE_POOL(DebounceTimerContext);
    }

    if (NoiseFilterTimerContext != NULL) {
        if (NoiseFilterTimerContext->NoiseFilterTimer != NULL) {
            ExDeleteTimer(NoiseFilterTimerContext->NoiseFilterTimer,
                          TRUE,
                          TRUE,
                          NULL);
        }

        GPIO_FREE_POOL(NoiseFilterTimerContext);
    }

    if (DebounceNoiseTimerWorker != NULL) {
        WdfObjectDelete(DebounceNoiseTimerWorker);
    }

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopDisableInterrupt: Client driver failed disable "
                    "interrupt request! Gsiv = %#x, Status = %#x\n",
                    Virq,
                    Status);

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopDisableInterrupt: Successfully disconnected GPIO "
                    "interrupt! Extn = 0x%p, Gsiv = %#x, Bank ID = 0x%x, "
                    "RelativePin = %#x\n",
                    GpioExtension,
                    Virq,
                    BankId,
                    RelativePin);
    }

    return Status;
}

__drv_sameIRQL
VOID
GpiopCheckEnabledInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine bugchecks if the GPIO hardware does not agree with the GPIO
    class extension on which interrupts are enabled for the specified bank.

    To "agree" means the hardware believes that a non-strict subset of
    the interrupts, that the class extension believes are enabled, are enabled.

    If the global variable GpioCheckEnabledInterrupts is FALSE, this routine
    does nothing.

    N.B. Caller requirements:

           Off-SoC:
             * The bank lock must be held (i.e. PASSIVE_LEVEL)
             * This may or may not be in the interrupt context

           On-SoC:
             * Interrupt context: Interrupt lock must be held
               (i.e. >= DISPATCH_LEVEL)
             * Non-interrupt context (e.g. enabling or disabling an interrupt):
               The bank lock must be held (i.e. PASSIVE_LEVEL)

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

Return Value:

    None.

--*/

{

    BOOLEAN AcquireInterruptLock;
    PGPIO_CLIENT_REGISTRATION_PACKET ClientPacket;
    ULONG64 PendingDisconnectMask;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 HardwareEnabledMask;
    BOOLEAN PassiveGpio;
    BOOLEAN RequestInInterruptContext;
    NTSTATUS Status;
    ULONG64 SoftwareEnabledMask;
    ULONG64 HardwareOnlyEnabledMask;

    GpioExtension = GpioBank->GpioExtension;
    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

#if DBG

    if (PassiveGpio != FALSE) {
        GPIO_ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);
        GPIO_ASSERT(GpioBank->PassiveLockOwner == KeGetCurrentThread());

    } else {
        GPIO_ASSERT((GpioBank->PassiveLockOwner == KeGetCurrentThread()) ||
                    (GpioBank->InterruptLockOwner ==
                     KeGetCurrentThread()));

    }

#endif

    if (GpioCheckEnabledInterrupts == FALSE) {
        return;
    }

    ClientPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    if (ClientPacket->CLIENT_QueryEnabledInterrupts == NULL) {
        return;
    }

    //
    // Off-Soc GPIO:
    //   The bank lock is already held in all cases so no special handling is
    //   required.
    //
    // On-SoC GPIO:
    //   If called from an interrupt context, the interrupt lock is already
    //   held so no special handling is required there either.
    //
    //   If not called from an interrupt context, acquire the interrupt lock
    //   to be consistent.
    //

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);
    RequestInInterruptContext = GPIO_IS_REQUEST_IN_INTERRUPT_CONTEXT(GpioBank);
    if ((PassiveGpio == FALSE) && (RequestInInterruptContext == FALSE)) {
        AcquireInterruptLock = TRUE;
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);

    } else {
        AcquireInterruptLock = FALSE;
    }

    //
    // Call into the GPIO client driver to query enabled interrupts.
    //

    Status = GpioClnInvokeQueryEnabledInterrupts(GpioBank,
                                                 &HardwareEnabledMask);

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Ignore failure to query enabled interrupts.
    //

    if (!NT_SUCCESS(Status)) {
        return;
    }

    SoftwareEnabledMask = GPIO_READ_ENABLE_REGISTER_VALUE(GpioBank);

    //
    // Determine which pins are enabled in hardware but are not enabled
    // according to the class extension's point of view (software).
    //
    // Pins that are enabled in software, but not hardware, are not
    // interesting as they cannot cause an interrupt storm. It could be
    // that:
    //
    // a) A software enable bit was set by GpiopEnableInterrupt but the
    //    client driver has not yet enabled the respective pin in hardware.
    // b) The client driver has disabled a pin but the respective software
    //    enable bit has not yet been cleared by GpiopDisableInterrupt.
    //

    HardwareOnlyEnabledMask = HardwareEnabledMask & ~SoftwareEnabledMask;

    if (PassiveGpio != FALSE) {

        //
        // Off-SoC case:
        //   If an off-SoC client driver failed a disable request, it is possible
        //   that the software enable bit for the pin has been prematurely cleared
        //   by GpiopDisableInterrupt prior to:
        //
        //   a) each subsequent retry of the request.
        //   b) the controller's interrupt on the parent being disabled due to
        //      multiple retries failing.
        //
        //   In both cases, the software pending disconnect bit will be set and the
        //   pin is not considered to be logically enabled.
        //
        //   Note that the bank lock is held during the following:
        //   1. the software pending disconnect bit being manipulated.
        //   2. the client driver being called to disable the pin.
        //   3. the current function being called.
        //
        // On-SoC case:
        //   It is not possible that the software enable bit for a bit will be
        //   prematurely cleared, so the pending disconnect bit is unimportant.
        //   This is because the client driver is always invoked to disable a pin
        //   prior to the software enable bit being cleared. Furthermore, the class
        //   extension will bugcheck if the disable request fails.
        //

        PendingDisconnectMask = GPIO_READ_PENDING_DISCONNECT_VALUE(GpioBank);
        HardwareOnlyEnabledMask &= ~PendingDisconnectMask;
    }

    if (HardwareOnlyEnabledMask != 0) {
        KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                     ENABLED_MASK_MISMATCH,
                     (ULONG_PTR)GpioBank,
                     (ULONG_PTR)(HardwareOnlyEnabledMask >> 32),
                     (ULONG)HardwareOnlyEnabledMask);
    }
}

__drv_sameIRQL
NTSTATUS
GpiopMaskInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Flags,
    __in ULONG Virq
    )

/*++

Routine Description:

    This routine masks the specified interrupt line behind a GPIO controller.
    Masking the interrupt line will prevent the CPU from being interrupted
    when the line fires again.

    N.B. 1. This routine may be called directly from within the context of
            the ISR (GPIO ISR --> kernel secondary interrupt dispatch -> target
            ISR is marked for passive-level dispatching, so HAL will attempt to
            mask the interrupt).

         2. This routine is called at DIRQL level if the GPIO controller is
            directly CPU-accessible (i.e. memory-mapped).

         3. This routine can be called at DISPATCH_LEVEL (by the kernel) even
            in the case of off-SoC GPIOs.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Flags - Supplies flags that control mask/unmask behavior. Valid flags
        for mask request:
        GPIO_REQUEST_FLAG_SERVICING_DEFERRED - ISR routine has been deferred
            due to IRQL requirements.

    Virq - Supplies the VIRQ for interrupt line that should be masked.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    BOOLEAN DispatchSynchronously;
    ULONG64 FailedMask;
    PGPIO_BANK_ENTRY GpioBank;
    KIRQL Irql;
    LONG MaskCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER RelativePin;
    BOOLEAN RequestInInterruptContext;
    NTSTATUS Status;

    //
    // Check whether the device is still in its D0 state. If the device is
    // not in D0 state (possible if it got (surprise) removed) then pretend as
    // if the operation was successful as no further interrupts should
    // originate from the device.
    //

    if (GPIO_GET_DEVICE_STATE(GpioExtension) != DEVICE_STATE_STARTED) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "%s: Device not in its started state! Bailing out on mask "
                    "request! Extn = 0x%p, State = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    GpioExtension->DeviceState);

        Status = STATUS_SUCCESS;
        goto MaskInterruptEnd;
    }

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to enable an
    // interrupt if it manages the supplied VIRQ.
    //

    FailedMask = 0;
    Status = STATUS_SUCCESS;
    PinInformation = GpiopGetPinEntryFromVirq(GpioExtension,
                                              Virq,
                                              &GpioBank,
                                              &BankId,
                                              &RelativePin);

    if (PinInformation == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopMaskInterrupt: Gsiv supplied is invalid! "
                    "Extn = 0x%p, Gsiv = %#x\n",
                    GpioExtension,
                    Virq);

        Status = STATUS_INVALID_PARAMETER;
        goto MaskInterruptEnd;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopMaskInterrupt: Attempting to mask GPIO interrupt! "
                "Extn = 0x%p, Gsiv = %#x, Bank ID = 0x%x, RelativePin = %#x\n",
                GpioExtension,
                Virq,
                BankId,
                RelativePin);

    //
    // Only a pin configured for interrupt can be masked.
    //

    if (PinInformation->Mode.Interrupt == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopMaskInterrupt: Pin not configured for interrupt! "
                    "Mode = %#x\n",
                    PinInformation->Mode.AsULONG);

        Status = STATUS_INVALID_PARAMETER;
        goto MaskInterruptEnd;
    }

    //
    // Increment the mask count on the pin and set appropriate bitmasks.
    //

    MaskCount = GpiopIncrementMaskCount(GpioBank, PinInformation, RelativePin);

    //
    // If the pin is already masked or there is an outstanding mask request
    // (pending mask), then no further processing is required.
    //

    if (MaskCount > 1) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopMaskInterrupt: Successfully masked GPIO interrupt! "
                    "Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, Mask count = %d\n",
                    GpioExtension,
                    Virq,
                    RelativePin,
                    MaskCount);

        goto MaskInterruptEnd;
    }

    //
    // Determine whether the request should be dispatched immediately or
    // deferred. Couple of cases exist:
    //
    // 1. If the request arrives in the interrupt context, then defer it.
    //    All such requests are batched and will be dispatched from within the
    //    ISR servicing routine.
    //
    // 2. If the incoming IRQL is greater than the synchronization IRQL of
    //    the controller, then dispatch the request asynchronously.
    //

    DispatchSynchronously = TRUE;
    RequestInInterruptContext = GPIO_IS_REQUEST_IN_INTERRUPT_CONTEXT(GpioBank);
    if ((CHECK_FLAG(Flags, GPIO_REQUEST_FLAG_SERVICING_DEFERRED) != FALSE) ||
        (RequestInInterruptContext != FALSE)) {

        DispatchSynchronously = FALSE;

    } else {
        Irql = KeGetCurrentIrql();
        if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
            if (Irql > GpioExtension->SynchronizationIrql) {
                DispatchSynchronously = FALSE;
            }

        } else if (Irql > PASSIVE_LEVEL) {
            DispatchSynchronously = FALSE;
        }
    }

    //
    // Set the DelayPendingStatusClear flag if required.
    //

    if ((CHECK_FLAG(Flags, GPIO_REQUEST_FLAG_SERVICING_DEFERRED) != FALSE) ||
        (RequestInInterruptContext != FALSE)) {

        GPIO_SET_PIN_DELAY_PENDING_STATUS_CLEAR(GpioBank, RelativePin);
    }

    //
    // If the mask request cannot be synchronously issued, then schedule a
    // deferred worker.
    //

    if (DispatchSynchronously == FALSE) {
        Status = GpiopScheduleDeferredInterruptActivities(GpioBank);
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopMaskInterrupt: Scheduled a deferred mask request! "
                    "Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, Status = %#x\n",
                    GpioExtension,
                    Virq,
                    RelativePin,
                    Status);

    } else {

        //
        // Make sure the GPIO bank is active. A power reference should have
        // been taken when the interrupt was connected.
        //

        GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

        Status = GpiopDeferredMaskRequestHandler(GpioBank);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "GpiopMaskInterrupt: Client driver failed mask "
                        "interrupt request! Gsiv = %#x, Status = %#x\n",
                        Virq,
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "GpiopMaskInterrupt: Successfully masked GPIO "
                        "interrupt! Extn = 0x%p, Gsiv = %#x, PinNumber = %#x\n",
                        GpioExtension,
                        Virq,
                        RelativePin);
        }
    }

MaskInterruptEnd:
    return Status;
}

__drv_sameIRQL
NTSTATUS
GpiopUnmaskInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Flags,
    __in ULONG Virq
    )

/*++

Routine Description:

    This routine unmasks the specified interrupt line behind a GPIO controller.
    Unmasking the interrupt line will enable the CPU to be interrupted when the
    line fires again.

    Entry IRQL:
        1. This routine can be called at DISPATCH_LEVEL (by the kernel) even
           in the case of off-SoC GPIOs. It can also be called at passive-level
           for off-SoC GPIOs.

        2. This routine can be called at D-IRQL, DISPATCH_LEVEL or passive-level
           for memory-mapped GPIOs.

    Calling contexts:
        1. This routine may be invoked synchronously from within the ISR
           context. The bank interrupt lock will be held for on-SoC GPIOs and
           passive-level bank lock will be held for off-SoC GPIOs.

        2. The request may be dispatched asynchronously from within the
           debouncing module with bank interrupt lock held for on-SoC GPIOs
           and passive-level bank lock held for off-SoC GPIOs.


Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Flags - Supplies flags that control mask/unmask behavior. Relevant flags
        defined are:
            GPIO_REQUEST_FLAG_SERVICING_COMPLETE

    Virq - Supplies the VIRQ for interrupt line that should be unmasked.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    BANK_ID BankId;
    BOOLEAN DispatchSynchronously;
    PGPIO_BANK_ENTRY GpioBank;
    KIRQL Irql;
    LONG MaskCount;
    BOOLEAN PassiveGpio;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER RelativePin;
    BOOLEAN SkipUnmaskRequest;
    NTSTATUS Status;

#if DBG

    ULONG64 EnableRegister;

#endif

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to enable an
    // interrupt if it manages the supplied VIRQ.
    //
    // Note if the device has gone away and is no longer in D0, then this
    // translation will fail (as all interrupts were cleaned up when the
    // device went away).
    //

    Status = STATUS_SUCCESS;
    PinInformation = GpiopGetPinEntryFromVirq(GpioExtension,
                                              Virq,
                                              &GpioBank,
                                              &BankId,
                                              &RelativePin);

    if (PinInformation == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopUnmaskInterrupt: Invalid Gsiv! Extn = 0x%p, "
                    "Gsiv = %#x\n",
                    GpioExtension,
                    Virq);

        Status = STATUS_INVALID_PARAMETER;
        goto UnmaskInterruptEnd;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopUnmaskInterrupt: Attempting to unmask GPIO interrupt! "
                "Extn = 0x%p, Gsiv = %#x, PinNumber = %#x\n",
                GpioExtension,
                Virq,
                RelativePin);

    //
    // Only a pin configured for interrupt can be unmasked.
    //

    if (PinInformation->Mode.Interrupt == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopUnmaskInterrupt: Pin not configured for interrupt!"
                    "Mode = %#x\n",
                    PinInformation->Mode.AsULONG);

        Status = STATUS_INVALID_PARAMETER;
        goto UnmaskInterruptEnd;
    }

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken when the interrupt was connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    MaskCount =
        InterlockedCompareExchange(&PinInformation->InterruptMaskCount, 0, 0);

    //
    // If the mask count is already zero, then it implies there is an extra
    // dereference (unmask) that happened somewhere, which is a bug in the
    // component that triggered the extra dereference.
    //

    if (MaskCount == 0) {


        //
        // Driver verifier complains if you complete an IRP with a warning code.
        // Thus return success in this case as the interrupt is unmasked from
        // the callers point of view.
        //
        // Status = STATUS_GPIO_INTERRUPT_ALREADY_UNMASKED;
        //

        Status = STATUS_SUCCESS;
        goto UnmaskInterruptEnd;
    }

    //
    // Decrement the mask count on the pin.
    //
    //

    MaskCount = InterlockedDecrement(&PinInformation->InterruptMaskCount);
    GPIO_ASSERT(MaskCount >= 0);

    //
    // If the pin is still supposed to be masked, then no further processing is
    // required.
    //

    if (MaskCount > 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopUnmaskInterrupt: Successfully unmasked GPIO interrupt"
                    "! Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, MaskCount = %d\n",
                    GpioExtension,
                    Virq,
                    RelativePin,
                    MaskCount);

        goto UnmaskInterruptEnd;
    }

    //
    // Determine whether the request should be dispatched immediately or
    // deferred. Couple of cases exist:
    //
    // 1. If the request arrives in the interrupt context, then defer it.
    //    All such requests are batched and will be dispatched from within the
    //    ISR servicing routine.
    //
    // 2. If the incoming IRQL is greater than than the synchronization IRQL of
    //    the controller, then dispatch the request asynchronously.
    //

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);
    DispatchSynchronously = TRUE;
    Irql = KeGetCurrentIrql();
    if (PassiveGpio == FALSE) {
        if (Irql > GpioExtension->SynchronizationIrql) {
            DispatchSynchronously = FALSE;
        }

    } else if (Irql > PASSIVE_LEVEL) {
        DispatchSynchronously = FALSE;
    }

    if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;
    }

    //
    // For a 1 -> 0, acquire the lock to mark the interrupt request as pending.
    //
    // The interrupt mask count (0->1 or 1->0 transitions) are protected by
    // the bank interrupt lock *even in cases of off-SoC GPIO as well*
    //
    // The lock is already acquired for an on-SoC GPIO if the request originates
    // within the context of the ISR and thus need not be acquired.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Clear the delayed status clear flag to let subsequent status clear
    // requests flow directly to the controller. If there were status clear
    // requests pending on this pin, then this will also cause them to be
    // committed when the unmask request is dispatched.
    //
    // N.B. The DelayPendingStatusClear flag must be updated under the interrupt
    //      lock.
    //

    if (CHECK_FLAG(Flags, GPIO_REQUEST_FLAG_SERVICING_COMPLETE) != FALSE) {
        GPIO_CLEAR_PIN_DELAY_PENDING_STATUS_CLEAR(GpioBank, RelativePin);
    }

    //
    // Re-read the interrupt mask count now under the lock. There could be
    // a mask request that raced ahead of this unmask request, thereby
    // cancelling it.
    //

    MaskCount = InterlockedCompareExchange(
                    &PinInformation->InterruptMaskCount,
                    0,
                    0);

    //
    // If the unmask request is still valid, then mark the pin has having a
    // unmask request outstanding. This implies that any pin mask requests
    // should not be attempted. Treat the pin as effectively unmasked from this
    // point on.
    //

    if (MaskCount == 0) {


#if DBG

        EnableRegister = GPIO_READ_REGISTER_VALUE(
                            GpioBank,
                            &GpioBank->InterruptData.EnableRegister);

        GPIO_ASSERT ((EnableRegister & (1ULL << RelativePin)) != 0);

#endif

        GPIO_CLEAR_PIN_PENDING_INTERRUPT_MASK(GpioBank, RelativePin);
        GPIO_SET_PIN_PENDING_INTERRUPT_UNMASK(GpioBank, RelativePin);
        GPIO_CLEAR_PIN_INTERRUPT_MASK(GpioBank, RelativePin);
        SkipUnmaskRequest = FALSE;

    } else {
        SkipUnmaskRequest = TRUE;
        Status = STATUS_SUCCESS;
        __fallthrough;
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    if (SkipUnmaskRequest != FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopUnmaskInterrupt: Unmask request skipped due a "
                    "parallel mask request! Extn = 0x%p, Gsiv = %#x, "
                    "PinNumber = %#x, Mask count = %d\n",
                    GpioExtension,
                    Virq,
                    RelativePin,
                    MaskCount);

        goto UnmaskInterruptEnd;
    }

    //
    // If the unmask request cannot be synchronously issued, then schedule a
    // deferred worker.
    //

    if (DispatchSynchronously == FALSE) {
        Status = GpiopScheduleDeferredInterruptActivities(GpioBank);
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "GpiopUnmaskInterrupt: Scheduled a deferred unmask request!"
                    "Extn = 0x%p, Gsiv = %#x, PinNumber = %#x, Status = %#x\n",
                    GpioExtension,
                    Virq,
                    RelativePin,
                    Status);

    } else {

        //
        // Make sure the GPIO bank is active. A power reference should have
        // been taken when the interrupt was connected.
        //

        GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

        Status = GpiopDeferredUnmaskRequestHandler(GpioBank);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "GpiopUnmaskInterrupt: Client driver failed unmask "
                        "interrupt request! Gsiv = %#x, Status = %#x\n",
                        Virq,
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "GpiopUnmaskInterrupt: Successfully unmasked GPIO "
                        "interrupt! Extn = 0x%p, Gsiv = %#x, PinNumber = %#x\n",
                        GpioExtension,
                        Virq,
                        RelativePin);
        }
    }

UnmaskInterruptEnd:
    return Status;
}

__drv_sameIRQL
NTSTATUS
GpiopRequestInterrupt (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Virq,
    __out PULONG NextGsiv
    )

/*++

Routine Description:

    This routine requests that the specified interrupt line should be considered
    to have asserted the next time this interrupt controllers line has asserted.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO controller extension.

    Virq - Supplies the VIRQ for interrupt line that should be asserted.

    NextGsiv - Supplies a pointer to a variable which receives the GSIV that
        needs to be asserted to ensure the eventual servicing the requested
        interrupt, if applicable.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PGPIO_BANK_ENTRY GpioBank;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    LONG64 PreviousReplayRegister;
    PIN_NUMBER RelativePin;
    NTSTATUS Status;

    //
    // Get the entry from the pin information table that maps to this VIRQ.
    // The hub driver will only request the class extension to request an
    // interrupt if it manages the supplied VIRQ.
    //

    Status = STATUS_SUCCESS;
    PinInformation = GpiopGetPinEntryFromVirq(GpioExtension,
                                              Virq,
                                              &GpioBank,
                                              &BankId,
                                              &RelativePin);

    if (PinInformation == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopRequestInterrupt: Gsiv supplied is invalid! "
                    "Extn = 0x%p, Gsiv = %#x\n",
                    GpioExtension,
                    Virq);

        Status = STATUS_INVALID_PARAMETER;
        goto RequestInterruptEnd;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopRequestInterrupt: Attempting to request GPIO interrupt! "
                "Extn = 0x%p, Gsiv = %#x, Bank ID = 0x%x, RelativePin = %#x\n",
                GpioExtension,
                Virq,
                BankId,
                RelativePin);

    //
    // Only a pin configured for interrupt can be asserted.
    //

    if (PinInformation->Mode.Interrupt == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopRequestInterrupt: Pin not configured for interrupt!"
                    "Mode = %#x\n",
                    PinInformation->Mode.AsULONG);

        Status = STATUS_INVALID_PARAMETER;
        goto RequestInterruptEnd;
    }

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken when the interrupt was connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    // Set the corresponding bit to indicate the interrupt needs to be replayed.
    //

    PreviousReplayRegister =
        InterlockedOr64(&GpioBank->InterruptData.ReplayRegister,
                        (1ULL << RelativePin));

    if ((PreviousReplayRegister & (1ULL << RelativePin)) == 0) {
        *NextGsiv = GpioBank->InterruptData.Gsiv;

        //
        // If the interrupt was already pending, then terminate the replay as
        // previous request would have already queued up all the appropriate
        // interrupts.  Distinguish this case from a virtual controller which
        // had no output pin by setting a unique status code.
        //

    } else {

        *NextGsiv = GPIO_UNSPECIFIED_INTERRUPT;
        Status = STATUS_SECONDARY_INTERRUPT_ALREADY_PENDING;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "GpiopRequestInterrupt: Successfully requested GPIO interrupt!"
                " Extn = 0x%p, Gsiv = %#x, PinNumber = %#x\n",
                GpioExtension,
                Virq,
                RelativePin);

RequestInterruptEnd:
    return Status;
}

LONG
GpiopIncrementMaskCount (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine increments the interrupt mask count on the specified pin. It
    returns the resulting mask count. A mask count of 1 implies that the caller
    has to schedule a mask request.

    Entry IRQL:
        1. This routine can be called at D-IRQL, DISPATCH_LEVEL or PASSIVE_LEVEL
           for memory-mapped GPIOs.

        2. This routine can be called at DISPATCH_LEVEL or PASSIVE_LEVEL for
           off-SoC GPIOs.

    Calling contexts:
         1. The caller may have already acquired the bank interrupt lock prior
            to invocation.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinInformation - Supplies a pointer to the pin information for the specified
        pin.

    PinNumber - Supplies the bank-relative pin number to be masked.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    LONG MaskCount;

    //
    // Determine whether the interrupt lock need to be acquired or not.
    //
    // Note the interrupt mask count is protected by the bank interrupt lock
    // *even in cases of off-SoC GPIO as well*.
    //
    // The lock is already acquired for an on-SoC GPIO if the request originates
    // within the context of the ISR and thus need not be acquired.
    //

    if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Increment the mask count on the pin.
    //

    MaskCount = InterlockedIncrement(&PinInformation->InterruptMaskCount);

    //
    // Treat the pin as effectively masked from this point on.
    //

    if (MaskCount == 1) {
        GPIO_CLEAR_PIN_PENDING_INTERRUPT_UNMASK(GpioBank, PinNumber);
        GPIO_SET_PIN_PENDING_INTERRUPT_MASK(GpioBank, PinNumber);
        GPIO_SET_PIN_INTERRUPT_MASK(GpioBank, PinNumber);
    }

    //
    // If the bank interrupt lock was acquired, then release it now.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    return MaskCount;
}

__drv_sameIRQL
NTSTATUS
GpiopDeferredMaskRequestHandler (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine flushes all interrupt mask requests on lines for which masking
    was previously deferred. Note this routine can be called:
    1. Synchronously in the context of the mask request, or
        - No lock held

    2. Lazily in the context of the interrupt service routine, or
        - Interrupt lock or passive-level lock held as appropriate

    3. Lazily in the context of a worker
        - No lock held

    N.B. 1. This does NOT wait for all activity related to these lines to
            complete -- see e.g. GpiopWaitForStateMachineCompletionOfDebouncedMaskedPin().

         2. This routine can be called at DIRQL or PASSIVE_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank to be masked.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    BOOLEAN AcquirePassiveLock;
    ULONG64 FailedMask;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 OldPendingMask;
    BOOLEAN PassiveGpio;
    NTSTATUS Status;

#if DBG

    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER PinNumber;
    ULONG64 TemporaryMaskValue;

#endif

    //
    // Read the pending interrupt mask register value. If it is zero, then
    // there is no work to be done.
    //

    GpioExtension = GpioBank->GpioExtension;
    Status = STATUS_SUCCESS;
    OldPendingMask = GPIO_READ_PENDING_MASK_REGISTER_VALUE(GpioBank);
    if (OldPendingMask == 0x0) {
        goto DeferredMaskRequestHandlerEnd;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "%s: Masking pins for Bank = %p, PinMask = 0x%I64x\n",
                __FUNCTION__,
                GpioBank,
                OldPendingMask);

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    GPIO_ASSERT((PassiveGpio == FALSE) ||
                (KeGetCurrentIrql() == PASSIVE_LEVEL));

    //
    // Determine whether passive-level and/or interrupt lock need to be
    // acquired.
    //

    if ((PassiveGpio != FALSE) &&
        (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) == FALSE)) {

        AcquirePassiveLock = TRUE;

    } else {
        AcquirePassiveLock = FALSE;
    }

    //
    // The mask request can be called in the context of the interrupt service
    // routine). In such case, the lock is already acquired by the GPIO ISR
    // routine but only for on-SoC GPIOs. It is also acquired for on-SoC GPIOs
    // when the request is issued from the debounce engine.
    //
    // Note the pending interrupt mask/unmask registers are protected by the
    // bank interrupt lock even for off-SoC GPIOs. Thus the lock should
    // not be acquired in those cases also.
    //

    if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
        AcquireInterruptLock = TRUE;

    } else {

        GPIO_ASSERT(PassiveGpio == FALSE);

        AcquireInterruptLock = FALSE;
    }

    //
    // Acquire the passive-level and/or interrupt locks in the correct order.
    //

    if (AcquirePassiveLock != FALSE) {
        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Atomically read and clear the pending interrupt mask register value.
    //

    OldPendingMask = GPIO_SET_PENDING_MASK_REGISTER_VALUE(GpioBank, 0x0);

    //
    // Clear off interrupts that have been already unmasked (due to an
    // unmask request that came prior to the current mask attempt),
    // or disconnected before the interrupt could be successfully masked.
    //
    // N.B. The interrupt mask value needs to be read under the interrupt lock.
    //

    OldPendingMask &= GpioBank->InterruptData.InterruptMask;
    if (OldPendingMask != 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "%s: Masking pins. Bank = %p, PinMask = 0x%I64x\n",
                    __FUNCTION__,
                    GpioBank,
                    OldPendingMask);

        //
        // On CHK builds, validate the each pin being masked is still
        // enabled for interrupt. Note the interrupt mask count is decremented
        // outside the bank interrupt lock and thus could have dropped to zero
        // already. The behavior in such case would be that the pin would get
        // masked here and then immediately unmasked by the thread handling the
        // unmask request.
        //

#if DBG

        TemporaryMaskValue = OldPendingMask;
        while (TemporaryMaskValue > 0) {
            PinNumber = RtlFindLeastSignificantBit(TemporaryMaskValue);
            TemporaryMaskValue &= ~(1ULL << PinNumber);
            PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

            GPIO_ASSERT((PinInformation != NULL) &&
                        (PinInformation->Mode.Interrupt == 1));
        }

#endif

        //
        // For passive-level GPIOs, drop the interrupt lock (DPC_LEVEL spinlock)
        // before calling into the client driver.
        //

        if (PassiveGpio != FALSE) {
            GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        //
        // Call into the client driver to mask the interrupts.
        //

        FailedMask = 0x0;
        Status = GpioClnInvokeMaskInterrupts(GpioBank,
                                             OldPendingMask,
                                             &FailedMask);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: Failed masking pins. Bank = %p, PinMask = 0x%I64x,"
                        " FailedMask = 0x%I64x, Status = %#x\n",
                        __FUNCTION__,
                        GpioBank,
                        OldPendingMask,
                        FailedMask,
                        Status);
        }

        GPIO_ASSERT(!NT_SUCCESS(Status) || (FailedMask == 0x0));
        GPIO_ASSERT((FailedMask & ~OldPendingMask) == 0);

        //
        // Work around buggy client drivers: Ignore pins that the client driver
        // failed to mask but that the class extension never requested be
        // masked.
        //

        FailedMask &= OldPendingMask;

        //
        // For passive-level GPIOs, re-acquire the interrupt lock to set the
        // interrupt mask values.
        //

        if (PassiveGpio != FALSE) {
            GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        GpiopHandleInterruptMaskRequestCompletion(GpioBank,
                                                  OldPendingMask,
                                                  FailedMask);
    }

    //
    // Release the passive-level and/or interrupt locks in the correct order.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    if (AcquirePassiveLock != FALSE) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

    //
    // For on-SoC GPIO controllers, a mask request cannot fail (otherwise
    // the system will bugcheck). For off-SoC GPIO controllers, this operation
    // can fail but the request will be retried asynchronously. So the line is
    // effectively masked from the caller's point of view.
    //
    // Hence always return STATUS_SUCCESS.
    //

DeferredMaskRequestHandlerEnd:
    return STATUS_SUCCESS;
}

__drv_sameIRQL
NTSTATUS
GpiopDeferredUnmaskRequestHandler (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine flushes all interrupt unmask requests on lines for which
    the operation was previously deferred. Note this routine can be called:
    1. Synchronously in the context of the unmask request, or
        - No lock held

    2. Lazily in the context of the interrupt service routine, or
        - Interrupt lock or passive-level lock held as appropriate

    3. In the context of debouncing, or
        - Interrupt lock or passive-level lock held as appropriate

    4. Lazily in the context of a worker
        - No lock held

    N.B. This routine can be called at DIRQL or PASSIVE_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank to be unmasked.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    BOOLEAN AcquirePassiveLock;
    ULONG64 FailedUnmask;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 OldPendingUnmask;
    BOOLEAN PassiveGpio;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER PinNumber;
    NTSTATUS Status;
    ULONG64 TemporaryUnmaskValue;

#if DBG

    LONG MaskCount;
    KIRQL OldIrql;

    OldIrql = KeGetCurrentIrql();

#endif

    //
    // Read the pending interrupt unmask register value. If it is zero, then
    // there is no work to be done.
    //

    Status = STATUS_SUCCESS;
    OldPendingUnmask = GPIO_READ_PENDING_UNMASK_REGISTER_VALUE(GpioBank);
    if (OldPendingUnmask == 0x0) {
        goto DeferredUnmaskRequestHandlerEnd;
    }

    //
    // The unmask request can be called in the context of the interrupt service
    // routine). In such case, the lock is already acquired by the GPIO ISR
    // routine but only for on-SoC GPIOs. The pending interrupt mask/unmask
    // registers are protected by the bank interrupt lock. Thus the lock should
    // not be acquired in that (off-SoC GPIO) case again.
    //

    GpioExtension = GpioBank->GpioExtension;
    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    GPIO_ASSERT((PassiveGpio == FALSE) ||
                (KeGetCurrentIrql() == PASSIVE_LEVEL));

    //
    // Determine whether passive-level and/or interrupt lock need to be
    // acquired.
    //
    // For passive-GPIOs, since this request always arrives at passive-level,
    // the thread will never own the interrupt lock on entry.
    //

    if ((PassiveGpio != FALSE) &&
        (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) == FALSE)) {

        AcquirePassiveLock = TRUE;

    } else {
        AcquirePassiveLock = FALSE;
    }

    if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;
    }

    //
    // Acquire the passive-level and/or interrupt locks in the correct order.
    //

    if (AcquirePassiveLock != FALSE) {
        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Atomically read and clear the pending interrupt unmask register value.
    //

    OldPendingUnmask = GPIO_SET_PENDING_UNMASK_REGISTER_VALUE(GpioBank, 0x0);

    //
    // Clear off interrupts that have been already masked (due to a mask
    // request that came prior to the current unmask attempt),
    // or disconnected before the interrupt could be successfully unmasked.
    //
    // N.B. The interrupt mask value needs to be read under the interrupt lock.
    //

    OldPendingUnmask &= ~GpioBank->InterruptData.InterruptMask;
    OldPendingUnmask &= GpioBank->InterruptData.EnableRegister;
    if (OldPendingUnmask != 0) {

        //
        // GPIO_INTERRUPT_DATA::PendingInterruptMask and
        // GPIO_PIN_INFORMATION_ENTRY::InterruptMaskCount are consistent when
        // the interrupt lock (DPC_LEVEL spinlock) is held.
        //

#if DBG

        TemporaryUnmaskValue = OldPendingUnmask;
        while (TemporaryUnmaskValue > 0) {
            PinNumber = RtlFindLeastSignificantBit(TemporaryUnmaskValue);
            TemporaryUnmaskValue &= ~(1ULL << PinNumber);
            PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

            GPIO_ASSERT((PinInformation != NULL) &&
                        (PinInformation->Mode.Interrupt == 1));

            MaskCount = InterlockedCompareExchange(
                            &PinInformation->InterruptMaskCount,
                            0,
                            0);

            GPIO_ASSERT(MaskCount == 0);
        }

#endif

        //
        // For passive-level GPIOs, drop the interrupt lock (DPC_LEVEL spinlock)
        // before calling into the client driver.
        //

        if (PassiveGpio != FALSE) {
            GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        //
        // If status clear was pended due to ISR servicing being deferred or
        // interrupt reconfigure during debouncing, then perform it now.
        // Note this won't have any effect on pins that are still marked for
        // deferred status clear (as is desired).
        //

        GpiopDeferredStatusClearRequestHandler(GpioBank);

        FailedUnmask = 0x0;
        TemporaryUnmaskValue = OldPendingUnmask;
        while (TemporaryUnmaskValue > 0) {
            PinNumber = RtlFindLeastSignificantBit(TemporaryUnmaskValue);
            TemporaryUnmaskValue &= ~(1ULL << PinNumber);
            PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

            GPIO_ASSERT(PinInformation != NULL);

            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Unmasking pins. Bank = %p, Pin number = %#x\n",
                        __FUNCTION__,
                        GpioBank,
                        PinNumber);

            //
            // Call into the GPIO client driver to unmask the interrupt.
            //

            Status = GpioClnInvokeUnmaskInterrupt(
                        GpioBank,
                        PinNumber,
                        PinInformation->CurrentInterruptMode,
                        PinInformation->CurrentInterruptPolarity,
                        PinInformation->PullConfiguration,
                        (USHORT)PinInformation->DebounceTimeout);

            if (!NT_SUCCESS(Status)) {
                FailedUnmask |= ((ULONG64)1 << PinNumber);
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_INTERRUPT,
                            "%s: Failed unmasking pin. Bank = %p, Pin number = "
                            "%#x, Status = %#x\n",
                            __FUNCTION__,
                            GpioBank,
                            PinNumber,
                            Status);
            }
        }

        //
        // For passive-level GPIOs, reacquire the interrupt lock to set the
        // interrupt unmask values.
        //

        if (PassiveGpio != FALSE) {
            GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        GpiopHandleInterruptUnmaskRequestCompletion(GpioBank,
                                                    OldPendingUnmask,
                                                    FailedUnmask);
    }

    //
    // Release the passive-level and/or interrupt locks in the correct order.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    if (AcquirePassiveLock != FALSE) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

    //
    // For on-SoC GPIO controllers, an unmask request cannot fail (otherwise
    // the system will bugcheck). For off-SoC GPIO controllers, this operation
    // can fail but the request will be retried asynchronously (N times). So
    // return STATUS_SUCCESS to treat this line as effectively unmasked from
    // caller's perspective.
    //
    // N.B. It is possible that all N attempts to unmask the line could fail
    //      but given that is being tried asynchronously, there is no way
    //      to communicate this error to the driver. The driver would stop
    //      receiving interrupts and may stop functioning.
    //

DeferredUnmaskRequestHandlerEnd:

    GPIO_ASSERT(KeGetCurrentIrql() == OldIrql);

    return STATUS_SUCCESS;
}

__drv_sameIRQL
NTSTATUS
GpiopDeferredStatusClearRequestHandler (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine flushes all status clear requests on lines for which the
    operation was previously deferred. Note this routine can be called:
    1. Synchronously in the context of the unmask request, or
        - No lock held

    2. Lazily in the context of the interrupt service routine, or
        - Interrupt lock or passive-level lock held as appropriate

    3. Lazily in the context of a worker
        - No lock held

    N.B. 1. This routine can be called at DIRQL or PASSIVE_LEVEL.
         2. This routine may be called to clear status bits for non-enabled
            interrupts as well.

Arguments:

   GpioBank - Supplies a pointer to the GPIO's bank to be lazily masked.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    BOOLEAN AcquirePassiveLock;
    ULONG64 FailedStatusClear;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 OldPendingStatusClear;
    BOOLEAN PassiveGpio;
    NTSTATUS Status;
    ULONG64 StatusClearMask;

    //
    // Read the pending status clear register value. If it is zero, then
    // there is no work to be done.
    //

    Status = STATUS_SUCCESS;
    OldPendingStatusClear = GPIO_READ_PENDING_STATUS_CLEAR_VALUE(GpioBank);
    if (OldPendingStatusClear == 0x0) {
        goto DeferredStatusClearRequestHandlerEnd;
    }

    //
    // The unmask request can be called in the context of the interrupt service
    // routine). In such case, the lock is already acquired by the GPIO ISR
    // routine but only for on-SoC GPIOs. The pending interrupt mask/unmask
    // registers are protected by the bank interrupt lock. Thus the lock should
    // not be acquired in that (off-SoC GPIO) case again.
    //

    GpioExtension = GpioBank->GpioExtension;
    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    GPIO_ASSERT((PassiveGpio == FALSE) ||
                (KeGetCurrentIrql() == PASSIVE_LEVEL));

    //
    // Serialize "status clear" operations on the GPIO controller. If the
    // controller is not memory-mapped (i.e. behind a serial bus), then the
    // serialization is done using the passive-level bank lock. Otherwise, the
    // bank interrupt lock is used.
    //

    AcquireInterruptLock = FALSE;
    AcquirePassiveLock = FALSE;
    if (PassiveGpio == FALSE) {
        if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
            AcquireInterruptLock = TRUE;
        }

    } else {
        if (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) == FALSE) {
            AcquirePassiveLock = TRUE;
        }
    }

    //
    // Acquire the passive-level or interrupt lock.
    //

    if (AcquirePassiveLock != FALSE) {
        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Atomically read and clear the pending status clear register.
    //

    OldPendingStatusClear = GPIO_SET_PENDING_STATUS_CLEAR_VALUE(GpioBank, 0x0);

    //
    // Remove ones for which interrupts have already been disabled. Since the
    // interrupt is not enabled, the status register being set should not
    // cause an interrupt to be generated.
    //

    OldPendingStatusClear &= GpioBank->InterruptData.EnableRegister;

    //
    // If there pins for which status clear needs to be processed, then process
    // them now.
    //

    if (OldPendingStatusClear != 0) {

        //
        // Remove ones with delay pending status clear set.
        //

        StatusClearMask = OldPendingStatusClear;
        StatusClearMask &= ~GpioBank->InterruptData.DelayPendingStatusClearMask;

        //
        // Put back the pending ones that aren't going to be cleared in this
        // round.
        //

        OldPendingStatusClear &= ~StatusClearMask;
        if (OldPendingStatusClear != 0x0) {
            GPIO_SET_PENDING_STATUS_CLEAR_PINMASK(GpioBank,
                                                  OldPendingStatusClear);
        }

        //
        // If there are pins to be cleared, then issue a request now.
        //

        FailedStatusClear = 0x0;
        if (StatusClearMask > 0) {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Clearing status for pins. Bank = %p, "
                        "PinMask = 0x%I64x\n",
                        __FUNCTION__,
                        GpioBank,
                        StatusClearMask);

            Status = GpioClnInvokeClearActiveInterrupts(GpioBank,
                                                        StatusClearMask,
                                                        &FailedStatusClear);

            if (!NT_SUCCESS(Status)) {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_INTERRUPT,
                            "%s: Failed clearing status for pins. Bank = %p, "
                            "PinMask = 0x%I64x, FailedClear = 0x%I64x, "
                            "Status = %#x\n",
                            __FUNCTION__,
                            GpioBank,
                            StatusClearMask,
                            FailedStatusClear,
                            Status);
            }
        }

        GPIO_ASSERT(!NT_SUCCESS(Status) || (FailedStatusClear == 0x0));

        GpiopHandleStatusClearRequestCompletion(GpioBank,
                                                StatusClearMask,
                                                FailedStatusClear);
    }

    //
    // Release the passive-level or interrupt lock.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    if (AcquirePassiveLock != FALSE) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

DeferredStatusClearRequestHandlerEnd:
    return Status;
}

NTSTATUS
GpiopDeferredDisableInterruptHandler (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in BOOLEAN SynchronousRequest
    )

/*++

Routine Description:

    This routine flushes all interrupt disconnect on lines for which disconnect
    operation failed previously. Note this routine can be called:
    1. Synchronously in the context of the disconnect request, or
        - Passive-level bank lock held.

    2. Lazily in the context of the interrupt service routine, or
        - Interrupt lock or passive-level lock held

    3. Lazily in the context of a worker
        - No lock held

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank to be lazily masked.

    SynchronousRequest - Supplies whether the invocation is being done in
        the context of disconnect request (TRUE) or lazily (FALSE).

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN ClientRetryFlag;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 OldPendingDisconnect;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PIN_NUMBER PinNumber;
    BOOLEAN RequestInInterruptContext;
    BOOLEAN Retry;
    NTSTATUS Status;

    //
    // If this request is called from within the interrupt context or a DPC,
    // then bail out. The client driver is supposed to be called at
    // PASSIVE_LEVEL for disconnecting interrupts even for memory-mapped GPIOs.
    //

    Status = STATUS_SUCCESS;
    if (KeGetCurrentIrql() >= DISPATCH_LEVEL) {
        goto DeferredDisableInterruptHandlerEnd;
    }

    //
    // Check whether the device is still in its D0 state. If the device is
    // not in D0 state (possible if it got (surprise) removed) then fail all
    // further operations.
    //

    GpioExtension = GpioBank->GpioExtension;
    if (GPIO_GET_DEVICE_STATE(GpioExtension) != DEVICE_STATE_STARTED) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: Device not in its started state! Terminating"
                    "deferred disable operation! Extn = %p, State = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    GpioExtension->DeviceState);

        Status = STATUS_UNSUCCESSFUL;
        goto DeferredDisableInterruptHandlerEnd;
    }

    //
    // Read the pending disconnect register value. If it is zero, then
    // there is no work to be done.
    //

    OldPendingDisconnect = GPIO_READ_PENDING_DISCONNECT_VALUE(GpioBank);
    if (OldPendingDisconnect == 0x0) {
        goto DeferredDisableInterruptHandlerEnd;
    }

    //
    // Serialize operations on the GPIO controller.
    //
    // This routine can be called synchronously in the context of the disconnect
    // request or in the context of the interrupt service routine (in case of
    // passive-level interrupts). In both cases, the lock is already acquired
    // by the caller and thus should not be acquired again. It is safe to
    // access the relevant bank-related data.
    //

    RequestInInterruptContext = GPIO_IS_REQUEST_IN_INTERRUPT_CONTEXT(GpioBank);
    if ((SynchronousRequest == FALSE) && (RequestInInterruptContext == FALSE)) {
        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    //
    // Atomically read and clear the pending disconnect register.
    //

    OldPendingDisconnect = GPIO_SET_PENDING_DISCONNECT_VALUE(GpioBank, 0x0);
    if (OldPendingDisconnect != 0) {
        Retry = FALSE;
        while (OldPendingDisconnect > 0) {
            PinNumber = RtlFindLeastSignificantBit(OldPendingDisconnect);
            OldPendingDisconnect &= ~(1ULL << PinNumber);
            PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

            GPIO_ASSERT(PinInformation != NULL);

            //
            // Check if this a first disable attempt or a retry.
            //

            if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) &&
                (PinInformation->DisconnectFailureCount > 0)) {

                ClientRetryFlag = TRUE;

            } else {
                ClientRetryFlag = FALSE;
            }

            //
            // Call into the GPIO client driver to disable the interrupt.
            //

            Status = GpioClnInvokeDisableInterrupt(GpioBank,
                                                   PinNumber,
                                                   ClientRetryFlag);

            if (!NT_SUCCESS(Status)) {
                Retry = TRUE;
                GPIO_SET_PIN_PENDING_DISCONNECT(GpioBank, PinNumber);
            }

            //
            // Check even on failure (where the pending disconnect bit has been
            // set for the pin) to enforce an invariant.
            //

            GpiopCheckEnabledInterrupts(GpioBank);
        }

        if (Retry != FALSE) {
            GpiopHandleDisableInterruptRequestFailure(GpioBank);
        }
    }

    //
    // If the passive-level controller lock was acquired, then release it now.
    //

    if ((SynchronousRequest == FALSE) && (RequestInInterruptContext == FALSE)) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

DeferredDisableInterruptHandlerEnd:
    return Status;
}

VOID
GpiopServiceActiveBothInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine services ActiveBoth interrupts. If ActiveBoth interrupts are
    being emulated, then it flips the interrupt polarity (and mode) on each
    interrupt for emulated pins that have asserted.

    Entry IRQL:
        1. This routine can be called at D-IRQL for memory-mapped GPIO
           controllers and PASSIVE_LEVEL for off-SoC GPIO controllers.

    Calling contexts: This routine may be invoked:
        1. Synchronously in the context of the interrupt service routine, or
            - Bank interrupt lock held for memory-mapped GPIOs and
              passive-level bank lock held for off-SoC GPIOs.

        2. In the context of interrupt debouncing (downgrading from "possibly
           enhanced" to Legacy.
            - Bank interrupt lock held for memory-mapped GPIOs and
              passive-level bank lock held for off-SoC GPIOs.

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank.

Return Value:

    None.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN PassiveGpio;
    ULONG64 PendingActiveBoth;
    PIN_NUMBER PinNumber;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;

    //
    // Atomically read and clear the pending ActiveBoth register value.
    //

    PendingActiveBoth = GPIO_SET_PENDING_ACTIVE_BOTH_VALUE(GpioBank, 0x0);
    if (PendingActiveBoth == 0x0) {
        goto ServiceActiveBothInterruptsEnd;
    }

    GpioExtension = GpioBank->GpioExtension;

    //
    // This routine should only be entered if ActiveBoth interrupts are being
    // emulated.
    //

    GPIO_ASSERT(GPIO_IS_ACTIVE_BOTH_EMULATED(GpioExtension) != FALSE);

    //
    // Validate the IRQL and locking requirements. This routine should only be
    // entered at >= DISPATCH_LEVEL for memory-mapped GPIOs and PASSIVE_LEVEL
    // for off-SoC GPIOs.
    //

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);
    if (PassiveGpio != FALSE) {

        GPIO_ASSERT((KeGetCurrentIrql() == PASSIVE_LEVEL) &&
                    (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) != FALSE));

    } else {

        GPIO_ASSERT((KeGetCurrentIrql() >= DISPATCH_LEVEL) &&
                    (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) != FALSE));
    }

    //
    // Clear off interrupts that are no longer connected.
    //

    PendingActiveBoth &= GpioBank->InterruptData.EnableRegister;

    //
    // Walk through the bitmask of pins that need to be flipped and request
    // each pin to be reconfigured.
    //

    while (PendingActiveBoth > 0) {
        PinNumber = RtlFindLeastSignificantBit(PendingActiveBoth);
        PendingActiveBoth &= ~(1ULL << PinNumber);
        PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

        GPIO_ASSERT(PinInformation != NULL);

        //
        // Request the pin to be reconfigured.
        //

        GpiopReconfigureInterrupts(GpioBank, PinNumber);

        //
        // Unmask the pin (it was masked in the ISR when the interrupt fired).
        //

        GpiopUnmaskInterrupt(GpioExtension,
                             GPIO_REQUEST_FLAG_NONE,
                             PinInformation->Virq);
    }

ServiceActiveBothInterruptsEnd:
    return;
}

__drv_sameIRQL
NTSTATUS
GpiopFlipPinPolarity (
    _In_ PGPIO_BANK_ENTRY GpioBank,
    _In_ PIN_NUMBER PinNumber
    )

/*++

    This routine flips the interrupt polarity and mode of an emulated
    ActiveBoth or emulated debounce pin that has asserted.

    An interrupt should not be reconfigured while it is still active. This
    routine assumes that the client driver believes that the interrupt is
    masked. It does not matter if there is a pending unmask and
    GPIO_PIN_INFORMATION_ENTRY::InterruptMaskCount is 0, as long as the
    client driver is not aware of that.

    N.B. 1. Entry IRQL: D-IRQL for memory-mapped GPIOs and PASSIVE_LEVEL for
            off-SoC GPIOs.

         2. This routine is invoked with the D-IRQL bank interrupt lock held
            for memory-mapped GPIO controllers and passive-level bank lock held
            for off-SoC GPIO controllers.

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank.

    PinNumber - Supplies the number of bank-relative pin.

Return Value:

    NTSTATUS code.

--*/

{

    KINTERRUPT_MODE CurrentInterruptMode;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    BOOLEAN PassiveGpio;
    KINTERRUPT_POLARITY Polarity;
    ULONG RetryCount;
    NTSTATUS Status;
    BOOLEAN TolerateFailure;

    //
    // Validate the IRQL and locking requirements.
    //

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioBank->GpioExtension);
    if (PassiveGpio != FALSE) {

        GPIO_ASSERT((GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) != FALSE) &&
                    (KeGetCurrentIrql() == PASSIVE_LEVEL));

    } else {

        GPIO_ASSERT(GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) != FALSE);
    }

    PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

    GPIO_ASSERT((PinInformation != NULL) &&
                (PinInformation->Mode.Interrupt == 1) &&
                ((PinInformation->Mode.EmulateActiveBoth == 1) ||
                 (PinInformation->Mode.EmulateDebounce == 1)));

    //
    // For edge-triggered pins that are purely being reconfigured due to
    // debounce (i.e., they are not emulated ActiveBoth pins), also flip
    // the interrupt mode on each invocation.
    //

    CurrentInterruptMode = PinInformation->CurrentInterruptMode;
    if ((PinInformation->Mode.EmulateActiveBoth == 0) &&
        (PinInformation->InterruptMode == Latched)) {

        if (CurrentInterruptMode == Latched) {
            CurrentInterruptMode = LevelSensitive;

        } else {
            CurrentInterruptMode = Latched;
        }
    }

    if (PinInformation->CurrentInterruptPolarity == InterruptActiveHigh) {
        Polarity = InterruptActiveLow;

    } else {

        GPIO_ASSERT(PinInformation->CurrentInterruptPolarity ==
                    InterruptActiveLow);

        Polarity = InterruptActiveHigh;
    }

    RetryCount = RECONFIGURE_FAILURE_COUNT_THRESHOLD;
    TolerateFailure =
        GPIO_RECONFIGURE_FAILURE_ALLOWED(&PinInformation->DebounceContext);

    //
    // Attempt to reconfigure the pin.
    //

    do {
        Status = GpioClnInvokeReconfigureInterrupt(GpioBank,
                                                   PinNumber,
                                                   CurrentInterruptMode,
                                                   Polarity,
                                                   TolerateFailure);

        //
        // On success, record the updated polarity as the current polarity
        // for the line.
        //
        // N.B. The interrupt may immediately fire on reconfigure. But the
        //      current polarity value can be safely updated as the
        //      interrupt lock is still held.
        //

        if (NT_SUCCESS(Status)) {
            PinInformation->CurrentInterruptPolarity = Polarity;
            PinInformation->CurrentInterruptMode = CurrentInterruptMode;
            TraceEvents(GpioBank->GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Reconfigured interrupt. Bank = %p, Pin number"
                        " = %#x, New Mode = %#x, New Polarity = %#x\n",
                        __FUNCTION__,
                        GpioBank,
                        PinNumber,
                        CurrentInterruptMode,
                        Polarity);

        } else {
            TraceEvents(GpioBank->GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Failed reconfiguring interrupt! Bank = %p, "
                        "Pin number = %#x, New Mode = %#x, New Polarity"
                        " = %#x Status = %#x\n",
                        __FUNCTION__,
                        GpioBank,
                        PinNumber,
                        CurrentInterruptMode,
                        Polarity,
                        Status);
        }

        RetryCount -= 1;

    } while (!NT_SUCCESS(Status) && (RetryCount > 0));

    return Status;
}

__drv_sameIRQL
NTSTATUS
GpiopDeferredReconfigureRequestHandler (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine reconfigures interrupts to service emulated ActiveBoth and
    emulated debounce interrupts. This routine flips the interrupt polarity and
    mode on each interrupt for emulated ActiveBoth or emulated debounce pins
    that have asserted.

    An interrupt should not be reconfigured while it is still active. This
    routine assumes that the caller has masked the interrupt prior to
    invocation.

    Entry IRQL:
        1. This routine can be called at D-IRQL for memory-mapped GPIO
           controllers and PASSIVE_LEVEL for off-SoC GPIO controllers.

    Calling contexts:
        This routine may be invoked:
        1. Synchronously in the context of the interrupt service routine, or
            - Interrupt lock or passive-level lock held as appropriate

        2. In the context of interrupt debouncing, or
            - Interrupt lock or passive-level lock held as appropriate

        3. Asynchronously in the context of DPC (deferred DPC handler),
            - No lock held.

        3. Lazily in the context of a worker (for off-SoC GPIOs only)
            - No lock held

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN AcquireInterruptLock;
    BOOLEAN AcquirePassiveLock;
    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN PassiveGpio;
    ULONG64 PendingReconfigure;
    PIN_NUMBER PinNumber;
    NTSTATUS Status;

#if DBG

    LONG MaskCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;

#endif

    //
    // Read the pending pending reconfigure register value. If it is zero, then
    // there is no work to be done.
    //

    Status = STATUS_SUCCESS;
    PendingReconfigure = GPIO_READ_PENDING_RECONFIGURE_VALUE(GpioBank);
    if (PendingReconfigure == 0x0) {
        goto DeferredReconfigureRequestHandlerEnd;
    }

    GpioExtension = GpioBank->GpioExtension;

    //
    // This routine should only be entered if ActiveBoth interrupts are being
    // emulated or debouncing is being emulated.
    //

    GPIO_ASSERT((GPIO_IS_ACTIVE_BOTH_EMULATED(GpioExtension) != FALSE) ||
                (GPIO_IS_DEBOUNCING_EMULATED(GpioExtension) != FALSE));

    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    GPIO_ASSERT((PassiveGpio == FALSE) ||
                (KeGetCurrentIrql() == PASSIVE_LEVEL));

    //
    // Serialize reconfigure operations on the GPIO controller. If the
    // controller is memory-mapped, then serialization is done using the
    // bank interrupt lock. Otherwise, the passsive-level bank lock is used
    // for serialization.
    //
    // Note the lock may be already acquired when called from ISR or debouncing
    // context.
    //

    AcquireInterruptLock = FALSE;
    AcquirePassiveLock = FALSE;
    if (PassiveGpio == FALSE) {
        if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
            AcquireInterruptLock = TRUE;
        }

    } else {
        if (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) == FALSE) {
            AcquirePassiveLock = TRUE;
        }
    }

    //
    // Acquire the passive-level bank lock or bank interrrupt lock, whichever
    // is necessary.
    //

    if (AcquirePassiveLock != FALSE) {
        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Atomically read and clear the pending reconfigure register value.
    //

    PendingReconfigure = GPIO_SET_PENDING_RECONFIGURE_VALUE(GpioBank, 0x0);

    //
    // Clear off interrupts that are no longer connected.
    //

    PendingReconfigure &= GpioBank->InterruptData.EnableRegister;

    //
    // Walk through the bitmask of pins that need to be flipped and reconfigure
    // each of them. For emulated pins, they need to be flipped to be
    // level-triggered (to avoid missing edges) and opposite of current
    // polarity.
    //

    while (PendingReconfigure > 0) {
        PinNumber = RtlFindLeastSignificantBit(PendingReconfigure);
        PendingReconfigure &= ~(1ULL << PinNumber);

#if DBG

        PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

        GPIO_ASSERT(PinInformation != NULL);

        MaskCount = InterlockedCompareExchange(
                        &PinInformation->InterruptMaskCount,
                        0,
                        0);

        GPIO_ASSERT(MaskCount > 0);

#endif

        Status = GpiopFlipPinPolarity(GpioBank, PinNumber);
    }

    //
    // Release the passive-level bank lock or bank interrrupt lock, whichever
    // was acquired.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    if (AcquirePassiveLock != FALSE) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

DeferredReconfigureRequestHandlerEnd:
    return Status;
}

NTSTATUS
GpiopReconfigureInterrupts (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_opt PIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine handles request for reconfiguring interrupt pins.
    Reconfiguration is required when servicing debounced interrupts and/or
    emulating ActiveBoth interrupts.

    Entry IRQL:
        1. This routine can be called at D-IRQL for memory-mapped GPIO
           controllers and PASSIVE_LEVEL for off-SoC GPIO controllers.


    Calling contexts:
        1. This routine may be invoked synchronously from within the ISR
           context (for non-debounced ActiveBoth interrupts) or outside of it
           (for enhanced debounce cases).

        2. The bank interrupt will be held for memory-mapped GPIOs. The
           passive-level bank lock may be held for off-SoC GPIOs if invoked from
           ISR context.

Arguments:

    GpioBank - Supplies a pointer to the GPIO's bank.

    PinNumber - Supplies the bank-relative pin number to be reconfigured.
        Note if this value is INVALID_PIN then assume that the caller has
        already set the mask of requisite pins to be reconfigured.

Return Value:

    None.

--*/

{

    BOOLEAN DispatchSynchronously;
    PDEVICE_EXTENSION GpioExtension;
    ULONG64 PendingReconfigure;
    NTSTATUS Status;

    //
    // If a valid pin number is specified, then update the activeboth mask to
    // cause the specified pin configuration to be flipped.
    //

    if (PinNumber != INVALID_PIN_NUMBER) {
        GPIO_SET_PIN_PENDING_RECONFIGURE(GpioBank, PinNumber);
    }

    //
    // Determine if there are any pending reconfigurations. Bail out if
    // there are none (the request may have been picked up by someone other
    // thread).
    //

    PendingReconfigure = GPIO_READ_PENDING_RECONFIGURE_VALUE(GpioBank);
    if (PendingReconfigure == 0x0) {
        Status = STATUS_SUCCESS;
        goto ReconfigureInterruptsEnd;
    }

    //
    // Determine whether the request should be dispatched immediately or
    // deferred. Couple of cases exist:
    //
    // For memory-mapped controllers, the request can be always dispatched
    // synchronously. For off-SoC GPIOs, dispatch requests that arrive at
    // PASSIVE_LEVEL (either directly from within ISR context or from a
    // worker thread) synchronously. Defer requests that arrive at
    // DISPATCH_LEVEL for off-SoC GPIO controllers.
    //

    GpioExtension = GpioBank->GpioExtension;
    if ((GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) ||
        (KeGetCurrentIrql() == PASSIVE_LEVEL)) {

        DispatchSynchronously = TRUE;

    } else {
        DispatchSynchronously = FALSE;
    }

    //
    // If the unmask request cannot be synchronously issued, then schedule a
    // deferred worker.
    //

    if (DispatchSynchronously == FALSE) {
        Status = GpiopScheduleDeferredInterruptActivities(GpioBank);
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "%s: Scheduled a deferred reconfigure request!"
                    "Extn = 0x%p, PinNumber = %#x, Status = %#x\n",
                    __FUNCTION__,
                    GpioExtension,
                    PinNumber,
                    Status);

    } else {

        //
        // Make sure the GPIO bank is active. A power reference should have
        // been taken when the interrupt was connected.
        //

        GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

        Status = GpiopDeferredReconfigureRequestHandler(GpioBank);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: Client driver failed reconfigure interrupt!"
                        "Extn = %p, PinNumber = %#x, Status = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        PinNumber,
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Successfully reconfigured interrupt!"
                        "Extn = %p, PinNumber = %#x, Status = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        PinNumber,
                        Status);
        }
    }

ReconfigureInterruptsEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
WDF_TRI_STATE
GpiopCheckLevelReconfigurationSupported (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in KINTERRUPT_MODE InterruptMode,
    __in KINTERRUPT_POLARITY Polarity,
    __in BOOLEAN EmulatedActiveBoth
    )

/*++

Routine Description:

    This routine determines whether the given interrupt can be configured as
    (Level, <opposite polarity>) or not. This is needed by the debounce engine
    to determine if the line is stable during an interval of time.

    Note for emulated-ActiveBoth pins, this may return WdfTrue even though it
    eventually turns out that the controller doesn't support level
    reconfiguration. It is assumed that the caller will invoke the debounce
    state machine to update the debounce model.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    InterruptMode - Supplies the trigger mode (edge or level) associated with
        this interrupt.

    Polarity - Supplies the polarity (active low or active high) associated with
        this interrupt.

    EmulatedActiveBoth - Supplies whether the pin is configured for emulated
        ActiveBoth or not.

Return Value:

    WdfTrue if the interrupt can be configured for [Level, <opposite polarity>].
    WdfFalse if the interrupt cannot be reconfigured for level-mode.
    WdfUseDefault if it cannot be determined whether reconfiguration is
        supported or not.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    BOOLEAN MemoryMapped;
    WDF_TRI_STATE ReconfigurationSupported;

    UNREFERENCED_PARAMETER(Polarity);

    GpioExtension = GpioBank->GpioExtension;
    ReconfigurationSupported = WdfFalse;

    //
    // Determine whether the controller is memory-mapped or not.
    //

    if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE) {
        MemoryMapped = TRUE;

    } else {
        MemoryMapped = FALSE;
    }

    //
    // Reconfiguration for (Level, <opposite polarity> is likely supported if
    // the pin is an emulated ActiveBoth pin.
    //
    // Else, if it is not a hardware ActiveBoth pin, the client driver
    // implements reconfigure interrupt, and:
    //
    // 1. If the interrupt is level-triggered: then likely the controller
    //    supports reconfiguration.
    //
    // 2. If the controller is memory-mapped: then it is likely to support
    //    level reconfiguration as well.
    //
    // 3. Participates in ActiveBoth emulation for other pins: then
    //    reconfiguration is possibly supported.
    //
    // Hardware ActiveBoth pins cannot be reconfigured because the GPIO class
    // extension has no idea which level stays asserted after an edge triggers.
    //

    if (EmulatedActiveBoth != FALSE) {
        ReconfigurationSupported = WdfTrue;

    } else if (Polarity != InterruptActiveBoth) {
        if (GpioExtension->CombinedPacket.CLIENT_ReconfigureInterrupt != NULL) {
            if ((InterruptMode == LevelSensitive) ||
                (MemoryMapped != FALSE) ||
                (GPIO_IS_ACTIVE_BOTH_EMULATED(GpioExtension) != FALSE)) {

                ReconfigurationSupported = WdfUseDefault;
            }
        }
    }

    return ReconfigurationSupported;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFlushPendingPinInterruptActivities (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine quiesces all the interrupt activities on the pin (e.g
    debouncing or noise filtering) such that is safe to disconnect the
    interrupt. This prevents interrupt disconnect from racing with debouncing
    or interrupt servicing activites. It will also prevent a disconnect
    race with an unmask that drops the count to zero.

    This routine assumes the caller has invoked it at PASSIVE_LEVEL with no
    bank locks held for both memory-mapped GPIOs and off-SoC GPIOs.

    Refer to GpiopDisableInterrupt() on reasons this routine is not marked as
    PAGED.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinInformation - Supplies a pointer to the pin information for the specified
        pin.

    PinNumber - Supplies the bank relative pin number.

Return Value:

    None.

--*/

{

    WDFWORKITEM DebounceNoiseWorker;
    PVOID DebounceTimer;
    PDEVICE_EXTENSION GpioExtension;
    ULONG Gsiv;
    KIRQL Irql;
    PVOID NoiseFilterTimer;

    DebounceNoiseWorker = NULL;
    DebounceTimer = NULL;
    GpioExtension = GpioBank->GpioExtension;
    Gsiv = PinInformation->Virq;
    NoiseFilterTimer = NULL;

    //
    // The steps to be taken are:
    //
    // 1. Mask the interrupt to prevent further interrupts from being generated
    //    from the pin.
    //

    GpiopMaskInterrupt(GpioExtension, GPIO_REQUEST_FLAG_NONE, Gsiv);

    //
    // 2. Acquire the passive-level bank lock and bank interrupt lock to
    //    synchronize with any pending interrupt and debounce/noise
    //    filtering activities on this pin.
    //
    //    For ACPI event pins, additionally acquire the ACPI event lock.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);

    //
    // Attempt to cancel any pending request to queue debounce or noise timers.
    //

    GPIO_CLEAR_PIN_DEBOUNCE_TIMER_MASK(GpioBank, PinNumber);

    GPIO_CLEAR_PIN_NOISE_FILTER_TIMER_MASK(GpioBank, PinNumber);

    if (PinInformation->DebounceTimerContext != NULL) {
        DebounceTimer = ((PGPIO_DEBOUNCE_TIMER_CONTEXT)
            PinInformation->DebounceTimerContext)->DebounceTimer;

        DebounceNoiseWorker = ((PGPIO_DEBOUNCE_TIMER_CONTEXT)
            PinInformation->DebounceTimerContext)->DebounceNoiseTimerWorker;
    }

    if (PinInformation->NoiseFilterTimerContext != NULL) {
        NoiseFilterTimer = ((PGPIO_NOISE_FILTER_TIMER_CONTEXT)
            PinInformation->NoiseFilterTimerContext)->NoiseFilterTimer;
    }

    GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);

    //
    // Attempt to cancel any pending ACPI event method evaluation request.
    //
    // N.B. This can't be done with the bank interrupt lock acquired for
    //      off-SoC GPIOs.
    //

    if (PinInformation->Mode.AcpiEvent == 1) {

        GPIO_ACQUIRE_ACPI_EVENT_LOCK(GpioExtension, &Irql);

        GpioUtilUpdatePinValue(GpioBank,
                               PinNumber,
                               AcpiRegisterTypePendingEvents,
                               FALSE);

        GPIO_RELEASE_ACPI_EVENT_LOCK(GpioExtension, Irql);
    }

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // Cancel any pending debounce and noise filtering timers on this pin.
    //

    if (DebounceTimer != NULL) {
        ExCancelTimer(DebounceTimer, NULL);
    }

    if (NoiseFilterTimer != NULL) {
        ExCancelTimer(NoiseFilterTimer, NULL);
    }

    //
    // 3. Flush all queued DPCs to force any pending DPCs to finish.
    //

    KeFlushQueuedDpcs();

    //
    // 4. Flush the worker queue and debounce noise worker.
    //
    // N.B. Clearing the pending event with ACPI lock acquired is sufficient
    //      for ACPI events.
    //
    //

    if (DebounceNoiseWorker != NULL) {
        WdfWorkItemFlush(DebounceNoiseWorker);
    }

    return;
}

//
// ------------------------------------------------- Function config routines
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopConnectFunctionConfigPinsReserve (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in PLARGE_INTEGER ConnectionId
    )

/*++

Routine Description:

    This routine reserves the specified set of lines for function config.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    ConnectionId - Supplies the Connection ID value corresponding to the
        request.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    BANK_ID CurrentBankId;
    BOOLEAN DelayedConnect;
    PUSHORT DescriptorPinTable;
    PPNP_FUNCTION_CONFIG_DESCRIPTOR FunctionConfigDescriptor;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG Index;
    ULONG Length;
    ULONG MinimumLength;
    ULONG PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    PPIN_NUMBER PinNumberTable;
    ULONG PinTableSize;
    PIN_NUMBER RelativePin;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER RhBuffer;
    NTSTATUS Status;
    BOOLEAN Valid;
    USHORT Value;

    PAGED_CODE();

    DelayedConnect = FALSE;
    Index = 0;
    PinInformationTable = NULL;
    PinNumberTable = NULL;
    RhBuffer = NULL;

    GPIO_ASSERT(ConnectionId != NULL);

    FunctionConfigDescriptor = NULL;
    
    //
    // Query the Resource Hub to fetch the BIOS descriptor associated with this
    // request.
    //

    Status = GpiopQueryResourceHubBiosDescriptor(GpioExtension->Device,
                                                 &GpioExtension->ResourceHubTarget,
                                                 FALSE,
                                                 ConnectionId->QuadPart,
                                                 &RhBuffer);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: Failed to query BIOS descriptor "
                    "from the Resource Hub! ID = [0x%I64x], Status = %#x\n",
                    __FUNCTION__,
                    ConnectionId->QuadPart,
                    Status);

        goto ConnectFunctionConfigPinsReserveEnd;
    }

    FileContext->ResourceBuffer = RhBuffer;

    //
    // The descriptor should at least be of the minimum required length for
    // a function config descriptor.
    //

    MinimumLength = (ULONG)sizeof(PNP_FUNCTION_CONFIG_DESCRIPTOR);
    if (RhBuffer->PropertiesLength < MinimumLength) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: BIOS function config "
                    "descriptor length is invalid! "
                    "Extn = 0x%p, Length = %d, Reqd = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    RhBuffer->PropertiesLength,
                    MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    FunctionConfigDescriptor =
        (PPNP_FUNCTION_CONFIG_DESCRIPTOR)RhBuffer->ConnectionProperties;

    //
    // Validate the function config descriptor buffer.
    //

    Valid = 
        GpioUtilIsBiosFunctionConfigDescriptorValid(FunctionConfigDescriptor,
                                                    RhBuffer->PropertiesLength);

    if (Valid == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: BIOS function config descriptor length is "
                    "invalid! Extn = 0x%p, Length = %d, Reqd = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    RhBuffer->PropertiesLength,
                    (ULONG)MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    Value = FunctionConfigDescriptor->ResourceSourceOffset;

    GPIO_ASSERT(Value >= FunctionConfigDescriptor->PinTableOffset);

    PinCount =
        (Value - FunctionConfigDescriptor->PinTableOffset) / sizeof(USHORT);

    DescriptorPinTable =
        ((PUSHORT)Add2Ptr(FunctionConfigDescriptor,
                          FunctionConfigDescriptor->PinTableOffset));

    //
    // Ensure that pin count is valid.
    //

    if (PinCount == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: BIOS function config descriptor pin "
                    "count s zero! Extn = 0x%p, ID = [0x%I64x]\n",
                    __FUNCTION__,
                    GpioExtension,
                    ConnectionId->QuadPart);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    //
    // Currently all pins referenced within a descriptor must fall within
    // the same GPIO bank. This restriction is enforced because:
    //
    // 1. Multiple pins per descriptor are meant mainly for atomic access,
    //    which is not guaranteed if pins span across banks.
    //
    // 2. Greatly simplifies implementation of the IO operations in the class
    //    extension.
    //

    BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension,
                                          DescriptorPinTable[0]);

    GpioBank = GpiopGetBankEntry(GpioExtension, BankId);
    if (GpioBank == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: No bank associated with "
                    "specified pins! Pin number is invalid! "
                    " Absolute pin = %#x, Bank ID = 0x%x\n",
                    __FUNCTION__,
                    DescriptorPinTable[0],
                    BankId);

        Status = STATUS_INVALID_PARAMETER;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    for (Index = 1; Index < PinCount; Index += 1) {
        CurrentBankId = GPIO_BANK_ID_FROM_PIN_NUMBER(
                            GpioExtension,
                            DescriptorPinTable[Index]);

        if (BankId != CurrentBankId) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Function config pins span across "
                        "banks which is not supported! Extn = 0x%p, "
                        "Bank ID = 0x%x, Current Bank ID = 0x%x, "
                        "Absolute pin = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        BankId,
                        CurrentBankId,
                        DescriptorPinTable[Index]);

            Status = STATUS_NOT_SUPPORTED;
            goto ConnectFunctionConfigPinsReserveEnd;
        }
    }

    //
    // Allocate a buffer to hold the pins represented by this request. The
    // pin table is accessed at DIRQL for memory-mapped controllers (read/write
    // are performed at DIRQL).
    //

    Length = PinCount * sizeof(PIN_NUMBER);
    PinNumberTable = GPIO_ALLOCATE_POOL(NonPagedPoolNx, Length);
    if (PinNumberTable  == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: Unable to allocate memory for"
                    " pin table!\n",
                    __FUNCTION__);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    //
    // Allocate a buffer to hold the pin information for the each pin. The
    // pin table should only be accessed from passive level.
    //

    PinTableSize = PinCount * sizeof(PGPIO_PIN_INFORMATION_ENTRY);
    PinInformationTable = GPIO_ALLOCATE_POOL(PagedPool, PinTableSize);
    if (PinInformationTable  == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: Unable to allocate memory for pin"
                    " information table!\n",
                    __FUNCTION__);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    //
    // Zero out all the buffers.
    //

    RtlZeroMemory(PinInformationTable, PinTableSize);
    RtlZeroMemory(PinNumberTable, Length);
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_FUNCCONFIG,
                "%s: Attempting to connect pins for function config!"
                " Extn = 0x%p\n",
                __FUNCTION__,
                GpioExtension);

    //
    // Initalize the pin table from the BIOS descriptor and also get the pin
    // information entry for each pin. If the pin information isn't allocated,
    // then it will be allocated.
    //

    Status = STATUS_SUCCESS;
    for (Index = 0; Index < PinCount; Index += 1) {
        RelativePin = GPIO_ABSOLUTE_TO_RELATIVE_PIN(GpioExtension,
                                                    DescriptorPinTable[Index]);

        PinNumberTable[Index] = RelativePin;
        PinInformation = GpiopCreatePinEntry(GpioBank, RelativePin);
        if (PinInformation == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Pin information cannot be found! "
                        " Extn = 0x%p, Absolute pin = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        DescriptorPinTable[Index]);

            Status = STATUS_INSUFFICIENT_RESOURCES;
            break;
        }

        PinInformationTable[Index] = PinInformation;
    }

    if (!NT_SUCCESS(Status)) {
        goto ConnectFunctionConfigPinsReserveEnd;
    }

    GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    //
    // Update the file context with the buffers allocated above. Connection mode
    // is not used for function config pins.
    //

    FileContext->ConnectMode = ConnectModeInvalid;
    FileContext->PinNumberTable = PinNumberTable;
    FileContext->PinCount = PinCount;
    FileContext->PinInformationTable = PinInformationTable;

    //
    // Before connecting the function config pins, mark the GPIO
    // controller as not stoppable or removable until the peripheral driver
    // disconnects the function config pins. Note this
    // is done during the reserve phase. 
    //
    //
    // N.B. WDF ref-counts the StaticStopRemove enables and disables. So only
    //      last enable would actually enable stop/remove.
    //

    WdfDeviceSetStaticStopRemove(GpioExtension->Device, FALSE);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken before the request reaches to this point. Also, all operations
    // should have ceased by the time the device left the STARTED state.
    //

    GPIO_ASSERT((GpioBank->PowerData.IsActive != FALSE) &&
                (GPIO_GET_DEVICE_STATE(GpioExtension) == DEVICE_STATE_STARTED));
    
    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    Status = GpiopConnectFunctionConfigPinsLocked(GpioExtension,
                                                  GpioBank,
                                                  FileContext,
                                                  PinUnreserved);

    //
    // If the connection was successful, then add the file context to the
    // per-bank IO operation list to ensure the pins are disconnected if the
    // controller gets removed.
    //

    if (NT_SUCCESS(Status)) {
        InsertTailList(&GpioBank->ConnectedPinsQueue, &FileContext->ListEntry);
    } 

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    if (NT_SUCCESS(Status)) {
        PoFxActivateComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0);

        //
        // Print trace messages.
        //

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_FUNCCONFIG,
                    "%s: Successfully connected function"
                    " config pins!, Extn = 0x%p\n",
                    __FUNCTION__,
                    GpioExtension);

        PinNumberTable = FileContext->PinNumberTable;
        PinCount = FileContext->PinCount;
        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    } else {
        WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

        //
        // Reset the file object context fields on failure.
        //

        FileContext->ConnectMode = ConnectModeInvalid;
        FileContext->PinNumberTable = NULL;
        FileContext->PinCount = 0;
        FileContext->PinInformationTable = NULL;

        //
        // Print trace messages.
        //

        if (Status == STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Connect mode not compatible"
                        " with mode for pin! Absolute pin = %#x, Status = %#x\n",
                        __FUNCTION__,
                        DescriptorPinTable[Index],
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Client driver failed to"
                        " connect pins! Status = %#x\n",
                        __FUNCTION__,
                        Status);
        }
    }

ConnectFunctionConfigPinsReserveEnd:
    if (!NT_SUCCESS(Status)) {
        if (PinInformationTable != NULL) {
            GPIO_FREE_POOL(PinInformationTable);
        }

        if (PinNumberTable != NULL) {
            GPIO_FREE_POOL(PinNumberTable);
        }

        if (RhBuffer != NULL)  {
            GPIO_FREE_POOL(RhBuffer);
            FileContext->ResourceBuffer = NULL;
        }
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopConnectFunctionConfigPinsCommit (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext
    )

/*++

Routine Description:

    This routine commit the specified set of lines for function config.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG PinCount;
    PPIN_NUMBER PinNumberTable;
    NTSTATUS Status;

    PAGED_CODE();

    PinNumberTable = FileContext->PinNumberTable;
    PinCount = FileContext->PinCount;

    //
    // Bank based validation is already completed in reserve phase.
    //
    
    BankId = FileContext->BankId;
    GpioBank = GpiopGetBankEntry(GpioExtension, BankId);

    GPIO_ASSERT(GpioBank != NULL);

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    Status = GpiopConnectFunctionConfigPinsLocked(GpioExtension,
                                                  GpioBank,
                                                  FileContext,
                                                  PinReserved);

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    if (NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_FUNCCONFIG,
                    "%s: Successfully connected function"
                    " config pins! Extn = 0x%p\n",
                    __FUNCTION__,
                    GpioExtension);

        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: Client driver failed to connect pins! Status = %#x\n",
                    __FUNCTION__,
                    Status);
    }

    //
    // Free resource buffer on successful commit. Keep it around on failure
    // to allow for commit to be retried.
    //

    if ((NT_SUCCESS(Status)) && (FileContext->ResourceBuffer != NULL)) {
        GPIO_FREE_POOL(FileContext->ResourceBuffer);
        FileContext->ResourceBuffer = NULL;
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopConnectFunctionConfigPinsLocked(
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in FUNCTION_CONFIG_PIN_STATE ExpectedFunctionConfigState
    )

/*++

Routine Description:

    This routine configures the specified set of lines for function config.

    N.B. This routine assumes the caller has acquired the GPIO bank lock
    prior to invocation.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    GpioBank - Supplies a pointer to the GPIO bank.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    ConnectionId - Supplies the Connection ID value corresponding to the
        request.

    ExpectedFunctionConfigState - The expected connection state for a function
        config pin.

Return Value:

    NTSTATUS code.

--*/

{

    BOOLEAN DescriptorIsShared;
    PPNP_FUNCTION_CONFIG_DESCRIPTOR FunctionConfigDescriptor;
    ULONG Index;
    ULONG PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    PPIN_NUMBER PinNumberTable;
    USHORT RelativePin;
    NTSTATUS Status;

    Status = STATUS_SUCCESS;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;
    PinInformationTable = FileContext->PinInformationTable;
    FunctionConfigDescriptor =
        (PPNP_FUNCTION_CONFIG_DESCRIPTOR)
        FileContext->ResourceBuffer->ConnectionProperties;

    DescriptorIsShared =
        (FunctionConfigDescriptor->Flags & FUNCTION_CONFIG_SHARED) != 0;

    //
    // Walk over all pins being connected and check if the pin is allowed
    // to be connected in the specified mode.
    //

    for (Index = 0; Index < PinCount; Index += 1) {
        RelativePin = PinNumberTable[Index];
        PinInformation = PinInformationTable[Index];

        //
        // If already connected for interrupt or I/O, then reject this request
        // to configure for function config mode.
        //

        if ((PinInformation->Mode.Interrupt != 0) ||
            (PinInformation->Mode.Input != 0) ||
            (PinInformation->Mode.Output != 0)) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Pin cannot be configured for function config"
                        " Absolute pin = %#x, "
                        " Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If the pin is already opened as exclusive or if an open pin is
        // being reopened in exclusive, then fail the request. Only do
        // this when we are first attempting to reserve the pin, since 
        // the FunctionConfig and ExclusiveMode bits are set in the reserve
        // phase.
        //

        if ((PinInformation->FunctionConfigPinState == PinUnreserved) &&
            ((DescriptorIsShared != FALSE) || 
            (PinInformation->Mode.FunctionConfig != 0))) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Function config pin is being opened in"
                        " non-exclusive mode or already configured for"
                        " function config."
                        " Absolute pin = %#x, Mode = %#x, IsShared = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG,
                        DescriptorIsShared);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If pin has been reserved for function config, block
        // any outstanding request.
        //

        if (PinInformation->FunctionConfigPinState != ExpectedFunctionConfigState) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: Pin cannot be configured for function config"
                        " Absolute pin = %#x, "
                        " Expected function config state = %#x"
                        " Actual function config state = %#x",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        ExpectedFunctionConfigState,
                        PinInformation->FunctionConfigPinState);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }
    }

    if (!NT_SUCCESS(Status)) {
        goto ConnectFunctionConfigPinsLockedEnd;
    }

    //
    // Walk over all pins and mark them as reserved. Also mark the pins as
    // for function config, since from a software perspective, they are
    // configured as such already.
    //

    for (Index = 0; Index < PinCount; Index += 1) {
        PinInformation = PinInformationTable[Index];

        //
        // Only update pin information during reserve phase
        //

        if (PinInformation->FunctionConfigPinState == PinUnreserved) {
            PinInformation->Mode.FunctionConfig = 1;
            PinInformation->FunctionNumber =
                FUNCTION_CONFIG_BIOS_DESCRIPTOR_FUNCTION(FunctionConfigDescriptor);

            if (DescriptorIsShared == FALSE) {
                PinInformation->Mode.ExclusiveMode = 1;

            } else {
                PinInformation->Mode.ExclusiveMode = 0;
            }

            if (PinInformation->Mode.Interrupt == 0) {

                //
                // Delete any stale interrupt-related values.
                //

                PinInformation->InterruptMode = 0;
                PinInformation->Polarity = 0;
                PinInformation->InterruptCallbackContext = 0;
                InterlockedExchange((PLONG)&PinInformation->Virq, 0);
                PinInformation->DebounceTimeout = 0;
                PinInformation->DriveStrength = 0;
                PinInformation->PullConfiguration =
                    FUNCTION_CONFIG_BIOS_DESCRIPTOR_PULL_CONFIGURATION(FunctionConfigDescriptor);
            }
        }

        PinInformation->FunctionConfigPinState += 1;
    }

    //
    // If this is the commit phase, call into client driver.
    //

    if (ExpectedFunctionConfigState == PinReserved) {
        Status = GpioClnInvokeConnectFunctionConfigPins(GpioBank,
                                                        PinNumberTable,
                                                        PinCount,
                                                        FunctionConfigDescriptor);
    }

ConnectFunctionConfigPinsLockedEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopDisconnectFunctionConfigPins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine disconnects a previously configured set of lines.
    
    Note this routine may be called twice on the same file context. Once
    in the context of surprise-removal cleanup and subsequently when the
    handle is closed. The second invocation would effectively be ignored.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Suprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_BANK_ENTRY GpioBank;
    ULONG PinCount;
    PPIN_NUMBER PinNumberTable;
    NTSTATUS Status;

    PAGED_CODE();

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_FUNCCONFIG,
                "%s: Attempting to disconnect"
                " function config pins! Extn = 0x%p\n",
                __FUNCTION__,
                GpioExtension);

    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    GPIO_ASSERT(GpioBank != NULL);

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Check if the file context has already been unlinked. If so, then
    // it has already been cleaned up and thus no processing is necessary.
    // Otherwise unlink it from the chain to prevent it from getting cleaned
    // up twice.
    //
    // N.B. This check needs to be done under the protection of the bank lock.
    //

    if (IsListEmpty(&FileContext->ListEntry) == FALSE) {
        RemoveEntryList(&FileContext->ListEntry);
        InitializeListHead(&FileContext->ListEntry);

    } else {

        //
        // Release the bank lock prior to exit.
        //

        GPIO_RELEASE_BANK_LOCK(GpioBank);

        Status = STATUS_SUCCESS;
        goto DisconnectFunctionConfigPinsEnd;
    }

    //
    // The disconnect pin table was allocated when the pin was connected.
    //

    Status = STATUS_SUCCESS;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;

    GPIO_ASSERT(PinNumberTable != NULL);

    GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken on such pins when they were first connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    // Call into the routine that will call into the client driver to actually
    // disconnect the pin.
    //

    GpiopDisconnectFunctionConfigPinsLocked(GpioExtension,
                                            FileContext,
                                            InSurpriseRemovalContext);

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // Drop the power reference that was previously taken on the bank/component
    // when the pin was connected.
    //

    PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);

    //
    // As the pin is no longer connected, mark the driver as stoppable and
    // removable.
    //
    // N.B. WDF ref-counts the StaticStopRemove enables and disables. So only
    //      last enable would actually enable stop/remove.
    //

    WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

    //
    // Print trace message.
    //

    if (NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_FUNCCONFIG,
                    "%s: Successfully disconnected function config "
                    "pins! Extn = 0x%p\n",
                    __FUNCTION__,
                    GpioExtension);

        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_FUNCCONFIG,
                    "%s: Client driver failed to disconnect function config "
                    "pins! Status = %#x\n",
                    __FUNCTION__,
                    Status);
    }

    //
    // The handle will be closed on return to the caller and thus all the
    // data associated with the file context must be destroyed even on failure.
    //

    if (FileContext->PinInformationTable != NULL) {
        GPIO_FREE_POOL(FileContext->PinInformationTable);
        FileContext->PinInformationTable = NULL;
    }

    if (FileContext->PinNumberTable != NULL) {
        GPIO_FREE_POOL(FileContext->PinNumberTable);
        FileContext->PinNumberTable = NULL;
    }

    if (FileContext->ResourceBuffer != NULL) {
        GPIO_FREE_POOL(FileContext->ResourceBuffer);
        FileContext->ResourceBuffer = NULL;
    }

    FileContext->PinCount = 0;

DisconnectFunctionConfigPinsEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDisconnectFunctionConfigPinsLocked (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine will call into the client driver to actually disconnect the
    pins. Note the caller is responsible for removing the file context from the
    per-bank IO operations list on the final disconnect.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Suprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    GPIO_DISCONNECT_FUNCTION_CONFIG_PINS_FLAGS DisconnectFlags;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG Index;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    ULONG PinCount;
    PPIN_NUMBER PinNumberTable;
    NTSTATUS Status;

    DisconnectFlags.AsULONG = 0;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;
    PinInformationTable = FileContext->PinInformationTable;
    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    for (Index = 0; Index < PinCount; Index += 1) {
        PinInformation = PinInformationTable[Index];

        //
        // Reset reserve flag regardless of whether the pin
        // was truly enabled in HW.
        //

        PinInformation->FunctionConfigPinState = PinUnreserved;

        //
        // If this pin has not been configured (in HW) as function config,
        // assert and disconnect it anyway.
        //

        GPIO_ASSERT(PinInformation->Mode.FunctionConfig != 0);

        //
        // Update the pin information to reflect the disconnect.
        //
        // N.B. The pin is assumed disconnected irrespective of whether the
        //      client driver could successfully disconnect it or not. This is
        //      because the IO manager will destroy the file handle and no
        //      further disconnect can arrive on that handle.
        //
        //      A subsequent connect would reconnect these pins which should
        //      reprogram those pins for the new mode. So leaving the pins
        //      connected should be harmless in most cases. If doing this could
        //      cause issues for some GPIO controllers, then it is the
        //      responsibility of the client driver to handle subsequent connects
        //      correctly (i.e., it should skip connecting a pin if it has
        //      already been connected).
        //

        PinInformation->Mode.FunctionConfig = 0;
        PinInformation->Mode.ExclusiveMode = 0;
    }

    //
    // Call into the GPIO client driver to disconnect the specified pins.
    //

    if (InSurpriseRemovalContext == FALSE) {
        Status = GpioClnInvokeDisconnectFunctionConfigPins(GpioBank,
                                                           PinNumberTable,
                                                           PinCount,
                                                           DisconnectFlags.AsULONG);
    }

    //
    // Mark the connect mode as invalid.
    //

    FileContext->ConnectMode = ConnectModeInvalid;
    return;
}

//
// ---------------------------------------------------------------- IO routines
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopConnectGpioPins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in GPIO_CONNECT_IO_PINS_MODE ConnectMode,
    __in PLARGE_INTEGER ConnectionId
    )

/*++

Routine Description:

    This routine configures the specified set of lines for input or output.

    N.B. When opening for input, the input descriptor may be an INTERRUPT
         descriptor.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    ConnectMode - Supplies the mode in which the pins should be configured
        (input or output).

    ConnectionId - Supplies the Connection ID value corresponding to the
        request.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PVOID ClientIoContext;
    PPIN_NUMBER ConnectPinTable;
    BANK_ID CurrentBankId;
    BOOLEAN DelayedConnect;
    PUSHORT DescriptorPinTable;
    BOOLEAN FailCreateContext;
    PGPIO_BANK_ENTRY GpioBank;
    PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR GpioDescriptor;
    ULONG Index;
    ULONG Length;
    ULONG MinimumLength;
    USHORT PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    PPIN_NUMBER PinNumberTable;
    ULONG PinTableSize;
    PIN_NUMBER RelativePin;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER RhBuffer;
    NTSTATUS Status;
    BOOLEAN Valid;
    USHORT Value;

    PAGED_CODE();

    ClientIoContext = NULL;
    ConnectPinTable = NULL;
    DelayedConnect = FALSE;
    FailCreateContext = FALSE;
    Index = 0;
    PinInformationTable = NULL;
    PinNumberTable = NULL;
    RhBuffer = NULL;

    NT_ASSERT(ConnectionId != NULL);

    //
    // GPIO connect mode can either be input or output but not both.
    //

    GpioDescriptor = NULL;
    if ((ConnectMode != ConnectModeInput) &&
        (ConnectMode != ConnectModeOutput)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopConnectGpioPins: Invalid connect mode! Extn = 0x%p, "
                    "ID = [0x%I64x], Mode = %#x\n",
                    GpioExtension,
                    ConnectionId->QuadPart,
                    ConnectMode);

        Status = STATUS_INVALID_PARAMETER;
        goto ConnectGpioPinsEnd;
    }

    //
    // Query the Resource Hub to fetch the BIOS descriptor associated with this
    // request.
    //

    Status = GpiopQueryResourceHubBiosDescriptor(
                 GpioExtension->Device,
                 &GpioExtension->ResourceHubTarget,
                 FALSE,
                 ConnectionId->QuadPart,
                 &RhBuffer);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopConnectGpioPins: Failed to query BIOS descriptor "
                    "from the Resource Hub! ID = [0x%I64x], Status = %#x\n",
                    ConnectionId->QuadPart,
                    Status);

        goto ConnectGpioPinsEnd;
    }

    //
    // The descriptor should atleast be of the minimum required length.
    //

    MinimumLength = (ULONG)sizeof(PNP_GPIO_INTERRUPT_IO_DESCRIPTOR);
    if (RhBuffer->PropertiesLength < MinimumLength) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopConnectGpioPins: BIOS GPIO descriptor length is "
                    "invalid! Extn = 0x%p, Length = %d, Reqd = %d\n",
                    GpioExtension,
                    RhBuffer->PropertiesLength,
                    MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectGpioPinsEnd;
    }

    GpioDescriptor =
        (PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR)RhBuffer->ConnectionProperties;

    Valid = GpioUtilIsBiosGpioDescriptorValid(GpioDescriptor,
                                              RhBuffer->PropertiesLength);

    if (Valid == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopConnectGpioPins: BIOS GPIO descriptor length is "
                    "invalid! Extn = 0x%p, Length = %d, Reqd = %d\n",
                    GpioExtension,
                    RhBuffer->PropertiesLength,
                    (ULONG)MinimumLength);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectGpioPinsEnd;
    }

    GPIO_ASSERT((GpioDescriptor->DescriptorType ==
                    PNP_GPIO_IRQ_DESCRIPTOR_TYPE_IO) ||
                (GpioDescriptor->DescriptorType ==
                    PNP_GPIO_IRQ_DESCRIPTOR_TYPE_INTERRUPT));

    Value = GpioDescriptor->ResourceSourceOffset;
    NT_ASSERT(Value >= GpioDescriptor->PinTableOffset);

    PinCount = (Value - GpioDescriptor->PinTableOffset) / sizeof(USHORT);
    DescriptorPinTable =
        ((PUSHORT)Add2Ptr(GpioDescriptor, GpioDescriptor->PinTableOffset));

    //
    // Ensure that pin count is valid.
    //

    if (PinCount == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopConnectGpioPins: BIOS GPIO descriptor pin count is "
                    "zero! Extn = 0x%p, ID = [0x%I64x], Mode = %d\n",
                    GpioExtension,
                    ConnectionId->QuadPart,
                    ConnectMode);

        Status = STATUS_UNSUCCESSFUL;
        goto ConnectGpioPinsEnd;
    }

    //
    // Currently all pins referenced within a descriptor must fall within
    // the same GPIO bank. This restriction is enforced because:
    // 1. Multiple pins per descriptor are meant mainly for atomic access,
    //    which is not guaranteed if pins span across banks.
    //
    // 2. Greatly simplifies implementation of the IO operations in the class
    //    extension.
    //

    BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension,
                                          DescriptorPinTable[0]);

    GpioBank = GpiopGetBankEntry(GpioExtension, BankId);
    if (GpioBank == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "GpiopConnectGpioPins: No bank associated with specified "
                    "pins! Pin number is invalid! Absolute pin = %#x, "
                    "Bank ID = 0x%x\n",
                    DescriptorPinTable[0],
                    BankId);

        Status = STATUS_INVALID_PARAMETER;
        goto ConnectGpioPinsEnd;
    }

    for (Index = 1; Index < PinCount; Index += 1) {
        CurrentBankId = GPIO_BANK_ID_FROM_PIN_NUMBER(
                            GpioExtension,
                            DescriptorPinTable[Index]);

        if (BankId != CurrentBankId) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "GpiopConnectGpioPins: GPIO pins span across banks "
                        "which is not supported! Extn = 0x%p, Bank ID = 0x%x, "
                        "Current Bank ID = 0x%x, Absolute pin = %#x\n",
                        GpioExtension,
                        BankId,
                        CurrentBankId,
                        DescriptorPinTable[Index]);

            Status = STATUS_NOT_SUPPORTED;
            goto ConnectGpioPinsEnd;
        }
    }

    //
    // Allocate a buffer to hold the pins represented by this request. The
    // pin table is accessed at DIRQL for memory-mapped controllers (read/write
    // are performed at DIRQL).
    //

    Length = PinCount * sizeof(PIN_NUMBER);
    PinNumberTable = GPIO_ALLOCATE_POOL(NonPagedPoolNx, Length);
    if (PinNumberTable  == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopConnectGpioPins: Unable to allocate memory for pin "
                    "table!\n");

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ConnectGpioPinsEnd;
    }

    //
    // Allocate a buffer to hold the pin information for the each pin. The
    // pin table should only be accessed from passive level.
    //

    PinTableSize = PinCount * sizeof(PGPIO_PIN_INFORMATION_ENTRY);
    PinInformationTable = GPIO_ALLOCATE_POOL(PagedPool, PinTableSize);
    if (PinInformationTable  == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopConnectGpioPins: Unable to allocate memory for pin "
                    "information table!\n");

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ConnectGpioPinsEnd;
    }

    //
    // Allocate a buffer to hold the pins that need to be connected. Some pins
    // may already have been connected due to some other request.
    //

    ConnectPinTable = GPIO_ALLOCATE_POOL(PagedPool,
                                         PinCount * sizeof(PIN_NUMBER));

    if (ConnectPinTable == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopConnectGpioPins: Unable to allocate memory for "
                    "connect pin table!\n");

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto ConnectGpioPinsEnd;
    }

    //
    // Zero out all the buffers.
    //

    RtlZeroMemory(PinInformationTable, PinTableSize);
    RtlZeroMemory(ConnectPinTable, PinCount * sizeof(PIN_NUMBER));

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_IO,
                "%s: Attempting to connect GPIO pins for IO!"
                " Extn = 0x%p, Mode = %#x\n",
                __FUNCTION__,
                GpioExtension,
                ConnectMode);

    //
    // Initalize the pin table from the BIOS descriptor and also get the pin
    // information entry for each pin. If the pin information isn't allocated,
    // then it will be allocated.
    //

    Status = STATUS_SUCCESS;
    for (Index = 0; Index < PinCount; Index += 1) {
        RelativePin = GPIO_ABSOLUTE_TO_RELATIVE_PIN(
                          GpioExtension,
                          DescriptorPinTable[Index]);

        PinNumberTable[Index] = RelativePin;
        PinInformation = GpiopCreatePinEntry(GpioBank, RelativePin);
        if (PinInformation == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Pin information cannot be found! "
                        " Extn = 0x%p, Absolute pin = %#x\n",
                        __FUNCTION__,
                        GpioExtension,
                        DescriptorPinTable[Index]);

            Status = STATUS_INSUFFICIENT_RESOURCES;
            break;
        }

        PinInformationTable[Index] = PinInformation;
    }

    if (!NT_SUCCESS(Status)) {
        goto ConnectGpioPinsEnd;
    }

    GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    //
    // Update the file context with the buffers allocated above.
    //

    FileContext->ConnectMode = ConnectModeInvalid;
    FileContext->PinNumberTable = PinNumberTable;
    FileContext->PinCount = PinCount;
    FileContext->PinInformationTable = PinInformationTable;
    FileContext->ConnectPinTable = ConnectPinTable;
    FileContext->ResourceBuffer = RhBuffer;

    //
    // Before connecting IO lines, mark the GPIO controller as not stoppable or
    // removable until the peripheral driver disconnects the IO lines.
    //
    //
    // N.B. WDF ref-counts the StaticStopRemove enables and disables. So only
    //      last enable would actually enable stop/remove.
    //

    WdfDeviceSetStaticStopRemove(GpioExtension->Device, FALSE);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken before the request reaches to this point. Also, all operations
    // should have ceased by the time the device left the STARTED state.
    //

    GPIO_ASSERT((GpioBank->PowerData.IsActive != FALSE) &&
                (GPIO_GET_DEVICE_STATE(GpioExtension) == DEVICE_STATE_STARTED));

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Call into the GPIO client driver to let it create an IO context that
    // will be associated with this handle.
    //
    // Force the call early to prevent clients from assuming IO contexts
    // always pair with connect (and disconnects). If the pins are already
    // connected, then the client driver would not be called to connect those
    // pins again.
    //
    // Note this callback is only supported for internal clients.
    //

    Status = GpioClnInvokeCreateIoContext(GpioBank,
                                          PinNumberTable,
                                          PinCount,
                                          ConnectMode,
                                          &ClientIoContext);

    if (!NT_SUCCESS(Status)) {
        FailCreateContext = TRUE;
        goto ConnectGpioPinsReleaseLockEnd;
    }

    Status = GpiopConnectGpioPinsLocked(GpioExtension,
                                        GpioBank,
                                        FileContext,
                                        ConnectMode,
                                        ConnectionId);

    //
    // If the connection was successful, then add the file context to the
    // per-bank IO operation list to ensure the pins are disconnected if the
    // controller gets removed.
    //

    if (NT_SUCCESS(Status)) {
        InsertTailList(&GpioBank->ConnectedPinsQueue, &FileContext->ListEntry);

    } else {
        GpioClnInvokeDeleteIoContext(GpioExtension, ClientIoContext);
        ClientIoContext = NULL;
    }

ConnectGpioPinsReleaseLockEnd:

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    if (NT_SUCCESS(Status)) {
        FileContext->ClientIoContext = ClientIoContext;

        //
        // Take a power reference on the bank/component. This will prevent the
        // PEP from transitioning the bank to F1 while there is a pin that is
        // actively connected for input or output.
        //

        PoFxActivateComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0);

        //
        // Print trace messages.
        //

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "GpiopConnectGpioPins: Successfully connected GPIO pins!"
                    " Extn = 0x%p, Mode = %#x, Deferred = %d\n",
                    GpioExtension,
                    ConnectMode,
                    DelayedConnect);

        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    } else {

        //
        // If the error happened after the driver was marked non-stoppable or
        // removable, then undo that operation.
        //

        WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

        //
        // Reset the file object context fields on failure.
        //

        FileContext->ConnectMode = ConnectModeInvalid;
        FileContext->PinNumberTable = NULL;
        FileContext->PinCount = 0;
        FileContext->PinInformationTable = NULL;
        FileContext->ConnectPinTable = NULL;
        FileContext->ResourceBuffer = NULL;

        //
        // The entry should not be on the global list if it failed to connect
        // (otherwise a disconnect attempt will happen if the controller gets
        // removed).
        //

        NT_ASSERT(IsListEmpty(&FileContext->ListEntry) != FALSE);

        //
        // Print trace messages.
        //

        if (FailCreateContext != FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopConnectGpioPins: Client driver failed to create "
                        "context! Status = %#x\n",
                        Status);

        } else if (Status == STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopConnectGpioPins: Connect mode not compatible with"
                        " mode for pin! Absolute pin = %#x, Status = %#x\n",
                        DescriptorPinTable[Index],
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopConnectGpioPins: Client driver failed to connect"
                        " pins! Status = %#x\n",
                        Status);
        }
    }

ConnectGpioPinsEnd:
    if (!NT_SUCCESS(Status)) {
        if (PinInformationTable != NULL) {
            GPIO_FREE_POOL(PinInformationTable);
        }

        if (PinNumberTable != NULL) {
            GPIO_FREE_POOL(PinNumberTable);
        }

        if (ConnectPinTable != NULL) {
            GPIO_FREE_POOL(ConnectPinTable);
        }

        if (RhBuffer != NULL)  {
            GPIO_FREE_POOL(RhBuffer);
        }
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopConnectGpioPinsLocked (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in GPIO_CONNECT_IO_PINS_MODE ConnectMode,
    __in PLARGE_INTEGER ConnectionId
    )

/*++

Routine Description:

    This routine configures the specified set of lines for input or output.
    Note the caller is responsible for adding the file context to the
    per-bank IO operations list on the initial connect.

    N.B. 1. This routine assumes the caller has acquired the GPIO bank lock
            prior to invocation.

         2. When opening for input, the input descriptor may be an INTERRUPT
            descriptor.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    GpioBank - Supplies a pointer to the GPIO bank.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    ConnectMode - Supplies the mode in which the pins should be configured
        (input or output).

    ConnectionId - Supplies the Connection ID value corresponding to the
        request.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG ConnectCount;
    PPIN_NUMBER ConnectPinTable;
    UCHAR DescriptorOpenMode;
    PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR GpioDescriptor;
    ULONG Index;
    UCHAR IoRestriction;
    ULONG PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    PPIN_NUMBER PinNumberTable;
    USHORT RelativePin;
    BOOLEAN SkipEntry;
    NTSTATUS Status;

    ConnectCount = 0;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;
    PinInformationTable = FileContext->PinInformationTable;
    ConnectPinTable = FileContext->ConnectPinTable;
    GpioDescriptor =
        (PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR)
            FileContext->ResourceBuffer->ConnectionProperties;

    DescriptorOpenMode = (GpioDescriptor->InterruptIoFlags & GPIO_SHARED);

    if (GpioDescriptor->DescriptorType == PNP_GPIO_IRQ_DESCRIPTOR_TYPE_IO) {
        IoRestriction = GPIO_BIOS_DESCRIPTOR_IO_RESTRICTION(GpioDescriptor);

    } else {
        IoRestriction = (UCHAR)GPIO_IORESTRICTION_NONE;
    }

    //
    // Ensure that the connect mode satisfies any IORestrictions specified in
    // the GPIO descriptor.
    //

    if ((IoRestriction == GPIO_IORESTRICTION_INPUT_ONLY) &&
        (ConnectMode != ConnectModeInput)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: IORestriction is input only...failing output "
                    "connect request! Extn = 0x%p, ID = [0x%I64x], Mode = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    ConnectionId->QuadPart,
                    ConnectMode);

        Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
        goto ConnectGpioPinsLockedEnd;

    } else if ((IoRestriction == GPIO_IORESTRICTION_OUTPUT_ONLY) &&
               (ConnectMode != ConnectModeOutput)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: IORestriction is output only...failing input "
                    "connect request! Extn = 0x%p, ID = [0x%I64x], Mode = %d\n",
                    __FUNCTION__,
                    GpioExtension,
                    ConnectionId->QuadPart,
                    ConnectMode);

        Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
        goto ConnectGpioPinsLockedEnd;
    }

    //
    // Walk over all pins being connected and check if the pin is allowed
    // to be connected in the specified mode.
    //

    Status = STATUS_SUCCESS;
    for (Index = 0; Index < PinCount; Index += 1) {
        SkipEntry = FALSE;
        RelativePin = PinNumberTable[Index];
        PinInformation = PinInformationTable[Index];

        //
        // If connected (or connecting) for function config, reject all other
        // connect requests.
        //

        if (PinInformation->Mode.FunctionConfig == 1) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Pin already configured for function config!"
                        " Absolute pin = %#x, FunctionConfigPinState = %#x"
                        " Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->FunctionConfigPinState,
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If already connected for interrupt, then reject output connect
        // requests unless a registry override is present to allow this
        // configuration (for test purposes).
        //

        if ((PinInformation->Mode.Interrupt != 0) &&
            (ConnectMode == ConnectModeOutput) &&
            (GpioAllowInterruptOutput == FALSE)) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Interrupt pin cannot be opened in output mode"
                        " without registry override! Absolute pin = %#x, "
                        "Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If connected for input, then reject output connect requests unless
        // the input is implict due to interrupt mode and Interrupt+Output
        // is allowed.
        //

        if ((PinInformation->Mode.Input == 1) &&
            (ConnectMode == ConnectModeOutput) &&
            ((PinInformation->Mode.Interrupt == 0) ||
             (GpioAllowInterruptOutput == FALSE))) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Input pin cannot be opened in output mode! "
                        "Absolute pin = %#x, Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If connected for output, then reject input connect requests.
        //

        if ((PinInformation->Mode.Output == 1) &&
            (ConnectMode == ConnectModeInput)) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Output pin cannot be opened in input mode! "
                        "Absolute pin = %#x, Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        //
        // If the pin is already opened as exclusive or if an open pin is
        // being reopened in exclusive, then fail the request.
        //

        if (((PinInformation->Mode.Input != 0) ||
             (PinInformation->Mode.Output != 0)) &&
             ((PinInformation->Mode.ExclusiveMode == 1) ||
             (DescriptorOpenMode == GPIO_EXCLUSIVE))) {

            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "%s: Pin already opened in exclusive mode!"
                        " Absolute pin = %#x, Mode = %#x\n",
                        __FUNCTION__,
                        GPIO_RELATIVE_TO_ABSOLUTE_PIN(GpioBank, RelativePin),
                        PinInformation->Mode.AsULONG);

            Status = STATUS_GPIO_INCOMPATIBLE_CONNECT_MODE;
            break;
        }

        if ((ConnectMode == ConnectModeInput) &&
            (PinInformation->Mode.Input == 1) &&
            (PinInformation->InputEnableCount > 0)) {

            SkipEntry = TRUE;
        }

        if ((ConnectMode == ConnectModeInput) &&
            (PinInformation->Mode.Interrupt == 1)) {

            SkipEntry = TRUE;
        }

        if ((ConnectMode == ConnectModeOutput) &&
            (PinInformation->Mode.Output == 1) &&
            (PinInformation->OutputEnableCount > 0)) {

            SkipEntry = TRUE;
        }

        if (SkipEntry == FALSE) {
            ConnectPinTable[ConnectCount] = PinNumberTable[Index];
            ConnectCount += 1;
        }
    }

    if (!NT_SUCCESS(Status)) {
        goto ConnectGpioPinsLockedEnd;
    }

    //
    // Call into the GPIO client driver to enable the specified pins. If the
    // pins are being configured for output only, then delay the connect
    // until the first write on those pins. This is done to avoid causing
    // glitches on the output lines.
    //

    FileContext->ConnectCount = ConnectCount;
    if (ConnectCount > 0) {
        if (ConnectMode == ConnectModeInput) {
            Status = GpioClnInvokeConnectPins(GpioBank,
                                              ConnectPinTable,
                                              ConnectCount,
                                              ConnectMode,
                                              GpioDescriptor);

        } else {
            GPIO_ASSERT(ConnectMode == ConnectModeOutput);

            FileContext->Flags.DelayedConnect = TRUE;
        }
    }

    //
    // If the connect succeeded, then update the pin information.
    //
    // N.B. The caller is responsible for adding the file context to the
    //      per-bank IO operations list on the first connect.
    //

    if (NT_SUCCESS(Status)) {

        //
        // Set the new connect mode for the pin.
        //

        FileContext->ConnectMode = ConnectMode;

        //
        // Walk over all pins being connected and update their enable counts.
        //

        for (Index = 0; Index < PinCount; Index += 1) {
            PinInformation = PinInformationTable[Index];
            if (ConnectMode == ConnectModeInput) {
                PinInformation->Mode.Input = 1;
                PinInformation->InputEnableCount += 1;

            } else {
                PinInformation->Mode.Output = 1;
                PinInformation->OutputEnableCount += 1;
            }

            if (DescriptorOpenMode == GPIO_EXCLUSIVE) {
                PinInformation->Mode.ExclusiveMode = 1;
            } else {
                PinInformation->Mode.ExclusiveMode = 0;
            }

            //
            // Set the preserve flag (reset information from a previous open).
            //

            if (IoRestriction != GPIO_IORESTRICTION_NONE) {
                PinInformation->Mode.PreserveConfiguration = 1;

            } else {
                PinInformation->Mode.PreserveConfiguration = 0;
            }

            if ((PinInformation->Mode.Interrupt == 0) &&
                ((PinInformation->InputEnableCount == 1) ||
                 (PinInformation->OutputEnableCount == 1))) {

                //
                // Delete any stale interrupt-related values.
                //

                PinInformation->InterruptMode = 0;
                PinInformation->Polarity = 0;
                PinInformation->InterruptCallbackContext = 0;
                InterlockedExchange((PLONG)&PinInformation->Virq, 0);

                if (GpioDescriptor != NULL) {
                    PinInformation->DebounceTimeout =
                        GpioDescriptor->DebounceTimeout;

                    PinInformation->PullConfiguration =
                        GPIO_BIOS_DESCRIPTOR_PULL_CONFIGURATION(GpioDescriptor);

                    PinInformation->DriveStrength =
                        GpioDescriptor->DriveStrength;

                } else {
                    PinInformation->DebounceTimeout = 0;
                    PinInformation->DriveStrength = 0;
                    PinInformation->PullConfiguration =
                        GPIO_PIN_PULL_CONFIGURATION_DEFAULT;
                }
            }
        }

        //
        // If this is a delayed connect and this is the first output
        // enable on a pin, then mark it as such.
        //
        // N.B. The pin information in the pin table and the connect table
        //      at the same index may be different (the connect table is
        //      a subset of the pin table).
        //

        if (FileContext->Flags.DelayedConnect != FALSE) {
            for (Index = 0; Index < PinCount; Index += 1) {
                if (PinInformationTable[Index]->OutputEnableCount == 1) {
                    PinInformationTable[Index]->OutputConnected = FALSE;
                }
            }
        }
    }

ConnectGpioPinsLockedEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopDisconnectGpioPins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine disconnects a previously configured set of lines. It will
    call into the client driver to disconnect the pins and destroy all IO
    context associated with the file handle.

    Note this routine may be called twice on the same file context. Once
    in the context of surprise-removal cleanup and subsequently when the
    handle is closed. The second invocation would effectively be ignored.

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Suprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG DisconnectMode;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG PinCount;
    PPIN_NUMBER PinNumberTable;
    NTSTATUS Status;

    //
    // A disconnect operation should not race with any other IO operation.
    // If it does, then it is a bug in the consumer of the pin (i.e. some
    // peripheral). Thus the connect mode can be read here without any
    // race concerns.
    //

    DisconnectMode = FileContext->ConnectMode;

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_IO,
                "GpiopDisconnectGpioPins: Attempting to disconnect GPIO pins!"
                " Extn = 0x%p, Mode = %#x\n",
                GpioExtension,
                DisconnectMode);

    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    GPIO_ASSERT(GpioBank != NULL);

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Check if the file context has already been unlinked. If so, then
    // it has already been cleaned up and thus no processing is necessary.
    // Otherwise unlink it from the chain to prevent it from getting cleaned
    // up twice.
    //
    // N.B. This check needs to be done under the protection of the bank lock.
    //

    if (IsListEmpty(&FileContext->ListEntry) == FALSE) {
        RemoveEntryList(&FileContext->ListEntry);
        InitializeListHead(&FileContext->ListEntry);

    } else {

        //
        // Release the bank lock prior to exit.
        //

        GPIO_RELEASE_BANK_LOCK(GpioBank);

        Status = STATUS_SUCCESS;
        goto DisconnectGpioPinsEnd;
    }

    //
    // The disconnect pin table was allocated when the pin was connected.
    //

    Status = STATUS_SUCCESS;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;

    GPIO_ASSERT(PinNumberTable != NULL);

    GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken on such pins when they were first connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    // Call the client driver to delete the context associated with the IO
    // request.
    //

    GpioClnInvokeDeleteIoContext(GpioExtension, FileContext->ClientIoContext);
    FileContext->ClientIoContext = NULL;

    //
    // Call into the routine that will call into the client driver to actually
    // disconnect the pin.
    //
    // N.B. The current connect mode may be invalid if the pin(s) was
    //      disconnected previously to switch direction but failed to connect
    //      subsequently. The client driver should not be called again for such
    //      pins.
    //

    if ((DisconnectMode == ConnectModeInput) ||
        (DisconnectMode == ConnectModeOutput)) {

        GpiopDisconnectGpioPinsLocked(GpioExtension,
                                      FileContext,
                                      InSurpriseRemovalContext,
                                      FALSE);
    }

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // Drop the power reference that was previously taken on the bank/component
    // when the pin was connected.
    //

    PoFxIdleComponent(GpioExtension->PowerHandle, GpioBank->BankId, 0x0);

    //
    // As the pin is no longer connected, mark the driver as stoppable and
    // removable.
    //
    // N.B. WDF ref-counts the StaticStopRemove enables and disables. So only
    //      last enable would actually enable stop/remove.
    //

    WdfDeviceSetStaticStopRemove(GpioExtension->Device, TRUE);

    //
    // Print trace message.
    //

    if (NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "%s: Successfully disconnected GPIO "
                    "pins! Extn = 0x%p, Mode = %#x\n",
                    __FUNCTION__,
                    GpioExtension,
                    DisconnectMode);

        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);

    } else {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: Client driver failed to disconnect pins! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);
    }

    //
    // The handle will be closed on return to the caller and thus all the
    // data associated with the file context must be destroyed even on failure.
    //

    if (FileContext->PinInformationTable != NULL) {
        GPIO_FREE_POOL(FileContext->PinInformationTable);
        FileContext->PinInformationTable = NULL;
    }

    if (FileContext->PinNumberTable != NULL) {
        GPIO_FREE_POOL(FileContext->PinNumberTable);
        FileContext->PinNumberTable = NULL;
        FileContext->PinCount = 0;
    }

    if (FileContext->ConnectPinTable != NULL) {
        GPIO_FREE_POOL(FileContext->ConnectPinTable);
        FileContext->ConnectPinTable = NULL;
        FileContext->ConnectCount = 0;
    }

    if (FileContext->ResourceBuffer != NULL) {
        GPIO_FREE_POOL(FileContext->ResourceBuffer);
        FileContext->ResourceBuffer = NULL;
    }

DisconnectGpioPinsEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDisconnectGpioPinsLocked (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in BOOLEAN InSurpriseRemovalContext,
    __in BOOLEAN ForcePreserveConfiguration
    )

/*++

Routine Description:

    This routine will call into the client driver to actually disconnect the
    pins. Note the caller is responsible for removing the file context from the
    per-bank IO operations list on the final disconnect.

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

    N.B. 1. This routine assumes the caller has acquired the GPIO bank lock
            prior to invocation.

         2. This routine assumes the pin has not already been disconnected.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Suprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

    ForcePreserveConfiguration - Supplies whether the pin configuration
        should be preserved by the controller after disconnect.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG DisconnectCount;
    GPIO_DISCONNECT_IO_PINS_FLAGS DisconnectFlags;
    ULONG DisconnectMode;
    PPIN_NUMBER DisconnectPinTable;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG Index;
    ULONG PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PGPIO_PIN_INFORMATION_ENTRY *PinInformationTable;
    PPIN_NUMBER PinNumberTable;
    BOOLEAN SkipEntry;
    NTSTATUS Status;

    DisconnectCount = 0;
    DisconnectFlags.AsULONG = 0;
    DisconnectMode = FileContext->ConnectMode;
    DisconnectPinTable = FileContext->ConnectPinTable;
    PinCount = FileContext->PinCount;
    PinNumberTable = FileContext->PinNumberTable;
    PinInformationTable = FileContext->PinInformationTable;
    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    GPIO_ASSERT((DisconnectMode == ConnectModeInput) ||
                (DisconnectMode == ConnectModeOutput));

    if (ForcePreserveConfiguration != FALSE) {
        DisconnectFlags.PreserveConfiguration = 1;
    }

    for (Index = 0; Index < PinCount; Index += 1) {
        SkipEntry = FALSE;
        PinInformation = PinInformationTable[Index];

        //
        // If this is not the last input disconnect, then no need to disconnect
        // it.
        //

        if ((DisconnectMode == ConnectModeInput) &&
            (PinInformation->Mode.Input == 1) &&
            (PinInformation->InputEnableCount > 1)) {

            SkipEntry = TRUE;
        }

        //
        // If this is not the last output disconnect or the pin was never
        // connected for output (delayed connect), then no need to disconnect
        // it.
        //

        if ((DisconnectMode == ConnectModeOutput) &&
            (PinInformation->Mode.Output == 1) &&
            ((PinInformation->OutputEnableCount > 1) ||
             (PinInformation->OutputConnected == FALSE))) {

            SkipEntry = TRUE;
        }

        if (SkipEntry == FALSE) {
            DisconnectPinTable[DisconnectCount] = PinNumberTable[Index];
            if (PinInformation->Mode.PreserveConfiguration == 1) {
                DisconnectFlags.PreserveConfiguration = 1;
            }

            DisconnectCount += 1;
        }
    }

    //
    // Call into the GPIO client driver to disconnect the specified pins.
    //

    if ((DisconnectCount > 0) && (InSurpriseRemovalContext == FALSE)) {
        Status = GpioClnInvokeDisconnectPins(GpioBank,
                                             DisconnectPinTable,
                                             DisconnectCount,
                                             DisconnectMode,
                                             DisconnectFlags.AsULONG);
    }

    //
    // Mark the connect mode as invalid.
    //

    FileContext->ConnectMode = ConnectModeInvalid;

    //
    // Update the pin information to reflect the disconnect.
    //
    // N.B. The pin is assumed disconnected irrespective of whether the
    //      client driver could successfully disconnect it or not. This is
    //      because the IO manager will destroy the file handle and no
    //      further disconnect can arrive on that handle.
    //
    //      A subsequent connect would reconnect these pins which should
    //      reprogram those pins for the new mode. So leaving the pins
    //      connected should be harmless in most cases. If doing this could
    //      cause issues for some GPIO controllers, then it is the
    //      responsibility of the client driver to handle subsequent connects
    //      correctly (i.e., it should skip connecting a pin if it has
    //      already been connected).
    //

    for (Index = 0; Index < PinCount; Index += 1) {
        PinInformation = PinInformationTable[Index];
        if (DisconnectMode == ConnectModeInput) {
            GPIO_ASSERT(PinInformation->InputEnableCount >= 1);

            PinInformation->InputEnableCount -= 1;
            if (PinInformation->InputEnableCount == 0) {
                PinInformation->Mode.Input = 0;
            }

        } else {        // ConnectModeOutput

            GPIO_ASSERT(PinInformation->OutputEnableCount >= 1);

            PinInformation->OutputEnableCount -= 1;
            if (PinInformation->OutputEnableCount == 0) {
                PinInformation->Mode.Output = 0;
                PinInformation->OutputConnected = FALSE;
            }
        }

        //
        // Reset the exclusive and preserve flags.
        //

        if ((PinInformation->Mode.Input == 0) &&
            (PinInformation->Mode.Output == 0)) {

            if (PinInformation->Mode.ExclusiveMode == 1) {
                PinInformation->Mode.ExclusiveMode = 0;
            }

            PinInformation->Mode.PreserveConfiguration = 0;
        }
    }

    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSwitchDirectionPins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in GPIO_CONNECT_IO_PINS_MODE NewDirection,
    __in BOOLEAN FastIoMode
    )

/*++

Routine Description:

    This routine configures the specified set of lines for input or output.

    N.B. This routine assumes the caller has acquired the GPIO bank lock
         prior to invocation.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    GpioBank - Supplies a pointer to the GPIO bank.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    NewDirection - Supplies the new mode in which the pins should be configured
        (input or output).

    FastIoMode - Supplies whether this routine is called in the context of a
        fast I/O operation.

Return Value:

    NTSTATUS code.

--*/

{

    NTSTATUS Status;

    if (FastIoMode == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
            Info,
            DBG_IO,
            "%s: Switching pin direction to %s! Extn = 0x%p, "
            "PinTable = 0x%p, PinCount = %#x\n",
            __FUNCTION__,
            (NewDirection == ConnectModeInput) ? "input" : "output",
            GpioExtension,
            FileContext->PinNumberTable,
            FileContext->PinCount);

        GpioUtilTracePrintPinNumbers(GpioBank,
            FileContext->PinNumberTable,
            FileContext->PinCount);
    }

    //
    // To switch direction, the current direction must be different than the
    // desired direction.
    //

    GPIO_ASSERT(FileContext->ConnectMode != NewDirection);

    //
    // If the pin is currently connected, then disconnnect it first.
    //
    // For now, always force the current configuration to be preserved
    // when switching direction for pins opened in read-write mode. This is
    // required to support one-wire protocol scenario where the pin direction
    // must be switched atomically to avoid glitches.
    //

    if (FileContext->ConnectMode != ConnectModeInvalid) {
        GpiopDisconnectGpioPinsLocked(GpioExtension, FileContext, FALSE, TRUE);
    }

    Status = GpiopConnectGpioPinsLocked(GpioExtension,
                                        GpioBank,
                                        FileContext,
                                        NewDirection,
                                        &FileContext->ConnectionId);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "%s: Failed to switch pin direction to %s! Extn = 0x%p, "
                    "PinTable = 0x%p, PinCount = %#x, Status = %#x\n",
                    __FUNCTION__,
                    (NewDirection == ConnectModeInput) ? "input" : "output",
                    GpioExtension,
                    FileContext->PinNumberTable,
                    FileContext->PinCount,
                    Status);

        goto SwitchDirectionPinsEnd;

    } else {
        if (FastIoMode == FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_IO,
                "%s: Successfully switched pin direction to %s! "
                "Extn = 0x%p, PinTable = 0x%p, PinCount = %#x\n",
                __FUNCTION__,
                (NewDirection == ConnectModeInput) ? "input" : "output",
                GpioExtension,
                FileContext->PinNumberTable,
                FileContext->PinCount);
        }
    }

SwitchDirectionPinsEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopReadPins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in ULONG Flags,
    __out PVOID Buffer,
    __in ULONG BufferSizeInBits,
    __out PULONG BitsRead
    )

/*++

Routine Description:

    This routine reads from the specified set of lines from the GPIO controller.

    For pins opened in read-write mode, the direction will be switched to
    input prior to the read operations if those pins are currently in the
    write/output mode.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    Flags - Flags used to indicate the following:

        GPIO_IO_FLAG_NONE - No special processing is needed.

        GPIO_IO_FLAG_FAST_IO_MODE - Current I/O request is using
            the fast I/O interface. Skip any unnecessary tracing that may
            reduce performance.

    Request - Supplies a handle to a framework request object.

    Buffer - Supplies a buffer that will receive the read values.

    BufferSizeInBits - Supplies the size of the buffer (in bits).

    BitsRead - Supplies a pointer to a variable that receives the total
        number of bits read. If the input buffer size is too small, then
        this will contain the buffer size required (in units of bits).

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_BANK_ENTRY GpioBank;
    BOOLEAN IoFail;
    ULONG PinCount;
    PPIN_NUMBER PinNumberTable;
    ULONG64 PinValues;
    BOOLEAN PinsWriteConfigured;
    NTSTATUS Status;
    BOOLEAN FastIoMode;

    //
    // Determine whether fast I/O interface is being used
    //

    FastIoMode = CHECK_FLAG(Flags, GPIO_IO_FLAG_FAST_IO_MODE);

    //
    // Read operation is permitted if the pins are opened for input or output.
    //
    // It is possible that the connect mode is currently invalid if the pin
    // was opened in read-write mode but the attempt to switch the pin
    // direction failed. It will be re-attemped as part of this read operation.
    //

    IoFail = FALSE;
    if ((FileContext->ConnectMode != ConnectModeInput) &&
        (FileContext->ConnectMode != ConnectModeOutput) &&
        (FileContext->Flags.ReadWriteSupported == FALSE)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopReadPins: Pins not connected for input. Read is not "
                    "permitted! Extn = 0x%p, ConnectMode = %#x\n",
                    GpioExtension,
                    FileContext->ConnectMode);

        Status = STATUS_GPIO_OPERATION_DENIED;
        goto ReadPinsEnd;
    }

    //
    // The size of the input data must match the number pins specified during
    // connect. Reading from only a subset of those pins is not supported. If
    // this is desired, the subset must be opened separately.
    //

    PinValues = 0;
    PinNumberTable = FileContext->PinNumberTable;
    PinCount = FileContext->PinCount;
    if ((PinNumberTable == NULL) || (PinCount == 0)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopReadPins: Invalid handle - Pin number table or count "
                    "is not valid! Extn = 0x%p PinNumberTable = 0x%p, "
                    "PinCount = %#x\n",
                    GpioExtension,
                    PinNumberTable,
                    PinCount);

        Status = STATUS_INVALID_HANDLE;
        goto ReadPinsEnd;
    }

    if (BufferSizeInBits < PinCount) {
        Status = STATUS_BUFFER_TOO_SMALL;
        if (ARGUMENT_PRESENT(BitsRead) != FALSE) {
            *BitsRead = PinCount;
        }

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopReadPins: Input buffer size to small...failing read!"
                    "Extn = 0x%p, Size = %d, Reqd = %d, PinCount = %#x\n",
                    GpioExtension,
                    BufferSizeInBits / 8,
                    (ULONG)(ALIGN_RANGE_UP(PinCount, 8) / 8),
                    PinCount);

        goto ReadPinsEnd;
    }
        
    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    GPIO_ASSERT(GpioBank != NULL);
    
    if (FastIoMode == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "GpiopReadPins: Attempting to read GPIO IO pins! Extn = 0x%p\n",
                    GpioExtension);
        
        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);
    }
    
    //
    // Check whether the device is still in its D0 state. If the device is
    // not in D0 state (possible if it got (surprise) removed) then fail all
    // further operations.
    //

    if (GPIO_GET_DEVICE_STATE(GpioExtension) != DEVICE_STATE_STARTED) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopReadPins: Device not in its started state! Failing "
                    "read request! Extn = 0x%p, PinCount = %#x, State = %d\n",
                    GpioExtension,
                    PinCount,
                    GpioExtension->DeviceState);

        Status = STATUS_UNSUCCESSFUL;
        goto ReadPinsEnd;
    }

    Status = STATUS_SUCCESS;

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken on such pins when they were first connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Check to make sure that a write has occurred prior to the read. Since
    // output configuration is delayed until after a write is performed,
    // checking if the configuration has happened or not is sufficient.
    //
    // This needs to be done under the protection of the passive-level
    // controller lock.
    //

    if ((FileContext->Flags.ReadWriteSupported == FALSE) &&
        (FileContext->ConnectMode == ConnectModeOutput)) {

        PinsWriteConfigured = TRUE;

    } else {
        PinsWriteConfigured = FALSE;
    }

    if ((PinsWriteConfigured != FALSE) &&
        (FileContext->Flags.WriteIssued == FALSE)) {

        Status = STATUS_GPIO_OPERATION_DENIED;
        goto ReadPinsReleaseLockEnd; // release lock and exit
    }

    //
    // For pins opened in read-write mode, switch the direction to be input
    // if the pins are not already in input mode.
    //
    // N.B. Since read operation is permitted for pins opened in write-only
    //      mode, the direction should be changed for only pins opened in
    //      read-write mode. (For write-only pins, a read is supposed to
    //      return the value of the output register/last write value and not
    //      sample the input line).
    //

    if ((FileContext->Flags.ReadWriteSupported != FALSE) &&
        (FileContext->ConnectMode != ConnectModeInput)) {

        Status = GpiopSwitchDirectionPins(GpioExtension,
                                          GpioBank,
                                          FileContext,
                                          ConnectModeInput,
                                          FastIoMode);

        if (!NT_SUCCESS(Status)) {
            goto ReadPinsReleaseLockEnd; // release lock and exit
        }

        GPIO_ASSERT(FileContext->ConnectMode == ConnectModeInput);
    }

    //
    // Acquire the bank interrupt lock as read operations for memory-mapped
    // GPIOs are invoked at DIRQL. Note this will be skipped if the client
    // driver has set the IndependentIoHwSupported flag bit.
    //

    if (GPIO_IS_INDEPENDENT_IO_HW_SUPPORTED(GpioExtension) == FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK_IF_MEMORY_MAPPED(GpioBank);
    }

    Status = GpioClnInvokeReadPins(GpioBank,
                                   PinNumberTable,
                                   PinCount,
                                   Buffer,
                                   &PinValues,
                                   FileContext->ClientIoContext,
                                   PinsWriteConfigured);

    if (GPIO_IS_INDEPENDENT_IO_HW_SUPPORTED(GpioExtension) == FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK_IF_MEMORY_MAPPED(GpioBank);
    }

    if (!NT_SUCCESS(Status)) {
        IoFail = TRUE;
    }

ReadPinsReleaseLockEnd:
    GPIO_RELEASE_BANK_LOCK(GpioBank);

    //
    // Format the mask values if requested by the client driver.
    //

    if (NT_SUCCESS(Status)) {
        GpiopFormatRequestForRead(GpioBank,
                                  PinNumberTable,
                                  PinCount,
                                  PinValues,
                                  Buffer);
    }

    if (NT_SUCCESS(Status) && (ARGUMENT_PRESENT(BitsRead) != FALSE)) {
        *BitsRead = PinCount;
    }

    if (!NT_SUCCESS(Status)) {
        if (IoFail != FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopReadPins: Client driver failed to read IO pins! "
                        "Extn = 0x%p, Size = %d, PinCount = %#x, "
                        "Status = %#x\n",
                        GpioExtension,
                        BufferSizeInBits / 8,
                        PinCount,
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopReadPins: Client driver failed to connect input"
                        " pins! Status = %#x\n",
                        Status);
        }

    } else {
        if (FastIoMode == FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "GpiopReadPins: Successfully read IO pins! Extn = 0x%p, "
                    "Buffer = 0x%p\n",
                    GpioExtension,
                    Buffer);

            GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);
        }
    }

ReadPinsEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopWritePins (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PGPIO_FILE_OBJECT_CONTEXT FileContext,
    __in WDFREQUEST Request,
    __in ULONG Flags,
    __in PVOID Buffer,
    __in ULONG BufferSizeInBits,
    __out PULONG BitsWritten
    )

/*++

Routine Description:

    This routine writes to the specified set of lines on the GPIO controller.

    For pins opened in read-write mode, the direction will be switched to
    output prior to the read operations if those pins are currently in the
    read/input mode.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    FileContext - Supplies a pointer to context associated with the open
        handle.

    Request - Supplies a handle to a framework request object.

    Flags - Flags used to indicate the following:

        GPIO_IO_FLAG_NONE - No special processing is needed
        
        GPIO_IO_FLAG_FAST_IO_MODE - Current I/O request is using
            the fast I/O interface. Skip any unnecessary tracing that
            may reduce performance.

    Buffer - Supplies a buffer that will receive the read values.

    BufferSizeInBits - Supplies the size of the buffer (in bits).

    BitsWritten - Supplies a pointer to a variable that receives the total
        number of bits written.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG64 ClearMask;
    ULONG ConnectMode;
    PPIN_NUMBER ConnectPinTable;
    ULONG CurrentConnectCount;
    PGPIO_BANK_ENTRY GpioBank;
    PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR GpioDescriptor;
    ULONG Index;
    BOOLEAN IoFail;
    ULONG PinCount;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    PPIN_NUMBER PinNumberTable;
    PIN_NUMBER RelativePin;
    ULONG64 SetMask;
    NTSTATUS Status;
    BOOLEAN FastIoMode;

    UNREFERENCED_PARAMETER(Request);

    //
    // Determine whether fast I/O interface is being used
    //

    FastIoMode = CHECK_FLAG(Flags, GPIO_IO_FLAG_FAST_IO_MODE);

    //
    // Write operation is only permitted if the pins were opened for output.
    //

    IoFail = FALSE;
    GpioDescriptor = NULL;
    PinNumberTable = FileContext->PinNumberTable;
    PinCount = FileContext->PinCount;
    ConnectMode = FileContext->ConnectMode;
    if ((ConnectMode != ConnectModeOutput) &&
        (FileContext->Flags.ReadWriteSupported == FALSE)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopWritePins: Pins not connected for output. Write is "
                    "not permitted! Extn = 0x%p, ConnectMode = %#x\n",
                    GpioExtension,
                    FileContext->ConnectMode);

        Status = STATUS_GPIO_OPERATION_DENIED;
        goto WritePinsEnd;
    }

    if ((PinNumberTable == NULL) || (PinCount == 0)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopWritePins: Invalid handle - Pin number table or count"
                    " is not valid! Extn = 0x%p PinNumberTable = 0x%p, "
                    "PinCount = %#x\n",
                    GpioExtension,
                    PinNumberTable,
                    PinCount);

        Status = STATUS_INVALID_HANDLE;
        goto WritePinsEnd;
    }

    //
    // The size of the input data must match the number pins specified during
    // connect. Writing to only a subset of those pins is not supported. If
    // this is desired, the subset must be opened separately.
    //

    if (BufferSizeInBits < PinCount) {
        Status = STATUS_BUFFER_TOO_SMALL;
        if (ARGUMENT_PRESENT(BitsWritten) != FALSE) {
            *BitsWritten = PinCount;
        }

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopWritePins: Input buffer to small...failing write!"
                    "Extn = 0x%p, Size = %d, Reqd = %d, PinCount = %#x\n",
                    GpioExtension,
                    BufferSizeInBits / 8,
                    (ULONG)(ALIGN_RANGE_UP(PinCount, 8) / 8),
                    PinCount);

        goto WritePinsEnd;
    }

    GpioBank = GpiopGetBankEntry(GpioExtension, FileContext->BankId);

    GPIO_ASSERT(GpioBank != NULL);

    if (FastIoMode == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_IO,
                    "GpiopWritePins: Attempting to write to GPIO IO pins! "
                    "Extn = 0x%p Request = 0x%p\n",
                    GpioExtension,
                    Request);
        
        GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);
    }

    //
    // Check whether the device is still in its D0 state. If the device is
    // not in D0 state (possible if it got (surprise) removed) then fail all
    // further operations.
    //

    if (GPIO_GET_DEVICE_STATE(GpioExtension) != DEVICE_STATE_STARTED) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_IO,
                    "GpiopWritePins: Device not in its started state! Failing "
                    "write request! Extn = 0x%p, PinCount = %#x, State = %d\n",
                    GpioExtension,
                    PinCount,
                    GpioExtension->DeviceState);

        Status = STATUS_UNSUCCESSFUL;
        goto WritePinsEnd;
    }

    //
    // Make sure the GPIO bank is active. A power reference should have been
    // taken on such pins when they were first connected.
    //

    GPIO_ASSERT(GpioBank->PowerData.IsActive != FALSE);

    //
    // Format the values into masks if requested by the client driver.
    //

    GpiopFormatRequestForWrite(GpioBank,
                               FileContext->PinNumberTable,
                               PinCount,
                               Buffer,
                               &SetMask,
                               &ClearMask);

    //
    // Serialize operations on the GPIO controller.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // For pins opened in read-write mode, switch the direction to be output
    // if the pins are not already in output mode.
    //

    if (ConnectMode != ConnectModeOutput) {

        GPIO_ASSERT(FileContext->Flags.ReadWriteSupported != FALSE);

        Status = GpiopSwitchDirectionPins(GpioExtension,
                                          GpioBank,
                                          FileContext,
                                          ConnectModeOutput,
                                          FastIoMode);

        if (!NT_SUCCESS(Status)) {
            goto WritePinsReleaseLockEnd; // release lock and exit
        }

        GPIO_ASSERT(FileContext->ConnectMode == ConnectModeOutput);
    }

    //
    // Acquire the bank interrupt lock as write operations for memory-mapped
    // GPIOs are invoked at DIRQL. Note this will be skipped if the client
    // driver has set the IndependentIoHwSupported flag bit.
    //

    if (GPIO_IS_INDEPENDENT_IO_HW_SUPPORTED(GpioExtension) == FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK_IF_MEMORY_MAPPED(GpioBank);
    }

    Status = GpioClnInvokeWritePins(GpioBank,
                                    FileContext->PinNumberTable,
                                    PinCount,
                                    Buffer,
                                    SetMask,
                                    ClearMask,
                                    FileContext->ClientIoContext);

    if (GPIO_IS_INDEPENDENT_IO_HW_SUPPORTED(GpioExtension) == FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK_IF_MEMORY_MAPPED(GpioBank);
    }

    if (!NT_SUCCESS(Status)) {
        IoFail = TRUE;
        goto WritePinsReleaseLockEnd;
    }

    //
    // If the connect was delayed for any pin, then connect it now and
    // supply the output value. It is possible that some other intervening
    // write happened on some of those pins, which caused them to have been
    // already connected.
    //

    FileContext->Flags.WriteIssued = TRUE;
    if (FileContext->Flags.DelayedConnect != FALSE) {

        GPIO_ASSERT(FileContext->ConnectCount > 0);

        ConnectPinTable = FileContext->ConnectPinTable;
        CurrentConnectCount = 0;
        for (Index = 0; Index < FileContext->ConnectCount; Index += 1) {
            RelativePin = ConnectPinTable[Index];
            PinInformation = GpiopGetPinEntry(GpioBank, RelativePin);
            if (PinInformation->OutputConnected == FALSE) {
                ConnectPinTable[CurrentConnectCount] = ConnectPinTable[Index];
                CurrentConnectCount += 1;
            }
        }

        if (CurrentConnectCount > 0) {
            if (FileContext->ResourceBuffer != NULL) {
                GpioDescriptor =
                    (PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR)
                    FileContext->ResourceBuffer->ConnectionProperties;
            }

            Status = GpioClnInvokeConnectPins(GpioBank,
                                              ConnectPinTable,
                                              CurrentConnectCount,
                                              ConnectModeOutput,
                                              GpioDescriptor);
        }

        //
        // If the connect is successful, then reset delayed connect to prevent
        // the connect being issued a second time.
        //

        if (NT_SUCCESS(Status)) {
            FileContext->Flags.DelayedConnect = FALSE;
            for (Index = 0; Index < CurrentConnectCount; Index += 1) {
                RelativePin = ConnectPinTable[Index];
                PinInformation = GpiopGetPinEntry(GpioBank, RelativePin);
                PinInformation->OutputConnected = TRUE;
            }
        }
    }

WritePinsReleaseLockEnd:

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    if (!NT_SUCCESS(Status)) {
        if (IoFail != FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopWritePins: Client driver failed to write IO pins!"
                        " Status = %#x\n",
                        Status);

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_IO,
                        "GpiopWritePins: Client driver failed to connect output"
                        " pins! Status = %#x\n",
                        Status);
        }

    } else {
        if (FastIoMode == FALSE) {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_IO,
                        "GpiopWritePins: Successfully wrote to IO pins! Extn = "
                        "0x%p, Request = 0x%p, Buffer = 0x%p, BitsWritten = %#x\n",
                        GpioExtension,
                        Request,
                        Buffer,
                        PinCount);

            GpioUtilTracePrintPinNumbers(GpioBank, PinNumberTable, PinCount);
        }
    }

WritePinsEnd:
    if (NT_SUCCESS(Status)) {
        *BitsWritten = PinCount;
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryEmulatedActiveBothInitialPolarities (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine, for controllers that emulate ActiveBoth, evaluates the _DSM
    method to determine which ActiveBoth pins should be programmed to have an
    initial polarity of ActiveHigh. It will do nothing if the particular _DSM
    function is not supported. It will bugcheck if the _DSM method exists
    but returns malformed output.

    For other controllers, this routine does nothing and returns success.

    If the routine fails for any reason, no initial ActiveHigh polarities will
    be recorded so the pins will default to ActiveLow.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    STATUS_INSUFFICIENT_RESOURCES on failure to allocate memory.

    STATUS_SUCCESS if the controller does not emulate ActiveBoth, or _DSM is
                   not supported, or the output from _DSM was successfully
                   processed.

    Otherwise a status code from a function call.

--*/

{

    ULONG BugCheckCause;
    USHORT ControllerPin;
    ULONG Index;
    PACPI_METHOD_ARGUMENT IntegerArgument;
    PGPIO_PIN_INFORMATION_ENTRY PinEntry;
    PACPI_EVAL_OUTPUT_BUFFER OutputBuffer;
    ULONG OutputBufferSize;
    PACPI_EVAL_INPUT_BUFFER_COMPLEX ParametersBuffer;
    NTSTATUS Status;
    BOOLEAN Supported;

    PAGED_CODE();

    BugCheckCause = 0;
    OutputBuffer = NULL;
    ParametersBuffer = NULL;

    //
    // If ActiveBoth is not emulated, we do not need to execute _DSM.
    //

    if (GPIO_IS_ACTIVE_BOTH_EMULATED(GpioExtension) == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "%s: ActiveBoth is not emulated. Skipping _DSM call.\n",
                    __FUNCTION__);

        Status = STATUS_SUCCESS;
        goto QueryEmulatedActiveBothInitialPolaritiesEnd;
    }

    //
    // Evaluating a _DSM method is a two-step process. In the first step, check
    // if a _DSM function for ActiveBothPolarity is supported. If TRUE, then
    // evaluate this function and return an appropriate result to the caller.
    //

    Supported = FALSE;
    Status = GpioUtilIsDsmFunctionSupported(
                GpioExtension->Device,
                GPIO_CONTROLLER_DSM_ACTIVE_BOTH_POLARITY_FUNCTION_INDEX,
                GPIO_CONTROLLER_DSM_REVISION,
                &Supported);

    if (!NT_SUCCESS(Status)) {

        //
        // _DSM is optional so if it does not exist, it is not a failure.
        //

        if (Status == STATUS_OBJECT_NAME_NOT_FOUND) {
            TraceEvents(GpioExtension->LogHandle,
                        Info,
                        DBG_INTERRUPT,
                        "%s: Optional _DSM does not exist.\n",
                       __FUNCTION__);

            Status = STATUS_SUCCESS;

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: _DSM query function failed to execute.\n",
                        __FUNCTION__);
        }

        goto QueryEmulatedActiveBothInitialPolaritiesEnd;
    }

    //
    // _DSM exists but the requested function for this revision is not
    // supported.  Since it's optional, this is not a failure.
    //

    if (Supported == FALSE) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "%s: _DSM does not support optional ActiveBothPolarity "
                    "function.\n",
                    __FUNCTION__);

        Status = STATUS_SUCCESS;
        goto QueryEmulatedActiveBothInitialPolaritiesEnd;
    }

    //
    // Evaluate this method for real.
    //

    Status = GpioUtilPrepareInputParametersForDsmMethod(
                GPIO_CONTROLLER_DSM_ACTIVE_BOTH_POLARITY_FUNCTION_INDEX,
                GPIO_CONTROLLER_DSM_REVISION,
                &ParametersBuffer,
                NULL);

    if (!NT_SUCCESS(Status)) {
        goto QueryEmulatedActiveBothInitialPolaritiesEnd;
    }

    //
    // Invoke a helper function to send an IOCTL to ACPI to evaluate this
    // control method.
    //

    Status = GpioUtilEvaluateAcpiMethod(GpioExtension->Device,
                                        ParametersBuffer->MethodNameAsUlong,
                                        ParametersBuffer,
                                        &OutputBuffer,
                                        &OutputBufferSize);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: _DSM ActiveBothPolarity function failed to execute.\n",
                    __FUNCTION__);

        goto QueryEmulatedActiveBothInitialPolaritiesEnd;
    }

    if (OutputBuffer == NULL) {
        BugCheckCause = 1;
        goto QueryEmulatedActiveBothInitialPolaritiesBugCheck;
    }

    //
    // The result data is a package of integers, each indicating the number of a
    // controller-relative pin that should have an initial polarity of ActiveHigh.
    // This package can be empty.
    //

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_INTERRUPT,
                "%s: _DSM ActiveBothFunction returned a package of %#x pin "
                "numbers,\n",
                __FUNCTION__,
                OutputBuffer->Count);

    GPIO_ASSERT(Add2Ptr(&OutputBuffer->Argument[0],
                        OutputBuffer->Count * sizeof(ACPI_METHOD_ARGUMENT)) <=
                Add2Ptr(OutputBuffer, OutputBuffer->Length));

    //
    // Loop for each pin number in the package.
    //

    for (Index = 0; Index < OutputBuffer->Count; Index++) {
        IntegerArgument = &OutputBuffer->Argument[Index];
        if(IntegerArgument->Type != ACPI_METHOD_ARGUMENT_INTEGER) {
            BugCheckCause = 2;
            goto QueryEmulatedActiveBothInitialPolaritiesBugCheck;
        }

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INTERRUPT,
                    "%s: Controller-relative pin #%#x is ActiveHigh.\n",
                    __FUNCTION__,
                    IntegerArgument->Argument);

        if (IntegerArgument->Argument > MAXUSHORT) {

            //
            // The pin number is out-of-range so we bugcheck.
            //

            BugCheckCause = 3;
            goto QueryEmulatedActiveBothInitialPolaritiesBugCheck;
        }

        ControllerPin = (USHORT)IntegerArgument->Argument;

        //
        // The firmware vendor may have defined the Buffer in their ASL as
        // containing the values of a fixed number of variables representing
        // ActiveHigh pin numbers.  We want to give them a way to conditionally
        // unmark a pin from being ActiveHigh, without changing the definition
        // of that Buffer, so we let them set one or more of those variables to
        // INVALID_PIN_NUMBER.
        //

        if (ControllerPin == INVALID_PIN_NUMBER) {
            continue;
        }

        if (ControllerPin >= GpioExtension->TotalPins) {

            //
            // The pin number is out-of-range so we bugcheck.
            //

            BugCheckCause = 4;
            goto QueryEmulatedActiveBothInitialPolaritiesBugCheck;
        }

        PinEntry = GpiopCreatePinEntryFromPinNumber(GpioExtension,
                                                    ControllerPin);

        if (PinEntry == NULL) {
            Status = STATUS_INSUFFICIENT_RESOURCES;
            goto QueryEmulatedActiveBothInitialPolaritiesEnd;
        }

        //
        // Record that the pin has an initial polarity of ActiveHigh.
        //

        PinEntry->EmulatedActiveBothInitialPolarityIsHigh = TRUE;
    }

QueryEmulatedActiveBothInitialPolaritiesEnd:
    if (ParametersBuffer != NULL) {
        GPIO_FREE_POOL(ParametersBuffer);
    }

    if (OutputBuffer != NULL) {
        GPIO_FREE_POOL(OutputBuffer);
    }

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Exiting with Status = %#x\n",
                    __FUNCTION__,
                    Status);
    }

    return Status;

QueryEmulatedActiveBothInitialPolaritiesBugCheck:
    KeBugCheckEx(
        GPIO_CONTROLLER_DRIVER_ERROR,
        DSM_OUTPUT_MALFORMED,
        (ULONG_PTR)GPIO_CONTROLLER_DSM_REVISION,
        (ULONG_PTR)GPIO_CONTROLLER_DSM_ACTIVE_BOTH_POLARITY_FUNCTION_INDEX,
        BugCheckCause);
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopCleanupAllBanksInterruptIo (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine disconnects previously configured set of interrupt or IO lines
    across all GPIO banks.

    Note this routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Surprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PGPIO_BANK_ENTRY GpioBank;

    GpioBank = NULL;
    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        GpiopCleanupBankInterruptsIo(GpioBank, InSurpriseRemovalContext);
    }

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopCleanupBankInterruptsIo (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in BOOLEAN InSurpriseRemovalContext
    )

/*++

Routine Description:

    This routine disconnects previously configured set of lines. It will
    call into the client driver to disconnect the pins and destroy all IO
    context associated with the file handle. By this point all the primary
    and secondary IO queues have been stopped, so no new requests should be
    arriving.

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InSurpriseRemovalContext - Supplies whether the disable is being performed
        within the normal context or as part of Suprise Removal. In the latter
        case, the GPIO class extension or GPIO client driver is no longer
        allowed to touch hardware. Thus only state clean up is performed and
        GPIO client driver not called.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_FILE_OBJECT_CONTEXT FileContext;
    PDEVICE_EXTENSION GpioExtension;
    ULONG Gsiv;
    KINTERRUPT_MODE InterruptMode;
    KINTERRUPT_POLARITY Polarity;
    PIN_NUMBER Index;
    PLIST_ENTRY NextEntry;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;

    GpioExtension = GpioBank->GpioExtension;

    //
    // Serialize operations on the GPIO bank.
    //

    GPIO_ACQUIRE_BANK_LOCK(GpioBank);

    //
    // Cleanup all the interrupts.
    //

    for (Index = 0; Index < GpioBank->PinCount; Index += 1) {
        PinInformation = GpiopGetPinEntry(GpioBank, Index);
        if (PinInformation == NULL) {
            continue;
        }

        if (PinInformation->Mode.Interrupt == 1) {
            Gsiv = PinInformation->Virq;
            InterruptMode = PinInformation->InterruptMode;
            Polarity = PinInformation->Polarity;

            //
            // Release the passive-level bank lock.
            //

            GPIO_RELEASE_BANK_LOCK(GpioBank);

            //
            // Disable the interrupt. This routine calls into the HAL to
            // ensure HAL's state for secondary interrupts is consistent
            // with GPIO class extension state.
            //

            GpioHalDisableInterrupt(Gsiv, InterruptMode, Polarity);

            //
            // Serialize operations on the GPIO bank.
            //

            GPIO_ACQUIRE_BANK_LOCK(GpioBank);
        }
    }

    while(IsListEmpty(&GpioBank->ConnectedPinsQueue) == FALSE) {
        NextEntry = GpioBank->ConnectedPinsQueue.Flink;
        FileContext = CONTAINING_RECORD(NextEntry,
                                        GPIO_FILE_OBJECT_CONTEXT,
                                        ListEntry);

        //
        // Release the passive-level bank lock.
        //

        GPIO_RELEASE_BANK_LOCK(GpioBank);

        //
        // Disconnect the pins specified in the file context. This will also
        // unlink the entry from the list.
        //

        GpiopDisconnectGpioPins(GpioExtension,
                                FileContext,
                                InSurpriseRemovalContext);

        //
        // Serialize operations on the GPIO bank.
        //

        GPIO_ACQUIRE_BANK_LOCK(GpioBank);
    }

    //
    // Release the passive-level bank lock.
    //

    GPIO_RELEASE_BANK_LOCK(GpioBank);

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopWaitForStateMachineCompletionOfDebouncedMaskedPin (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber,
    __in BOOLEAN WaitForTimerDpcCompletion
    )

/*++

Routine Description:

    This routine wraps GpiopDebounceWaitForStateMachineCompletionOfMaskedPin().

    This routine is not marked PAGED as it may be called before/after
    the boot device is in D0/D3 if boot device has GPIO dependencies.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinInformation - Supplies a pointer to the pin information for the specified
        pin.

    PinNumber - Supplies the bank-relative pin number.

    WaitForTimerDpcCompletion - Supplies whether to wait for the debounce
        and/or noise timer DPCs to complete.  If FALSE, the caller should
        subsequently call KeFlushQueuedDpcs().

Return Value:

    None.

--*/

{

    GPIO_INTERRUPT_DEBOUNCE_CONTROLLER_CONTEXT ControllerContext;

    //
    // Initialize the controller context to be passed to the debouncing
    // module.
    //

    RtlZeroMemory(&ControllerContext,
                  sizeof(GPIO_INTERRUPT_DEBOUNCE_CONTROLLER_CONTEXT));

    ControllerContext.GpioBank = GpioBank;
    ControllerContext.PinNumber = PinNumber;
    ControllerContext.PinInformation = PinInformation;
    GpiopDebounceWaitForStateMachineCompletionOfMaskedPin(
        &PinInformation->DebounceContext,
        &ControllerContext,
        WaitForTimerDpcCompletion);
}

//
// ------------------------------------------------------------------ Functions
//

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitialize (
    __in ULONG Phase,
    __in PDRIVER_OBJECT WdmDriverObject,
    __in WDFDRIVER WdfDriver,
    __in PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine perform initialization of the GPIO class extension. This is
    called from the class extension's driver initialization entry point
    (i.e., DriverEntry). This routine initializes the global client driver list
    and its lock.

Arguments:

    Phase - Supplies the phase of the initialization.

    WdmDriverObject - Supplies a pointer to the GPIO class extension's WDM
        driver object.

    WdfDriver - Supplies a handle to the WDF driver object.

    RegistryPath - Supplies a pointer to the driver specific registry key.

Return Value:

    None.

--*/

{

    NTSTATUS Status;

    PAGED_CODE();

    UNREFERENCED_PARAMETER(RegistryPath);

    if (Phase == 0) {

        //
        // Initialize the GPIO client driver list and the list lock.
        //

        InitializeListHead(&GpioClientDriverList);
        ExInitializeFastMutex(&GpioClientGlobalListLock);

        //
        // Perform phase 0 initialization of the GPIO hub.
        //

        Status = GpioHubpInitialize(0, WdmDriverObject, WdfDriver);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioLogHandle,
                        Error,
                        DBG_INIT,
                        "GpiopInitialize: Failed to initialize the HUB device "
                        "(phase 0 init)! Status = %#x\n",
                        Status);
        }

    } else {
        GPIO_ASSERT(Phase == 1);


        //
        // Query the registry for flags. Do not treat a failure as fatal.
        //
        //

        GpiopInitializeRegistryOptions(WdmDriverObject,
                                       WdfDriver,
                                       RegistryPath);

        //
        // Perform phase 1 initialization of the GPIO hub.
        //

        Status = GpioHubpInitialize(1, WdmDriverObject, WdfDriver);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioLogHandle,
                        Error,
                        DBG_INIT,
                        "GpiopInitialize: Failed to initialize the HUB device "
                        "(phase 1 init)! Status = %#x\n",
                        Status);
        }
    }

    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopUninitialize (
    __in PDRIVER_OBJECT  DriverObject
    )

/*++

Routine Description:

    This routine perform cleanup of the GPIO class extension prior to unload.

Arguments:

    DriverObject - Pointer to the driver object created by the I/O manager.

Return Value:

    None.

--*/

{

    UNREFERENCED_PARAMETER(DriverObject);

    PAGED_CODE();

    //
    // Signal the emergency thread to terminate.
    //

    if (GpioHubEmergencyThread != NULL) {
        KeSetEvent(&GpioHubEmergencyTerminateEvent, IO_NO_INCREMENT, FALSE);
    }

    //
    // Cleanup the hub device object in the meantime.
    //

    if (GpioHubDevice != NULL) {
        GpioClxEvtDeviceContextCleanup(GpioHubDevice);
    }

    //
    // Wait for the emergency service thread to terminate.
    //

    if (GpioHubEmergencyThread != NULL) {
        KeWaitForSingleObject(GpioHubEmergencyThread,
                              Executive,
                              KernelMode,
                              FALSE,
                              NULL);

        ObDereferenceObject(GpioHubEmergencyThread);
    }

    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopCreateClient (
    __in WDFDRIVER DriverObject,
    __in PGPIO_CLIENT_REGISTRATION_PACKET RegistrationPacket
    )

/*++

Routine Description:

    This routine creates a new client driver object and adds it to the global
    list of registered clients. The client driver is associated with the driver
    object handle and can be later located using that handle.

Arguments:

    DriverObject - Pointer to the driver object created by the I/O manager.

    RegistrationPacket - Supplies the registration packet that contains
        information about the client driver and all its callbacks.

Return Value:

    NTSTATUS code.

--*/

{

    PGPIO_CLIENT_DRIVER Client;
    ULONG Flags;
    USHORT Size;
    NTSTATUS Status;
    BOOLEAN Valid;

    PAGED_CODE();

    Valid = GpiopValidateRegistrationPacketPhase1(RegistrationPacket);
    if (Valid == FALSE) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "GpiopCreateClient: Registration packet is not valid!\n");

        Status = STATUS_GPIO_INVALID_REGISTRATION_PACKET;
        goto CreateClientEnd;
    }

    //
    // Allocate memory to store the client driver registration information.
    //

    Client = GPIO_ALLOCATE_POOL(NonPagedPoolNx, sizeof(GPIO_CLIENT_DRIVER));
    if (Client == NULL) {
        TraceEvents(GpioLogHandle,
                    Error,
                    DBG_INIT,
                    "GpiopCreateClient: Failed to allocate memory for client!"
                    " driver\n");

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto CreateClientEnd;
    }

    RtlZeroMemory(Client, sizeof(GPIO_CLIENT_DRIVER));
    Client->DriverObject = DriverObject;

    //
    // For regular GPIO client drivers only copy the public registration packet.
    // For internal clients, copy the combined registration packet.
    //
    // N.B. Can't use GPIO_IS_CLIENT_INTERNAL() here as the combined packet is
    //      not yet initialized.
    //

    Flags = RegistrationPacket->Flags;
    if (CHECK_FLAG(Flags, GPIO_CLIENT_REGISTRATION_FLAGS_INTERNAL) == FALSE) {
        Size = MIN(RegistrationPacket->Size,
                   sizeof(GPIO_CLIENT_REGISTRATION_PACKET));

    } else {
        Size = MIN(RegistrationPacket->Size,
                   sizeof(GPIO_CLIENT_REGISTRATION_PACKET_COMBINED));
    }

    //
    // Copy the supplied registration packet and adjust the size.
    //

    RtlCopyMemory(&Client->CombinedPacket, RegistrationPacket, Size);
    Client->CombinedPacket.Size = Size;

    //
    // Insert the client driver into the global list of registered clients.
    //

    GPIO_ACQUIRE_CLIENT_LOCK();
    InsertTailList(&GpioClientDriverList, &Client->ListEntry);
    GPIO_RELEASE_CLIENT_LOCK();
    Status = STATUS_SUCCESS;

CreateClientEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopDeleteClient (
    __in WDFDRIVER DriverObject
    )

/*++

Routine Description:

    This routine removes the client driver associated with the specified
    driver object from the global client driver list and deletes the
    client driver object.

Arguments:

    DriverObject - Pointer to the driver object created by the I/O manager.

Return Value:

    STATUS_SUCCESS if the client driver object was found (and deleted)
    successfully.

    STATUS_NOT_FOUND if the client driver object was not found.

--*/

{

    PGPIO_CLIENT_DRIVER Client;
    NTSTATUS Status;

    PAGED_CODE();

    Status = STATUS_NOT_FOUND;

    //
    // Multiple deletions could happen simultaneously. Thus the "find and
    // delete" need to happen with the lock acquired.
    //

    GPIO_ACQUIRE_CLIENT_LOCK();
    Client = GpiopFindClientNoLock(DriverObject);
    if (Client != NULL) {
        RemoveEntryList(&Client->ListEntry);
        Status = STATUS_SUCCESS;
    }

    GPIO_RELEASE_CLIENT_LOCK();

    if (Client != NULL) {
        GPIO_FREE_POOL(Client);
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
PGPIO_CLIENT_DRIVER
GpiopFindClient (
    __in WDFDRIVER DriverObject
    )

/*++

Routine Description:

    This routine finds the client driver associated with the given driver object
    in the global client driver list.

    N.B. This routine assumes the global client driver lock is not held.

Arguments:

    DriverObject - Supplies a pointer to a driver object.

Return Value:

    Pointer to client driver if found. NULL otherwise.

--*/

{

    PGPIO_CLIENT_DRIVER MatchingClient;

    PAGED_CODE();

    GPIO_ACQUIRE_CLIENT_LOCK();
    MatchingClient = GpiopFindClientNoLock(DriverObject);
    GPIO_RELEASE_CLIENT_LOCK();

    return MatchingClient;
}

_Must_inspect_result_
__drv_maxIRQL(APC_LEVEL)
PGPIO_CLIENT_DRIVER
GpiopFindClientNoLock (
    __in WDFDRIVER DriverObject
    )

/*++

Routine Description:

    This routine finds the client driver associated with the given driver
    object.

    N.B. This routine assumes the caller holds the global client driver lock.

Arguments:

    DriverObject - Supplies a pointer to a driver object.

Return Value:

    Pointer to client driver if found. Otherwise NULL.

--*/

{

    PGPIO_CLIENT_DRIVER MatchingClient;
    PGPIO_CLIENT_DRIVER ClientEntry;
    PLIST_ENTRY NextEntry;

    PAGED_CODE();

    MatchingClient = NULL;
    NextEntry = &GpioClientDriverList;
    if (IsListEmpty(NextEntry) == FALSE) {
        NextEntry = GpioClientDriverList.Flink;
    }

    //
    // Walk the client driver list and find a client driver entry that matches
    // the specified driver object.
    //

    while (NextEntry != &GpioClientDriverList) {
        ClientEntry = (PGPIO_CLIENT_DRIVER)
            CONTAINING_RECORD(NextEntry, GPIO_CLIENT_DRIVER, ListEntry);

        if (ClientEntry->DriverObject == DriverObject) {
            MatchingClient = ClientEntry;
            break;
        }

        NextEntry = NextEntry->Flink;
    }

    return MatchingClient;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank,
    __out PCLIENT_QUERY_BANK_POWER_INFORMATION_OUTPUT PowerInformation
   )

/*++

Routine Description:

    This routine queries the per-bank information from the client driver and
    validates it.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank to be initialized.

    BankInformation - Supplies a pointer to a buffer that receives the
        information returned by the client driver.

Return Value:

    NT status code.

--*/

{

    CLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT BankInformation;
    NTSTATUS Status;
    BOOLEAN Valid;

    PAGED_CODE();

    //
    // Initialize the output buffer and its size and issue the query.
    //

    RtlZeroMemory(&BankInformation, sizeof(BankInformation));
    BankInformation.Size = sizeof(BankInformation);
    Status = GpioClnInvokeQueryBankPowerInformation(GpioBank, &BankInformation);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioBank->GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver failed query for bank information! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto QueryBankInformationEnd;
    }

    Valid = GpiopValidateClientBankPowerInformation(GpioBank, &BankInformation);
    if (Valid == FALSE) {
        TraceEvents(GpioBank->GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver returned invalid bank information!\n",
                    __FUNCTION__);

        Status = STATUS_UNSUCCESSFUL;
        goto QueryBankInformationEnd;
    }

    RtlCopyMemory(PowerInformation,
                  &BankInformation.BankPowerInformation,
                  sizeof(CLIENT_QUERY_BANK_POWER_INFORMATION_OUTPUT));

QueryBankInformationEnd:
    return Status;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryControllerInformation (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine queries the attributes from the GPIO controller and initializes
    the GPIO device extension appropriately.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    CLIENT_CONTROLLER_BASIC_INFORMATION ClientInformation;
    ULONG Size;
    NTSTATUS Status;
    BOOLEAN Valid;

    PAGED_CODE();

    //
    // Query information regarding debouncing timeouts and other attributes of
    // the GPIO controller.
    //

    Size = sizeof(CLIENT_CONTROLLER_BASIC_INFORMATION);
    RtlZeroMemory(&ClientInformation, Size);
    Status = GpioClnInvokeQueryControllerBasicInformation(
                 GpioExtension,
                 &ClientInformation);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver failed query device attributes! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto InitializeControllerAttributesEnd;
    }

    //
    // Validate the interrupt and IO information supplied by the client driver.
    //

    Valid = GpiopValidateClientControllerBasicInformation(GpioExtension,
                                                          &ClientInformation);

    if (Valid == FALSE) {
        Status = STATUS_GPIO_CLIENT_INFORMATION_INVALID;
        goto InitializeControllerAttributesEnd;
    }

    RtlCopyMemory(&GpioExtension->ClientInformation, &ClientInformation, Size);

    //
    // Now that basic information has been queried from the client driver,
    // perform phase 2 of registration packet validation.
    //

    Valid = GpiopValidateRegistrationPacketPhase2(GpioExtension);

InitializeControllerAttributesEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitializeBanksAndInterrupts (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw
    )

/*++

Routine Description:

    This routine builds the GPIO banks and initializes the interrupt
    information for each bank.

    This routine assumes the total number of pins on the controller have been
    queried from the client driver.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

Return Value:

    NTSTATUS code.

--*/

{

    NTSTATUS Status;

    PAGED_CODE();

    //
    // Now that the total number of pins have been queried, build all the
    // GPIO banks.
    //

    GpioExtension->TotalPins = GpioExtension->ClientInformation.TotalPins;
    GpioExtension->PinsPerBank =
        GpioExtension->ClientInformation.NumberOfPinsPerBank;

    Status = GpiopBuildGpioBanks(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Failed to setup banks for the GPIO controller! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto InitializeBanksAndInterruptsEnd;
    }

    //
    // For each of the GPIO banks, setup it interrupt information based on
    // the resources supplied to the device. Note the resources may not be
    // described in the order of the banks.
    //

    Status = GpiopSetupBankInterrupts(GpioExtension,
                                      ResourcesTranslated,
                                      ResourcesRaw);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Failed to setup interrupts for GPIO banks!"
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto InitializeBanksAndInterruptsEnd;
    }

    //
    // Allocate all the necessary resources required for interrupt processing.
    //

    Status = GpiopAllocateInterruptProcessingResources(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "GpioClxEvtDevicePrepareHardware: Failed to allocate "
                    "resources for interrupt processing! Status = %#x\n",
                    Status);

        goto InitializeBanksAndInterruptsEnd;
    }

    //
    // N.B. On failure, the allocated buffers will be freed when the device
    //      extension is deleted during driver unload.
    //

InitializeBanksAndInterruptsEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildDeviceExtension (
    __in WDFDRIVER Driver,
    __in WDFDEVICE Device,
    __in PGPIO_CLIENT_DRIVER Client,
    __in RECORDER_LOG RecorderLogHandle
    )

/*++

Routine Description:

    This routine builds the GPIO class extension's device extension. This
    routine is called once the client driver creates the framework device
    object and supplies it to the class extension.

Arguments:

    Driver - Supplies a handle to the driver object created in DriverEntry.

    Device - Supplies a handle to the client driver's framework device object.

Return Value:

    NTSTATUS code.

--*/

{

    UNICODE_STRING DeviceNameString = {0};
    PDEVICE_EXTENSION GpioExtension;
    USHORT Length;
    PWCHAR NameBuffer;
    WDF_OBJECT_ATTRIBUTES ObjectAttributes;
    NTSTATUS Status;
    WDFSTRING StringHandle;

    UNREFERENCED_PARAMETER(Driver);

    PAGED_CODE();

    StringHandle = NULL;
    GpioExtension = GpioClxGetDeviceExtension(Device);
    RtlZeroMemory(GpioExtension, sizeof(DEVICE_EXTENSION));
    GpioExtension->Signature = GPIO_DEVICE_EXTENSION_SIGNATURE;
    GpioExtension->Device = Device;
    GpioExtension->LogHandle = RecorderLogHandle;

    Status = WdfStringCreate(NULL, WDF_NO_OBJECT_ATTRIBUTES, &StringHandle);
    if (NT_SUCCESS(Status)) {
        Status = WdfDeviceRetrieveDeviceName(Device, StringHandle);
        if (NT_SUCCESS(Status)) {
            WdfStringGetUnicodeString(StringHandle, &DeviceNameString);
        }
    }

    if ((DeviceNameString.Buffer == NULL) ||
        (DeviceNameString.Length <= sizeof(WCHAR))) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "GpiopBuildDeviceExtension: Failed to retrieve device "
                    "name!\n");

        Status = STATUS_UNSUCCESSFUL;
        goto BuildDeviceExtensionEnd;
    }

    //
    // The string may or may not be NULL terminated, so account for it
    // separately.
    //

    Length = DeviceNameString.Length + sizeof(WCHAR);
    NameBuffer = GPIO_ALLOCATE_POOL(PagedPool, Length);
    if (NameBuffer == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "GpiopBuildDeviceExtension: Failed to allocate memory for "
                    "device name string!\n");

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto BuildDeviceExtensionEnd;
    }

    RtlCopyMemory(NameBuffer, DeviceNameString.Buffer, DeviceNameString.Length);
    NameBuffer[(Length / sizeof(WCHAR)) - 1] = UNICODE_NULL;
    GpioExtension->TargetName.Buffer = NameBuffer;
    GpioExtension->TargetName.Length = Length - sizeof(WCHAR);
    GpioExtension->TargetName.MaximumLength = Length;

    //
    // Register the fact that the next D0 entry transition would be the first
    // ever for this device.
    //

    GpioExtension->InitialD0Entry = TRUE;

    //
    // Create the spin-lock that is used to synchronize state during device
    // powering down.
    //

    WDF_OBJECT_ATTRIBUTES_INIT(&ObjectAttributes);
    ObjectAttributes.ParentObject = Device;
    Status = WdfSpinLockCreate(&ObjectAttributes,
                               &GpioExtension->PowerDownLock);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%!FUNC! - WdfSpinLockCreate() failed with %!status!.\n",
                    Status);

        goto BuildDeviceExtensionEnd;
    }

    //
    // Initialize the device D0 transition worker.
    //

    Status = GpiopInitializeDevicePowerWorker(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s - GpiopInitializeDevicePowerWorker() failed with "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto BuildDeviceExtensionEnd;
    }

    //
    // Initialize the base interrupt GSIV and synchronization IRQL.
    //

    GpioExtension->BaseInterruptGsiv = GPIO_UNSPECIFIED_INTERRUPT;
    GpioExtension->SynchronizationIrql = DISPATCH_LEVEL + 1;

    //
    // Copy over the registration data supplied by the client driver to the
    // device extension. This eliminates the need to walk over the client
    // driver list every time. Only the valid size was copied over to the
    // internal copy at registration time.
    //
    // For internal clients, this will also copy over the internal packet.
    //

    GPIO_ASSERT(Client->CombinedPacket.Size <=
                    sizeof(GPIO_CLIENT_REGISTRATION_PACKET_COMBINED));

    RtlCopyMemory(&GpioExtension->CombinedPacket,
                  &Client->CombinedPacket,
                  Client->CombinedPacket.Size);

    //
    // Copy the registration packet to the publicly-visible registration
    // packet (used for debugging only).
    //

    RtlCopyMemory(&GpioExtension->RegistrationPacketDebug,
                  &GpioExtension->CombinedPacket,
                  sizeof(GPIO_CLIENT_REGISTRATION_PACKET));

    GpioExtension->RegistrationPacketDebug.Size =
            MIN(GpioExtension->RegistrationPacketDebug.Size,
                sizeof(GPIO_CLIENT_REGISTRATION_PACKET));

    //
    // Propogate the internal flag to the device extension.
    //

    if (GPIO_IS_CLIENT_INTERNAL(Client) != FALSE) {
        GpioExtension->IsInternalClient = TRUE;
    }

    ExInitializeRundownProtection(
        &GpioExtension->EventData.EventWorkerReferences);

    //
    // Initialize the mutex used to synchronize ACPI event state machine
    // transitions.
    //

    KeInitializeEvent(&GpioExtension->EventData.MutexEvent,
                      SynchronizationEvent,
                      TRUE);

    //
    // Initialize the mutex used to synchronize access to ACPI event data.
    //

    ExInitializeFastMutex(&GpioExtension->EventData.EventLock);

    //
    // Initialize the spinlock used to synchronize access to ACPI event data.
    //

    KeInitializeSpinLock(&GpioExtension->EventData.EventInterruptLock);

BuildDeviceExtensionEnd:
    if(StringHandle != NULL) {
        WdfObjectDelete(StringHandle);
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopBuildGpioBanks (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine builds and initializes all the GPIO banks.

    N.B. On failure, the allocated buffers will be freed when the device
         extension is deleted during driver unload.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG AlignedPinCount;
    PGPIO_BANK_ENTRY GpioBank;
    BANK_ID Index;
    ULONG Size;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // The information to determine the total banks must have been setup
    // prior to invocation.
    //

    GPIO_ASSERT(GpioExtension->PinsPerBank > 0);
    GPIO_ASSERT(GpioExtension->TotalPins >= GpioExtension->PinsPerBank);

    AlignedPinCount = ROUND_UP(GpioExtension->TotalPins,
                               GpioExtension->PinsPerBank);

    GpioExtension->TotalBanks =
        (USHORT)AlignedPinCount / GpioExtension->PinsPerBank;

    Size = GpioExtension->TotalBanks * sizeof(GPIO_BANK_ENTRY);

    GPIO_ASSERT(GpioExtension->Banks == NULL);

    GpioExtension->Banks = GPIO_ALLOCATE_POOL(NonPagedPoolNx, Size);
    if (GpioExtension->Banks == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Failed to allocate memory for GPIO banks. "
                    "Size = %#x!\n",
                    __FUNCTION__,
                    Size);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto BuildGpioBanksEnd;
    }

    RtlZeroMemory(GpioExtension->Banks, Size);

    //
    // Walk through each bank and initialize its fields.
    //

    for (Index = 0; Index < GpioExtension->TotalBanks; Index += 1) {
        GpioBank = &GpioExtension->Banks[Index];
        GpioBank->BankId = Index;
        GpioBank->GpioExtension = GpioExtension;

        //
        // Create the passive-level bank lock. The wait lock is used to
        // synchronize access to the GPIO bank.
        //

        Status = WdfWaitLockCreate(WDF_NO_OBJECT_ATTRIBUTES,
                                   &GpioBank->PassiveLock);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "GpiopBuildDeviceExtension: WdfWaitLockCreate() failed! "
                        "Status = %#x\n",
                        Status);

            goto BuildGpioBanksEnd;
        }

        //
        // Initialize interrupt data structure with default values.
        //

        GpiopSetInterruptInformationFromResources(GpioBank, NULL, NULL);

        //
        // Initialize IO data structure. Since the bank is zeroed out, no
        // special initialization is required for IO data.
        //

        //
        // Initialize the power data. Query the per-bank power information from
        // the client driver and setup any resources necessary for bank power
        // management.
        //

        Status = GpiopInitializeBankPowerInformation(GpioBank);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to intialize bank power data! Extn = %p, "
                        "Bank ID = 0x%x!\n",
                        __FUNCTION__,
                        GpioExtension,
                        GpioBank->BankId);

            goto BuildGpioBanksEnd;
        }

        //
        // Initialize the pins in this bank. All banks upto the last one have
        // same pins per bank; the last one gets the residual value.
        //

        if (Index < (GpioExtension->TotalBanks - 1)) {
            GpioBank->PinCount = GpioExtension->PinsPerBank;

        } else {
            GpioBank->PinCount = GpioExtension->TotalPins -
                (GpioExtension->PinsPerBank * (GpioExtension->TotalBanks - 1));
        }

        //
        // The GPIO pin table is allocated on-demand -- when connecting a pin
        // for I/O or interrupts and, during bank creation, when marking a
        // pin's EmulatedActiveBothInitialPolarityIsHigh field.
        //

        GPIO_WRITE_DYNAMIC_PIN_TABLE_POINTER(GpioBank, NULL);

        InitializeListHead(&GpioBank->ConnectedPinsQueue);

        //
        // It's OK if an ActivityId can't be created.
        //

        (VOID)EtwActivityIdControl(EVENT_ACTIVITY_CTRL_CREATE_ID, &GpioBank->ActivityId);
    }

    Status = GpiopQueryEmulatedActiveBothInitialPolarities(GpioExtension);
    if (!NT_SUCCESS(Status)) {
        goto BuildGpioBanksEnd;
    }

    //
    // Now that number of banks (i.e. components) is known, setup a queue per
    // component.
    //

    Status = GpiopSetupBankSecondaryIoQueue(GpioExtension);

BuildGpioBanksEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopAllocateDebounceResources (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PGPIO_PIN_INFORMATION_ENTRY PinInformation,
    __in PIN_NUMBER PinNumber,
    __out PGPIO_DEBOUNCE_TIMER_CONTEXT *DebounceTimerContextOut,
    __out PGPIO_NOISE_FILTER_TIMER_CONTEXT *NoiseFilterTimerContextOut,
    __out WDFWORKITEM *DebounceNoiseTimerWorkerOut
    )

/*++

Routine Description:

    This routine allocates resources like debounce timer, noise interval timer
    and a debounce worker for emulating debouncing on an interrupt pin.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinInformation - Supplies a pointer to the pin information for the specified
        pin.

    PinNumber - Supplies the pin number for which the entry is to be retrieved.

    DebounceTimerContextOut - Supplies a pointer that receives the allocated
        debounce interval timer context.

    NoiseFilterTimerContextOut - Supplies a pointer that receives the allocated
        noise filter timer context.

    DebounceNoiseTimerWorkerOut - Supplies a pointer that receives the debounce
        and noise timer expiration worker. This is only allocated for
        off-SoC GPIO controllers.

Return Value:

    NTSTATUS code.

--*/

{

    WDF_OBJECT_ATTRIBUTES Attributes;
    PGPIO_DEBOUNCE_NOISE_TIMER_WORK_ITEM_CONTEXT DebounceNoiseWorkerContext;
    WDFWORKITEM DebounceNoiseTimerWorkerHandle;
    PGPIO_DEBOUNCE_TIMER_CONTEXT DebounceTimerContext;
    ULONG DebounceTimerContextSize;
    PDEVICE_EXTENSION GpioExtension;
    PVOID DebounceTimer;
    PVOID NoiseFilterTimer;
    PGPIO_NOISE_FILTER_TIMER_CONTEXT NoiseFilterTimerContext;
    ULONG NoiseFilterTimerContextSize;
    NTSTATUS Status;
    WDF_WORKITEM_CONFIG WorkItemConfiguration;

    PAGED_CODE();

    DebounceNoiseWorkerContext = NULL;
    DebounceNoiseTimerWorkerHandle = NULL;
    DebounceTimerContext = NULL;
    GpioExtension = GpioBank->GpioExtension;
    DebounceTimer = NULL;
    NoiseFilterTimer = NULL;
    NoiseFilterTimerContext = NULL;
    Status = STATUS_SUCCESS;

    //
    //
    // Create a workitem to process debounce and noise filters in a worker
    // thread. The worker is not needed for memory-mapped GPIO controllers.
    //

    if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) != FALSE) {
        WDF_OBJECT_ATTRIBUTES_INIT(&Attributes);
        Attributes.ParentObject = GpioExtension->Device;
        WDF_OBJECT_ATTRIBUTES_SET_CONTEXT_TYPE(
            &Attributes,
            GPIO_DEBOUNCE_NOISE_TIMER_WORK_ITEM_CONTEXT);

        //
        // Initialize the worker routine and create a new workitem.
        //

        WDF_WORKITEM_CONFIG_INIT(&WorkItemConfiguration,
                                 GpioClxEvtDebounceNoiseFilterExpirationWorker);

        //
        // Disable automatic serialization by the framework for the worker
        // thread. The parent device object is being serialized at device level
        // (i.e., WdfSynchronizationScopeDevice), and the framework requires it
        // be passive level (i.e., WdfExecutionLevelPassive) if automatic
        // synchronization is desired.
        //

        WorkItemConfiguration.AutomaticSerialization = FALSE;
        Status = WdfWorkItemCreate(&WorkItemConfiguration,
                                   &Attributes,
                                   &DebounceNoiseTimerWorkerHandle);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: Failed to allocate debounce/noise timer work item!"
                        " Status = %#x\n",
                        __FUNCTION__,
                        Status);

            goto AllocateDebounceResourcesEnd;
        }

        //
        // Initialize the context to be supplied to the workitem handler.
        //

        DebounceNoiseWorkerContext =
            GpiopGetDebounceNoiseTimerWorkItemContext(
                      DebounceNoiseTimerWorkerHandle);

        //
        // Partially initialize the debounce/noise timer expiration worker
        // context. The remaining fields will be initialized before the worker
        // is queued.
        //

        RtlZeroMemory(DebounceNoiseWorkerContext,
                      sizeof(GPIO_DEBOUNCE_NOISE_TIMER_WORK_ITEM_CONTEXT));

        DebounceNoiseWorkerContext->Bank = GpioBank;
    }

    //
    // Allocate memory for the debounce timer context.
    //

    DebounceTimerContextSize = sizeof(GPIO_DEBOUNCE_TIMER_CONTEXT);
    DebounceTimerContext = GPIO_ALLOCATE_POOL(NonPagedPoolNx,
                                              DebounceTimerContextSize);

    if (DebounceTimerContext == NULL) {
        Status = STATUS_NO_MEMORY;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Failed to allocate memory for debounce timer context!"
                    " Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto AllocateDebounceResourcesEnd;
    }

    //
    // Allocate an Enhanced timer for the debounce interval. The timer
    // needs to be high-res and IR-resilient.
    //
    // N.B. The wrapper specifies the (EX_TIMER_IDLE_RESILIENT |
    //      EX_TIMER_HIGH_RESOLUTION) attributes.
    //

    DebounceTimer = GpiopExAllocateTimerInternal(GpioClxEvtDebounceTimerHandler,
                                                 DebounceTimerContext);

    if (DebounceTimer == NULL) {
        Status = STATUS_INSUFFICIENT_RESOURCES;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Failed to allocate enhanced debounce timer!"
                    " Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto AllocateDebounceResourcesEnd;
    }

    //
    // Allocate memory for the noise filter timer context.
    //

    NoiseFilterTimerContextSize = sizeof(GPIO_NOISE_FILTER_TIMER_CONTEXT);
    NoiseFilterTimerContext = GPIO_ALLOCATE_POOL(NonPagedPoolNx,
                                                 NoiseFilterTimerContextSize);

    if (NoiseFilterTimerContext == NULL) {
        Status = STATUS_NO_MEMORY;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Failed to allocate memory for noise timer context!"
                    " Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto AllocateDebounceResourcesEnd;
    }

    //
    // Allocate an Enhanced timer for the noise filter interval. The timer
    // needs to be high-res and IR-resilient.
    //
    // N.B. The wrapper specifies the (EX_TIMER_IDLE_RESILIENT |
    //      EX_TIMER_HIGH_RESOLUTION) attributes.
    //

    NoiseFilterTimer =
        GpiopExAllocateTimerInternal(GpioClxEvtNoiseFilterTimerHandler,
                                     NoiseFilterTimerContext);

    if (NoiseFilterTimer == NULL) {
        Status = STATUS_INSUFFICIENT_RESOURCES;
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Failed to allocate enhanced noise filter timer!"
                    " Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto AllocateDebounceResourcesEnd;
    }

    //
    // Initialize the debounce timer context.
    //

    RtlZeroMemory(DebounceTimerContext, DebounceTimerContextSize);
    DebounceTimerContext->Bank = GpioBank;
    DebounceTimerContext->PinNumber = PinNumber;
    DebounceTimerContext->PinInformation = PinInformation;
    DebounceTimerContext->DebounceTimer = DebounceTimer;
    DebounceTimerContext->DebounceNoiseTimerWorker =
        DebounceNoiseTimerWorkerHandle;

    //
    // Initialize the noise filter timer context.
    //

    RtlZeroMemory(NoiseFilterTimerContext, NoiseFilterTimerContextSize);
    NoiseFilterTimerContext->Bank = GpioBank;
    NoiseFilterTimerContext->PinNumber = PinNumber;
    NoiseFilterTimerContext->PinInformation = PinInformation;
    NoiseFilterTimerContext->NoiseFilterTimer = NoiseFilterTimer;
    NoiseFilterTimerContext->DebounceNoiseTimerWorker =
        DebounceNoiseTimerWorkerHandle;

AllocateDebounceResourcesEnd:
    if (!NT_SUCCESS(Status)) {
        if (DebounceTimerContext != NULL) {
            GPIO_FREE_POOL(DebounceTimerContext);
            DebounceTimerContext = NULL;
        }

        if (NoiseFilterTimerContext != NULL) {
            GPIO_FREE_POOL(NoiseFilterTimerContext);
            NoiseFilterTimerContext = NULL;
        }

        if (DebounceTimer != NULL) {
            ExDeleteTimer(DebounceTimer, TRUE, FALSE, NULL);
            DebounceTimer = NULL;
        }

        if (NoiseFilterTimer != NULL) {
            ExDeleteTimer(NoiseFilterTimer, TRUE, FALSE, NULL);
            NoiseFilterTimer = NULL;
        }

        if (DebounceNoiseTimerWorkerHandle != NULL) {
            WdfObjectDelete(DebounceNoiseTimerWorkerHandle);
            DebounceNoiseTimerWorkerHandle = NULL;
        }
    }

    *DebounceTimerContextOut = DebounceTimerContext;
    *NoiseFilterTimerContextOut = NoiseFilterTimerContext;
    *DebounceNoiseTimerWorkerOut = DebounceNoiseTimerWorkerHandle;
    return Status;
}

VOID
GpiopScheduleDebounceTimers (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine starts timers for each pin that has interrupted and for
    which debouncing needs to be emulated. This routine accounts for the
    approximate spent since the interrupt fired and the timer is being
    scheduled. This is because significant amount of time could have passed
    in case of off-SoC GPIOs since an interrupt fires and the service routine
    actually runs. For off-SoC GPIOs, there is inherent ambiguity due to lack of
    knowledge of which pin asserted when the (DIRQL) ISR fires. Thus if the ISR
    is called multiple times due to multiple pins, then it can't be determined
    the order in which the pins asserted. Hence the timing delta computed here
    is approximate.

    Entry IRQL:
        1. This routine can be called at DISPATCH_LEVEL for memory-mapped GPIO
           controllers and PASSIVE_LEVEL for off-SoC GPIO controllers.

    Calling contexts:
        This routine may be invoked:
        1. Synchronously in the context of the interrupt service routine for
           off-SoC GPIOs only, or
            - passive-level lock held

        2. Asynchronously in the context of a DPC (due to interrupt debouncing)
           for memory-mapped GPIOs only, or
            - No lock held

        3. Lazily in the context of a worker (for off-SoC GPIOs only)
            - No lock held


Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

Return Value:

    None.

--*/

{

    BOOLEAN AcquireInterruptLock;
    ULONG DebounceTimeout;
    PEX_TIMER DebounceTimer;
    PGPIO_DEBOUNCE_TIMER_CONTEXT DebounceTimerContext;
    PDEVICE_EXTENSION GpioExtension;
    PEX_TIMER NoiseFilterTimer;
    PGPIO_NOISE_FILTER_TIMER_CONTEXT NoiseFilterTimerContext;
    BOOLEAN PassiveGpio;
    ULONG64 PendingTimerMask;
    PIN_NUMBER PinNumber;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    BOOLEAN RequestInInterruptContext;
    LONG64 TimerDueTime;

#ifdef DEBOUNCE_ELAPSED_TIME_ACCOUNTING

    LONG64 CurrentInterruptTime;
    LONG64 DeltaInterruptTime;
    LONG64 LastInterruptTime;

#endif

    //
    // Atomically read and clear the pending debounce timer value.
    //

    PendingTimerMask = GPIO_SET_DEBOUNCE_TIMER_MASK_VALUE(GpioBank, 0x0);
    if (PendingTimerMask == 0x0) {
        goto ScheduleDebounceTimersEnd;
    }

    GpioExtension = GpioBank->GpioExtension;
    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    GPIO_ASSERT((PassiveGpio == FALSE) ||
                (KeGetCurrentIrql() == PASSIVE_LEVEL));

    //
    // Determine whether passive-level and/or interrupt lock need to be
    // acquired.
    //

    RequestInInterruptContext = GPIO_IS_REQUEST_IN_INTERRUPT_CONTEXT(GpioBank);
    if ((PassiveGpio != FALSE) || (RequestInInterruptContext == FALSE)) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;

        GPIO_ASSERT(GpioBank->InterruptThread == KeGetCurrentThread());
    }

    //
    // Clear off interrupts that are no longer connected.
    //

    PendingTimerMask &= GpioBank->InterruptData.EnableRegister;
    while (PendingTimerMask > 0) {
        DebounceTimeout = 0x0;
        DebounceTimer = NULL;
        NoiseFilterTimer = NULL;
        PinNumber = RtlFindLeastSignificantBit(PendingTimerMask);
        PendingTimerMask &= ~(1ULL << PinNumber);

        if (AcquireInterruptLock != FALSE) {
            GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

        GPIO_ASSERT(PinInformation != NULL);

        if (PinInformation->Mode.Interrupt == 1) {

            GPIO_ASSERT(PinInformation->Mode.EmulateDebounce == 1);

            DebounceTimeout = PinInformation->DebounceTimeout;
            DebounceTimerContext = (PGPIO_DEBOUNCE_TIMER_CONTEXT)
                PinInformation->DebounceTimerContext;

            DebounceTimer = DebounceTimerContext->DebounceTimer;

            NoiseFilterTimerContext = (PGPIO_NOISE_FILTER_TIMER_CONTEXT)
                PinInformation->NoiseFilterTimerContext;

            NoiseFilterTimer = NoiseFilterTimerContext->NoiseFilterTimer;
        }

        if (AcquireInterruptLock != FALSE) {
            GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        if (DebounceTimer == NULL) {
            continue;
        }

        //
        // Before scheduling the debounce timer, cancel the noise filter timer
        // if it is still queued. It is OK for the timer to have already
        // fired and be racing with this thread. The interrupt epoch should
        // no longer match the new interrupt corresponding to this debounce
        // timer request.
        //

        if (NoiseFilterTimer != NULL) {
            ExCancelTimer(NoiseFilterTimer, NULL);
        }


#ifdef DEBOUNCE_ELAPSED_TIME_ACCOUNTING

        CurrentInterruptTime = (LONG64)KeQueryInterruptTime();
        LastInterruptTime = GpioBank->InterruptData.LastInterruptTime;
        if (CurrentInterruptTime > LastInterruptTime) {
            DeltaInterruptTime = CurrentInterruptTime - LastInterruptTime;

        } else {
            DeltaInterruptTime = 0;
        }

        //
        // Debounce time is specified in 100s of msec.
        //

        TimerDueTime = DebounceTimeout * 10 * 10;
        if (TimerDueTime > DeltaInterruptTime) {
            TimerDueTime -= DeltaInterruptTime;

        } else {
            TimerDueTime = 1;   // immediate timeout
        }

        TimerDueTime *= -1; // relative timeout

#else

        //
        // Debounce time is specified in 100s of msec.
        //

        TimerDueTime = WDF_REL_TIMEOUT_IN_US(DebounceTimeout * 10);

#endif

        EventWrite_DEBOUNCE_TIMER_START(
            &GpioBank->ActivityId,
            GpioExtension->BiosName.Buffer,
            GpioBank->BankId,
            PinNumber,
            (USHORT)DebounceTimeout);

        ExSetTimer(DebounceTimer, TimerDueTime, 0, NULL);
    }

ScheduleDebounceTimersEnd:
    return;
}

VOID
GpiopScheduleNoiseFilterTimers (
    __in PGPIO_BANK_ENTRY GpioBank
    )

/*++

Routine Description:

    This routine starts the noise filter timer for each pin that has such a
    request pending.

    Entry IRQL:
        1. This routine can be called at DISPATCH_LEVEL for memory-mapped GPIO
           controllers and PASSIVE_LEVEL for off-SoC GPIO controllers.

    Calling contexts:
        This routine may be invoked:
        1. Synchronously in the context of debounce DPC for memory-mapped GPIOs
           only, or
            - No lock held

        2. Lazily in the context of a debounce worker or noise expiration worker
            - passive-level bank lock held

    N.B. The bank interrupt lock is *not* held when invoked for memory-mapped
         or off-SoC GPIOs.


Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

Return Value:

    None.

--*/

{

    BOOLEAN AcquireInterruptLock;
    PGPIO_DEBOUNCE_PIN_CONTEXT DebounceContext;
    PDEVICE_EXTENSION GpioExtension;
    ULONG InterruptEpoch;
    ULONG64 NoiseFilterTimeout;
    PEX_TIMER NoiseFilterTimer;
    PGPIO_NOISE_FILTER_TIMER_CONTEXT NoiseFilterTimerContext;
    BOOLEAN PassiveGpio;
    ULONG64 PendingTimerMask;
    PIN_NUMBER PinNumber;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    BOOLEAN RequestInInterruptContext;
    LONG64 TimerDueTime;

    NoiseFilterTimerContext = NULL;

    //
    // Atomically read and clear the pending noise timer value.
    //

    PendingTimerMask = GPIO_SET_NOISE_FILTER_TIMER_MASK_VALUE(GpioBank, 0x0);
    if (PendingTimerMask == 0x0) {
        goto ScheduleNoisFilterTimersEnd;
    }

    GpioExtension = GpioBank->GpioExtension;
    PassiveGpio = GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension);
    RequestInInterruptContext = GPIO_IS_REQUEST_IN_INTERRUPT_CONTEXT(GpioBank);

    //
    // This routine should only be entered at PASSIVE_LEVEL for off-SoC GPIOs.
    //

    if (PassiveGpio != FALSE) {

        GPIO_ASSERT((KeGetCurrentIrql() == PASSIVE_LEVEL) &&
                    (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) != FALSE));

    //
    // The request should never arrive within the interrupt context for
    // memory-mapped GPIOs.
    //

    } else {

        GPIO_ASSERT(RequestInInterruptContext == FALSE);
    }

    //
    // Determine whether the bank interrupt lock needs to be acquired or not.
    //

    if (GPIO_BANK_INTERRUPT_LOCK_OWNER(GpioBank) == FALSE) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Clear off interrupts that are no longer connected.
    //

    PendingTimerMask &= GpioBank->InterruptData.EnableRegister;
    while (PendingTimerMask > 0) {
        NoiseFilterTimeout = 0x0;
        NoiseFilterTimer = NULL;
        PinNumber = RtlFindLeastSignificantBit(PendingTimerMask);
        PendingTimerMask &= ~(1ULL << PinNumber);

        PinInformation = GpiopGetPinEntry(GpioBank, PinNumber);

        GPIO_ASSERT(PinInformation != NULL);

        if (PinInformation->Mode.Interrupt == 1) {

            GPIO_ASSERT(PinInformation->Mode.EmulateDebounce == 1);

            NoiseFilterTimeout = PinInformation->NoiseFilterTimeout;
            NoiseFilterTimerContext = PinInformation->NoiseFilterTimerContext;
            NoiseFilterTimer = NoiseFilterTimerContext->NoiseFilterTimer;
        }

        DebounceContext = &PinInformation->DebounceContext;
        InterruptEpoch = GpiopDebounceQueryInterruptEpoch(DebounceContext);

        if (AcquireInterruptLock != FALSE) {
            GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
        }

        if (NoiseFilterTimer == NULL) {
            continue;
        }

        NoiseFilterTimerContext->InterruptEpoch = InterruptEpoch;

        //
        // Noise filter interval is specified in micro-seconds (minimum set to
        // 500us).
        //

        TimerDueTime = WDF_REL_TIMEOUT_IN_US(NoiseFilterTimeout);


        ExSetTimer(NoiseFilterTimer, TimerDueTime, 0, NULL);

        if (AcquireInterruptLock != FALSE) {
            GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
        }
    }

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

ScheduleNoisFilterTimersEnd:
    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetupBankInterrupts (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw
    )

/*++

Routine Description:

    This routine sets up a framework interrupt object for each GPIO bank. An
    interrupt may be shared across a subset of banks.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{


    WDF_OBJECT_ATTRIBUTES Attributes;
    BOOLEAN DirqlIsrRequired;
    PGPIO_BANK_ENTRY GpioBank;
    BANK_ID Index;
    WDF_OBJECT_ATTRIBUTES InterruptLockAttributes;
    PGPIO_INTERRUPT_OBJECT_CONTEXT InterruptContext;
    WDF_INTERRUPT_CONFIG InterruptConfiguration;
    WDF_INTERRUPT_CONFIG PassiveInterruptConfiguration;
    PGPIO_CLIENT_REGISTRATION_PACKET RegistrationPacket;
    ULONG InterruptCount;
    NTSTATUS Status;

    PAGED_CODE();

    GpiopDetermineInterruptCountAndIrql(GpioExtension,
                                        ResourcesTranslated,
                                        ResourcesRaw,
                                        &InterruptCount,
                                        &GpioExtension->SynchronizationIrql,
                                        &GpioExtension->BaseInterruptGsiv);

    Status = GpiopResolveInterruptBinding(GpioExtension,
                                          InterruptCount,
                                          ResourcesTranslated,
                                          ResourcesRaw);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Failed to resolve interrupt binding! Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto SetupInterruptsForBanksEnd;
    }

    //
    // Initialize the interrupt and DPC callbacks for the client driver's
    // interrupt. The framework will automatically connect the interrupt for
    // the driver prior to transitioning into D0.
    //

    WDF_INTERRUPT_CONFIG_INIT(&InterruptConfiguration,
                              GpioClxEvtInterruptIsr,
                              GpioClxEvtInterruptDpc);

    WDF_OBJECT_ATTRIBUTES_INIT_CONTEXT_TYPE(&Attributes,
                                            GPIO_INTERRUPT_OBJECT_CONTEXT);

    WDF_OBJECT_ATTRIBUTES_INIT(&InterruptLockAttributes);
    InterruptLockAttributes.ParentObject = GpioExtension->Device;

    //
    // Initialize the passive-interrupt configuration.
    //

    WDF_INTERRUPT_CONFIG_INIT(&PassiveInterruptConfiguration,
                              GpioClxEvtInterruptPassiveIsr,
                              NULL);

    PassiveInterruptConfiguration.PassiveHandling = TRUE;

    //
    // Determine whether a bank interrupt (i.e. DIRQL-level ISR) is required or
    // not for this GPIO.
    //
    // For on-SoC this is required for interrupt processing. For off-SoC
    // GPIOs, this required for invoking client driver's pre-process
    // interrupt callback as well as tracking the interrupt time for
    // debounce time adjustment (if that support is enabled). If there is no
    // pre-process callback,and debounce time need not be emulated (or time
    // adjustment support is enabled), then this is not required.
    //

    DirqlIsrRequired = TRUE;
    RegistrationPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    if ((GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) != FALSE) &&
        (RegistrationPacket->CLIENT_PreProcessControllerInterrupt == NULL)) {

        DirqlIsrRequired = FALSE;

#ifdef DEBOUNCE_ELAPSED_TIME_ACCOUNTING

        if (GPIO_IS_DEBOUNCING_EMULATED(GpioExtension) != FALSE)) {

            DirqlIsrRequired = TRUE;

            //
            // Since the bank (DIRQL) interrupt is only being connected for
            // debounce time adjustment for emulated-debouncing for off-SoC
            // GPIOs, actively manage it.
            //

            GpioExtension->ActivelyManageBankInterrupt = TRUE;
        }

#endif

    }

    //
    // Create a WDF interrupt object for each bank on the controller. If some
    // banks share an interrupt line, then number of interrupt resources
    // supplied to the driver will be less the interrupt objects created.
    // In such case, the additional interrupt objects will be unused by WDF.
    //

    for (Index = 0; Index < GpioExtension->TotalBanks; Index += 1) {
        GpioBank = &GpioExtension->Banks[Index];
        Status = WdfSpinLockCreate(&InterruptLockAttributes,
                                   &GpioBank->BankInterruptLock);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s - WdfSpinLockCreate() failed for bank interrupt"
                        " lock! Bank ID = 0x%x, Status = %#x.\n",
                        __FUNCTION__,
                        Index,
                        Status);

            goto SetupInterruptsForBanksEnd;
        }

        if (GpioBank->InterruptData.Gsiv != GPIO_UNSPECIFIED_INTERRUPT) {
            InterruptConfiguration.InterruptTranslated =
                WdfCmResourceListGetDescriptor(ResourcesTranslated,
                                               GpioBank->ResourceIndex);

            InterruptConfiguration.InterruptRaw =
               WdfCmResourceListGetDescriptor(ResourcesRaw,
                                              GpioBank->ResourceIndex);

            PassiveInterruptConfiguration.InterruptTranslated =
                InterruptConfiguration.InterruptTranslated;

            PassiveInterruptConfiguration.InterruptRaw =
                InterruptConfiguration.InterruptRaw;

        } else {
            InterruptConfiguration.InterruptTranslated = NULL;
            InterruptConfiguration.InterruptRaw = NULL;
            continue;
        }

        //
        // Create a DIRQL interrupt handler if the GPIO is on-SoC or if there
        // is a pre-process interrupt callback registered. The pre-process
        // callback needs to be called within the context of the interrupt
        // (and not passively).
        //

        if (DirqlIsrRequired != FALSE) {

            //
            // Initialize the per-bank spinlock as the lock that is used for
            // synchronizing ISR invocation for that bank.
            //

            InterruptConfiguration.SpinLock = GpioBank->BankInterruptLock;
            Status = WdfInterruptCreate(GpioExtension->Device,
                                        &InterruptConfiguration,
                                        &Attributes,
                                        &GpioBank->BankInterrupt);

            //
            // If the interrupt object was successfully created, then initialize
            // the interrupt context with the bank information.
            //

            if (NT_SUCCESS(Status)) {
                InterruptContext =
                    GpioClxGetInterruptObjectContext(GpioBank->BankInterrupt);

                InterruptContext->Bank = GpioBank;

            } else {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_PNP,
                            "%s: WdfInterruptCreate() failed! Status = %#x\n",
                            __FUNCTION__,
                            Status);

                goto SetupInterruptsForBanksEnd;
            }
        }

        //
        // Create a passive-level interrupt handler if the GPIO is off-SoC.
        //

        if (GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) != FALSE) {
            Status = WdfInterruptCreate(GpioExtension->Device,
                                        &PassiveInterruptConfiguration,
                                        &Attributes,
                                        &GpioBank->BankPassiveInterrupt);

            //
            // If the interrupt object was successfully created, then
            // initialize the interrupt context with the bank information.
            //

            if (NT_SUCCESS(Status)) {
                InterruptContext = GpioClxGetInterruptObjectContext(
                                       GpioBank->BankPassiveInterrupt);

                InterruptContext->Bank = GpioBank;

            } else {
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_PNP,
                            "%s: WdfInterruptCreate() failed for passive "
                            "interrupt! Status = %#x\n",
                            __FUNCTION__,
                            Status);

                goto SetupInterruptsForBanksEnd;
            }
        }
    }

    //
    // The interrupt objects would be parented to the device and thus would
    // automatically be deleted by WDF on failure.
    //

SetupInterruptsForBanksEnd:
    return Status;
}

_Must_inspect_result_
PGPIO_BANK_ENTRY
GpiopGetBankEntry (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BANK_ID BankId
    )

/*++

Routine Description:

    This routine returns the GPIO bank entry given the bank ID. This routine
    assumes banks are always consecutive numbered from 0 -> (n - 1). If the
    order can be different, then routine would need to be updated.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    BankId - Supplies the ID for the GPIO bank.

Return Value:

    Pointer to the bank entry if successful. NULL otherwise.

--*/

{

    PGPIO_BANK_ENTRY GpioBank;

    GPIO_ASSERT(GpioExtension->Banks != NULL);

    if (BankId < GpioExtension->TotalBanks) {
        GpioBank = &GpioExtension->Banks[BankId];

    } else {
        GpioBank = NULL;
    }

    return GpioBank;
}

_Must_inspect_result_
PGPIO_BANK_ENTRY
GpiopGetBankEntryFromVirq (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Virq
    )

/*++

Routine Description:

    This routine translates the given VIRQ to the corresponding GPIO pin.

    N.B. This routine can be called from DIRQL.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Virq - Supplies the VIRQ for which the pin information should be obtained.

    BankEntry - Supplies a variable that receives the GPIO bank entry.

    BankNumber - Supplies a variable that receives the GPIO bank ID.

    PinNumber - Supplies a variable that receives the relative pin number.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    BANK_ID BankId;
    PGPIO_PIN_INFORMATION_ENTRY *DynamicPinTable;
    PGPIO_PIN_INFORMATION_ENTRY PinEntry;
    PGPIO_BANK_ENTRY GpioBank;
    PIN_NUMBER Index;

    GpioBank = NULL;
    Index = 0;
    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        DynamicPinTable = GPIO_READ_DYNAMIC_PIN_TABLE_POINTER(GpioBank);
        if (DynamicPinTable == NULL) {
            continue;
        }

        for (Index = 0; Index < GpioBank->PinCount; Index += 1) {
            PinEntry = GPIO_READ_PIN_ENTRY_POINTER(DynamicPinTable, Index);
            if (PinEntry == NULL) {
                continue;
            }

            if (PinEntry->Virq == Virq) {
                return GpioBank;
            }
        }
    }

    return NULL;
}

_Must_inspect_result_
PGPIO_PIN_INFORMATION_ENTRY
GpiopGetPinEntryFromVirq (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG Virq,
    __out_opt PGPIO_BANK_ENTRY *BankEntry,
    __out_opt PBANK_ID BankNumber,
    __out_opt PPIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine translates the given VIRQ to the corresponding GPIO pin.

    N.B. This routine can be called from DIRQL.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Virq - Supplies the VIRQ for which the pin information should be obtained.

    BankEntry - Supplies a variable that receives the GPIO bank entry.

    BankNumber - Supplies a variable that receives the GPIO bank ID.

    PinNumber - Supplies a variable that receives the relative pin number.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    BANK_ID BankId;
    PGPIO_PIN_INFORMATION_ENTRY *DynamicPinTable;
    PGPIO_BANK_ENTRY GpioBank;
    PIN_NUMBER Index;
    PGPIO_PIN_INFORMATION_ENTRY PinEntry;
    PGPIO_PIN_INFORMATION_ENTRY FoundPinEntry;

    GpioBank = NULL;
    Index = 0;
    FoundPinEntry = NULL;

    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        DynamicPinTable = GPIO_READ_DYNAMIC_PIN_TABLE_POINTER(GpioBank);
        if (DynamicPinTable == NULL) {
            continue;
        }

        for (Index = 0; Index < GpioBank->PinCount; Index += 1) {
            PinEntry = GPIO_READ_PIN_ENTRY_POINTER(DynamicPinTable, Index);
            if (PinEntry == NULL) {
                continue;
            }

            if (PinEntry->Virq == Virq) {
                FoundPinEntry = PinEntry;
                goto FoundMatch;
            }
        }
    }

FoundMatch:
    if (FoundPinEntry != NULL) {
        if (ARGUMENT_PRESENT(BankEntry) != FALSE) {
            *BankEntry = GpioBank;
        }

        if (ARGUMENT_PRESENT(BankNumber) != FALSE) {
            *BankNumber = BankId;
        }

        if (ARGUMENT_PRESENT(PinNumber) != FALSE) {
            *PinNumber = Index;
        }
    }

    return FoundPinEntry;
}

_Must_inspect_result_
static
PGPIO_PIN_INFORMATION_ENTRY
GpiopGetPinEntryInternal (
    _In_ PGPIO_BANK_ENTRY GpioBank,
    _In_ PIN_NUMBER RelativePin,
    _In_ BOOLEAN CanCreate
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.

    Callers should use the GpiopGetPinEntry() and GpiopCreatePinEntry()
    wrappers instead.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) for which the
        entry is to be retrieved.

    CanCreate - Supplies whether the routine should allocate the entry if it
        has not been allocated yet.  If TRUE, the routine must be called at
        PASSIVE_LEVEL.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/
{

    BOOLEAN AcquirePassiveLock;
    PGPIO_PIN_INFORMATION_ENTRY *DynamicPinTable;
    PGPIO_PIN_INFORMATION_ENTRY PinEntry;
    ULONG PinTableSize;

    AcquirePassiveLock = FALSE;
    PinEntry = NULL;

    if (CanCreate != FALSE) {

        //
        // PASSIVE_LEVEL allows allocation from the pool.
        //

        GPIO_ASSERT(KeGetCurrentIrql() == PASSIVE_LEVEL);

        //
        // This synchronizes concurrent attempts to allocate the dynamic pin
        // table and the pin itself.
        //

        if (GPIO_BANK_PASSIVE_LOCK_OWNER(GpioBank) == FALSE) {
            GPIO_ACQUIRE_BANK_LOCK(GpioBank);
            AcquirePassiveLock = TRUE;
        }
    }

    if (RelativePin >= GpioBank->PinCount) {
        goto GetPinEntryInternalEnd;
    }

    DynamicPinTable = GPIO_READ_DYNAMIC_PIN_TABLE_POINTER(GpioBank);
    if (DynamicPinTable == NULL) {
        if (CanCreate == FALSE) {
            goto GetPinEntryInternalEnd;
        }

        //
        // Build the pin information table (out of NP pool since it is accessed
        // at DIRQL) and initialize it.
        //

        PinTableSize = GpioBank->PinCount * sizeof(PGPIO_PIN_INFORMATION_ENTRY);
        DynamicPinTable = GPIO_ALLOCATE_POOL(NonPagedPoolNx, PinTableSize);
        if (DynamicPinTable == NULL) {
            TraceEvents(GpioBank->GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to allocate memory for pin table! "
                        "Size = %#x\n",
                        __FUNCTION__,
                        PinTableSize);

            goto GetPinEntryInternalEnd;
        }

        RtlZeroMemory(DynamicPinTable, PinTableSize);
        GPIO_WRITE_DYNAMIC_PIN_TABLE_POINTER(GpioBank, DynamicPinTable);
    }

    PinEntry = GPIO_READ_PIN_ENTRY_POINTER(DynamicPinTable, RelativePin);
    if (PinEntry == NULL) {
        if (CanCreate == FALSE) {
            goto GetPinEntryInternalEnd;
        }

        PinEntry = GPIO_ALLOCATE_POOL(NonPagedPoolNx,
                                      sizeof(GPIO_PIN_INFORMATION_ENTRY));

        if (PinEntry == NULL) {
            TraceEvents(GpioBank->GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to allocate memory for pin!\n",
                        __FUNCTION__);

            goto GetPinEntryInternalEnd;
        }

        RtlZeroMemory(PinEntry, sizeof(GPIO_PIN_INFORMATION_ENTRY));
        GPIO_WRITE_PIN_ENTRY_POINTER(DynamicPinTable, RelativePin, PinEntry);
    }

GetPinEntryInternalEnd:
    if (AcquirePassiveLock != FALSE) {
        GPIO_RELEASE_BANK_LOCK(GpioBank);
    }

    return PinEntry;
}

_Must_inspect_result_
PGPIO_PIN_INFORMATION_ENTRY
GpiopGetPinEntry (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PIN_NUMBER RelativePin
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.

    N.B. This routine can be called from within the ISR.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) for which the
        entry is to be retrieved.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    return GpiopGetPinEntryInternal(GpioBank, RelativePin, FALSE);
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
PGPIO_PIN_INFORMATION_ENTRY
GpiopCreatePinEntry (
    _In_ PGPIO_BANK_ENTRY GpioBank,
    _In_ PIN_NUMBER RelativePin
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.  If the entry has not been allocated yet, the routine will
    allocate it.

    N.B. This routine must be called at PASSIVE_LEVEL.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    RelativePin - Supplies the pin number (relative to the bank) for which the
        entry is to be retrieved.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    return GpiopGetPinEntryInternal(GpioBank, RelativePin, TRUE);
}

_Must_inspect_result_
static
PGPIO_PIN_INFORMATION_ENTRY
GpiopGetPinEntryFromPinNumberInternal (
    _In_ PDEVICE_EXTENSION GpioExtension,
    _In_ PIN_NUMBER PinNumber,
    _In_ BOOLEAN CanCreate
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.

    Callers should use the GpiopGetPinEntryFromPinNumberInternal() and
    GpiopCreatePinEntryFromPinNumber() wrappers instead.

    CanCreate - Supplies whether the routine should allocate the entry if it
        has not been allocated yet.  If TRUE, the routine must be called at
        PASSIVE_LEVEL.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    PinNumber - Supplies the pin number for which the entry is to be retrieved.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    BANK_ID BankId;
    PGPIO_BANK_ENTRY GpioBank;
    PGPIO_PIN_INFORMATION_ENTRY PinEntry;
    PIN_NUMBER RelativePin;

    if (PinNumber >= GpioExtension->TotalPins) {
        PinEntry = NULL;

    } else {
        BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension, PinNumber);
        RelativePin = GPIO_ABSOLUTE_TO_RELATIVE_PIN(GpioExtension, PinNumber);
        GpioBank = GpiopGetBankEntry(GpioExtension, BankId);
        if (GpioBank != NULL) {
            PinEntry = GpiopGetPinEntryInternal(GpioBank,
                                                RelativePin,
                                                CanCreate);

        } else {
            PinEntry = NULL;
        }
    }

    return PinEntry;
}

_Must_inspect_result_
PGPIO_PIN_INFORMATION_ENTRY
GpiopGetPinEntryFromPinNumber (
    _In_ PDEVICE_EXTENSION GpioExtension,
    _In_ PIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.

    N.B. This routine can be called from within the ISR.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    PinNumber - Supplies the pin number for which the entry is to be retrieved.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    return GpiopGetPinEntryFromPinNumberInternal(GpioExtension,
                                                 PinNumber,
                                                 FALSE);
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
PGPIO_PIN_INFORMATION_ENTRY
GpiopCreatePinEntryFromPinNumber (
    _In_ PDEVICE_EXTENSION GpioExtension,
    _In_ PIN_NUMBER PinNumber
    )

/*++

Routine Description:

    This routine returns the entry in the pin information table given the
    pin number.  If the entry has not been allocated yet, the routine will
    allocate it.

    N.B. This routine must be called at PASSIVE_LEVEL.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    PinNumber - Supplies the pin number for which the entry is to be retrieved.

Return Value:

    Pointer to the pin entry if successful. NULL otherwise.

--*/

{

    return GpiopGetPinEntryFromPinNumberInternal(GpioExtension,
                                                 PinNumber,
                                                 TRUE);
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDeleteDeviceExtension (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine frees the specified device extension. This routine is called
    when the controller is being removed in the context of IRP_MN_REMOVE_DEVICE
    when the framework deletes the device object.

    This does not clobber GpioExtension->IsInternalClient, CombinedPacket, and
    Device, since they're used by GpioClnInvokeControllerCleanupCallback.

    N.B. Since all the debounce timers are parented to the device object, they
         would be automatically deleted by WDF.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{

    BANK_ID BankId;
    PGPIO_PIN_INFORMATION_ENTRY *DynamicPinTable;
    PGPIO_BANK_ENTRY GpioBank;
    PGPIO_PIN_INFORMATION_ENTRY PinInformation;
    ULONG Index;

    PAGED_CODE();

    GPIO_ASSERT(GPIO_IS_HUB_DEVICE(GpioExtension->Device) == FALSE);

    //
    // Delete the self IO target.
    //

    if (GpioExtension->SelfIoTarget != NULL) {
        WdfIoTargetClose(GpioExtension->SelfIoTarget);
        WdfObjectDelete(GpioExtension->SelfIoTarget);
        GpioExtension->SelfIoTarget = NULL;
    }

    //
    // Delete the resource hub target.
    //

    if (GpioExtension->ResourceHubTarget != NULL) {
        WdfIoTargetClose(GpioExtension->ResourceHubTarget);
        WdfObjectDelete(GpioExtension->ResourceHubTarget);
        GpioExtension->ResourceHubTarget = NULL;
    }

    //
    // Free the NT device name and the BIOS device name buffers.
    //

    if (GpioExtension->TargetName.Buffer != NULL) {
        GPIO_FREE_POOL(GpioExtension->TargetName.Buffer);
        RtlZeroMemory(&GpioExtension->TargetName,
                      sizeof(GpioExtension->TargetName));
    }

    if (GpioExtension->BiosName.Buffer != NULL) {
        GPIO_FREE_POOL(GpioExtension->BiosName.Buffer);
        RtlZeroMemory(&GpioExtension->BiosName,
                      sizeof(GpioExtension->BiosName));
    }

    //
    // Delete ACPI event related data.
    //

    GpiopManageAcpiEventing(GpioExtension, AcpiEventStateDestroy, TRUE);

    //
    // Delete interrupt-related data. The DPC and workitem are parented to the
    // device and will automatically be deleted by the framework. So there is
    // nothing to do here.
    //

    //
    // Delete per-bank data including pin information table.
    //

    if (GpioExtension->Banks != NULL) {
        for(BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
            GpioBank = &GpioExtension->Banks[BankId];
            DynamicPinTable = GPIO_READ_DYNAMIC_PIN_TABLE_POINTER(GpioBank);
            if (DynamicPinTable != NULL) {
                for (Index = 0; Index < GpioBank->PinCount; Index += 1) {
                    PinInformation = GPIO_READ_PIN_ENTRY_POINTER(DynamicPinTable, Index);
                    if (PinInformation != NULL) {
                        GPIO_FREE_POOL(PinInformation);
                    }
                }

                GPIO_FREE_POOL(DynamicPinTable);
            }
        }

        GPIO_FREE_POOL(GpioExtension->Banks);
    }

#if defined(EVENT_TRACING)

    //
    // Delete the in-flight recorder log.
    //

    if ((GpioExtension->LogHandle != NULL) &&
        (GpioExtension->LogHandle != GpioLogHandle)) {

        WppRecorderLogDelete(GpioExtension->LogHandle);
    }

#endif

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFormatRequestForRead (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in ULONG64 PinValues,
    __out_bcount((PinCount + 7) / 8) PUCHAR Buffer
    )

/*++

Routine Description:

    This routine formats the request for read operation. It processes the
    data read from the client driver and places them at the proper locations in
    the user buffer.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence (or the deep-idle sequence).

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    PinValues - Supplies the values for each of the pins.

    Buffer - Supplies a buffer that will receive the read values. This routine
        assumes the buffer has been checked to be of appropriate size.

Return Value:

    None.

--*/

{

    CONTROLLER_ATTRIBUTE_FLAGS Flags;
    ULONG Index;
    ULONG64 Value;

    //
    // If the client driver did not request the IO requests to be formatted,
    // then the buffer already has the correct data. Otherwise, format the
    // buffer with the data returned by client driver.
    //
    // Walk through all the bits in the client driver returned data and set
    // the bits in the supplied buffer as appropriate.
    //

    Flags = GpioBank->GpioExtension->ClientInformation.Flags;
    if (Flags.FormatIoRequestsAsMasks != 0) {
        RtlZeroMemory(Buffer, ALIGN_RANGE_UP(PinCount, 8) / 8);
        for (Index = 0; Index < PinCount; Index += 1) {
            Value  = PinValues & ((ULONG64)1 << PinNumberTable[Index]);
            if (Value != 0) {
                Buffer[Index / 8] |= (1 << (Index % 8));
            }
        }
    }

    return;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopFormatRequestForWrite (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_ecount(PinCount) PPIN_NUMBER PinNumberTable,
    __in ULONG PinCount,
    __in_bcount((PinCount + 7) / 8) PUCHAR Buffer,
    __out PULONG64 SetMaskOutput,
    __out PULONG64 ClearMaskOutput
    )

/*++

Routine Description:

    This routine formats the request for write operation. It processes the
    data in the specified buffer and places them at the proper locations in
    the bitmask to be given to the client driver.

    N.B. This routine is called at PASSIVE_LEVEL but is not marked as
         PAGED_CODE as it could be executed late in the hibernate or
         early in resume sequence (or the deep-idle sequence).

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank.

    PinNumberTable - Supplies an array containing the pins to be connected.

    PinCount - Supplies a count of entries in the pin number table.

    Buffer - Supplies a buffer that contains values to be written. This routine
        assumes the buffer has been checked to be of appropriate size.

    SetMaskOutput - Supplies a pointer that receives the bitmask of pins to
        be set.

    ClearMaskOutput - Supplies a pointer that receives the bitmask of pins to
        be cleared.

Return Value:

    None.

--*/

{

    ULONG64 ClearMask;
    CONTROLLER_ATTRIBUTE_FLAGS Flags;
    ULONG Index;
    ULONG64 SetMask;
    UCHAR Value;

    ClearMask = 0;
    SetMask = 0;

    //
    // Walk through all the bits in the buffer and set the bits in either
    // the set or clear bitmask as appropriate.
    //

    Flags = GpioBank->GpioExtension->ClientInformation.Flags;
    if (Flags.FormatIoRequestsAsMasks != 0) {
        for (Index = 0; Index < PinCount; Index += 1) {
            Value = Buffer[Index / 8] & (1 << (Index % 8));
            if (Value != 0) {
                SetMask |= (ULONG64)1 << PinNumberTable[Index];

            } else {
                ClearMask |= (ULONG64)1 << PinNumberTable[Index];
            }
        }
    }

    *SetMaskOutput = SetMask;
    *ClearMaskOutput = ClearMask;
    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryBankIdFromGsivOrConnectionId (
    __in PDEVICE_EXTENSION GpioExtension,
    __in BOOLEAN QueryOnGsiv,
    __in ULONG64 GsivOrConnectionId,
    __out PGPIO_BANK_ENTRY *GpioBankOut,
    __out_opt PUCHAR IoRestriction,
    __out_opt PBOOLEAN FunctionConfig
    )

/*++

Routine Description:

    This routine queries the BIOS descriptor corresponding to the supplied
    GSIV or connection ID and determines the GPIO bank the request is
    associated with.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    QueryOnGsiv - Supplies whether the search should be done on GSIV or
        the Connection ID.

    GsivOrConnectionId - Supplies the GSIV or 64-bit Connection ID value
        depending on the search criterion.

    GpioBankOut - Supplies a variable that receives the GPIO bank entry.

    IoRestriction - Supplies an optional pointer that receives the IO
        restriction value from the BIOS descriptor. This value is only
        valid if the descriptor is for GPIO I/O operation.

    FunctionConfig - Supplies an optional PBOOLEAN that receives
        whether the queried resource descriptor is an function config
        descriptor.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PUCHAR BiosDescriptor;
    PPNP_FUNCTION_CONFIG_DESCRIPTOR FunctionConfigDescriptor;
    PGPIO_BANK_ENTRY GpioBank;
    PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR GpioDescriptor;
    ULONG MinimumLength;
    USHORT PinCount;
    PIN_NUMBER PinNumber;
    ULONG PinTableOffset;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER RhBuffer;
    NTSTATUS Status;
    USHORT Value;

    PAGED_CODE();

    RhBuffer = NULL;

    //
    // Query the Resource Hub to fetch the BIOS descriptor associated with this
    // request.
    //

    Status = GpiopQueryResourceHubBiosDescriptor(
                 GpioExtension->Device,
                 &GpioExtension->ResourceHubTarget,
                 QueryOnGsiv,
                 GsivOrConnectionId,
                 &RhBuffer);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: Failed to query BIOS descriptor from the Resource Hub!"
                    " ID = [0x%I64x], Status = %#x\n",
                    __FUNCTION__,
                    GsivOrConnectionId,
                    Status);

        goto QueryBankIdFromGsivOrConnectionIdEnd;
    }

    BiosDescriptor = (PUCHAR)RhBuffer->ConnectionProperties;

    //
    // The descriptor tag is in the same location for all descriptor types.
    // Check it first before proceeding to descriptor-type specific processing.
    //
    
    FunctionConfigDescriptor = (PPNP_FUNCTION_CONFIG_DESCRIPTOR)BiosDescriptor;

    if (FunctionConfigDescriptor->Tag == FUNCTION_CONFIG_DESCRIPTOR) {
        MinimumLength = sizeof(PNP_FUNCTION_CONFIG_DESCRIPTOR);
        if (RhBuffer->PropertiesLength < MinimumLength) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: BIOS function config descriptor length is"
                        " invalid! Extn = 0x%p, Length = %d, Reqd = %d\n",
                        __FUNCTION__,
                        GpioExtension,
                        RhBuffer->PropertiesLength,
                        (ULONG)MinimumLength);

            Status = STATUS_UNSUCCESSFUL;
            goto QueryBankIdFromGsivOrConnectionIdEnd;
        }

        Value = FunctionConfigDescriptor->ResourceSourceOffset;

        NT_ASSERT(Value >= FunctionConfigDescriptor->PinTableOffset);

        PinCount = 
            (Value - FunctionConfigDescriptor->PinTableOffset) / sizeof(USHORT);

        PinTableOffset = FunctionConfigDescriptor->PinTableOffset;
        if (PinCount == 0) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_FUNCCONFIG,
                        "%s: BIOS function config descriptor pin count is"
                        " zero! Extn = 0x%p\n",
                        __FUNCTION__,
                        GpioExtension);

            Status = STATUS_UNSUCCESSFUL;
            goto QueryBankIdFromGsivOrConnectionIdEnd;
        }

        if (ARGUMENT_PRESENT(IoRestriction) != FALSE) {
            *IoRestriction = (UCHAR)GPIO_IORESTRICTION_NONE;
        }

        if (ARGUMENT_PRESENT(FunctionConfig) != FALSE) {
            *FunctionConfig = TRUE;
        }

    } else {
        MinimumLength = sizeof(PNP_GPIO_INTERRUPT_IO_DESCRIPTOR);
        
        if (RhBuffer->PropertiesLength < MinimumLength) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: BIOS GPIO descriptor length is invalid! "
                        "Extn = 0x%p, Length = %d, Reqd = %d\n",
                        __FUNCTION__,
                        GpioExtension,
                        RhBuffer->PropertiesLength,
                        (ULONG)MinimumLength);

            Status = STATUS_UNSUCCESSFUL;
            goto QueryBankIdFromGsivOrConnectionIdEnd;
        }

        GpioDescriptor = (PPNP_GPIO_INTERRUPT_IO_DESCRIPTOR)BiosDescriptor;
        Value = GpioDescriptor->ResourceSourceOffset;

        NT_ASSERT(Value >= GpioDescriptor->PinTableOffset);

        PinCount = (Value - GpioDescriptor->PinTableOffset)/sizeof(USHORT);
        PinTableOffset = GpioDescriptor->PinTableOffset;
        if (PinCount == 0) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INTERRUPT,
                        "%s: BIOS GPIO descriptor pin count is zero!"
                        " Extn = 0x%p\n",
                        __FUNCTION__,
                        GpioExtension);

            Status = STATUS_UNSUCCESSFUL;
            goto QueryBankIdFromGsivOrConnectionIdEnd;
        }

        if (ARGUMENT_PRESENT(IoRestriction) != FALSE) {
            if (GpioDescriptor->DescriptorType == PNP_GPIO_IRQ_DESCRIPTOR_TYPE_IO) {
                *IoRestriction =
                    GPIO_BIOS_DESCRIPTOR_IO_RESTRICTION(GpioDescriptor);

            } else {
                *IoRestriction = (UCHAR)GPIO_IORESTRICTION_NONE;
            }
        }

        if (ARGUMENT_PRESENT(FunctionConfig) != FALSE) {
            *FunctionConfig = FALSE;
        }
    }

    PinNumber = *((PUSHORT)Add2Ptr(BiosDescriptor, PinTableOffset));
    BankId = GPIO_BANK_ID_FROM_PIN_NUMBER(GpioExtension, PinNumber);
    GpioBank = GpiopGetBankEntry(GpioExtension, BankId);
    if (GpioBank == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INTERRUPT,
                    "%s: No bank associated with specified pins! Pin number "
                    "is invalid! PinNumber = %#x, Bank ID = 0x%x\n",
                    __FUNCTION__,
                    PinNumber,
                    BankId);

        Status = STATUS_INVALID_PARAMETER;
        goto QueryBankIdFromGsivOrConnectionIdEnd;
    }

    *GpioBankOut = GpioBank;

QueryBankIdFromGsivOrConnectionIdEnd:
    if (RhBuffer != NULL) {
        GPIO_FREE_POOL(RhBuffer);
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryEmulatedActiveBothInitialPolarity (
    __in WDFDEVICE Device,
    __inout __deref WDFIOTARGET *ResourceHubTarget,
    __in ULONG Virq,
    __out PKINTERRUPT_POLARITY InitialPolarity
    )

/*++

Routine Description:

    This function queries the Resource Hub for the initial polarity for the
    specified emulated active both interrupt.

Arguments:

    Device - Supplies a handle to the framework device object.

    Virq - Supplies the VIRQ for interrupt line that should be queried.

    InitialPolarity - Supplies a pointer to variable that receives the asserted
        level for the specified interrupt.

Return Value:

    NTSTATUS code.

--*/

{
    ULONG_PTR BytesReturned;
    RH_QUERY_ACTIVE_BOTH_INITIAL_POLARITY_INPUT_BUFFER InputBuffer;
    WDF_MEMORY_DESCRIPTOR InputMemoryDescriptor;
    RH_QUERY_ACTIVE_BOTH_INITIAL_POLARITY_OUTPUT_BUFFER OutputBuffer;
    WDF_MEMORY_DESCRIPTOR OutputMemoryDescriptor;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Open a handle to the resource hub if it has not been opened already.
    //

    if (*ResourceHubTarget == NULL) {
        Status = GpiopOpenResourceHubTarget(Device, ResourceHubTarget);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioLogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to open resource hub target! "
                        "Status = %#x\n",
                        __FUNCTION__,
                        Status);

            goto QueryEmulatedActiveBothInitialPolarityEnd;
        }
    }

    GPIO_ASSERT(*ResourceHubTarget != NULL);

    //
    // Set up a WDF memory descriptor for the input and output parameters.
    //

    RtlZeroMemory(&InputBuffer, sizeof(InputBuffer));
    InputBuffer.Version = RH_QUERY_ACTIVE_BOTH_INITIAL_POLARITY_INPUT_VERSION;
    InputBuffer.InterruptVector = Virq;

    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&InputMemoryDescriptor,
                                      &InputBuffer,
                                      sizeof(InputBuffer));

    RtlZeroMemory(&OutputBuffer, sizeof(OutputBuffer));
    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&OutputMemoryDescriptor,
                                      &OutputBuffer,
                                      sizeof(OutputBuffer));

    //
    // Send the request to update the connection.
    //

    Status = WdfIoTargetSendIoctlSynchronously(
                *ResourceHubTarget,
                NULL,
                IOCTL_RH_QUERY_ACTIVE_BOTH_INITIAL_POLARITY,
                &InputMemoryDescriptor,
                &OutputMemoryDescriptor,
                NULL,
                &BytesReturned);

    if (!NT_SUCCESS(Status)) {
        goto QueryEmulatedActiveBothInitialPolarityEnd;
    }

    if (OutputBuffer.InitialPolarity == PNP_GPIO_IRQ_POLARITY_LOW) {
        *InitialPolarity = InterruptActiveLow;

    } else if (OutputBuffer.InitialPolarity == PNP_GPIO_IRQ_POLARITY_HIGH) {
        *InitialPolarity = InterruptActiveHigh;

    } else {
        *InitialPolarity = InterruptPolarityUnknown;
    }

QueryEmulatedActiveBothInitialPolarityEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopUpdateEmulatedActiveBothInitialPolarity (
    __in WDFDEVICE Device,
    __inout __deref WDFIOTARGET *ResourceHubTarget,
    __in ULONG Virq,
    __in KINTERRUPT_POLARITY InitialPolarity
    )

/*++

Routine Description:

    This function uses the Resource Hub to update the initial polarity for the
    specified emulated active both interrupt.

Arguments:

    Device - Supplies a handle to the framework device object.

    Virq - Supplies the

    InitialPolarity - Supplies the new initial polarity for the specified
        interrupt.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG_PTR BytesReturned;
    RH_UPDATE_ACTIVE_BOTH_INITIAL_POLARITY_INPUT_BUFFER InputBuffer;
    WDF_MEMORY_DESCRIPTOR InputMemoryDescriptor;
    RH_UPDATE_ACTIVE_BOTH_INITIAL_POLARITY_OUTPUT_BUFFER OutputBuffer;
    WDF_MEMORY_DESCRIPTOR OutputMemoryDescriptor;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Open a handle to the resource hub if it has not been opened already.
    //

    if (*ResourceHubTarget == NULL) {
        Status = GpiopOpenResourceHubTarget(Device, ResourceHubTarget);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioLogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to open resource hub target! "
                        "Status = %#x\n",
                        __FUNCTION__,
                        Status);

            goto UpdateEmulatedActiveBothInitialPolarityEnd;
        }
    }

    GPIO_ASSERT(*ResourceHubTarget != NULL);

    //
    // Set up a WDF memory descriptor for the input and output parameters.
    //

    RtlZeroMemory(&InputBuffer, sizeof(InputBuffer));
    InputBuffer.Version = RH_UPDATE_ACTIVE_BOTH_INITIAL_POLARITY_INPUT_VERSION;
    InputBuffer.InterruptVector = Virq;
    if (InitialPolarity == InterruptActiveLow) {
        InputBuffer.InitialPolarity = FALSE;

    } else if (InitialPolarity == InterruptActiveHigh) {
        InputBuffer.InitialPolarity = TRUE;
    }

    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&InputMemoryDescriptor,
                                      &InputBuffer,
                                      sizeof(InputBuffer));

    RtlZeroMemory(&OutputBuffer, sizeof(OutputBuffer));
    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&OutputMemoryDescriptor,
                                      &OutputBuffer,
                                      sizeof(OutputBuffer));


    //
    // Send the request to update the connection.
    //

    Status = WdfIoTargetSendIoctlSynchronously(
                *ResourceHubTarget,
                NULL,
                IOCTL_RH_UPDATE_ACTIVE_BOTH_INITIAL_POLARITY,
                &InputMemoryDescriptor,
                &OutputMemoryDescriptor,
                NULL,
                &BytesReturned);

UpdateEmulatedActiveBothInitialPolarityEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopQueryResourceHubBiosDescriptor (
    __in WDFDEVICE Device,
    __inout __deref WDFIOTARGET *ResourceHubTarget,
    __in BOOLEAN QueryOnGsiv,
    __in ULONG64 GsivOrConnectionId,
    __out __deref __drv_allocatesMem(Mem)
        PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER *BiosDescriptorOut
    )

/*++

Routine Description:

    This routine queries the BIOS descriptor corresponding to the supplied
    GSIV or (Controller ID, Perhiperal ID, Connection ID) tuple from the
    Resource Hub. If the caller does not specify a resource hub handle, then
    this routine will optionally open one on behalf of the caller.

Arguments:

    Device - Supplies a handle to the framework device object.

    ResourceHubTarget - Supplies a pointer to a variable that contains the
        resource hub handle. If the target pointer is NULL, then this routine
        will store the resource hub handle value into this variable on output.
        It is the responsiblity of the device to delete this handle when the
        device goes away.

    QueryOnGsiv - Supplies whether the search should be done on GSIV or
        the Connection ID.

    GsivOrConnectionId - Supplies the GSIV or 64-bit Connection ID value
        depending on the search criterion.

    BiosDescriptorOut - Supplies a pointer that receives the buffer containing
        the BIOS descriptor. The caller is responsible for freeing the buffer.

Return Value:

    NTSTATUS code.

--*/

{

    //
    // Assume the device is defined a maximum of 12 levels deep and each level
    // needs 5 characters including the separator ("ABCD.")
    //

#define INITIAL_ACPI_STRING_SIZE (64)

    ULONG_PTR BytesReturned;
    PRH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER DescriptorBuffer;
    RH_QUERY_CONNECTION_PROPERTIES_INPUT_BUFFER InputBuffer;
    WDF_MEMORY_DESCRIPTOR InputMemoryDescriptor;
    ULONG Length;
    WDF_MEMORY_DESCRIPTOR OutputMemoryDescriptor;
    NTSTATUS Status;

    PAGED_CODE();

    DescriptorBuffer = NULL;

    //
    // Open a handle to the resource hub if it has not been opened already.
    //

    if (*ResourceHubTarget == NULL) {
        Status = GpiopOpenResourceHubTarget(Device, ResourceHubTarget);
        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioLogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to open resource hub target! "
                        "Status = %#x\n",
                        __FUNCTION__,
                        Status);

            goto QueryResourceHubBiosDescriptorEnd;
        }
    }

    GPIO_ASSERT(*ResourceHubTarget != NULL);

    //
    // Set up a WDF memory descriptor for the input and output parameters.
    //

    RtlZeroMemory(&InputBuffer, sizeof(InputBuffer));
    InputBuffer.Version = RH_QUERY_CONNECTION_PROPERTIES_INPUT_VERSION;
    Length =
        FIELD_OFFSET(RH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER,
                          ConnectionProperties) +
        sizeof(PNP_GPIO_INTERRUPT_IO_DESCRIPTOR) +
        INITIAL_ACPI_STRING_SIZE;

    if (QueryOnGsiv != FALSE) {
        InputBuffer.QueryType = InterruptVectorType;
        InputBuffer.u.InterruptVector = (ULONG)GsivOrConnectionId;

    } else {
        InputBuffer.QueryType = ConnectionIdType;

        NT_ASSERT(sizeof(InputBuffer.u.ConnectionId) == sizeof(GsivOrConnectionId));

        RtlCopyMemory(&InputBuffer.u.ConnectionId,
                      &GsivOrConnectionId,
                      sizeof(InputBuffer.u.ConnectionId));
    }

    WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&InputMemoryDescriptor,
                                      &InputBuffer,
                                      sizeof(InputBuffer));

    //
    // Query the BIOS descriptor from the Resource hub.
    //

    do {

        DescriptorBuffer = GPIO_ALLOCATE_POOL(PagedPool, Length);
        if (DescriptorBuffer == NULL) {
            Status = STATUS_INSUFFICIENT_RESOURCES;
            goto QueryResourceHubBiosDescriptorEnd;
        }

        WDF_MEMORY_DESCRIPTOR_INIT_BUFFER(&OutputMemoryDescriptor,
                                          DescriptorBuffer,
                                          Length);

        //
        // Send the request to query the size of the BIOS descriptor.
        //

        Status = WdfIoTargetSendIoctlSynchronously(
                     *ResourceHubTarget,
                     NULL,
                     IOCTL_RH_QUERY_CONNECTION_PROPERTIES,
                     &InputMemoryDescriptor,
                     &OutputMemoryDescriptor,
                     NULL,
                     &BytesReturned);

        if (Status == STATUS_BUFFER_TOO_SMALL) {
            Length = DescriptorBuffer->PropertiesLength +
                     FIELD_OFFSET(RH_QUERY_CONNECTION_PROPERTIES_OUTPUT_BUFFER,
                                  ConnectionProperties);

            GPIO_FREE_POOL(DescriptorBuffer);
        }

    } while (Status == STATUS_BUFFER_TOO_SMALL);

QueryResourceHubBiosDescriptorEnd:
    if (!NT_SUCCESS(Status)) {
        if (DescriptorBuffer != NULL) {
            GPIO_FREE_POOL(DescriptorBuffer);
        }

    } else {
        *BiosDescriptorOut = DescriptorBuffer;
    }

    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopResolveInterruptBinding (
    __in PDEVICE_EXTENSION GpioExtension,
    __in ULONG InterruptCount,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw
   )

/*++

Routine Description:

    This routine determines the association between banks and resources
    supplied to the devices. A controller may have:

    1. No interrupt resources - possible if it only supports IO or is the
       virtual root GPIO controller tree (test environment).

    2. Only one interrupt resource, in which case, all banks share the same
       interrupt line.

    3. Interrupt resources >= Number of banks,
       A. No interrupt binding callback specified by the client, then the
          interrupt resources are described in the order of the banks.

       B. An interrupt binding callback is specified, in which case, the client
          driver will resolve the mapping.

    4. Interrupt resources < Number of banks. An interrupt binding
       callback must be specified. The client driver will resolve the mapping.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    InterruptCount - Supplies a pointer that receives a count of the
        interrupt resources.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

Return Value:

    STATUS_SUCCESS always.

--*/

{

    BANK_ID BankIndex;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR Descriptor;
    PGPIO_BANK_ENTRY GpioBank;
    ULONG Index;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR InterruptDescriptor;
    PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT BindingOutput;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR RawInterruptDescriptor;
    ULONG ResourceIndex;
    PULONG ResourceMapping;
    BOOLEAN Result;
    ULONG Size;
    NTSTATUS Status;
    ULONG TotalCount;

    UNREFERENCED_PARAMETER(ResourcesRaw);

    PAGED_CODE();

    //
    // If there are no interrupt resources, then bail out.
    //

    Status = STATUS_SUCCESS;
    if (InterruptCount == 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_INIT,
                    "%s: No interrupt resources described. Bindings skipped!\n",
                    __FUNCTION__);

        goto ResolveInterruptBindingEnd;

    } else if (InterruptCount == 1) {

        InterruptDescriptor = NULL;
        RawInterruptDescriptor = NULL;
        TotalCount = WdfCmResourceListGetCount(ResourcesTranslated);
        for (Index = 0; Index < TotalCount; Index += 1) {
            Descriptor = WdfCmResourceListGetDescriptor(ResourcesTranslated,
                                                        Index);

            if (Descriptor->Type == CmResourceTypeInterrupt) {
                InterruptDescriptor = Descriptor;
                RawInterruptDescriptor =
                    WdfCmResourceListGetDescriptor(ResourcesRaw, Index);

                break;
            }
        }

        GPIO_ASSERT((InterruptDescriptor != NULL) &&
                    (RawInterruptDescriptor != NULL));

        for (BankIndex = 0;
             BankIndex < GpioExtension->TotalBanks;
             BankIndex += 1) {

            GpioBank = &GpioExtension->Banks[BankIndex];
            GpioBank->ResourceIndex = Index;
            GpiopSetInterruptInformationFromResources(GpioBank,
                                                      InterruptDescriptor,
                                                      RawInterruptDescriptor);
        }

    } else {

        //
        // Allocate a buffer to store banks <--> resource index mapping.
        //

        Size = FIELD_OFFSET(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT,
                            BankInterruptBinding.ResourceMapping);

        Size += GpioExtension->TotalBanks * sizeof(ULONG);
        BindingOutput = GPIO_ALLOCATE_POOL_EX(PagedPool,
                                              Size,
                                              GPIO_CLX_INTERRUPT_POOL_TAG);

        if (BindingOutput == NULL) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to allocate memory for mapping indexes!\n",
                        __FUNCTION__);

            Status = STATUS_INSUFFICIENT_RESOURCES;
            goto ResolveInterruptBindingEnd;
        }

        //
        // Call into the client if it supports query interrupt binding callback
        // to resolve the interrupts. This is to ensure the client driver
        // is always given precedence in resolving interrupt binding.
        //

        memset(BindingOutput, 0xFF, Size);
        BindingOutput->Version = 0;
        BindingOutput->Size = (USHORT)Size;
        Status = GpioClnInvokeQueryInterruptBinding(
                     GpioExtension,
                     ResourcesTranslated,
                     ResourcesRaw,
                     InterruptCount,
                     BindingOutput);

        if (NT_SUCCESS(Status)) {
            Result = GpiopValidateInterruptBindings(GpioExtension,
                                                    ResourcesTranslated,
                                                    ResourcesRaw,
                                                    BindingOutput,
                                                    Size);

            if (Result == FALSE) {
                Status = STATUS_UNSUCCESSFUL;
                TraceEvents(GpioExtension->LogHandle,
                            Error,
                            DBG_INIT,
                            "%s: Client driver returned invalid binding "
                            "information! Status = %#x\n",
                            __FUNCTION__,
                            Status);
            }

        } else {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Failed to query binding information from client! "
                        "Status = %#x\n",
                        __FUNCTION__,
                        Status);
        }

        //
        // Walk over each bank and setup the interrupt information (GSIV,
        // IRQL, affinity etc.
        //

        ResourceMapping =
            (PULONG)&BindingOutput->BankInterruptBinding.ResourceMapping;

        for (BankIndex = 0;
             NT_SUCCESS(Status) && (BankIndex < GpioExtension->TotalBanks);
             BankIndex += 1) {

            ResourceIndex = ResourceMapping[BankIndex];
            if (ResourceIndex != GPIO_BANK_INTERRUPT_BINDING_RESERVED_INDEX) {
                InterruptDescriptor = WdfCmResourceListGetDescriptor(
                                          ResourcesTranslated,
                                          ResourceIndex);

                GPIO_ASSERT(InterruptDescriptor->Type ==
                                CmResourceTypeInterrupt);

                RawInterruptDescriptor = WdfCmResourceListGetDescriptor(
                                             ResourcesRaw,
                                             ResourceIndex);

            } else {
                InterruptDescriptor = NULL;
                RawInterruptDescriptor = NULL;
            }

            GpioBank = &GpioExtension->Banks[BankIndex];
            GpioBank->ResourceIndex = ResourceIndex;
            GpiopSetInterruptInformationFromResources(GpioBank,
                                                      InterruptDescriptor,
                                                      RawInterruptDescriptor);
         }

         if (BindingOutput != NULL) {
            GPIO_FREE_POOL_EX(BindingOutput, GPIO_CLX_INTERRUPT_POOL_TAG);
         }
    }

ResolveInterruptBindingEnd:
    return Status;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopDetermineInterruptCountAndIrql (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw,
    __out PULONG InterruptCount,
    __out KIRQL *SynchronizationIrql,
    __out ULONG *BaseInterruptGsiv
   )

/*++

Routine Description:

    This routine determines a count of interrupt resources in the supplied
    resources and also the GPIO controller interrupt synchronization IRQL.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

    InterruptCount - Supplies a pointer that receives a count of the
        interrupt resources.

    SynchronizationIrql - Supplies a pointer that receives the GPIO controller
        interrupt synchronization IRQL.

    BaseInterruptGsiv - Supplies a pointer that receives the GSIV for the
        interrupt that correponds to the synchronization IRQL.

Return Value:

    None.

--*/

{

    ULONG BaseGsiv;
    ULONG Count;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR Descriptor;
    ULONG Index;
    KIRQL Irql;
    PCM_PARTIAL_RESOURCE_DESCRIPTOR RawInterruptDescriptor;
    ULONG TotalCount;

    UNREFERENCED_PARAMETER(GpioExtension);
    UNREFERENCED_PARAMETER(ResourcesRaw);

    PAGED_CODE();

    //
    // Walk through the translated resources list trying to find descriptors
    // describing the interrupt resources. It is valid for the interrupt
    // resource to be missing.
    //

    BaseGsiv = GPIO_UNSPECIFIED_INTERRUPT;
    Irql = DISPATCH_LEVEL;
    Count = 0;
    TotalCount = WdfCmResourceListGetCount(ResourcesTranslated);
    for (Index = 0; Index < TotalCount; Index += 1) {
        Descriptor = WdfCmResourceListGetDescriptor(ResourcesTranslated, Index);
        if (Descriptor->Type == CmResourceTypeInterrupt) {
            Count += 1;
            if ((KIRQL)Descriptor->u.Interrupt.Level > Irql) {
                Irql = (KIRQL)Descriptor->u.Interrupt.Level;

                RawInterruptDescriptor =
                    WdfCmResourceListGetDescriptor(ResourcesRaw, Index);

                BaseGsiv = RawInterruptDescriptor->u.Interrupt.Vector;
            }
        }
    }

    *InterruptCount = Count;
    *SynchronizationIrql = Irql;
    *BaseInterruptGsiv = BaseGsiv;
    return;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopInitializeBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank
   )

/*++

Routine Description:

    This routine initializes all the power-releated information for each
    bank.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank to be initialized.

Return Value:

    NT status code.

--*/

{

    CLIENT_QUERY_BANK_POWER_INFORMATION_OUTPUT BankInformation;
    PCLIENT_CONTROLLER_BASIC_INFORMATION ClientInformation;
    PPO_FX_COMPONENT_IDLE_STATE F1Parameters;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Query the max F-state supported by the bank. The query is only required
    // if the controller specified at least one of the banks supports
    // power-management in the basic information query.
    //

    RtlZeroMemory(&BankInformation, sizeof(BankInformation));
    ClientInformation = &GpioBank->GpioExtension->ClientInformation;
    Status = STATUS_SUCCESS;
    if (ClientInformation->Flags.BankIdlePowerMgmtSupported != FALSE) {
        Status = GpiopQueryBankPowerInformation(GpioBank, &BankInformation);
        if (!NT_SUCCESS(Status)) {
            goto InitializePowerInformationEnd;
        }
    }

    if (BankInformation.F1StateSupported != FALSE) {
        GPIO_ASSERT(
            GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioBank->GpioExtension) == FALSE);

        F1Parameters = &BankInformation.F1IdleStateParameters;


        if (F1Parameters->TransitionLatency == 0x0) {
            F1Parameters->TransitionLatency = GPIO_TRANSITION_LATENCY_F1;
        }

        if (F1Parameters->ResidencyRequirement == 0x0) {
            F1Parameters->ResidencyRequirement = GPIO_RESIDENCY_REQUIREMENT_F1;
        }

        //
        // If the F1 power state is supported, then ensure that the client
        // driver has specified valid values for latency and residency.
        //

        if ((F1Parameters->TransitionLatency == 0x0) ||
            (F1Parameters->ResidencyRequirement == 0x0)) {

            TraceEvents(GpioBank->GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%s: Transition or residency for F1 is zero! "
                        "Bank ID = 0x%x, Latency = 0x%I64x, "
                        "Residency = 0x%I64x\n",
                        __FUNCTION__,
                        GpioBank->BankId,
                        F1Parameters->TransitionLatency,
                        F1Parameters->ResidencyRequirement);

            goto InitializePowerInformationEnd;
        }

        GpioBank->PowerData.F1StateParameters = *F1Parameters;
        GpioBank->PowerData.MaximumFState = FStateF1;

    } else {
        GpioBank->PowerData.MaximumFState = FStateF0;
    }

InitializePowerInformationEnd:
    return Status;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetInterruptInformationFromResources (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in_opt PCM_PARTIAL_RESOURCE_DESCRIPTOR InterruptDescriptor,
    __in_opt PCM_PARTIAL_RESOURCE_DESCRIPTOR RawInterruptDescriptor
   )

/*++

Routine Description:

    This routine extracts the GPIO controller's interrupt information (like
    GSIV, mode, polarity, IRQL etc.) from the supplies raw and translated
    resource descriptors.

Arguments:

    GpioBank - Supplies a pointer to the GPIO bank to be initialized.

    InterruptDescriptor - Supplies a pointer to the framework translated
        interrupt descriptor. This value may be null.

    RawInterruptDescriptor - Supplies a pointer to the framework raw
        interrupt descriptor. This value may be null.

Return Value:

    NT status code.

--*/

{

    PGPIO_INTERRUPT_DATA InterruptData;

    PAGED_CODE();

    //
    // If the controller has an interrupt described in its resources, then set
    // the values per the interrupt descriptor. Pull the GSIV value from the
    // raw descriptor and IRQL and affinity values from the translated
    // descriptor.
    //
    // Otherwise, set the values to default. In this case, the interrupt
    // information should never be used.
    //

    InterruptData = &GpioBank->InterruptData;
    if (ARGUMENT_PRESENT(InterruptDescriptor) != FALSE) {

        GPIO_ASSERT(InterruptDescriptor->Type == CmResourceTypeInterrupt);

        InterruptData->Gsiv = RawInterruptDescriptor->u.Interrupt.Vector;
        InterruptData->Irql = (KIRQL)InterruptDescriptor->u.Interrupt.Level;

        //
        // As only one interrupt per-GPIO controller is supported currently,
        // the synchronization IRQL is the same as the interrupt IRQL level.
        //
        // In case of multiple interrupts, this should be the highest IRQL
        // assigned to the device.
        //

        InterruptData->SynchronizeIrql = InterruptData->Irql;

        //
        // Set group affinity and affinity mask.
        //

        RtlZeroMemory(&InterruptData->Affinity, sizeof(GROUP_AFFINITY));
        InterruptData->Affinity.Group = InterruptDescriptor->u.Interrupt.Group;
        InterruptData->Affinity.Mask =
            InterruptDescriptor->u.Interrupt.Affinity;

        if (InterruptDescriptor->Flags & CM_RESOURCE_INTERRUPT_LATCHED) {
            InterruptData->InterruptMode = Latched;

        } else {
            InterruptData->InterruptMode = LevelSensitive;
        }

        //
        // The partial descriptor does not supply the polarity information and
        // thus set it to unknown.
        //

        InterruptData->Polarity = InterruptPolarityUnknown;

    //
    // Otherwise set the interrupt data to default (error) values.
    //

    } else {
        InterruptData->Gsiv = GPIO_UNSPECIFIED_INTERRUPT;
        InterruptData->Irql = 0;
        InterruptData->Affinity.Group = ALL_PROCESSOR_GROUPS;
        InterruptData->Affinity.Mask = (KAFFINITY)-1;
        InterruptData->InterruptMode = LevelSensitive;
        InterruptData->Polarity = InterruptPolarityUnknown;
    }

    return STATUS_SUCCESS;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
NTSTATUS
GpiopSetupBankSecondaryIoQueue (
    __in PDEVICE_EXTENSION GpioExtension
)

/*++

Routine Description:

    This routine creates all the secondary I/O queue for each bank. The queue
    is set as non-power managed.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    BANK_ID BankId;
    PGPIO_BANK_ENTRY GpioBank;
    WDF_OBJECT_ATTRIBUTES IoQueueAttributes;
    WDF_IO_QUEUE_CONFIG QueueConfiguration;
    PGPIO_BANK_QUEUE_CONTEXT QueueContext;
    NTSTATUS Status;

    PAGED_CODE();

    //
    // Create the per-bank IO queues. Mark the queues as non-power managed
    // to prevent WDF from synchronizing these queues with device Dx power
    // state. The queues would be synchronized using the Fx state for the
    // component.
    //

    WDF_IO_QUEUE_CONFIG_INIT(&QueueConfiguration, WdfIoQueueDispatchSequential);
    QueueConfiguration.EvtIoDefault = GpioClxBankEvtProcessIoDefault;
    QueueConfiguration.EvtIoDeviceControl = GpioClxBankEvtProcessDeviceIoControl;
    QueueConfiguration.EvtIoInternalDeviceControl = GpioClxBankEvtProcessInternalIoControl;
    QueueConfiguration.EvtIoCanceledOnQueue = GpioClxBankEvtProcessIoCanceledOnQueue;
    QueueConfiguration.PowerManaged = WdfFalse;

    //
    // Initialize the attributes of the queue. Set the execution level to
    // passive to force all IRPs to be sent to the driver at passive IRQL.
    //

    WDF_OBJECT_ATTRIBUTES_INIT(&IoQueueAttributes);
    WDF_OBJECT_ATTRIBUTES_SET_CONTEXT_TYPE(&IoQueueAttributes,
                                           GPIO_BANK_QUEUE_CONTEXT);

    IoQueueAttributes.SynchronizationScope = WdfSynchronizationScopeQueue;
    IoQueueAttributes.ExecutionLevel = WdfExecutionLevelPassive;
    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        Status = WdfIoQueueCreate(GpioExtension->Device,
                                  &QueueConfiguration,
                                  &IoQueueAttributes,
                                  &GpioBank->IoQueue);

        if (!NT_SUCCESS(Status)) {
            TraceEvents(GpioExtension->LogHandle,
                        Error,
                        DBG_INIT,
                        "%!FUNC! - WdfIoQueueCreate() failed! Status = %#x "
                        "Bank ID = 0x%x\n",
                        Status,
                        BankId);

            goto SetupBankSecondaryIoQueueEnd;
        }

        //
        // Store the bank pointer in the queue extension.
        //

        QueueContext = GpiopBankQueueGetContext(GpioBank->IoQueue);
        RtlZeroMemory(QueueContext, sizeof(GPIO_BANK_QUEUE_CONTEXT));
        QueueContext->Bank = GpioBank;
    }

    Status = STATUS_SUCCESS;

SetupBankSecondaryIoQueueEnd:
    return Status;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateClientBankPowerInformation (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT Information
    )

/*++

Routine Description:

    This routine validates the bank information supplied by the client
    driver.

Arguments:

    Information - Supplies a pointer to a buffer containing the bank
        information supplied by the client driver.

Return Value:

    TRUE if the attribute information is valid. FALSE otherwise.

--*/

{

    USHORT MinimumSize;
    BOOLEAN Valid;
    USHORT Version;

    PAGED_CODE();

    Version = Information->Version;
    if (Version < GPIO_BANK_POWER_INFORMATION_OUTPUT_VERSION_MIN) {
        TraceEvents(GpioBank->GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver bank information version is not "
                    "supported! Actual = %d, Min = %d\n",
                    __FUNCTION__,
                    Information->Version,
                    GPIO_BANK_POWER_INFORMATION_OUTPUT_VERSION_MIN);

        Valid = FALSE;
        goto ValidateClientBankInformationEnd;
    }

    MinimumSize =
        FIELD_OFFSET_AND_SIZE(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT,
                              BankPowerInformation);

    if (Information->Size < MinimumSize) {
        TraceEvents(GpioBank->GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver bank information buffer size is not "
                    "valid! Actual = %d, Min required = %d\n",
                    __FUNCTION__,
                    Information->Size,
                    MinimumSize);

        Valid = FALSE;
        goto ValidateClientBankInformationEnd;
    }

    Valid = TRUE;

ValidateClientBankInformationEnd:
    return Valid;
}

__checkReturn
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateClientControllerBasicInformation (
    __in PDEVICE_EXTENSION GpioExtension,
    __in PCLIENT_CONTROLLER_BASIC_INFORMATION Information
    )

/*++

Routine Description:

    This routine validates the attribute information supplied by the client
    driver.

Arguments:

    Information - Supplies a pointer to a buffer containing the
        information supplied by the client driver.

Return Value:

    TRUE if the attribute information is valid. FALSE otherwise.

--*/

{

    USHORT MinimumSize;
    BOOLEAN Valid;
    USHORT Version;

    PAGED_CODE();

    Version = Information->Version;
    if (Version < GPIO_CONTROLLER_BASIC_INFORMATION_VERSION_MIN) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver attributes version not supported! "
                    "Actual = %d, Min = %d\n",
                    __FUNCTION__,
                    Information->Version,
                    GPIO_CONTROLLER_BASIC_INFORMATION_VERSION_MIN);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    MinimumSize =
        FIELD_OFFSET_AND_SIZE(CLIENT_CONTROLLER_BASIC_INFORMATION, Flags);

    if (Information->Size < MinimumSize) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver information buffer size not valid! "
                    "Actual = %d, Min required = %d\n",
                    __FUNCTION__,
                    Information->Size,
                    MinimumSize);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    if ((Information->TotalPins == 0) ||
        (Information->NumberOfPinsPerBank == 0) ||
        (Information->NumberOfPinsPerBank > Information->TotalPins)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver total pins or pins per bank information "
                    "is incorrect! Total Pins = %#x, PinsPerBank = %#x\n",
                    __FUNCTION__,
                    Information->TotalPins,
                    Information->NumberOfPinsPerBank);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    if (Information->NumberOfPinsPerBank > 64) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Pins per bank > 0x40! PinsPerBank = %#x\n",
                    __FUNCTION__,
                    Information->NumberOfPinsPerBank);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    if (Information->Flags.Reserved != 0) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver passed non-zero value for reserved "
                    "field!\n",
                    __FUNCTION__);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    if ((Information->Flags.BankIdlePowerMgmtSupported != FALSE) &&
        (Information->Flags.MemoryMappedController == FALSE)) {

        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: F-state power management is not supported for "
                    "off-SoC GPIOs!\n",
                    __FUNCTION__);

        Valid = FALSE;
        goto ValidateClientControllerBasicInformationEnd;
    }

    Valid = TRUE;

ValidateClientControllerBasicInformationEnd:
    return Valid;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateInterruptBindings (
    __in PDEVICE_EXTENSION GpioExtension,
    __in WDFCMRESLIST ResourcesTranslated,
    __in WDFCMRESLIST ResourcesRaw,
    __in_bcount(BindingBufferSize)
        PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT BindingOutput,

    __in ULONG BindingBufferSize
    )

/*++

Routine Description:

    This routine validates that the GPIO banks <--> interrupt resources
    mapping returned by the client driver is valid.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    ResourcesRaw - Supplies a handle to a collection of framework resource
        objects. This collection identifies the raw (bus-relative) hardware
        resources that have been assigned to the device.

    ResourcesTranslated - Supplies a handle to a collection of framework
        resource objects. This collection identifies the translated
        (system-physical) hardware resources that have been assigned to the
        device. The resources appear from the CPU's point of view.

    BindingOutput - Supplies a buffer that receives the mappings filled
        by the client driver.

    BindingBufferSize - Supplies the size of the output binding buffer.


Return Value:

    TRUE if the mapping information is valid. FALSE otherwise.

--*/

{

    PCM_PARTIAL_RESOURCE_DESCRIPTOR Descriptor;
    ULONG Index;
    PULONG ResourceMapping;
    ULONG TotalResourceCount;
    ULONG Value;
    BOOLEAN Valid;
    USHORT Version;

    PAGED_CODE();

    UNREFERENCED_PARAMETER(ResourcesRaw);

    Version = BindingOutput->Version;
    if (Version < GPIO_INTERRUPT_BINDING_INFORMATION_OUTPUT_VERSION_MIN) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver interrupt binding version not supported!"
                    " Actual = %d, Min = %d\n",
                    __FUNCTION__,
                    Version,
                    GPIO_INTERRUPT_BINDING_INFORMATION_OUTPUT_VERSION_MIN);

        Valid = FALSE;
        goto ValidateInterruptBindingsEnd;
    }

    //
    // It is OK for the client driver to fill-in less than the supplied buffer
    // but above it. The client driver may only support any older revision of
    // this structure.
    //

    if (BindingOutput->Size > BindingBufferSize) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver information buffer size not valid! "
                    "Actual = %d, Min required = %d\n",
                    __FUNCTION__,
                    BindingOutput->Size,
                    BindingBufferSize);

        Valid = FALSE;
        goto ValidateInterruptBindingsEnd;
    }

    //
    // For each GPIO bank,  verify that the resource index specified in the
    // mapping buffer is within bounds and also that the corresponding
    // resource describes an interrupt descriptor (CmResourceTypeInterrupt).
    //

    ResourceMapping =
        (PULONG)&BindingOutput->BankInterruptBinding.ResourceMapping;

    TotalResourceCount = WdfCmResourceListGetCount(ResourcesTranslated);
    Valid = TRUE;
    for (Index = 0; Index < GpioExtension->TotalBanks; Index += 1) {

        //
        // If the index value is reserved, then it implies the bank does not
        // have an interrupt mapping and thus should be skipped.
        //

        Value = ResourceMapping[Index];
        if (Value == GPIO_BANK_INTERRUPT_BINDING_RESERVED_INDEX) {
            continue;
        }

        if (Value >= TotalResourceCount) {
            Valid = FALSE;
            break;
        }

        Descriptor = WdfCmResourceListGetDescriptor(ResourcesTranslated, Value);
        if (Descriptor->Type != CmResourceTypeInterrupt) {
            Valid = FALSE;
            break;
        }
    }

ValidateInterruptBindingsEnd:
    return Valid;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateControllerFunctionBankMappings (
    __in PDEVICE_EXTENSION GpioExtension,
    __out_xcount(Information->Size)
        PCLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT Information
    )

/*++

Routine Description:

    This routine validates the IOCTL <-> bank information supplied by the client
    driver.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

    Information - Supplies a pointer to a buffer containing the bank
        information supplied by the client driver.

Return Value:

    TRUE if the attribute information is valid. FALSE otherwise.

--*/

{

    USHORT MinimumSize;
    BOOLEAN Valid;
    USHORT Version;

    PAGED_CODE();

    UNREFERENCED_PARAMETER(GpioExtension);

    Valid = FALSE;
    Version = Information->Version;
    if (Version < GPIO_IOCTL_BANK_MAPPING_INFORMATION_VERSION_MIN) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver bank information version is not "
                    "supported! Actual = %d, Min = %d\n",
                    __FUNCTION__,
                    Information->Version,
                    GPIO_IOCTL_BANK_MAPPING_INFORMATION_VERSION_MIN);

        goto ValidateControllerFunctionBankMappingsEnd;
    }

    MinimumSize = FIELD_OFFSET(CLIENT_CONTROLLER_QUERY_SET_INFORMATION_OUTPUT,
                               ControllerFunctionBankMapping.Mapping);

    if (Information->Size < MinimumSize) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Client driver bank information buffer size is not "
                    "valid! Actual = %d, Min required = %d\n",
                    __FUNCTION__,
                    Information->Size,
                    MinimumSize);

        goto ValidateControllerFunctionBankMappingsEnd;
    }

    Valid = TRUE;

ValidateControllerFunctionBankMappingsEnd:
    return Valid;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateRegistrationPacketPhase1 (
    __in_xcount(RegistrationPacket->Size)
        PGPIO_CLIENT_REGISTRATION_PACKET RegistrationPacket
    )

/*++

Routine Description:

    This routine checks whether the registration packet supplied the client
    driver is valid or not.

Arguments:

    RegistrationPacket - Supplies the registration packet that contains
        information about the client driver and all its callbacks.

Return Value:

    TRUE if the registration data is valid; FALSE otherwise.

--*/

{

    PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED CombinedPacket;
    ULONG Flags;
    USHORT MinimumSize;
    BOOLEAN Valid;
    USHORT Version;

    PAGED_CODE();

    //
    // Check whether the version is supported or not.
    //

    Valid = FALSE;
    Version = RegistrationPacket->Version;
    if (Version < GPIO_CLIENT_SUPPORTED_VERSION_MIN) {
       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: Client version not supported! Version = %d, "
                   "Minimum supported = %d\n",
                   __FUNCTION__,
                   Version,
                   GPIO_CLIENT_SUPPORTED_VERSION_MIN);

        goto ValidateRegistrationPacketPhase1End;
    }

    //
    // The structure should meet the minimum size requirements.
    //

    MinimumSize = FIELD_OFFSET_AND_SIZE(GPIO_CLIENT_REGISTRATION_PACKET,
                                        CLIENT_PreProcessControllerInterrupt);

    if (RegistrationPacket->Size < MinimumSize) {
       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: Client packet size mismatch! Expected = %#x, "
                   "Actual = %#x\n",
                   __FUNCTION__,
                   MinimumSize,
                   RegistrationPacket->Size);

        goto ValidateRegistrationPacketPhase1End;
    }


    if ((RegistrationPacket->CLIENT_PrepareController == NULL) ||
        (RegistrationPacket->CLIENT_ReleaseController == NULL) ||
        (RegistrationPacket->CLIENT_StartController == NULL) ||
        (RegistrationPacket->CLIENT_StopController == NULL) ||
        (RegistrationPacket->CLIENT_QueryControllerBasicInformation == NULL)) {

       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: Mandatory device init-related interface(s) not "
                   "supplied! Packet = %p\n",
                   __FUNCTION__,
                   RegistrationPacket);

        goto ValidateRegistrationPacketPhase1End;
    }

    //
    // If enabling interrupt is supported, then all other interrupt interfaces
    // must be supplied. Note ClearActiveInterrupts() interface is optional if
    // auto-clear on read is specified.
    //

    if ((RegistrationPacket->CLIENT_EnableInterrupt != NULL) &&
        ((RegistrationPacket->CLIENT_DisableInterrupt == NULL) ||
         (RegistrationPacket->CLIENT_MaskInterrupts == NULL) ||
         (RegistrationPacket->CLIENT_QueryActiveInterrupts == NULL))) {

       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: All interrupt-related interface not supplied! "
                   "Packet = %p\n",
                   __FUNCTION__,
                   RegistrationPacket);

        goto ValidateRegistrationPacketPhase1End;
    }

    //
    // If connecting pins for IO is supported, then the disconnect interface
    // must be supplied and one of the read or write interfaces must be
    // supplied.
    //

    if (RegistrationPacket->CLIENT_ConnectIoPins != NULL) {
        if (RegistrationPacket->CLIENT_DisconnectIoPins == NULL) {

           TraceEvents(GpioLogHandle,
                       Error,
                       DBG_INIT,
                       "%s: DisconnectIo interface not supplied! Packet = %p\n",
                       __FUNCTION__,
                       RegistrationPacket);

            goto ValidateRegistrationPacketPhase1End;
        }

        if ((RegistrationPacket->CLIENT_WriteGpioPins == NULL) &&
            (RegistrationPacket->CLIENT_ReadGpioPins == NULL)) {

           TraceEvents(GpioLogHandle,
                       Error,
                       DBG_INIT,
                       "%s: Either read or write interface must be supplied! "
                       "Packet = %p\n",
                       __FUNCTION__,
                       RegistrationPacket);

           goto ValidateRegistrationPacketPhase1End;
        }
    }

    //
    // Checks for internal-only clients.
    //

    //
    // The client must be marked with the "internal" flag to be considered
    // internal client.
    //

    Flags = RegistrationPacket->Flags;
    if (CHECK_FLAG(Flags, GPIO_CLIENT_REGISTRATION_FLAGS_INTERNAL) == FALSE) {
        Valid = TRUE;
        goto ValidateRegistrationPacketPhase1End;
    }

    //
    // Note internal clients need to always match the latest registration
    // packet revision. Revisioning is not supported for internal clients.
    //

    MinimumSize = sizeof(GPIO_CLIENT_REGISTRATION_PACKET_COMBINED);
    if (RegistrationPacket->Size != MinimumSize) {
       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: Registration packet does not meet the internal "
                   "structure size! Packet = %p, Actual = %#x, Required = %#x\n",
                   __FUNCTION__,
                   RegistrationPacket,
                   RegistrationPacket->Size,
                   MinimumSize);

        goto ValidateRegistrationPacketPhase1End;
    }

    CombinedPacket =
        (PGPIO_CLIENT_REGISTRATION_PACKET_COMBINED)RegistrationPacket;

    //
    // CreateIoContext() and DeleteIoContext() are optional. But if
    // one is specified, then the other must also be.
    //

    if ((CombinedPacket->CLIENT_CreateIoContext != NULL) &&
        (CombinedPacket->CLIENT_DeleteIoContext == NULL)) {

       TraceEvents(GpioLogHandle,
                   Error,
                   DBG_INIT,
                   "%s: CreateIoContext interface is supplied but not the "
                   "DeleteIoContext! Packet = %p\n",
                   __FUNCTION__,
                   RegistrationPacket);

        goto ValidateRegistrationPacketPhase1End;
    }

    Valid = TRUE;

ValidateRegistrationPacketPhase1End:
    return Valid;
}

_Must_inspect_result_
__drv_requiresIRQL(PASSIVE_LEVEL)
BOOLEAN
GpiopValidateRegistrationPacketPhase2 (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine performs phase 2 validation of the client driver registration
    packet supplied. This phase is performed after the client driver
    basic information has been queried.

    Note this routine assumes that basic information buffer has been already
    validated and that Phase1 validation was successful.

Arguments:

    RegistrationPacket - Supplies the registration packet that contains
        information about the client driver and all its callbacks.

    Information - Supplies a buffer containing the basic information supplied
        by the client driver.

Return Value:

    TRUE if the registration data is valid; FALSE otherwise.

--*/

{

    USHORT ConnectFunctionConfigSize;
    USHORT DisconnectFunctionConfigSize;
    PCLIENT_CONTROLLER_BASIC_INFORMATION Information;
    USHORT QueryEnabledInterruptsSize;
    PGPIO_CLIENT_REGISTRATION_PACKET RegistrationPacket;
    BOOLEAN Valid;

    PAGED_CODE();

    //
    // Check whether the version is supported or not.
    //

    Information = &GpioExtension->ClientInformation;
    RegistrationPacket = GPIO_GET_CLIENT_REGISTRATION_PACKET(GpioExtension);
    Valid = FALSE;
    if ((Information->Flags.EmulateActiveBoth == 1) &&
        (RegistrationPacket->CLIENT_ReconfigureInterrupt == NULL)) {

       TraceEvents(GpioExtension->LogHandle,
                   Error,
                   DBG_INIT,
                   "%s: ActiveBoth is being emulated but reconfigure interrupt"
                   " interface is supplied! Extension = %p, Device = %p\n",
                   __FUNCTION__,
                   GpioExtension,
                   GpioExtension->Device);

        goto ValidateRegistrationPacketPhase2End;
    }

    Valid = TRUE;

    //
    // Ignore Version 2.0 functions that do not exist in older drivers.
    //
    // This will catch any older but not-so-well-written clients that pass in a
    // Size field that is longer than the length (including any padding) of
    // that older version's GPIO_CLIENT_REGISTRATION_PACKET.
    //

    if (RegistrationPacket->Version < 2) {
        QueryEnabledInterruptsSize = FIELD_OFFSET_AND_SIZE(
                                        GPIO_CLIENT_REGISTRATION_PACKET,
                                        CLIENT_QueryEnabledInterrupts);

        if ((RegistrationPacket->Size >= QueryEnabledInterruptsSize) &&
            (RegistrationPacket->CLIENT_QueryEnabledInterrupts != NULL)) {

           GPIO_ASSERT(!"The registration packet contains the Version 2 "
                        "CLIENT_QueryEnabledInterrupts but the Version field "
                        "is too old");

           RegistrationPacket->CLIENT_QueryEnabledInterrupts = NULL;
           __fallthrough;
        }
    }

    //
    // Ignore Version 3.0 functions that do not exist in older drivers.
    //
    // This will catch any older but not-so-well-written clients that pass in a
    // Size field that is longer than the length (including any padding) of
    // that older version's GPIO_CLIENT_REGISTRATION_PACKET.
    //

    if (RegistrationPacket->Version < 3) {
        ConnectFunctionConfigSize = FIELD_OFFSET_AND_SIZE(
            GPIO_CLIENT_REGISTRATION_PACKET,
            CLIENT_ConnectFunctionConfigPins);

        if ((RegistrationPacket->Size >= ConnectFunctionConfigSize) &&
            (RegistrationPacket->CLIENT_ConnectFunctionConfigPins != NULL)) {

            GPIO_ASSERT(!"The registration packet contains non-null ptr to "
                "CLIENT_ConnectFunctionConfigPins but the Version field "
                "is too old");

            RegistrationPacket->CLIENT_ConnectFunctionConfigPins = NULL;
            __fallthrough;
        }

        DisconnectFunctionConfigSize = FIELD_OFFSET_AND_SIZE(
            GPIO_CLIENT_REGISTRATION_PACKET,
            CLIENT_DisconnectFunctionConfigPins);

        if ((RegistrationPacket->Size >= DisconnectFunctionConfigSize) &&
            (RegistrationPacket->CLIENT_DisconnectFunctionConfigPins != NULL)) {

            GPIO_ASSERT(!"The registration packet contains non-null ptr to "
                "CLIENT_DisconnectFunctionConfigPins but the Version field "
                "is too old");

            RegistrationPacket->CLIENT_DisconnectFunctionConfigPins = NULL;
            __fallthrough;
        }
    }

ValidateRegistrationPacketPhase2End:
    return Valid;
}

__drv_requiresIRQL(PASSIVE_LEVEL)
VOID
GpiopInitializeRegistryOptions (
    __in PDRIVER_OBJECT WdmDriverObject,
    __in WDFDRIVER WdfDriver,
    __in PUNICODE_STRING RegistryPath
    )

/*++

Routine Description:

    This routine queries options/flags set in the registry and intializes
    respective globals.

Arguments:

    WdmDriverObject - Supplies a pointer to the GPIO class extension's WDM
        driver object.

    WdfDriver - Supplies a handle to the WDF driver object.

    RegistryPath - Supplies a pointer to the driver specific registry key.

Return Value:

    None.

--*/

{

    ULONG EmergencyFlag;
    NTSTATUS Status;

    DECLARE_CONST_UNICODE_STRING(EmergencyWorkerKeyName,
                                 GPIO_REGISTRY_FORCE_EMERGENCY_WORKER_KEY);

#ifdef DBG

    ULONG InterruptOutputFlag;

    DECLARE_CONST_UNICODE_STRING(InterruptOutputKeyName,
                                 GPIO_REGISTRY_ALLOW_INTERRUPT_OUTPUT_KEY);

#endif

    ULONG CheckEnabledInterruptsFlag;

    DECLARE_CONST_UNICODE_STRING(CheckEnabledInterruptsKeyName,
                                 GPIO_REGISTRY_CHECK_ENABLED_INTERRUPTS_KEY);

    ULONG DebounceModelOverride;

    DECLARE_CONST_UNICODE_STRING(DebounceModelOverrideKeyName,
                                 GPIO_REGISTRY_DEBOUNCE_MODEL_OVERRIDE_KEY);

#if defined(DBG)

    ULONG NoiseIntervalOverride;

    DECLARE_CONST_UNICODE_STRING(NoiseIntervalOverrideKeyName,
                                 GPIO_REGISTRY_DEBOUNCE_NOISE_INTERVAL_KEY);

#endif

    PAGED_CODE();

    UNREFERENCED_PARAMETER(WdmDriverObject);

    //
    // Check if the flag is set to force all worker activity onto the emergency
    // thread.
    //

    EmergencyFlag = 0x0;
    Status = GpioUtilQueryRegistryFlags(NULL,
                                        WdfDriver,
                                        RegistryPath,
                                        &EmergencyWorkerKeyName,
                                        &EmergencyFlag);

    if (NT_SUCCESS(Status)) {
        GpioForceEmergencyWorker = (BOOLEAN)EmergencyFlag;
        TraceEvents(GpioLogHandle,
                    Info,
                    DBG_INIT,
                    "%s: Force emergency worker flag = %d!\n",
                    __FUNCTION__,
                    GpioForceEmergencyWorker);
    }

    //
    // Check if the flag is set to allow Interrupt+Output configuration.
    // This is only supported on CHK versions of the GPIO class extension.
    //

#ifdef DBG

    InterruptOutputFlag = 0x0;
    Status = GpioUtilQueryRegistryFlags(NULL,
                                        WdfDriver,
                                        RegistryPath,
                                        &InterruptOutputKeyName,
                                        &InterruptOutputFlag);

    if (NT_SUCCESS(Status)) {
        GpioAllowInterruptOutput = (BOOLEAN)InterruptOutputFlag;
        TraceEvents(GpioLogHandle,
                    Info,
                    DBG_INIT,
                    "%s: Interrupt+Output configuration flag = %d!\n",
                    __FUNCTION__,
                    GpioAllowInterruptOutput);
    }

#endif

    CheckEnabledInterruptsFlag = 0x0;
    Status = GpioUtilQueryRegistryFlags(NULL,
                                        WdfDriver,
                                        RegistryPath,
                                        &CheckEnabledInterruptsKeyName,
                                        &CheckEnabledInterruptsFlag);

    if (NT_SUCCESS(Status)) {
        GpioCheckEnabledInterrupts = (BOOLEAN)CheckEnabledInterruptsFlag;

    } else {

        //
        // Checked builds and builds from development branches can afford the
        // performance hit of frequent checks of enabled interrupts.
        //

#if defined(DBG)

        GpioCheckEnabledInterrupts = TRUE;

#else
        GpioCheckEnabledInterrupts = FALSE;

#endif

    }

    TraceEvents(GpioLogHandle,
                Info,
                DBG_INIT,
                "%s: Check enabled interrupts flag = %d!\n",
                __FUNCTION__,
                GpioCheckEnabledInterrupts);

    //
    // Check if the flag is set to force the debounce model globally.
    //

    DebounceModelOverride = 0x0;
    Status = GpioUtilQueryRegistryFlags(NULL,
                                        WdfDriver,
                                        RegistryPath,
                                        &DebounceModelOverrideKeyName,
                                        &DebounceModelOverride);

    if (NT_SUCCESS(Status) && (DebounceModelOverride < DebounceModelMaximum)) {
        GpioGlobalDebounceModelOverride =
            (GPIO_DEBOUNCE_MODEL)DebounceModelOverride;

        TraceEvents(GpioLogHandle,
                    Info,
                    DBG_INIT,
                    "%s: Debounce model override = %d!\n",
                    __FUNCTION__,
                    GpioGlobalDebounceModelOverride);
    }

    //
    // Check if a noise interval override is set in the registry. The value
    // is specified in 100s of micro-second intervals. This value likely to
    // be changed only for debugging debouncing code and thus not enabled
    // for retail builds.
    //

#if defined(DBG)

    NoiseIntervalOverride = 0x0;
    Status = GpioUtilQueryRegistryFlags(NULL,
                                        WdfDriver,
                                        RegistryPath,
                                        &NoiseIntervalOverrideKeyName,
                                        &NoiseIntervalOverride);

    if (NT_SUCCESS(Status)) {
        if (NoiseIntervalOverride >
                GPIO_NOISE_FILTER_INTERVAL_OVERRIDE_MAXIMUM) {

            NoiseIntervalOverride = GPIO_NOISE_FILTER_INTERVAL_OVERRIDE_MAXIMUM;
        }

        GpioGlobalNoiseIntervalOverride = NoiseIntervalOverride * 100;

        TraceEvents(GpioLogHandle,
                    Info,
                    DBG_INIT,
                    "%s: Debounce noise filtering interval override "
                    "in 100s of us = %d!\n",
                    __FUNCTION__,
                    GpioGlobalNoiseIntervalOverride);
    }

#endif

    return;
}

