/*++

Copyright (C) Microsoft. All rights reserved.

Module Name:

    power.c

Abstract:

    This file implements GPIO class extension driver's power management
    functionality. The GPIO class extension supports banks being power-managed
    at OS runtime if they are not being used.

    The GPIO class extension supports the Windows 8 component/f-state based
    power management model, "PoFx". A GPIO controller could be partitioned into
    several banks, where each banks represents a component (unit of
    power management). Thus Banks == Components from GPIO class extension
    perspective. This module implements routines to power-manage individual
    GPIO banks.

    Note the GPIO class extension handles the PoFx callbacks exclusively and
    that WDF is not aware of, nor explicitly synchronized with, PoFx. This is
    because WDF does not yet support integration with PoFx for devices with
    multiple banks and components.


Environment:

    Kernel mode

--*/

//
// ------------------------------------------------------------------- Includes
//

#include "pch.h"

#if defined(EVENT_TRACING)
#include "power.tmh"         // auto-generated by WPP
#endif

#include "privdefs.h"

//
// -------------------------------------------------------------------- Defines
//

//
// Bank 0 is reserved by WDF and represents the device. Components 1 -> N+1 map
// to banks 0 -> N registered by the GPIO class extension.
//

#define BANK_ID_TO_COMPONENT_ID(BankId) ((BankId) + 1)

//
// ----------------------------------------------------------------- Prototypes
//

PO_FX_COMPONENT_ACTIVE_CONDITION_CALLBACK GpioClxBankActiveConditionCallback;
PO_FX_COMPONENT_IDLE_CONDITION_CALLBACK GpioClxBankIdleConditionCallback;
PO_FX_COMPONENT_IDLE_STATE_CALLBACK GpioClxBankIdleStateCallback;
PO_FX_DEVICE_POWER_REQUIRED_CALLBACK GpioClxDevicePowerRequiredCallback;
PO_FX_DEVICE_POWER_NOT_REQUIRED_CALLBACK GpioClxDevicePowerNotRequiredCallback;
PO_FX_COMPONENT_CRITICAL_TRANSITION_CALLBACK GpioClxCriticalTransitionCallback;

VOID
GpiopProcessBankIdleStateTransition (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in ULONG State,
    __in BOOLEAN CriticalTransition
    );

//
// -------------------------------------------------------------------- Pragmas
//

#pragma alloc_text (PAGE, GpiopRegisterWithPowerFramework)
#pragma alloc_text (PAGE, GpiopStartRuntimePowerManagement)
#pragma alloc_text (PAGE, GpiopUnregisterWithPowerFramework)

//
// -------------------------------------------------- Power framework callbacks
//

VOID
GpioClxBankActiveConditionCallback (
    __in PVOID Context,
    __in ULONG Component
    )

/*++

Routine Description:

    This routine is invoked by the power framework to switch a component's
    logical condition from Idle to Active.

    N.B. This routine may be called at DISPATCH_LEVEL.

Arguments:

    Context - Supplies the context value passed during registration.

    Component - Supplies the component index (GPIO bank ID).

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PDEVICE_EXTENSION GpioExtension;

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    GPIO_ASSERT(Component < GpioExtension->TotalBanks);

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto BankActiveConditionCallbackEnd;
    }

    //
    // Component is going active.
    //
    // If a bank/component is going active, then start the secondary I/O queue
    // associated with that bank. However, this callback could be racing
    // with PnP callback to power down the device, thus the queue needs to
    // started only if the power down sequence has not begun already.
    //
    // N.B. Even though the queue is started at DISPATCH_LEVEL, the request
    //      will be dispatched only at passive-level since
    //      WdfExecutionLevelPassive attribute is set on the queue.
    //

    GpioExtension->Banks[Component].PowerData.IsActive = TRUE;
    WdfSpinLockAcquire(GpioExtension->PowerDownLock);

    //
    // Don't change the state of the queues if the device is being powered
    // down.
    //

    if (GpioExtension->DevicePoweringDown == FALSE) {
        WdfIoQueueStart(GpioExtension->Banks[Component].IoQueue);
    }

    WdfSpinLockRelease(GpioExtension->PowerDownLock);

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Marked bank active! Extn = %p, Bank ID = 0x%x\n",
                __FUNCTION__,
                GpioExtension,
                Component);

BankActiveConditionCallbackEnd:
    return;
}

VOID
GpioClxBankIdleConditionCallback (
    __in PVOID Context,
    __in ULONG Component
    )

/*++

Routine Description:

    This routine is invoked by the power framework to switch a component's
    logical condition from Active to Idle.

    N.B. This routine may be called at DISPATCH_LEVEL.

Arguments:

    Context - Supplies the context value passed during registration.

    Component - Supplies the component index (GPIO bank ID).

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PPO_FX_COMPONENT_IDLE_STATE F1State;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    GPIO_ASSERT(Component < GpioExtension->TotalBanks);

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto BankIdleConditionCallbackEnd;
    }

    //
    // Component is going idle.
    //
    // NOTE: When we're being notified that the component is idle, this
    // callback is always invoked at dispatch level (because the power
    // framework is holding a spinlock while it invokes this callback). That
    // is why we've chosen to use a spinlock (instead of a passive-level
    // lock) for our PowerDownLock.
    //
    // NOTE: Given that the component is idle, there shouldn't be any
    // outstanding requests either sitting in the component queue or
    // delivered to the driver from that queue. In other words, the queue
    // should be empty now. We also know that no new requests will get added
    // to the queue while this callback is running. The reason is described
    // below.
    //
    // Our driver does not add any new request to this queue until the
    // PoFxActivateComponent call for that request has returned. If
    // PoFxActivateComponent is called on another thread while this callback
    // is running or about to run, then the power framework makes sure that
    // PoFxActivateComponent does not return until this callback has finished
    // running. Therefore we know that no new request would get added to
    // this queue while we are attempting to stop the queue within this
    // callback.
    //

    GpioBank = &GpioExtension->Banks[Component];
    GpioBank->PowerData.IsActive = FALSE;

    WdfSpinLockAcquire(GpioExtension->PowerDownLock);

    //
    // Stop the queue. Given that the component is idle, there shouldn't be
    // any outstanding requests either in the queue or being delivered to
    // the driver. The queue is stopped only to prevent new requests that
    // get added to the queue from being dispatched to the driver until the
    // component becomes active again.
    //
    // N.B. Don't change the state of the queues if the device is being
    //      powered down.
    //

    if (GpioExtension->DevicePoweringDown == FALSE) {
        WdfIoQueueStop(GpioBank->IoQueue, NULL, NULL);
    }

    WdfSpinLockRelease(GpioExtension->PowerDownLock);

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Marked bank idle! Extn = %p, Bank ID = 0x%x\n",
                __FUNCTION__,
                GpioExtension,
                Component);

    //
    // At this point, it would be desirable to put the bank into F1 if the
    // PEP desires so (if F1 is supported). Provide a hint to the PEP by setting
    // the latency and residency values. Note the PEP would ultimately decide
    // based on other factors if the device should be put into F1 or not.
    //

    if (GpioBank->PowerData.MaximumFState == FStateF1) {
        F1State = &GpioBank->PowerData.F1StateParameters;
        PoFxSetComponentLatency(GpioExtension->PowerHandle,
                                Component,
                                F1State->TransitionLatency);

        PoFxSetComponentResidency(GpioExtension->PowerHandle,
                                Component,
                                F1State->ResidencyRequirement);
    }

BankIdleConditionCallbackEnd:
    PoFxCompleteIdleCondition(GpioExtension->PowerHandle, Component);
    return;
}

VOID
GpioClxBankIdleStateCallback (
    __in PVOID Context,
    __in ULONG Component,
    __in ULONG State
    )

/*++

Routine Description:

    This routine is invoked by the power framework to change the F-state of
    the specified component. When transitioning to F0, driver is expected to
    restore hardware context as necessary. On F0 -> F1 transition, the
    context may need to be saved.

Arguments:

    Context - Supplies the context value passed during registration.

    Component - Supplies the component index (GPIO bank ID).

    State - Supplies the new F-state the component should transition to.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    GPIO_ASSERT(Component < GpioExtension->TotalBanks);
    GPIO_ASSERT(State < GPIO_TOTAL_F_STATES);

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto BankIdleStateCallbackEnd;
    }

    GpioBank = &GpioExtension->Banks[Component];

    //
    // Request the client driver to save the bank HW context if transitioning
    // into off (F1) state. Otherwise request the client driver to restore the
    // bank HW context.
    //
    // Note f-state based power management is only supported for on-SoC GPIO
    // controllers.
    //

    GPIO_ASSERT(GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE);

    GpiopProcessBankIdleStateTransition(GpioBank, State, FALSE);

    //
    // Since the transition was completed synchronously, notify PO as such.
    //

BankIdleStateCallbackEnd:
    PoFxCompleteIdleState(GpioExtension->PowerHandle, Component);
    return;
}

VOID
GpiopDevicePowerD0TransitionWorker (
    __in WDFWORKITEM WorkItem
    )

/*++

Routine Description:

    This routine is the worker routine for handling Dx -> D0 state transitions.
    It requests WDF to power up the device synchronously (i.e. send a D0 IRP)
    and then notifies PO about the transition completion.

Arguments:

    WorkItem - Supplies a handle to the workitem supplying the context.

Return Value:

    None.

--*/

{

    PGPIO_DEVICE_POWER_WORK_ITEM_CONTEXT PowerContext;
    PDEVICE_EXTENSION GpioExtension;
    NTSTATUS Status;

    PAGED_CODE();

    PowerContext = GpiopGetDevicePowerWorkItemContext(WorkItem);
    GpioExtension = PowerContext->GpioExtension;

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto BankIdleStateCallbackEnd;
    }

    //
    // If the power framework notifies that device is required, then it implies
    // that at least one bank requested to be active. In such case, request
    // WDF to suspend the idle-state management. It will be resumed when the
    // power framework notifies that device power is not required.
    //
    // N.B. The synchronous power on request is made as power framework needs
    //      to be notified post-D0 transition.
    //

    Status = WdfDeviceStopIdle(GpioExtension->Device, TRUE);

    GPIO_ASSERT((Status == STATUS_SUCCESS) ||
                (Status == STATUS_POWER_STATE_INVALID));

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - WdfStopIdle() called! Extn = %p, Status = %#x\n",
                __FUNCTION__,
                GpioExtension,
                Status);

BankIdleStateCallbackEnd:

    //
    // Notify Power framework about D0 transition completion. Note this needs
    // to be done even if the D0 transition actually failed
    // (i.e. WdfDeviceStopIdle() call above failed).
    //

    GPIO_ASSERT(GpioExtension->PowerHandle != NULL);
    PoFxReportDevicePoweredOn(GpioExtension->PowerHandle);
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - PoFxReportDevicePoweredOn() called! Extn = %p\n",
                __FUNCTION__,
                GpioExtension);

    return;
}

VOID
GpioClxDevicePowerRequiredCallback (
    __in PVOID Context
    )

/*++

Routine Description:

    This routine is invoked by the power framework to notify the driver when it
    it must enter the D0 state.

Arguments:

    Context - Supplies the context value passed during registration.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PDEVICE_EXTENSION GpioExtension;

    //
    // If the power framework notifies that the device power is not required,
    // then it implies that all banks have reported idle and they are in a
    // suitable F-state where PEP believes that device power can be cut. The
    // actual F-state can vary per-bank. In such case, request WDF to resume
    // idle-state Dx-based power management. If no IRPs are received within a
    // specified interval (default timeout), then WDF will send D3 down the
    // stack to turn off the controller.
    //

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // Queue a worker, GpiopDevicePowerD0TransitionWorker(), to accomplish the
    // D0 transition. This request is always handled asynchronously because it
    // can be dispatched to the driver in the context of a request arriving to
    // the power managed queue. Trying to synchronously power on the device in
    // such context can lead to WDF deadlock.
    //

    WdfWorkItemEnqueue(GpioExtension->DevicePowerWorkItem);
    return;
}

VOID
GpioClxDevicePowerNotRequiredCallback (
    __in PVOID Context
    )

/*++

Routine Description:

    This routine is invoked by the power framework to notify the driver when it
    may enter a low-power Dx state.

Arguments:

    Context - Supplies the context value passed during registration.

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PDEVICE_EXTENSION GpioExtension;

    //
    // If the power framework notifies that the device power is not required,
    // then it implies that all banks have reported idle and they are in a
    // suitable F-state where PEP believes that device power can be cut. The
    // actual F-state can vary per-bank. In such case, request WDF to resume
    // idle-state Dx-based power management. If no IRPs are received within a
    // specified interval (default timeout), then WDF will send D3 down the
    // stack to turn off the controller.
    //

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto DevicePowerNotRequiredCallbackEnd;
    }

    WdfDeviceResumeIdle(Device);
    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Received DevicePowerNotRequired notification..."
                "WdfDeviceResumeIdle() called! Extn = %p\n",
                __FUNCTION__,
                GpioExtension);

DevicePowerNotRequiredCallbackEnd:
    PoFxCompleteDevicePowerNotRequired(GpioExtension->PowerHandle);
    return;
}

VOID
GpioClxCriticalTransitionCallback (
    __in PVOID Context,
    __in ULONG Component,
    __in BOOLEAN Active
    )

/*++

Routine Description:

    This routine handles critical power transitions for a primary device.

Arguments:

    Context - Supplies the context value passed during registration.

    Component - Supplies the component index (GPIO bank ID).

    State - Supplies the new F-state the component should transition to.

    Active - Supplies a flag indicating the logical state to which the device
        will be transitioned to upon returning from this callback:
        TRUE => F0 state; FALSE => F1 state

Return Value:

    None.

--*/

{

    WDFDEVICE Device;
    PGPIO_BANK_ENTRY GpioBank;
    PDEVICE_EXTENSION GpioExtension;
    ULONG State;

    Device = (WDFDEVICE)Context;
    GpioExtension = GpioClxGetDeviceExtension(Device);

    GPIO_ASSERT(Component < GpioExtension->TotalBanks);

    if (GPIO_GET_IGNORE_PO_FX_CALLBACKS(GpioExtension) != FALSE) {
        goto CriticalTransitionCallbackEnd;
    }

    GpioBank = &GpioExtension->Banks[Component];

    //
    // Map the active/idle to appropriate F-state.
    //

    if (Active == FALSE) {
        State = FStateF1;

    } else {
        State = FStateF0;
    }

    GpiopProcessBankIdleStateTransition(GpioBank, State, TRUE);

CriticalTransitionCallbackEnd:
    return;
}

//
// ------------------------------------------------------------- Power routines
//

NTSTATUS
GpiopHandlePowerPostInterruptsEnabled (
    __in WDFDEVICE Device,
    __in WDF_POWER_DEVICE_STATE PreviousState
    )

/*

Routine Description:

    This routine performs power-related post-processing after interrupts
    have been re-enabled. On the first Dx -> D0 transition, there is nothing
    to do. On subsequent D0 transistions, it restarts all queues.

Arguments:

    Device - Supplies a handle to the framework device object.

    PreviousState - Supplies the previous device power state.

Return Value:

    NTSTATUS code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    ULONG BankId;

    UNREFERENCED_PARAMETER(PreviousState);

    //
    // If this is the first D0 entry for this device, the per-bank secondary
    // I/O queues need not be started. That is needed only for subsequent D0
    // transitions, where a previous D0 exit would have stopped the
    // power-managed queues.
    //

    GpioExtension = GpioClxGetDeviceExtension(Device);
    if (GpioExtension->InitialD0Entry) {
        GpioExtension->InitialD0Entry = FALSE;

    } else {

        //
        // Start the secondary queues for those components that were active.
        //

        for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
            if (GpioExtension->Banks[BankId].PowerData.IsActive == TRUE) {
                WdfIoQueueStart(GpioExtension->Banks[BankId].IoQueue);
            }
        }
    }

    return STATUS_SUCCESS;
}

NTSTATUS
GpiopHandlePowerPreInterruptsDisabled (
    __in WDFDEVICE Device
    )

/*

Routine Description:

    This routine performs power-related pre-processing before interrupts
    are disabled (i.e. before D0 -> Dx transition). This routine stops all
    the non-power managed queues.

    Note this routine may be called before/after the boot device is in D0/D3 if
    boot device has GPIO dependencies.

Arguments:

    Device - Supplies a handle to the framework device object.

    TargetState - Supplies the device power state which the device will be put
        in once the callback is complete.

Return Value:

    NTSTATUS code.

--*/

{

    PDEVICE_EXTENSION GpioExtension;
    ULONG BankId;

    GpioExtension = GpioClxGetDeviceExtension(Device);

    //
    // Stop the non-power-managed queues that we have associated with each of
    // our components. Recall that these non-power-managed queues are secondary
    // queues. The primary (top-level) queue is power-managed and would have
    // already been stopped by the framework before it invoked this callback.
    //

    //
    // Make a note that the device is powering down, so that the
    // FStateComponentActiveConditionCallback routine does not try to modify the
    // state of the secondary queues.
    //

    WdfSpinLockAcquire(GpioExtension->PowerDownLock);
    GpioExtension->DevicePoweringDown = TRUE;
    WdfSpinLockRelease(GpioExtension->PowerDownLock);

    //
    // Now stop the secondary queues for each of the components.
    //

    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        WdfIoQueueStopSynchronously(GpioExtension->Banks[BankId].IoQueue);
    }

    WdfSpinLockAcquire(GpioExtension->PowerDownLock);
    GpioExtension->DevicePoweringDown = FALSE;
    WdfSpinLockRelease(GpioExtension->PowerDownLock);

    return STATUS_SUCCESS;
}

NTSTATUS
GpiopInitializeDxIdlePowerPolicy (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine initializes the device idle-state Dx-based power management
    policy. The default policy is to put the controller into D3 if it is
    idle.

Arguments:

    GpioExtension - Supplies a pointer the GPIO device extension.

Return Value:

    NTSTATUS code.

--*/

{

    PCLIENT_CONTROLLER_BASIC_INFORMATION Information;
    WDF_DEVICE_POWER_POLICY_IDLE_SETTINGS IdleSettings;
    NTSTATUS Status;

    PAGED_CODE();

    Information = &GpioExtension->ClientInformation;
    if (Information->Flags.DeviceIdlePowerMgmtSupported == FALSE) {
        Status = STATUS_SUCCESS;
        goto InitializeDxIdlePowerPolicyEnd;
    }

    //
    // Init the idle policy structure. By idle timeout is set to the default
    // WDF timeout. The target Dx state will be D3 when IdleCannotWakeFromS0
    // is specified.
    //

    WDF_DEVICE_POWER_POLICY_IDLE_SETTINGS_INIT(&IdleSettings,
                                               IdleCannotWakeFromS0);

    if (Information->DeviceIdleTimeout != 0) {
        IdleSettings.IdleTimeout = Information->DeviceIdleTimeout;
    }

    IdleSettings.PowerUpIdleDeviceOnSystemWake = WdfTrue;
    Status = WdfDeviceAssignS0IdleSettings(GpioExtension->Device,
                                           &IdleSettings);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_PNP,
                    "%s: WdfDeviceAssignS0IdleSettings() failed! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);
    }

InitializeDxIdlePowerPolicyEnd:
    return Status;
}

NTSTATUS
GpiopInitializeDevicePowerWorker (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine initializes the device D0 power transition worker.


Arguments:

    GpioExtension - Supplies a pointer the GPIO device extension.

Return Value:

    NTSTATUS code.

--*/

{

    WDF_OBJECT_ATTRIBUTES Attributes;
    PGPIO_DEVICE_POWER_WORK_ITEM_CONTEXT Context;
    NTSTATUS Status;
    WDF_WORKITEM_CONFIG WorkItemConfiguration;

    PAGED_CODE();

    WDF_OBJECT_ATTRIBUTES_INIT(&Attributes);
    Attributes.ParentObject = GpioExtension->Device;
    WDF_OBJECT_ATTRIBUTES_SET_CONTEXT_TYPE(&Attributes,
                                           GPIO_DEVICE_POWER_WORK_ITEM_CONTEXT);

    //
    // Initialize the worker routine and create a new workitem.
    //

    WDF_WORKITEM_CONFIG_INIT(&WorkItemConfiguration,
                             GpiopDevicePowerD0TransitionWorker);

    //
    // Disable automatic serialization by the framework for the worker thread.
    // The parent device object is being serialized at device level (i.e.,
    // WdfSynchronizationScopeDevice), and the framework requires it be
    // passive level (i.e., WdfExecutionLevelPassive) if automatic
    // synchronization is desired.
    //

    WorkItemConfiguration.AutomaticSerialization = FALSE;

    //
    // Create the work item object and initialize the context.
    //

    Status = WdfWorkItemCreate(&WorkItemConfiguration,
                               &Attributes,
                               &GpioExtension->DevicePowerWorkItem);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_INIT,
                    "%s: Failed to allocate device power work item! "
                    "Status = %#x\n",
                    __FUNCTION__,
                    Status);

        goto InitializeDevicePowerWorkerEnd;
    }

    //
    // Initialize the context to be supplied to the workitem handler.
    //

    Context =
        GpiopGetDevicePowerWorkItemContext(GpioExtension->DevicePowerWorkItem);

    Context->GpioExtension = GpioExtension;

InitializeDevicePowerWorkerEnd:
    return Status;
}

VOID
GpiopProcessBankIdleStateTransition (
    __in PGPIO_BANK_ENTRY GpioBank,
    __in ULONG State,
    __in BOOLEAN CriticalTransition
    )

/*++

Routine Description:

    This routine processes requests to transition a GPIO bank into the
    specified f-state.

    N.B. This routine may be called at DISPATCH_LEVEL for memory-mapped/on-SoC
         GPIO controllers. For off-SoC controller, it will be called at
         PASSIVE_LEVEL (from the worker).

Arguments:

    GpioBank - Supplies the GPIO bank associated with the request.

    State - Supplies the new F-state the component should transition to.

    CriticalTransition - Supplies whether this idle state transition is being
        done as part of a critical transition (PEP putting the system into
        low power) or normal transition.

Return Value:

    None.

--*/

{

    BOOLEAN AcquireInterruptLock;
    PDEVICE_EXTENSION GpioExtension;
    PGPIO_BANK_POWER_DATA PowerData;
    BOOLEAN ShouldRestoreContext;
    BOOLEAN ShouldSaveContext;

    GpioExtension = GpioBank->GpioExtension;
    PowerData = &GpioBank->PowerData;

    //
    // Acquire the bank interrupt lock only for regular runtime f-state
    // transitions. For critical transitions, the invocation is made by the
    // PEP at HIGH_LEVEL, which exceeds the bank's IRQL.
    //
    // N.B. In case of critical transitions, the PEP makes the critical
    //      transition callback in the context of last process going idle and
    //      at HIGH_LEVEL, thus there is no race involved. One exception is
    //      test, which may call the critical transition callbacks at
    //      DISPATCH_LEVEL or below. Acquire the interrupt lock in such cases.
    //

    if ((CriticalTransition == FALSE) || (KeGetCurrentIrql() < HIGH_LEVEL)) {
        AcquireInterruptLock = TRUE;

    } else {
        AcquireInterruptLock = FALSE;
    }

    //
    // Notify (internal) client driver about an impending FState transition.
    //

    GpioClnInvokePreBankFStateTransitionCallback(GpioBank, State);

    //
    // If the bank is becoming active, then report the interrupts as active
    // again. This is done prior to calling into the client driver to restore
    // the context as an interrupt can fire the moment the interrupt lock
    // is dropped. Otherwise there is a possibility an interrupt may fire
    // without the handler being enabled.
    //
    // Note if this is a critical F0 transition, then no need to report as
    // active as it was never report inactive when going into F1.
    //

    if ((State == FStateF0) && (CriticalTransition == FALSE)) {
        GpiopSetInterruptState(GpioBank, GpioBank->BankInterrupt, TRUE);
        GpiopSetInterruptState(GpioBank, GpioBank->BankPassiveInterrupt, TRUE);
    }

    //
    // Synchronize operations on the GPIO controller. Since  f-state based power
    // management is only supported for on-SoC GPIO controllers, the bank
    // interrupt lock is used.
    //

    GPIO_ASSERT(GPIO_IS_PASSIVE_IRQL_ONLY_DEVICE(GpioExtension) == FALSE);

    if (AcquireInterruptLock != FALSE) {
        GPIO_ACQUIRE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // Invoke the client driver callback to save the context when going into
    // F1. Otherwise, invoke the client driver callback to restore the context.
    //
    // A GPIO controller may be managed by the:
    //
    // 1. Default PEP: For regular active/idle f-state management. When the
    //                 context is saved/restored, we call this a Regular
    //                 Save/Restore.
    // 2. Platform PEP: To put the controller into a certain Fx state while
    //                  transitioning into/out of deep-idle. When the context
    //                  is saved/restored, we call this a Critical
    //                  Save/Restore.
    //
    // Due to race scenarios existing with the default PEP transitioning a
    // component into Fx state and the platform entering/exiting deep-idle, the
    // platform PEP may not always precisely know if a given GPIO bank is in
    // certain Fx state or not. To simplify PEP implementation, it was decided
    // that the PEP could always call Critical Save/Restore on GPIO banks across
    // deep-idle transitions. The GPIO class extension will handle a Critical
    // Save and Restore pair occuring in between a Regular Save and a Regular
    // Restore, and only perform the save once & the restore once.
    //
    // The only valid combinations are:
    //
    // 1. Regular Save followed by a Regular Restore
    // 2. Critical Save followed by a Critical Restore
    // 3. Regular Save followed by a Critical Save (and not the opposite unless
    //    Critical Restore happened in between those two calls)
    // 4. Critical Restore followed by a Regular Restore (and not the opposite
    //    unless Critical Save happened in between those two calls).
    //

    if (State == FStateF1) {
        ShouldSaveContext = TRUE;

        if (CriticalTransition == FALSE) {

            //
            // Regular Save
            //
            // If we have already Regular Saved or Critical Saved, disallow
            // the double saving.
            //

            if ((PowerData->RegularContextSaved != FALSE) ||
                (PowerData->CriticalContextSaved != FALSE)) {

                KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                             INVALID_SAVE_BANK_HARDWARE_CONTEXT,
                             (ULONG_PTR)CriticalTransition,
                             (ULONG_PTR)PowerData->RegularContextSaved,
                             (ULONG_PTR)PowerData->CriticalContextSaved);
            }

            PowerData->RegularContextSaved = TRUE;

        } else {

            //
            // Critical Save
            //
            // 1. Disallow double Critical Saving.
            // 2. Allow a Critical Save after a Regular Save, but make it do
            //    nothing.
            //

            if (PowerData->CriticalContextSaved != FALSE) {
                KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                             INVALID_SAVE_BANK_HARDWARE_CONTEXT,
                             (ULONG_PTR)CriticalTransition,
                             (ULONG_PTR)PowerData->RegularContextSaved,
                             (ULONG_PTR)PowerData->CriticalContextSaved);

            } else if (PowerData->RegularContextSaved != FALSE) {
                ShouldSaveContext = FALSE;
            }

            PowerData->CriticalContextSaved = TRUE;
        }

        if (ShouldSaveContext != FALSE) {
            GpioClnInvokeSaveBankHardwareContext(GpioBank,
                                                 CriticalTransition);
        }

#if defined(EVENT_TRACING)

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_POWER,
                    "%s - %s SaveBankHardwareContext() for Extn = %p, "
                    "Bank ID = 0x%x, Critical = %d\n",
                    __FUNCTION__,
                    ((ShouldSaveContext == FALSE) ? "Ignored" : "Called"),
                    GpioExtension,
                    GpioBank->BankId,
                    CriticalTransition);

#endif

    } else {
        ShouldRestoreContext = TRUE;

        if (CriticalTransition == FALSE) {

            //
            // Regular Restore
            //
            // Only allow it if we have only Regular Saved before.
            //

            if ((PowerData->RegularContextSaved == FALSE) ||
                (PowerData->CriticalContextSaved != FALSE)) {

                KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                             INVALID_RESTORE_BANK_HARDWARE_CONTEXT,
                             (ULONG_PTR)CriticalTransition,
                             (ULONG_PTR)PowerData->RegularContextSaved,
                             (ULONG_PTR)PowerData->CriticalContextSaved);
            }

            PowerData->RegularContextSaved = FALSE;

        } else {

            //
            // Critical Restore
            //
            // 1. Disallow it if we haven't Critical Saved before.
            // 2. Allow a Critical Restore that followed a Critical Save that
            //    followed a Regular Save, but make it do nothing.
            //

            if (PowerData->CriticalContextSaved == FALSE) {
                KeBugCheckEx(GPIO_CONTROLLER_DRIVER_ERROR,
                             INVALID_RESTORE_BANK_HARDWARE_CONTEXT,
                             (ULONG_PTR)CriticalTransition,
                             (ULONG_PTR)PowerData->RegularContextSaved,
                             (ULONG_PTR)PowerData->CriticalContextSaved);

            } else if (PowerData->RegularContextSaved != FALSE) {
                ShouldRestoreContext = FALSE;
            }

            PowerData->CriticalContextSaved = FALSE;
        }

        if (ShouldRestoreContext != FALSE) {
            GpioClnInvokeRestoreBankHardwareContext(GpioBank,
                                                    CriticalTransition);
        }

#if defined(EVENT_TRACING)

        TraceEvents(GpioExtension->LogHandle,
                    Info,
                    DBG_POWER,
                    "%s - %s RestoreBankHardwareContext() for Extn = %p, "
                    "Bank ID = 0x%x, Critical = %d\n",
                    __FUNCTION__,
                    ((ShouldRestoreContext == FALSE) ? "Ignored" : "Called"),
                    GpioExtension,
                    GpioBank->BankId,
                    CriticalTransition);

#endif

    }

    //
    // Set the new F-state for the bank.
    //

    GpioBank->PowerData.FState = State;

    //
    // Release the bank interrupt lock that was acquired.
    //

    if (AcquireInterruptLock != FALSE) {
        GPIO_RELEASE_BANK_INTERRUPT_LOCK(GpioBank);
    }

    //
    // If the bank is becoming inactive, then report the interrupts as inactive
    // also. This is done post calling into the client driver (to save
    // its context) as an interrupt can fire until the client driver has
    // saved its state and disabled all interrupts. Otherwise there is a
    // possibility an interrupt may fire without the handler being enabled.
    //
    // Note if this is a critical F1 transition, then no need to report
    // interrupt as inactive. This is because reporting active/idle is primarily
    // a hint to the PEP to select the appropriate idle state. In this case,
    // the PEP already has made a  decision to enter deep-idle state and thus
    // the call is not necessary. Besides, the incoming IRQL is HIGH_LEVEL
    // which the IoReport* APIs do not support.
    //

    if ((State == FStateF1) && (CriticalTransition == FALSE)) {
        GpiopSetInterruptState(GpioBank, GpioBank->BankInterrupt, FALSE);
        GpiopSetInterruptState(GpioBank, GpioBank->BankPassiveInterrupt, FALSE);
    }

    //
    // Notify (internal) client driver about FState transition completion.
    //

    GpioClnInvokePostBankFStateTransitionCallback(GpioBank, State);

#if defined(EVENT_TRACING)

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Transitioned bank into new FState! Extn = %p, "
                "Bank ID = 0x%x State = %d\n",
                __FUNCTION__,
                GpioExtension,
                GpioBank->BankId,
                State);

#endif

    return;
}

NTSTATUS
GpiopRegisterWithPowerFramework (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine registers the GPIO controller with the power framework. This
    enables the PEP to perform runtime power management on the GPIO controller.

    Each bank is registered as a separate component. This is a contract imposed
    by the GPIO class extension that the GPIO client driver and the PEP need to
    honor. For each bank, only F0 (fully-on) and F1 (off) are supported today.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    NTSTATUS code.

--*/

{

    ULONG Alignment;
    BANK_ID BankId;
    ULONG BaseSize;
    PGPIO_BANK_ENTRY GpioBank;
    PPO_FX_COMPONENT_IDLE_STATE IdleState;
    ULONG IdleStateSize;
    ULONG IdleStateStartOffset;
    PVOID RuntimePower;
    ULONG Size;
    NTSTATUS Status;
    ULONG TotalBanks;
    ULONG TotalFStates;

    PAGED_CODE();

    //
    // Reserve enough space for all power components and max number of idle
    // states for each component.
    //

    BaseSize = GpiopGetBaseRegistrationBufferSize();
    IdleStateStartOffset = BaseSize +
           (GpioExtension->TotalBanks - 1) * sizeof(PO_FX_COMPONENT);


    Alignment = TYPE_ALIGNMENT(PO_FX_COMPONENT_IDLE_STATE);


    IdleStateStartOffset =
        ALIGN_UP_BY(IdleStateStartOffset, Alignment);

    //
    // Compute the total F-states supported across all banks.
    //

    TotalBanks = GpioExtension->TotalBanks;
    TotalFStates = 0;
    for (BankId = 0; BankId < TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        TotalFStates += GpioBank->PowerData.MaximumFState + 1;
    }

    GPIO_ASSERT(TotalFStates <= (TotalBanks * GPIO_TOTAL_F_STATES));

    IdleStateSize = TotalFStates * sizeof(PO_FX_COMPONENT_IDLE_STATE);
    Size =  IdleStateStartOffset + IdleStateSize;

    //
    // Allocate the buffer used for registration with the power framework.
    //

    RuntimePower = GPIO_ALLOCATE_POOL(PagedPool, Size);
    if (RuntimePower == NULL) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_POWER,
                    "%s: Failed to allocate memory for runtime power management"
                    " registration buffer!\n",
                    __FUNCTION__);

        Status = STATUS_INSUFFICIENT_RESOURCES;
        goto RegisterWithPowerFrameworkEnd;
    }

    GPIO_ASSERT(GpioExtension->TotalPins >= 1);

    //
    // Specify active/idle callbacks and context.
    //

    RtlZeroMemory(RuntimePower, Size);
    GpiopInitializePrimaryDevice(RuntimePower,
                                 GpioExtension->TotalBanks,
                                 GpioClxBankActiveConditionCallback,
                                 GpioClxBankIdleConditionCallback,
                                 GpioClxBankIdleStateCallback,
                                 GpioClxDevicePowerRequiredCallback,
                                 GpioClxDevicePowerNotRequiredCallback,
                                 NULL,
                                 (PVOID)GpioExtension->Device,
                                 GpioClxCriticalTransitionCallback);

    //
    // Initialize F-state-related information
    //

    IdleState = (PPO_FX_COMPONENT_IDLE_STATE)
                Add2Ptr(RuntimePower, IdleStateStartOffset);

    for (BankId = 0; BankId < GpioExtension->TotalBanks; BankId += 1) {
        GpioBank = &GpioExtension->Banks[BankId];
        GpiopInitializeIdleStateBufferAndCount(
            RuntimePower,
            BankId,
            IdleState,
            GpioBank->PowerData.MaximumFState + 1);

        //
        // Initialize the transition latency, residency requirement and nominal
        // power values for F0 state.
        //
        // Use 0 for nominal power until a value is defined for "unknown".
        //

        IdleState->TransitionLatency = 0;
        IdleState->ResidencyRequirement = 0;
        IdleState->NominalPower = 0;
        IdleState = Add2Ptr(IdleState, sizeof(PO_FX_COMPONENT_IDLE_STATE));

        //
        // If the bank supports F1, then initialize the transition latency,
        // residency requirement and nominal power values for F1 state.
        //

        if (GpioBank->PowerData.MaximumFState == FStateF1) {
            *IdleState = GpioBank->PowerData.F1StateParameters;
            IdleState = Add2Ptr(IdleState, sizeof(PO_FX_COMPONENT_IDLE_STATE));
        }
    }

    //
    // Register with the power framework.
    //

    Status = GpiopRegisterPrimaryDeviceInternal(
                WdfDeviceWdmGetPhysicalDevice(GpioExtension->Device),
                (PVOID)RuntimePower,
                &GpioExtension->PowerHandle);

    if (!NT_SUCCESS(Status)) {
        TraceEvents(GpioExtension->LogHandle,
                    Error,
                    DBG_POWER,
                    "%s - PoFxRegisterDevice() failed with Status = %#x",
                    __FUNCTION__,
                    Status);

        goto RegisterWithPowerFrameworkEnd;
    }

RegisterWithPowerFrameworkEnd:
    if (RuntimePower != NULL) {
        GPIO_FREE_POOL(RuntimePower);
    }

    return Status;
}

VOID
GpiopStartRuntimePowerManagement (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine requests power framework to start runtime power management
    of the GPIO controller.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{

    NTSTATUS Status;

    PAGED_CODE();

    //
    // Request WDF to stop idle power management on the device (Dx IRPs). It
    // will be restarted when all the banks go idle.
    //
    // This needs to be done prior to registering with the power framework.
    // Otherwise, PEP may callback into the driver requesting device power not
    // required, which calls WdfDeviceResumeIdle(). The WdfDeviceResumeIdle()
    // could race ahead of the WdfDeviceStopIdle() call.
    //

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Calling WdfStopIdle() to stop idle power mgmt..."
                "calling PoFxStartDevicePowerManagement()! Extn = %p\n",
                __FUNCTION__,
                GpioExtension);

    Status = WdfDeviceStopIdle(GpioExtension->Device, FALSE);

    GPIO_ASSERT((Status == STATUS_SUCCESS) || (Status == STATUS_PENDING));

    //
    // Request the power framework to start runtime power management
    //

    GPIO_ASSERT(GpioExtension->PowerHandle != NULL);

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Starting runtime power mgmt..."
                "calling PoFxStartDevicePowerManagement()! Extn = %p\n",
                __FUNCTION__,
                GpioExtension);

    PoFxStartDevicePowerManagement(GpioExtension->PowerHandle);

    return;
}

VOID
GpiopUnregisterWithPowerFramework (
    __in PDEVICE_EXTENSION GpioExtension
    )

/*++

Routine Description:

    This routine unregisters the GPIO controller from the runtime power
    framework.

Arguments:

    GpioExtension - Supplies a pointer to the GPIO's device extension.

Return Value:

    None.

--*/

{

    PAGED_CODE();

    //
    // This blocks until all components are put into F0.
    //

    if (GpioExtension->PowerHandle != NULL) {
        PoFxUnregisterDevice(GpioExtension->PowerHandle);
        GpioExtension->PowerHandle = NULL;
    }

    TraceEvents(GpioExtension->LogHandle,
                Info,
                DBG_POWER,
                "%s - Stopping runtime power mgmt..."
                "PoFxUnregisterDevice() called!\n",
                __FUNCTION__);

    return;
}


