/* Copyright (c) 1995-2000 Microsoft Corporation.  All rights reserved. */

/*
    mdmem.h - machine dependant memory management structures
*/

#ifndef MEM_PPC_H
#define MEM_PPC_H

//	The user mode virtual address space is 2GB split into 64 sections
// of 512 blocks of 16 4K pages. For some platforms, the # of blocks in
// a section may be limited to fewer than 512 blocks to reduce the size
// of the intermeditate tables. E.G.: Since the PeRP only has 5MB of total
// RAM, the sections are limited to 4MB. This results in 64 block sections.
//
// Virtual address format:
//	3322222 222221111 1111 110000000000
//	1098765 432109876 5432 109876543210
//	zSSSSSS BBBBBBBBB PPPP oooooooooooo

#define BLOCK_MASK      0x1FF
#define SECTION_MASK	0x03F
#define RESERVED_SECTIONS 1     // reserve section 0 for current process

// Bit offsets of page, block & section in a virtual address:
#define VA_BLOCK        16
#define VA_SECTION      25

#if PPC403
#define DCB_SIZE        16      // IBM PPC403: 16 byte D-cache block size
#define PAGE_SIZE       1024    // IBM PPC403GC: 1K Pages
#define VA_PAGE         10
#define PAGE_MASK       0x03F
#elif PPC821
#define DCB_SIZE        16      // Motorola MPC821: 16 byte D-cache block size
#define PAGE_SIZE       4096    // Motorola MPC821: 4K
#define VA_PAGE         12
#define PAGE_MASK       0x00F
#endif

#define PAGES_PER_BLOCK  (0x10000 / PAGE_SIZE)

//
//  TLB Page permission bits:
//

#define PG_PERMISSION_MASK  (PAGE_SIZE-1)
#define PG_PHYS_ADDR_MASK   (~PG_PERMISSION_MASK)

#if PPC403                                          // IBM PPC403

//
//  The low 10 bits of the PPC403 PTE are:
//      ---------------------
//      |X|W|R|-|-|-|-|I|M|V|  where:   X = Execute
//      ---------------------           W = Write
//                                      R = Read (S/W bit)
//                                      I = (Cache) Inhibit
//                                      M = Execute Mask bit (S/W bit)
//                                      V = Valid
//

#define PG_VALID_MASK       0x00000001
#define PG_CACHE_MASK       0x00000004
#define PG_CACHE            0
#define PG_NOCACHE          0x00000004

#define PG_PROT_READ        0x00000080              // Implied bit on PPC403
#define PG_PROT_WRITE       0x00000380
#define PG_PROT_EXECUTE     0x00000200

#define PG_EXECUTE_MASK     0x00000202              // 0x00000002 Unused by PPC403 hardware
#define PG_DIRTY_MASK       (PG_PROT_WRITE | PG_PROT_EXECUTE) // writable pages must be executable

#define PG_PROTECTION       (PG_PROT_READ | PG_PROT_WRITE | PG_EXECUTE_MASK)

#define PG_READ_WRITE       (PG_VALID_MASK | PG_PROT_WRITE)

#define PG_WRITABLE_MASK    0x00000100

#elif PPC821                                        // Motorola MPC821

/* Page permission bits */
#define PG_VALID_MASK       0x000000F1
#define PG_CACHE_MASK       0x00000002
#define PG_CACHE            0
#define PG_NOCACHE          0x00000002
#define PG_DIRTY_MASK       0x00000100
#define PG_GLOBAL_MASK      0x00000004
#define PG_EXECUTE_MASK     0x00000000              // undefined on MPC821

#define PG_PROTECTION       0x00000E00
#define PG_PROT_READ        0x00000C00
#define PG_PROT_WRITE       0x00000800
#define PG_PROT_EXECUTE     0x00000000              // undefined on MPC821
#define PG_PROT_URO_KRW		0x00000400

#define PG_READ_WRITE       (PG_VALID_MASK | PG_DIRTY_MASK | PG_CACHE | PG_PROT_WRITE)

#endif  // Page Perms

#define PG_SIZE_MASK		0x00000000
#define PG_GUARD			PG_CACHE_MASK

// MemBlock structure layout
#define mb_lock     0
#define mb_uses     4
#define mb_flags    5
#define mb_ixBase   6
#define mb_hPf		8
#define mb_pages    12


#define INVALID_POINTER_VALUE	0xE0000000

#define UNKNOWN_BASE    (~0)
#define BAD_PAGE        (~0xffful)

#define API_MAX			0xFFFF0000

#ifndef VERIFY_READ_FLAG
#define VERIFY_READ_FLAG    0
#define VERIFY_EXECUTE_FLAG 0
#define VERIFY_WRITE_FLAG   1
#define VERIFY_KERNEL_OK    2
#endif

// Trap id values shared between C & ASM code. The codes are
// generated by the h/w vector address divided by 256.
#define ID_RESCHEDULE	0x00	// NOP used to force a reschedule
#define ID_HW_INTERRUPT	0x05	// external h/w interrupt
#define ID_ALIGNMENT	0x06	// data alignment error
#define ID_PROGRAM		0x07	// program error
#define ID_DECREMENTER	0x09	// decrementer counter underflow

#if defined(PPC403)
#define ID_DPAGE_FAULT  0x03    // data page fault
#define ID_CPAGE_FAULT  0x04    // code page fault
#elif defined(PPC821)
#define ID_CPAGE_FAULT	0x13	// code page fault
#define ID_DPAGE_FAULT	0x14	// data page fault
#endif

// The following codes are used internally by the kernel and are
// not releated to h/w interrupt addresses.
#define ID_STACK_FAULT	0x30
#define ID_HW_BREAK		0x31

//
//  Kernel Virtual Address translation values. Some systems will use an
//  OEMMemoryMap to translation the KVA space (0x8000.0000-0xbfff.ffff).
//  These constants define the granularity of translation of this space:
//

#if defined(PPC403) || defined(PPC821)
#define KVA_MAP                             // Use OEMMemoryMap for translation
#endif

#define KVA_SIZE    (512*1024*1024)             // 512MB KVA Space
#define KPFN_SIZE   (16*1024*1024)              // 16MB KVA Page Frame Size
#define KPFN_CNT    (KVA_SIZE / KPFN_SIZE)      // Number of PFNS in KVA space
#define KPFN_SHIFT	24	                        // PFN shift for 16MB PFNs

#define KPFN_MASK   ((KVA_SIZE-1) ^ (KPFN_SIZE-1))
#define KPFN(kva)   KPfnMap[((ulong)(kva) & KPFN_MASK) >> KPFN_SHIFT]


#ifndef ASM_ONLY

/* Test the read/write access on a page table entry */
#define IsPageReadable(entry)	(((entry)&PG_VALID_MASK) == PG_VALID_MASK)

#if PPC821
#define IsPageWritable(entry)	(((entry)&(PG_PROTECTION|PG_DIRTY_MASK)) \
		== (PG_PROT_WRITE|PG_DIRTY_MASK))
#else
#define IsPageWritable(entry)	(((entry)&(PG_WRITABLE_MASK)) != 0)
#endif

#define SWITCHKEY(oldval, newval) ((oldval) = (pCurThread)->aky, CurAKey = (pCurThread)->aky = (newval))
#define GETCURKEY() ((pCurThread)->aky)
#define SETCURKEY(newval) (CurAKey = (pCurThread)->aky = (newval))


#define PFN_INCR  PAGE_SIZE

//
//  Macros for KVA and PFN manipulation:
//

#ifdef KVA_MAP

#define GetPFN(pgpi)\
	((KPFN(pgpi) << KPFN_SHIFT) | ((ulong)(pgpi) & 0x00FFFC00))

extern struct
{
	DWORD	KernelVirtualAddress;
	DWORD	PhysicalAddress;
	DWORD	Size;
} OEMMemoryMap[KPFN_CNT];    // KVA Memory map defined in HAL

PVOID Phys2Virt(DWORD pfn);

#else

#define GetPFN(pgpi)    ((ulong)(pgpi) & 0x1FFFF000)
#define Phys2Virt(pfn)  ((PVOID)((pfn) | 0x80000000))       // one-to-one KVA

#endif

#define NextPFN(pfn)            ((pfn) + PFN_INCR)
#define PFNfromEntry(entry)     ((entry) & ~(PAGE_SIZE-1))
#define PFNfrom256(pg256)	    ((ulong)(pg256)<<8 & ~(PAGE_SIZE-1))


/* Page Table Entry for the kernel data page. On the PowerPC, this
 * page is located at physical 0x5000 and virtual 0x5000. The virtual
 * address is setup to be read-only for user mode and read-write for
 * kernel mode */

#if defined(PPC821)




#define KPAGE_PTE	((KPAGE_BASE+0x800) & 0x1FFFF000)\
	| PG_VALID_MASK | PG_GLOBAL_MASK | PG_CACHE | PG_PROT_URO_KRW

#endif // PPC821

#if defined(PPC403)
#define KPAGE_PTE   0                   // Locked KPage entry on the PPC403
#endif

#endif // not ASM_ONLY

#endif // MEM_PPC_H

