//+-------------------------------------------------------------------------
//
//  Microsoft Windows
//  Copyright (C) 1992, Microsoft Corporation.
//
//  File:       expst.cxx
//
//  Contents:   CExposedStream code
//
//  History:    28-Feb-92   PhilipLa    Created.
//
//--------------------------------------------------------------------------

#include <exphead.cxx>
#pragma hdrstop                 //  Remove for MAC build

#include <pbstream.hxx>
#include <expst.hxx>
#include <lock.hxx>
#include <seekptr.hxx>
#ifndef REF
#include <marshl.hxx>
#endif //!REF
#include <logfile.hxx>

// Maximum stream size supported by exposed streams
// This is MAX_ULONG with one subtracted so that
// the seek pointer has a spot to sit even at the
// end of the stream
#define CBMAXSTREAM 0xfffffffeUL
// Maximum seek pointer value
#define CBMAXSEEK (CBMAXSTREAM+1)

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::CExposedStream, public
//
//  Synopsis:   Empty object constructor
//
//  History:    30-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_CES)   // Expst_Init_TEXT
#endif


CExposedStream::CExposedStream(void)
{
    olDebugOut((DEB_ITRACE, "In  CExposedStream::CExposedStream()\n"));
#ifndef REF
    _pdfb = NULL;
    _ppc = NULL;
    _fOwnContext = FALSE;
#endif //!REF
    _cReferences = 0;
    _psp = NULL;
    _pst = NULL;
    olDebugOut((DEB_ITRACE, "Out CExposedStream::CExposedStream\n"));
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::InitWithSeekPtr, public
//
//  Synopsis:   Base constructor
//
//  Arguments:  [pst] - Public stream
//              [pdfb] - DocFile basis
//              [ppc] - Context
//              [fOwnContext] - Whether this object owns the context
//              [psp] - Seek pointer or NULL for new seek pointer
//
//  Returns:    Appropriate status code
//
//  History:    28-Feb-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Init)
#endif

SCODE CExposedStream::Init(CPubStream *pst,
#ifndef REF
                           CDFBasis *pdfb,
                           CPerContext *ppc,
                           BOOL fOwnContext,
#endif //!REF
                           CSeekPointer *psp)
{
    SCODE sc;

#ifndef REF
    olDebugOut((DEB_ITRACE, "In  CExposedStream::Init("
                "%p, %p, %p, %u, %p)\n",
                pst, pdfb, ppc, fOwnContext, psp));
#else
    olDebugOut((DEB_ITRACE, "In  CExposedStream::Init("
                "%p, %p)\n", pst, psp));
#endif //!REF
    if (psp == NULL)
    {
        CSeekPointer *pspTemp;
        olMem(pspTemp = new (pdfb->GetMalloc()) CSeekPointer(0));
        _psp = P_TO_BP(CSeekPointer DFBASED *, pspTemp);
    }
    else
        _psp = P_TO_BP(CSeekPointer DFBASED *, psp);
#ifndef REF
    _ppc = ppc;
    _fOwnContext = fOwnContext;
#endif //!REF
    _pst = P_TO_BP(CPubStream DFBASED *, pst);
#ifndef REF
    _pdfb = P_TO_BP(CDFBasis DFBASED *, pdfb);
    _pdfb->vAddRef();
#endif //!REF
    _cReferences = 1;
    _sig = CEXPOSEDSTREAM_SIG;
    olDebugOut((DEB_ITRACE, "Out CExposedStream::InitWithSeekPtr\n"));
    return S_OK;

EH_Err:
    return sc;
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::~CExposedStream, public
//
//  Synopsis:   Destructor
//
//  Returns:    Appropriate status code
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_1CES)  // Expst_Shutdown_TEXT
#endif

#ifndef REF
#ifndef FLAT
inline
#endif
#else
inline
#endif //!REF
CExposedStream::~CExposedStream(void)
{
    olDebugOut((DEB_ITRACE, "In  CExposedStream::~CExposedStream\n"));
    olAssert(_cReferences == 0);
    _sig = CEXPOSEDSTREAM_SIGDEL;
    if (_pst)
        _pst->CPubStream::vRelease();
    if (_psp)
        _psp->CSeekPointer::vRelease();
#ifndef REF
    if (_pdfb)
        _pdfb->CDFBasis::vRelease();
    if (_fOwnContext && _ppc)
        _ppc->Release();
#endif //!REF
    olDebugOut((DEB_ITRACE, "Out CExposedStream::~CExposedStream\n"));
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Read, public
//
//  Synopsis:   Read from a stream
//
//  Arguments:  [pb] - Buffer
//              [cb] - Count of bytes to read
//              [pcbRead] - Return number of bytes read
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pcbRead]
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Read)  //  Expst_Read_TEXT
#endif

STDMETHODIMP CExposedStream::Read(VOID HUGEP *pb, ULONG cb, ULONG *pcbRead)
{
    SCODE sc;
#ifndef REF
    SAFE_SEM;
    SAFE_ACCESS;
#endif //!REF
    ULONG cbRead = 0;

    olLog(("%p::In  CExposedStream::Read(%p, %lu, %p)\n",
           this, pb, cb, pcbRead));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Read(%p, %lu, %p)\n",
                 pb, cb, pcbRead));

    if (pcbRead)
    {
        olChkTo(EH_BadPtr, ValidateOutBuffer(pcbRead, sizeof(ULONG)));
    }

    olChk(ValidateHugeOutBuffer(pb, cb));
    olChk(Validate());
#ifndef REF
    olChk(TakeSafeSem());
    SafeReadAccess();
#endif //!REF
    sc = _pst->ReadAt(_psp->GetPos(), pb, cb, (ULONG STACKBASED *)&cbRead);
    olAssert(CBMAXSEEK-_psp->GetPos() >= cbRead);
    _psp->SetPos(_psp->GetPos()+cbRead);

    olDebugOut((DEB_TRACE, "Out CExposedStream::Read => %lu\n", cbRead));

EH_Err:
    if (pcbRead)
    {
        // May fault and leave stream seek pointer changed
        // This is acceptable
        *pcbRead = cbRead;
        olLog(("%p::Out CExposedStream::Read().  *pcbRead == %lu, ret = %lx\n",
               this, *pcbRead, sc));
    }
    else
    {
        olLog(("%p::Out CExposedStream::Read().  ret == %lx\n", this, sc));
    }

EH_BadPtr:
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Write, public
//
//  Synopsis:   Write to a stream
//
//  Arguments:  [pb] - Buffer
//              [cb] - Count of bytes to write
//              [pcbWritten] - Return of bytes written
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pcbWritten]
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Write)  //  Expst_Write_TEXT
#endif

STDMETHODIMP CExposedStream::Write(
        VOID const HUGEP *pb,
        ULONG cb,
        ULONG *pcbWritten)
{
    SCODE sc;
#ifndef REF
    SAFE_SEM;
    SAFE_ACCESS;
#endif //!REF
    ULONG cbWritten = 0;

    olLog(("%p::In  CExposedStream::Write(%p, %lu, %p)\n",
           this, pb, cb, pcbWritten));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Write(%p, %lu, %p)\n",
                pb, cb, pcbWritten));

    if (pcbWritten)
    {
        olChkTo(EH_BadPtr, ValidateOutBuffer(pcbWritten, sizeof(ULONG)));
    }
    olChk(ValidateHugeBuffer(pb, cb));
    olChk(Validate());
#ifndef REF
    olChk(TakeSafeSem());
    SafeWriteAccess();
#endif //!REF
    sc = _pst->WriteAt(_psp->GetPos(), pb, cb,
                       (ULONG STACKBASED *)&cbWritten);
    olAssert(CBMAXSEEK-_psp->GetPos() >= cbWritten);
    _psp->SetPos(_psp->GetPos()+cbWritten);

    olDebugOut((DEB_TRACE, "Out CExposedStream::Write => %lu\n",
                cbWritten));
EH_Err:
    if (pcbWritten)
    {
        // May fault but that's acceptable
        *pcbWritten = cbWritten;
        olLog(("%p::Out CExposedStream::Write().  "
               "*pcbWritten == %lu, ret = %lx\n",
               this, *pcbWritten, sc));
    }
    else
    {
        olLog(("%p::Out CExposedStream::Write().  ret == %lx\n", this, sc));
    }

EH_BadPtr:
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Seek, public
//
//  Synopsis:   Seek to a point in a stream
//
//  Arguments:  [dlibMove] - Offset to move by
//              [dwOrigin] - SEEK_SET, SEEK_CUR, SEEK_END
//              [plibNewPosition] - Return of new offset
//
//  Returns:    Appropriate status code
//
//  Modifies:   [plibNewPosition]
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Seek)  //
#endif

STDMETHODIMP CExposedStream::Seek(LARGE_INTEGER dlibMove,
                                  DWORD dwOrigin,
                                  ULARGE_INTEGER *plibNewPosition)
{
    SCODE sc;
#ifndef REF
    SAFE_SEM;
    SAFE_ACCESS;
#endif //!REF
    LONG lMove;
    ULARGE_INTEGER ulPos;

    olLog(("%p::In  CExposedStream::Seek(%ld, %lu, %p)\n",
           this, LIGetLow(dlibMove), dwOrigin, plibNewPosition));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Seek(%ld, %lu, %p)\n",
                LIGetLow(dlibMove), dwOrigin, plibNewPosition));

    if (plibNewPosition)
    {
        olChk(ValidateOutBuffer(plibNewPosition, sizeof(ULARGE_INTEGER)));
        ULISet32(*plibNewPosition, 0);
    }
    if (dwOrigin != STREAM_SEEK_SET && dwOrigin != STREAM_SEEK_CUR &&
        dwOrigin != STREAM_SEEK_END)
        olErr(EH_Err, STG_E_INVALIDFUNCTION);

    // Truncate dlibMove to 32 bits
    if (dwOrigin == STREAM_SEEK_SET)
    {
        // Make sure we don't seek too far
        if (LIGetHigh(dlibMove) != 0)
            LISet32(dlibMove, 0xffffffff);
    }
    else
    {
        // High dword must be zero for positive values or -1 for
        // negative values
        // Additionally, for negative values, the low dword can't
        // exceed -0x80000000 because the 32nd bit is the sign
        // bit
        if (LIGetHigh(dlibMove) > 0 ||
            (LIGetHigh(dlibMove) == 0 &&
             LIGetLow(dlibMove) >= 0x80000000))
            LISet32(dlibMove, 0x7fffffff);
        else if (LIGetHigh(dlibMove) < -1 ||
                 (LIGetHigh(dlibMove) == -1 &&
                  LIGetLow(dlibMove) <= 0x7fffffff))
            LISet32(dlibMove, 0x80000000);
    }

    lMove = (LONG)LIGetLow(dlibMove);
    olChk(Validate());
    olChk(_pst->CheckReverted());
#ifndef REF
    olChk(TakeSafeSem());
    SafeReadAccess();
#endif //!REF
    
    ULISet32(ulPos, _psp->GetPos());
    switch(dwOrigin)
    {
    case STREAM_SEEK_SET:
        ULISetLow(ulPos, (ULONG)lMove);
        break;
        
    case STREAM_SEEK_END:
        ULONG cbSize;
        olChk(_pst->GetSize(&cbSize));
        if (lMove < 0)
        {
            if ((ULONG)(-lMove) > cbSize)
                olErr(EH_Err, STG_E_INVALIDFUNCTION);
        }
        else if ((ULONG)lMove > CBMAXSEEK-cbSize)
            lMove = (LONG)(CBMAXSEEK-cbSize);
        ULISetLow(ulPos, cbSize+lMove);
        break;
        
    case STREAM_SEEK_CUR:
        if (lMove < 0)
        {
            if ((ULONG)(-lMove) > _psp->GetPos())
                olErr(EH_Err, STG_E_INVALIDFUNCTION);
        }
        else if ((ULONG)lMove > CBMAXSEEK-_psp->GetPos())
            lMove = (LONG)(CBMAXSEEK-_psp->GetPos());
        ULISetLow(ulPos, _psp->GetPos()+lMove);
        break;
    }
    _psp->SetPos(ULIGetLow(ulPos));
    
    if (plibNewPosition)
        // May fault but that's acceptable
        *plibNewPosition = ulPos;
    
    olDebugOut((DEB_TRACE, "Out CExposedStream::Seek => %lu\n",
                ULIGetLow(ulPos)));
EH_Err:
    olLog(("%p::Out CExposedStream::Seek().  ulPos == %lu,  ret == %lx\n",
           this, ULIGetLow(ulPos), sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::SetSize, public
//
//  Synopsis:   Sets the size of a stream
//
//  Arguments:  [ulNewSize] - New size
//
//  Returns:    Appropriate status code
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_SetSize)
#endif

STDMETHODIMP CExposedStream::SetSize(ULARGE_INTEGER ulNewSize)
{
    SCODE sc;
#ifndef REF
    SAFE_SEM;
    SAFE_ACCESS;
#endif //!REF

    olLog(("%p::In  CExposedStream::SetSize(%lu)\n",
           this, ULIGetLow(ulNewSize)));
    olDebugOut((DEB_TRACE, "In  CExposedStream::SetSize(%lu)\n",
                ULIGetLow(ulNewSize)));

    if (ULIGetHigh(ulNewSize) != 0)
        olErr(EH_Err, STG_E_INVALIDFUNCTION);
    olChk(Validate());
#ifndef REF
    olChk(TakeSafeSem());
    SafeWriteAccess();
#endif //!REF
    sc = _pst->SetSize(ULIGetLow(ulNewSize));

    olDebugOut((DEB_TRACE, "Out CExposedStream::SetSize\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::SetSize().  ret == %lx\n", this, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::CopyTo, public
//
//  Synopsis:   Copies information from one stream to another
//
//  Arguments:  [pstm] - Destination
//              [cb] - Number of bytes to copy
//              [pcbRead] - Return number of bytes read
//              [pcbWritten] - Return number of bytes written
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pcbRead]
//              [pcbWritten]
//
//  History:    25-Mar-92       DrewB   Created
//              12-Jan-93       AlexT   Rewritten without recursion
//
//  Notes:      We do our best to handle overlap correctly.  This allows
//              CopyTo to be used to insert and remove space within a
//              stream.
//
//              In the error case, we make no gurantees as to the
//              validity of pcbRead, pcbWritten, or either stream's
//              seek position.
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_CopyTo)    // Expst_CopyTo_TEXT
#endif

STDMETHODIMP CExposedStream::CopyTo(IStream *pstm,
                                    ULARGE_INTEGER cb,
                                    ULARGE_INTEGER *pcbRead,
                                    ULARGE_INTEGER *pcbWritten)
{
#ifndef REF
    SAFE_SEM;
#endif //!REF
    SCODE sc;
    ULONG ulCopySize;
    ULONG ulSrcSize;
    ULONG ulSrcOrig;
    ULARGE_INTEGER uliDestOrig;
    LARGE_INTEGER liDestPos;
    BYTE *pb = NULL;
    BOOL fOverlap;
    ULONG ulBytesCopied = 0;

    olLog(("%p::In  CExposedStream::CopyTo(%p, %lu, %p, %p)\n",
           this, pstm, ULIGetLow(cb), pcbRead, pcbWritten));
    olDebugOut((DEB_TRACE, "In  CExposedStream::CopyTo("
                "%p, %lu, %p, %p)\n", pstm, ULIGetLow(cb),
                pcbRead, pcbWritten));

    if (pcbRead)
    {
        olChk(ValidateOutBuffer(pcbRead, sizeof(ULARGE_INTEGER)));
        ULISet32(*pcbRead, 0);
    }
    if (pcbWritten)
    {
        olChk(ValidateOutBuffer(pcbWritten, sizeof(ULARGE_INTEGER)));
        ULISet32(*pcbWritten, 0);
    }

    // We don't touch the out parameters during the actual work so we
    // don't expect any faults until the end
    
    olChk(ValidateInterface(pstm, IID_IStream));
    olChk(Validate());

    //  Bound the size of the copy
    //  1.  The maximum we can copy is 0xffffffff
    if (ULIGetHigh(cb) == 0)
        ulCopySize = ULIGetLow(cb);
    else
        ulCopySize = 0xffffffff;

    olChk(TakeSafeSem());
    
    //  2.  We can only copy what's available in the source stream
    SetReadAccess();
    sc = _pst->GetSize(&ulSrcSize);
    ClearReadAccess();
    olChk(sc);

    ulSrcOrig = _psp->GetPos();
    if (ulSrcSize < ulSrcOrig)
    {
        //  Nothing in source to copy
        ulCopySize = 0;
    }
    else if ((ulSrcSize - ulSrcOrig) < ulCopySize)
    {
        //  Shrink ulCopySize to fit bytes in source
        ulCopySize = ulSrcSize - ulSrcOrig;
    }

    //  3.  We can only copy what will fit in the destination
    LISet32(liDestPos, 0);
    olHChk(pstm->Seek(liDestPos, STREAM_SEEK_CUR, &uliDestOrig));
    olAssert(ULIGetHigh(uliDestOrig) == 0);

    if (ulCopySize > CBMAXSEEK - ULIGetLow(uliDestOrig))
        ulCopySize = CBMAXSEEK - ULIGetLow(uliDestOrig);

    //  We are allowed to fail here with out-of-memory
    olMem(pb = (BYTE *) DfMemAlloc(STREAMBUFFERSIZE));

    // Since we have no reliable way to determine if the source and
    // destination represent the same stream, we assume they
    // do and always handle overlap.

    fOverlap = (ULIGetLow(uliDestOrig) > ulSrcOrig &&
                ULIGetLow(uliDestOrig) < ulSrcOrig + ulCopySize);

    ULONG ulSrcCopyOffset;
    ULONG ulDstCopyOffset;
    if (fOverlap)
    {
        //  We're going to copy back to front, so determine the
        //  stream end positions
        ulSrcCopyOffset = ulSrcOrig + ulCopySize;

        //  uliDestOrig is the destination starting offset
        ulDstCopyOffset = ULIGetLow(uliDestOrig) + ulCopySize;
    }

    while (ulCopySize > 0)
    {
        //  We can only copy up to STREAMBUFFERSIZE bytes at a time
        ULONG cbPart = min(ulCopySize, STREAMBUFFERSIZE);

        if (fOverlap)
        {
            //  We're copying back to front so we need to seek to
            //  set up the streams correctly

            ulSrcCopyOffset -= cbPart;
            ulDstCopyOffset -= cbPart;

            //  Set source stream position
            _psp->SetPos(ulSrcCopyOffset);

            //  Set destination stream position
            LISet32(liDestPos, ulDstCopyOffset);
            olHChk(pstm->Seek(liDestPos, STREAM_SEEK_SET, NULL));
        }
        
        ULONG ulRead;
        olHChk(Read(pb, cbPart, &ulRead));
        if (cbPart != ulRead)
        {
            //  There was no error, but we were unable to read cbPart
            //  bytes.  Something's wrong (the underlying ILockBytes?)
            //  but we can't control it;  just return an error.
            olErr(EH_Err, STG_E_READFAULT);
        }
        
        ULONG ulWritten;
        olHChk(pstm->Write(pb, cbPart, &ulWritten));
        if (cbPart != ulWritten)
        {
            //  There was no error, but we were unable to write
            //  ulWritten bytes.  We can't trust the pstm
            //  implementation, so all we can do here is return
            //  an error.
            olErr(EH_Err, STG_E_WRITEFAULT);
        }
        
        olAssert(ulCopySize >= cbPart);
        ulCopySize -= cbPart;
        ulBytesCopied += cbPart;
    }
    
    if (fOverlap)
    {
        //  Set the seek pointers to the correct location
        _psp->SetPos(ulSrcOrig + ulBytesCopied);
        
        LISet32(liDestPos, ULIGetLow(uliDestOrig) + ulBytesCopied);
        olHChk(pstm->Seek(liDestPos, STREAM_SEEK_SET, NULL));
    }
    
    olDebugOut((DEB_TRACE, "Out CExposedStream::CopyTo => %lu, %lu\n",
                pcbRead ? ULIGetLow(*pcbRead) : 0,
                pcbWritten ? ULIGetLow(*pcbWritten) : 0));
    // Fall through
    
EH_Err:
    DfMemFree(pb);

    // We can fault here, but that's acceptable
    if (pcbRead)
        ULISet32(*pcbRead, ulBytesCopied);
    if (pcbWritten)
        ULISet32(*pcbWritten, ulBytesCopied);

    olLog(("%p::Out CExposedStream::CopyTo().  "
           "cbRead == %lu, cbWritten == %lu, ret == %lx\n",
           this, pcbRead ? ULIGetLow(*pcbRead) : 0,
           pcbWritten ? ULIGetLow(*pcbWritten) : 0, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Release, public
//
//  Synopsis:   Releases a stream
//
//  Returns:    Appropriate status code
//
//  History:    28-Feb-92       DrewB   Created from pbstream source
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Release)    // Expst_Release_TEXT
#endif

STDMETHODIMP_(ULONG) CExposedStream::Release(void)
{
    LONG lRet;

    olLog(("%p::In  CExposedStream::Release()\n", this));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Release()\n"));

    if (FAILED(Validate()))
        return 0;
    olAssert(_cReferences > 0);
    lRet = AtomicDec(&_cReferences);
    if (lRet == 0)
    {
#ifndef REF
        if (_fOwnContext || SUCCEEDED(_pst->CheckReverted()))
        {
            _pdfb->SetContext(_ppc);
        }
#endif //!REF
        delete this;
    }
    else if (lRet < 0)
        lRet = 0;

    olDebugOut((DEB_TRACE, "Out CExposedStream::Release\n"));
    olLog(("%p::Out CExposedStream::Release().  ret == %lu\n", this, lRet));
    FreeLogFile();
    return lRet;
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Stat, public
//
//  Synopsis:   Fills in a buffer of information about this object
//
//  Arguments:  [pstatstg] - Buffer
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pstatstg]
//
//  History:    24-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Stat)    // Stat_TEXT
#endif

_OLESTDMETHODIMP CExposedStream::Stat(STATSTGW *pstatstg, DWORD grfStatFlag)
{
    SCODE sc;
    SAFE_SEM;
    SAFE_ACCESS;
    STATSTGW stat;

    olLog(("%p::In  CExposedStream::Stat(%p)\n", this, pstatstg));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Stat(%p)\n",
                pstatstg));

    olChkTo(EH_RetSc, ValidateOutBuffer(pstatstg, sizeof(STATSTGW)));
    olChk(VerifyStatFlag(grfStatFlag));
    olChk(Validate());
#ifndef REF
    olChk(TakeSafeSem());
    SafeReadAccess();
#endif //!REF

    olChk(_pst->Stat(&stat, grfStatFlag));
    TRY
    {
        *pstatstg = stat;
        pstatstg->type = STGTY_STREAM;
        pstatstg->grfLocksSupported = 0;
        pstatstg->STATSTG_reserved = 0;
        pstatstg->ctime.dwLowDateTime = pstatstg->ctime.dwHighDateTime = 0;
        pstatstg->mtime.dwLowDateTime = pstatstg->mtime.dwHighDateTime = 0;
        pstatstg->atime.dwLowDateTime = pstatstg->atime.dwHighDateTime = 0;
    }
    CATCH(CException, e)
    {
        UNREFERENCED_PARM(e);
        if (stat.pwcsName)
            TaskMemFree(stat.pwcsName);
        sc = STG_E_INVALIDPOINTER;
    }
    END_CATCH

    olDebugOut((DEB_TRACE, "Out CExposedStream::Stat\n"));
    // Fall through
    
EH_Err:
#ifndef WIN32
    // This isn't safe for platforms where pstatstg can be invalid
    if (FAILED(sc))
        memset(pstatstg, 0, sizeof(STATSTGW));
#endif
EH_RetSc:
    olLog(("%p::Out CExposedStream::Stat().  ret == %lx\n",
           this, sc));
    return _OLERETURN(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Clone, public
//
//  Synopsis:   Clones a stream
//
//  Returns:    Appropriate status code
//
//  History:    28-Feb-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Clone)    //
#endif

STDMETHODIMP CExposedStream::Clone(IStream **ppstm)
{
    SafeCExposedStream pst;
    CSeekPointer *psp;
    SCODE sc;
#ifndef REF
    SAFE_SEM;
    SAFE_ACCESS;
#endif //!REF

    olLog(("%p::In  CExposedStream::Clone(%p)\n", this, ppstm));
    olDebugOut((DEB_TRACE, "In  CExposedStream::Clone(%p)\n", ppstm));

    olChk(ValidateOutPtrBuffer(ppstm));
    *ppstm = NULL;
    olChk(Validate());
    olChk(_pst->CheckReverted());
#ifndef REF
    olChk(TakeSafeSem());
    SafeReadAccess();
#endif //!REF
    olMem(psp = new (_pdfb->GetMalloc()) CSeekPointer(_psp->GetPos()));
    pst.Attach(new (_pdfb->GetMalloc()) CExposedStream);
    olMemTo(EH_psp, (CExposedStream *)pst);
#ifndef REF
    olChkTo(EH_pst, pst->Init(BP_TO_P(CPubStream *, _pst),
                              BP_TO_P(CDFBasis *, _pdfb),
                              _ppc, TRUE, psp));
    _ppc->AddRef();
#else
    olChkTo(EH_pst, pst->Init(_pst, psp));
#endif //!REF
    _pst->vAddRef();
    
    TRANSFER_INTERFACE(pst, IStream, ppstm);

    olDebugOut((DEB_TRACE, "Out CExposedStream::Clone => %p\n", *ppstm));

 EH_Err:
    olLog(("%p::Out CExposedStream::Clone().  *ppstm == %p, ret == %lx\n",
           this, *ppstm, sc));
    return ResultFromScode(sc);

EH_pst:
    delete pst;
EH_psp:
    psp->vRelease();
    goto EH_Err;
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::AddRef, public
//
//  Synopsis:   Increments the ref count
//
//  Returns:    Appropriate status code
//
//  History:    16-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_AddRef)
#endif

STDMETHODIMP_(ULONG) CExposedStream::AddRef(void)
{
    ULONG ulRet;

    olLog(("%p::In  CExposedStream::AddRef()\n", this));
    olDebugOut((DEB_TRACE, "In  CExposedStream::AddRef()\n"));

    if (FAILED(Validate()))
        return 0;
    AtomicInc(&_cReferences);
    ulRet = _cReferences;

    olDebugOut((DEB_TRACE, "Out CExposedStream::AddRef\n"));
    olLog(("%p::Out CExposedStream::AddRef().  ret == %lu\n", this, ulRet));
    return ulRet;
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::LockRegion, public
//
//  Synopsis:   Nonfunctional
//
//  Returns:    Appropriate status code
//
//  History:    16-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_LockRegion)
#endif

STDMETHODIMP CExposedStream::LockRegion(ULARGE_INTEGER libOffset,
                                        ULARGE_INTEGER cb,
                                        DWORD dwLockType)
{
    olDebugOut((DEB_TRACE, "In  CExposedStream::LockRegion("
                "%lu, %lu\n", ULIGetLow(cb), dwLockType));
    olDebugOut((DEB_TRACE, "Out CExposedStream::LockRegion\n"));
    olLog(("%p::INVALID CALL TO CExposedStream::LockRegion()\n"));
    return ResultFromScode(STG_E_INVALIDFUNCTION);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::UnlockRegion, public
//
//  Synopsis:   Nonfunctional
//
//  Returns:    Appropriate status code
//
//  History:    16-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_UnlockRegion)
#endif

STDMETHODIMP CExposedStream::UnlockRegion(ULARGE_INTEGER libOffset,
                                          ULARGE_INTEGER cb,
                                          DWORD dwLockType)
{
    olDebugOut((DEB_TRACE, "In  CExposedStream::UnlockRegion(%lu, %lu)\n",
                ULIGetLow(cb), dwLockType));
    olDebugOut((DEB_TRACE, "Out CExposedStream::UnlockRegion\n"));
    olLog(("%p::INVALID CALL TO CExposedStream::UnlockRegion()\n"));
    return ResultFromScode(STG_E_INVALIDFUNCTION);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Commit, public
//
//  Synopsis:   No-op in current implementation
//
//  Returns:    Appropriate status code
//
//  History:    16-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Commit)    // Expst_Commit_TEXT
#endif

STDMETHODIMP CExposedStream::Commit(DWORD grfCommitFlags)
{
    SCODE sc;
    SAFE_SEM;
    SAFE_ACCESS;
    
    olDebugOut((DEB_TRACE, "In  CExposedStream::Commit(%lu)\n",
                grfCommitFlags));
    olLog(("%p::In  CExposedStream::Commit(%lx)\n", this, grfCommitFlags));

    olChk(Validate());
#ifndef REF
    olChk(TakeSafeSem());
    SafeWriteAccess();
#endif //!REF
    
    sc = _pst->Commit(grfCommitFlags);
    
    olDebugOut((DEB_TRACE, "Out CExposedStream::Commit\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::Commit().  ret == %lx\n", this, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Revert, public
//
//  Synopsis:   No-op in current implementation
//
//  Returns:    Appropriate status code
//
//  History:    16-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Revert)    // Expst_Revert_TEXT
#endif

STDMETHODIMP CExposedStream::Revert(void)
{
    SCODE sc;

    olDebugOut((DEB_TRACE, "In  CExposedStream::Revert()\n"));
    sc = _pst->CheckReverted();
    olDebugOut((DEB_TRACE, "Out CExposedStream::Revert\n"));
    olLog(("%p::In  CExposedStream::Revert()\n", this));
    olLog(("%p::Out CExposedStream::Revert().  ret == %lx", this, sc));

    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::QueryInterface, public
//
//  Synopsis:   Returns an object for the requested interface
//
//  Arguments:  [iid] - Interface ID
//              [ppvObj] - Object return
//
//  Returns:    Appropriate status code
//
//  Modifies:   [ppvObj]
//
//  History:    26-Mar-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_QueryInterface)    //
#endif

STDMETHODIMP CExposedStream::QueryInterface(REFIID iid, void **ppvObj)
{
    SCODE sc;

    olLog(("%p::In  CExposedStream::QueryInterface(?, %p)\n",
           this, ppvObj));
    olDebugOut((DEB_TRACE, "In  CExposedStream::QueryInterface(?, %p)\n",
                ppvObj));

    olChk(ValidateOutPtrBuffer(ppvObj));
    *ppvObj = NULL;
    olChk(ValidateIid(iid));
    olChk(Validate());
    olChk(_pst->CheckReverted());

    sc = S_OK;
    if (IsEqualIID(iid, IID_IStream) || IsEqualIID(iid, IID_IUnknown))
    {
        *ppvObj = (IStream *)this;
        AddRef();
    }
#ifndef REF
    else if (IsEqualIID(iid, IID_IMarshal))
    {
        *ppvObj = (IMarshal *)this;
        AddRef();
    }
#endif //!REF
    else
        sc = E_NOINTERFACE;

    olDebugOut((DEB_TRACE, "Out CExposedStream::QueryInterface => %p\n",
                ppvObj));
EH_Err:
    olLog(("%p::Out CExposedStream::QueryInterface().  "
           "*ppvObj == %p, ret == %lx\n", this, *ppvObj, sc));
    return ResultFromScode(sc);
}

#ifndef REF
//+--------------------------------------------------------------
//
//  Member:     CExposedStream::Unmarshal, public
//
//  Synopsis:   Creates a duplicate stream from parts
//
//  Arguments:  [pstm] - Marshal stream
//              [ppv] - Object return
//              [mshlflags] - Marshal flags
//
//  Returns:    Appropriate status code
//
//  Modifies:   [ppv]
//
//  History:    26-Feb-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_Unmarshal)    // Marshal_TEXT
#endif

SCODE CExposedStream::Unmarshal(IStream *pstm,
                                void **ppv,
                                DWORD mshlflags)
{
    SCODE sc;
    CPerContext *ppc;
    CPubStream *pst;
    CDFBasis *pdfb;
    CExposedStream *pest;
    CSeekPointer *psp;

    olDebugOut((DEB_ITRACE, "In  CExposedStream::Unmarshal(%p, %p, %lu)\n",
                pstm, ppv, mshlflags));
    
    olChk(UnmarshalPointer(pstm, (void **)&pst));
    olChk(ValidateBuffer(pstm, sizeof(CPubStream)));
    olChkTo(EH_pst, UnmarshalPointer(pstm, (void **)&pdfb));
    olChkTo(EH_pdfb, UnmarshalPointer(pstm, (void **)&psp));
    olChkTo(EH_psp, UnmarshalContext(pstm, &ppc, mshlflags, FALSE, FALSE));
    olMemTo(EH_ppc, pest = new (pdfb->GetMalloc()) CExposedStream);
    olChkTo(EH_pest, pest->Init(pst, pdfb, ppc, TRUE, psp));
    
    pst->vAddRef();
    psp->vAddRef();
    *ppv = pest;

    olDebugOut((DEB_ITRACE, "Out CExposedStream::Unmarshal => %p\n", *ppv));
    return S_OK;

 EH_pest:
    delete pest;
 EH_ppc:
    ppc->Release();
 EH_psp:
 EH_pdfb:
 EH_pst:
 EH_Err:
    return sc;
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::GetUnmarshalClass, public
//
//  Synopsis:   Returns the class ID
//
//  Arguments:  [riid] - IID of object
//              [pv] - Unreferenced
//              [dwDestContext] - Unreferenced
//              [pvDestContext] - Unreferenced
//              [mshlflags] - Unreferenced
//              [pcid] - CLSID return
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pcid]
//
//  History:    04-May-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_GetUnmarshalClass)
#endif

STDMETHODIMP CExposedStream::GetUnmarshalClass(REFIID riid,
                                               void *pv,
                                               DWORD dwDestContext,
                                               LPVOID pvDestContext,
                                               DWORD mshlflags,
                                               LPCLSID pcid)
{
    SCODE sc;

    olLog(("%p::In  CExposedStream::GetUnmarshalClass("
           "riid, %p, %lu, %p, %lu, %p)\n",
           this, pv, dwDestContext, pvDestContext, mshlflags, pcid));
    olDebugOut((DEB_TRACE, "In  CExposedStream::GetUnmarshalClass:%p("
                "riid, %p, %lu, %p, %lu, %p)\n", this, pv, dwDestContext,
                pvDestContext, mshlflags, pcid));
    
    UNREFERENCED_PARM(pv);
    UNREFERENCED_PARM(mshlflags);
    
    olChk(ValidateOutBuffer(pcid, sizeof(CLSID)));
    memset(pcid, 0, sizeof(CLSID));
    olChk(ValidateIid(riid));
    olChk(Validate());
    olChk(_pst->CheckReverted());

#if WIN32 != 300
    if (dwDestContext != 0 || pvDestContext != NULL)
#else
    if (dwDestContext != 0)
    {
        IMarshal *pmsh;
        
        if (SUCCEEDED(sc = CoGetStandardMarshal(riid, (IUnknown *)pv,
                                                dwDestContext, pvDestContext,
                                                mshlflags, &pmsh)))
        {
            sc = GetScode(pmsh->GetUnmarshalClass(riid, pv, dwDestContext,
                                                  pvDestContext, mshlflags,
                                                  pcid));
            pmsh->Release();
        }
    }
    else if (pvDestContext != NULL)
#endif
    {
        sc = STG_E_INVALIDPARAMETER;
    }
    else
    {
        *pcid = CLSID_DfMarshal;
    }

    olDebugOut((DEB_TRACE, "Out CExposedStream::GetUnmarshalClass\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::GetUnmarshalClass().  ret == %lx\n",
           this, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::GetMarshalSizeMax, public
//
//  Synopsis:   Returns the size needed for the marshal buffer
//
//  Arguments:  [riid] - IID of object being marshaled
//              [pv] - Unreferenced
//              [dwDestContext] - Unreferenced
//              [pvDestContext] - Unreferenced
//              [mshlflags] - Marshal flags
//              [pcbSize] - Size return
//
//  Returns:    Appropriate status code
//
//  Modifies:   [pcbSize]
//
//  History:    04-May-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_GetMarshalSizeMax)
#endif

STDMETHODIMP CExposedStream::GetMarshalSizeMax(REFIID riid,
                                               void *pv,
                                               DWORD dwDestContext,
                                               LPVOID pvDestContext,
                                               DWORD mshlflags,
                                               LPDWORD pcbSize)
{
    SCODE sc;

    UNREFERENCED_PARM(pv);
    olLog(("%p::In  CExposedStream::GetMarshalSizeMax("
           "riid, %p, %lu, %p, %lu, %p)\n",
           this, pv, dwDestContext, pvDestContext, mshlflags, pcbSize));
    olDebugOut((DEB_TRACE, "In  CExposedStream::GetMarshalSizeMax:%p("
                "riid, %p, %lu, %p, %lu, %p)\n", this, pv, dwDestContext,
                pvDestContext, mshlflags, pcbSize));
    
    olChk(Validate());
    olChk(_pst->CheckReverted());
    
#if WIN32 != 300
    if (dwDestContext != 0 || pvDestContext != NULL)
#else
    if (dwDestContext != 0)
    {
        IMarshal *pmsh;
        
        if (SUCCEEDED(sc = CoGetStandardMarshal(riid, (IUnknown *)pv,
                                                dwDestContext, pvDestContext,
                                                mshlflags, &pmsh)))
        {
            sc = GetScode(pmsh->GetMarshalSizeMax(riid, pv, dwDestContext,
                                                  pvDestContext, mshlflags,
                                                  pcbSize));
            pmsh->Release();
        }
    }
    else if (pvDestContext != NULL)
#endif
    {
        sc = STG_E_INVALIDPARAMETER;
    }
    else
    {
        sc = GetStdMarshalSize(riid, IID_IStream, dwDestContext, pvDestContext,
                               mshlflags, pcbSize,
                               sizeof(CPubStream *)+sizeof(CDFBasis *)+
                               sizeof(CSeekPointer *), _ppc, FALSE);
    }

    olDebugOut((DEB_TRACE, "Out CExposedStream::GetMarshalSizeMax\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::GetMarshalSizeMax().  *pcbSize == %lu, "
           "ret == %lx\n", this, *pcbSize, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::MarshalInterface, public
//
//  Synopsis:   Marshals a given object
//
//  Arguments:  [pstStm] - Stream to write marshal data into
//              [riid] - Interface to marshal
//              [pv] - Unreferenced
//              [dwDestContext] - Unreferenced
//              [pvDestContext] - Unreferenced
//              [mshlflags] - Marshal flags
//
//  Returns:    Appropriate status code
//
//  History:    04-May-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_MarshalInterface)
#endif

STDMETHODIMP CExposedStream::MarshalInterface(IStream *pstStm,
                                              REFIID riid,
                                              void *pv,
                                              DWORD dwDestContext,
                                              LPVOID pvDestContext,
                                              DWORD mshlflags)
{
    SCODE sc;

    olLog(("%p::In  CExposedStream::MarshalInterface("
           "%p, riid, %p, %lu, %p, %lu).  Context == %lX\n",
           this, pstStm, pv, dwDestContext, pvDestContext,
           mshlflags, (ULONG)GetCurrentContextId()));
    olDebugOut((DEB_TRACE, "In  CExposedStream::MarshalInterface:%p("
                "%p, riid, %p, %lu, %p, %lu)\n", this, pstStm, pv,
                dwDestContext, pvDestContext, mshlflags));
    
    UNREFERENCED_PARM(pv);
    
    olChk(Validate());
    olChk(_pst->CheckReverted());

#if WIN32 != 300
    if (dwDestContext != 0 || pvDestContext != NULL)
#else
    if (dwDestContext != 0)
    {
        IMarshal *pmsh;
        
        if (SUCCEEDED(sc = CoGetStandardMarshal(riid, (IUnknown *)pv,
                                                dwDestContext, pvDestContext,
                                                mshlflags, &pmsh)))
        {
            sc = GetScode(pmsh->MarshalInterface(pstStm, riid, pv,
                                                 dwDestContext, pvDestContext,
                                                 mshlflags));
            pmsh->Release();
        }
    }
    else if (pvDestContext != NULL)
#endif
    {
        sc = STG_E_INVALIDPARAMETER;
    }
    else
    {
        olChk(StartMarshal(pstStm, riid, IID_IStream, mshlflags));
        olChk(MarshalPointer(pstStm, BP_TO_P(CPubStream *, _pst)));
        olChk(MarshalPointer(pstStm, BP_TO_P(CDFBasis *, _pdfb)));
        olChk(MarshalPointer(pstStm, BP_TO_P(CSeekPointer *, _psp)));
        sc = MarshalContext(pstStm, _ppc, dwDestContext, pvDestContext,
                            mshlflags, FALSE);
        
#ifdef WIN32
        if (SUCCEEDED(sc) && mshlflags != MSHLFLAGS_TABLEWEAK)
        {
            _pst->vAddRef();
            _pdfb->vAddRef();
            _psp->vAddRef();
        }
#endif
    }

    olDebugOut((DEB_TRACE, "Out CExposedStream::MarshalInterface\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::MarshalInterface().  ret == %lx\n",
           this, sc));
    return ResultFromScode(sc);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::UnmarshalInterface, public
//
//  Synopsis:   Non-functional
//
//  Arguments:  [pstStm] -
//              [riid] -
//              [ppvObj] -
//
//  Returns:    Appropriate status code
//
//  Modifies:   [ppvObj]
//
//  History:    04-May-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_UnmarshalInterface)    // unused
#endif

STDMETHODIMP CExposedStream::UnmarshalInterface(IStream *pstStm,
                                                REFIID riid,
                                                void **ppvObj)
{
    olLog(("%p::INVALID CALL TO CExposedStream::UnmarshalInterface()\n"));
    return ResultFromScode(STG_E_INVALIDFUNCTION);
}

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::StaticReleaseMarshalData, public static
//
//  Synopsis:   Releases any references held in marshal data
//
//  Arguments:  [pstStm] - Marshal data stream
//
//  Returns:    Appropriate status code
//
//  History:    02-Feb-94       DrewB   Created
//
//  Notes:      Assumes standard marshal header has already been read
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_ReleaseMarshalData)
#endif

#ifdef WIN32
SCODE CExposedStream::StaticReleaseMarshalData(IStream *pstStm,
                                               DWORD mshlflags)
{
    SCODE sc;
    CPubStream *pst;
    CDFBasis *pdfb;
    CSeekPointer *psp;

    olDebugOut((DEB_ITRACE, "In  CExposedStream::StaticReleaseMarshalData:("
                "%p, %lX)\n", pstStm, mshlflags));
    
    olChk(UnmarshalPointer(pstStm, (void **)&pst));
    olChk(UnmarshalPointer(pstStm, (void **)&pdfb));
    olChk(UnmarshalPointer(pstStm, (void **)&psp));
    sc = ReleaseContext(pstStm, FALSE, mshlflags);
    
    if (SUCCEEDED(sc) && mshlflags != MSHLFLAGS_TABLEWEAK)
    {
        pst->vRelease();
        pdfb->vRelease();
        psp->vRelease();
    }

    olDebugOut((DEB_ITRACE,
                "Out CExposedStream::StaticReleaseMarshalData\n"));
EH_Err:
    return sc;
}
#endif

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::ReleaseMarshalData, public
//
//  Synopsis:   Non-functional
//
//  Arguments:  [pstStm] - Stream
//
//  Returns:    Appropriate status code
//
//  History:    18-Sep-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_ReleaseMarshalData)    // unused
#endif

#ifdef WIN32
STDMETHODIMP CExposedStream::ReleaseMarshalData(IStream *pstStm)
{
    SCODE sc;
    DWORD mshlflags;
    IID iid;

    olLog(("%p::In  CExposedStream::ReleaseMarshalData(%p)\n", this, pstStm));
    olDebugOut((DEB_TRACE, "In  CExposedStream::ReleaseMarshalData:%p(%p)\n",
                this, pstStm));

    olChk(Validate());
    olChk(_pst->CheckReverted());
    olChk(SkipStdMarshal(pstStm, &iid, &mshlflags));
    olAssert(IsEqualIID(iid, IID_IStream));
    sc = StaticReleaseMarshalData(pstStm, mshlflags);

    olDebugOut((DEB_TRACE, "Out CExposedStream::ReleaseMarshalData\n"));
EH_Err:
    olLog(("%p::Out CExposedStream::ReleaseMarshalData().  ret == %lx\n",
           this, sc));
    return ResultFromScode(sc);
}
#else
STDMETHODIMP CExposedStream::ReleaseMarshalData(IStream *pstStm)
{
    olLog(("%p::INVALID CALL TO CExposedStream::ReleaseMarshalData()\n"));
    return ResultFromScode(STG_E_INVALIDFUNCTION);
}
#endif

//+--------------------------------------------------------------
//
//  Member:     CExposedStream::DisconnectObject, public
//
//  Synopsis:   Non-functional
//
//  Arguments:  [dwRevserved] -
//
//  Returns:    Appropriate status code
//
//  History:    18-Sep-92       DrewB   Created
//
//---------------------------------------------------------------

#ifdef CODESEGMENTS
#pragma code_seg(SEG_CES_DisconnectObject)    // unused
#endif

STDMETHODIMP CExposedStream::DisconnectObject(DWORD dwReserved)
{
    olLog(("%p::INVALID CALL TO CExposedStream::DisconnectObject()\n"));
    return ResultFromScode(STG_E_INVALIDFUNCTION);
}
#endif //!REF
