/*++

Copyright (c) 1990-1991  Microsoft Corporation


Module Name:

    ht.h


Abstract:

    This module contains all the public defines, constants, structures and
    functions declarations for accessing the DLL.

Author:

    15-Jan-1991 Tue 21:13:21 created  -by-  Daniel Chou (danielc)

[Environment:]

    GDI Device Driver - Halftone.


[Notes:]


Revision History:

    29-Oct-1991 Tue 14:33:43 updated  -by-  Daniel Chou (danielc)

        1) Change HALFTONEPATTERN data structure.

            a) 'Flags' field from WORD to BYTE
            b) 'MaximumHTDensityIndex' from WORD to BYTE
            c) Change the field order.

        2) Remove ReferenceWhite/ReferenceBlack from HTCOLORADJUSTMENT data
           structure.

        3)

--*/

#ifndef _HT_
#define _HT_

#if defined(__cplusplus)
extern "C" {
#endif

#include <stddef.h>
#include <stdio.h>
#include <memory.h>


#if defined(_OS2_) || defined(_OS_20_)

#include <os2.h>

#ifndef far
#define far
#endif

#ifndef FAR
#define FAR     far
#endif

typedef void FAR                *LPVOID;
typedef LPVOID                  HANDLE;
typedef unsigned char FAR       *LPBYTE;
typedef char          FAR       *LPSTR;

typedef unsigned short          WORD;
typedef WORD near               *PWORD;
typedef WORD far                *LPWORD;

typedef unsigned long           DWORD;
typedef DWORD near              *PDWORD;
typedef DWORD far               *LPDWORD;

typedef int                     INT;
typedef int near                *PINT;
typedef int far                 *LPINT;

typedef unsigned int            UINT;
typedef unsigned int            *PUINT;


#else               // if not os/2

#include <windows.h>

#ifndef far
#define far
#endif

#ifndef FAR
#define FAR     far
#endif

typedef CHAR FAR                *LPCHAR;
typedef SHORT FAR               *LPSHORT;
typedef UINT FAR                *LPUINT;

#endif  // _OS2_, _OS_20_


// these are also defined in LIMIT.H

#ifndef PRECOMPILED_GRE

#define MIN_USTYPE(Type)    (unsigned Type)0
#define MAX_USTYPE(Type)    (unsigned Type)~MIN_USTYPE(Type)
#define MAX_STYPE(Type)     (Type)(MAX_USTYPE(Type) >> 1)
#define MIN_STYPE(Type)     (Type)~(unsigned Type)MAX_STYPE(Type)

#define CHAR_MAX            (MAX_STYPE(char))
#define CHAR_MIN            (MIN_STYPE(char))
#define BYTE_MAX            (MAX_USTYPE(char))
#define BYTE_MIN            (MIN_USTYPE(char))
#define SHORT_MAX           (MAX_STYPE(short))
#define SHORT_MIN           (MIN_STYPE(short))
#define WORD_MAX            (MAX_USTYPE(short))
#define WORD_MIN            (MIN_USTYPE(short))
#define LONG_MAX            (MAX_STYPE(long))
#define LONG_MIN            (MIN_STYPE(long))
#define DWORD_MAX           (MAX_USTYPE(long))
#define DWORD_MIN           (MIN_USTYPE(long))
#define INT_MAX             (MAX_STYPE(int))
#define INT_MIN             (MIN_STYPE(int))
#define UINT_MAX            (MAX_USTYPE(int))
#define UINT_MIN            (MIN_USTYPE(int))
#endif

//
// The DECI4/UDECI4 is a special number used in halftone DLL, this number
// just like regular short, unsigned short number, except it using lower
// four decimal digits as right side of the decimal point, that is
// 10000 is as 1.0000, and -12345 will be -1.2345.
//

typedef short               DECI4;
typedef unsigned short      UDECI4;
typedef DECI4 FAR           *PDECI4;
typedef UDECI4 FAR          *PUDECI4;

#define DECI4_0             (DECI4)0
#define DECI4_1             (DECI4)10000
#define DECI4_Neg1          (DECI4)-10000
#define UDECI4_0            (UDECI4)0
#define UDECI4_1            (UDECI4)10000


#define SIZE_BYTE           sizeof(BYTE)
#define SIZE_CHAR           sizeof(CHAR)
#define SIZE_WORD           sizeof(WORD)
#define SIZE_SHORT          sizeof(SHORT)
#define SIZE_LONG           sizeof(LONG)
#define SIZE_DWORD          sizeof(DWORD)
#define SIZE_UINT           sizeof(UINT)
#define SIZE_INT            sizeof(INT)
#define SIZE_UDECI4         sizeof(UDECI4)
#define SIZE_DECI4          sizeof(DECI4)

#define COUNT_ARRAY(array)  (sizeof(array) / sizeof(array[0]))

#define B_BITPOS(x)         ((BYTE)(1 << (x)))
#define W_BITPOS(x)         ((WORD)(1 << (x)))
#define DW_BITPOS(x)        ((DWORD)(1 << (x)))
#define BIT_IF(b,t)         (((t)) ? (b) : ((b)-(b)))
#define SET_BIT(x,b)        ((x) |= (b))
#define CLR_BIT(x,b)        ((x) &= ~(b))
#define INV_BIT(x,b)        ((x) ^= (b))
#define HAS_BIT(x,b)        ((x) & (b))


//
// The following are the error return values for the HTHalftoneBitmap() call.
//

#define HTERR_WRONG_VERSION_HTINITINFO      -1
#define HTERR_INSUFFICIENT_MEMORY           -2
#define HTERR_CANNOT_DEALLOCATE_MEMORY      -3
#define HTERR_COLORTABLE_TOO_BIG            -4
#define HTERR_QUERY_SRC_BITMAP_FAILED       -5
#define HTERR_QUERY_DEST_BITMAP_FAILED      -6
#define HTERR_QUERY_SRC_MASK_FAILED         -7
#define HTERR_SET_DEST_BITMAP_FAILED        -8
#define HTERR_INVALID_SRC_FORMAT            -9
#define HTERR_INVALID_SRC_MASK_FORMAT       -10
#define HTERR_INVALID_DEST_FORMAT           -11
#define HTERR_INVALID_DHI_POINTER           -12
#define HTERR_SRC_MASK_BITS_TOO_SMALL       -13
#define HTERR_INVALID_HTPATTERN_INDEX       -14
#define HTERR_INVALID_HALFTONE_PATTERN      -15
#define HTERR_HTPATTERN_SIZE_TOO_BIG        -16
#define HTERR_NO_SRC_COLORTRIAD             -17
#define HTERR_INVALID_COLOR_TABLE           -18
#define HTERR_INVALID_COLOR_TYPE            -29
#define HTERR_INVALID_COLOR_TABLE_SIZE      -20
#define HTERR_INVALID_PRIMARY_SIZE          -21
#define HTERR_INVALID_PRIMARY_VALUE_MAX     -22
#define HTERR_INVALID_PRIMARY_ORDER         -23
#define HTERR_INVALID_COLOR_ENTRY_SIZE      -24
#define HTERR_INVALID_FILL_SRC_FORMAT       -25
#define HTERR_INVALID_FILL_MODE_INDEX       -26
#define HTERR_INVALID_STDMONOPAT_INDEX      -27
#define HTERR_INVALID_DEVICE_RESOLUTION     -28
#define HTERR_INVALID_TONEMAP_VALUE         -29
#define HTERR_NO_TONEMAP_DATA               -30
#define HTERR_TONEMAP_VALUE_IS_SINGULAR     -31
#define HTERR_INVALID_BANDRECT              -32
#define HTERR_STRETCH_RATIO_TOO_BIG         -33
#define HTERR_CHB_INV_COLORTABLE_SIZE       -34
#define HTERR_HALFTONE_INTERRUPTTED         -35
#define HTERR_NO_SRC_HTSURFACEINFO          -36
#define HTERR_NO_DEST_HTSURFACEINFO         -37
#define HTERR_8BPP_PATSIZE_TOO_BIG          -38
#define HTERR_16BPP_555_PATSIZE_TOO_BIG     -39
#define HTERR_INTERNAL_ERRORS_START         -10000


#define COLOR_TYPE_RGB          0
#define COLOR_TYPE_XYZ          1
#define COLOR_TYPE_YIQ          2
#define COLOR_TYPE_MAX          2

#define PRIMARY_ORDER_123       0
#define PRIMARY_ORDER_132       1
#define PRIMARY_ORDER_213       2
#define PRIMARY_ORDER_231       3
#define PRIMARY_ORDER_321       4
#define PRIMARY_ORDER_312       5
#define PRIMARY_ORDER_MAX       5

#define PRIMARY_ORDER_RGB       PRIMARY_ORDER_123
#define PRIMARY_ORDER_RBG       PRIMARY_ORDER_132
#define PRIMARY_ORDER_GRB       PRIMARY_ORDER_213
#define PRIMARY_ORDER_GBR       PRIMARY_ORDER_231
#define PRIMARY_ORDER_BGR       PRIMARY_ORDER_321
#define PRIMARY_ORDER_BRG       PRIMARY_ORDER_312

#define PRIMARY_ORDER_CMY       PRIMARY_ORDER_123
#define PRIMARY_ORDER_CYM       PRIMARY_ORDER_132
#define PRIMARY_ORDER_MCY       PRIMARY_ORDER_213
#define PRIMARY_ORDER_MYC       PRIMARY_ORDER_231
#define PRIMARY_ORDER_YMC       PRIMARY_ORDER_321
#define PRIMARY_ORDER_YCM       PRIMARY_ORDER_312

#define PRIMARY_ORDER_XYZ       PRIMARY_ORDER_123
#define PRIMARY_ORDER_XZY       PRIMARY_ORDER_132
#define PRIMARY_ORDER_YXZ       PRIMARY_ORDER_213
#define PRIMARY_ORDER_YZX       PRIMARY_ORDER_231
#define PRIMARY_ORDER_ZYX       PRIMARY_ORDER_321
#define PRIMARY_ORDER_ZXY       PRIMARY_ORDER_312

#define PRIMARY_ORDER_YIQ       PRIMARY_ORDER_123
#define PRIMARY_ORDER_YQI       PRIMARY_ORDER_132
#define PRIMARY_ORDER_IYQ       PRIMARY_ORDER_213
#define PRIMARY_ORDER_IQY       PRIMARY_ORDER_231
#define PRIMARY_ORDER_QIY       PRIMARY_ORDER_321
#define PRIMARY_ORDER_QYI       PRIMARY_ORDER_312

//
// COLORTRIAD
//
//  This data structure describe the source color informations
//
//  Type                - One of the following type may be specified.
//
//                          COLOR_TYPE_RGB  - primaries are RGB.
//                          COLOR_TYPE_XYZ  - primaries are CIE XYZ.
//                          COLOR_TYPE_YIQ  - primaries are NTSC YIQ.
//
//  BytesPerPrimary     - Specified how many bytes used per primary color, it
//                        must be one of the following
//
//                          1 - BYTE
//                          2 - WORD
//                          4 - DWORD
//
//                        All 3 primaries must be consecutive in memory.
//
//  BytesPerEntry       - Specified how many bytes used for color table entry,
//                        each entry specified 3 primaries colors.
//
//  PrimaryOrder        - The primaries order in the color table, it can be
//                        one of the defined PRIMARY_ORDER_abc, for each entry
//                        in the memory it defined as
//
//                          PRIMARY_ORDER_abc
//                                        |||
//                                        ||+-- highest memory location
//                                        ||
//                                        |+--- middle
//                                        |
//                                        +---- Lowest memory location
//
//                        All 3 primaries must be consecutive in memory.
//
//  PrimaryValueMax     - The maximum value for the primary color, this is used
//                        to nomalized the input colors, for example a 8-bit
//                        RGB color table will specified 255.
//
//  ColorTableEntries   - Total entries of the color table pointed by the
//                        pColorTable.
//
//  pColorTable         - Pointer to the start of color table, the size of the
//                        this color table must at least (BytesPerEntry *
//                        ColorTableEntries).
//
//                        If the first primary color in the color table entry
//                        is not at first byte of the pColorTable, then caller
//                        must specified the pColorTable at first primary
//                        color. (pColorTable += Offset(first primary).
//
//

typedef struct _COLORTRIAD {
    BYTE    Type;
    BYTE    BytesPerPrimary;
    BYTE    BytesPerEntry;
    BYTE    PrimaryOrder;
    LONG    PrimaryValueMax;
    DWORD   ColorTableEntries;
    LPVOID  pColorTable;
} COLORTRIAD, FAR *PCOLORTRIAD;


//
// HTSURFACEINFO
//
//  This data structure describe the the input/output surface in order for
//  halftone function to render the output, this data structure only used for
//  the memory device.
//
//  hSurface                - This is 32-bits handle which will be passed back
//                            to the caller's callback function.
//
//  Flags                   - One or more following flags may be defined
//
//                              HTSIF_SCANLINES_TOPDOWN
//
//                                  Specified that the first scan line is
//                                  represent the top of the bitmap, otherwise
//                                  the first scan line is represent the bottom
//                                  of the bitmap.
//
//  SurfaceFormat           - Following formats are defined
//
//                              BMF_1BPP
//
//                                  1-bit per pel format, this is the index
//                                  number (0 or 1) for the color table/palette.
//
//                              BMF_4BPP
//
//                                  4-bit per pel and pack two pels to a byte
//                                  starting from high nibble (bit 4-7) format,
//                                  this is the index number (0-7) for the
//                                  color table/palette. (ONLY LOW 3 bits of
//                                  the nibble is used)
//
//                              BMF_4BPP_VGA16
//
//                                  4-bit per pel and pack two pels to a byte
//                                  starting from high nibble (bit 4-7) format,
//                                  this is the index number (0-15) for the
//                                  standard VGA 16 colors table/palette.
//
//                                  The different from BMF_4BPP is this indices
//                                  are fixed to standard VGA 16 colors as
//
//                                      Index#  Colors      Lightness
//                                      ---------------------------------
//                                          0   Black         0%
//                                          1   Red          50%
//                                          2   Green        50%
//                                          3   Yellow       50%
//                                          4   Blue         50%
//                                          5   Magenata     50%
//                                          6   Cyan         50%
//                                          7   Gray         50%
//                                          8   Gray         75%
//                                          9   Red         100%
//                                         10   Green       100%
//                                         11   Yellow      100%
//                                         12   Blue        100%
//                                         13   Magenata    100%
//                                         14   Cyan        100%
//                                         15   White       100%
//
//                                  Notice that the color order is
//
//                                  Bit 2 = Blue, Bit 1 = Green, Bit 0 = Red
//
//                                  This format can only be used as destination
//                                  surface, when used as destination surface
//                                  the halftone dll automatically set it to
//                                  USE_ADDITIVE_PRIMS and set the primaries
//                                  order as PRIMARY_ORDER_BGR.
//
//                              BMF_8BPP
//
//                                  8-bit per pel format (1 byte each), this is
//                                  the index number (0-255) for the color
//                                  table/palette.  The format is not allowed
//                                  for the destination surface.
//
//                              BMF_8BPP_VGA256
//
//                                  8-bit per pel format (1 byte each), this is
//                                  the index number (0-255) for the color
//                                  table/palette.
//
//                                  The different from BMF_8BPP is this indices
//                                  are fixed to halftone special colors.
//
//                                  The color table (palette) is defined by
//                                  halftone.dll, the display should call
//                                  HT_Get8BPPFormatPalette() api call to get
//                                  the current palette used by the halftone.
//
//                                  The HT_GetBPPFormatPalette() will only need
//                                  to called once until next time the display
//                                  caliberation occurred.
//
//                                  Halftone.dll will not used all 256 colors
//                                  in the system palette, it will leave some
//                                  20 or more entries for the system colors.
//
//                              BMF_16BPP
//
//                                  16-bit per pel format (16 bits each), this
//                                  is the index number (0-65535) for the color
//                                  table/palette.  The format is not allowed
//                                  for the destination surface.
//
//                              BMF_16BPP_555
//
//                                  16-bit per pel format (only 15 bits used),
//                                  each primary occupy 5 bits, the layout of
//                                  bits as follow
//
//                                      bit 10-15   - Primary A
//                                      bit  5- 9   - Primary B
//                                      bit  0- 4   - Primary C
//
//                                  The order of the Primary A, B and C is
//                                  specfied by PRIMARY_ORDER_xxx.
//
//                                  for each primary there are 32 gradations,
//                                  and halftone.dll output is assume to be
//                                  linear. (non-gamma corrected), this format
//                                  only allowed for destination surface.
//
//                              BMF_24BPP
//
//                                  24-bit per pel format (8-bit per color),
//                                  the order of RGB color stored in the source
//                                  bitmap or color table.
//
//                              BMF_32BPP
//
//                                  Same as BMF_24BPP but with extra byte
//                                  packing, if the extra byte is packed at
//                                  begining (the first color is starting from
//                                  second byte of that 4 bytes) then caller
//                                  must set the pColorTable = pColorTable + 1
//                                  or set pPlane = pPlane + 1, to skip first
//                                  unused byte.
//
//                              BMF_1BPP_3PLANES
//
//                                  3 planes and 1-bit per pel for each plane,
//                                  this format can only used for destination
//                                  surface.
//
//                              NOTE: Allowed source formats are
//
//                                      1) BMF_1BPP
//                                      2) BMF_4BPP
//                                      3) BMF_8BPP
//                                      4) BMF_16BPP
//                                      5) BMF_24BPP
//                                      6) BMF_32BPP
//
//                                    Allowed destination formats are
//
//                                      1) BMF_1BPP
//                                      2) BMF_4BPP
//                                      3) BMF_4BPP_VGA16
//                                      4) BMF_8BPP_VGA256
//                                      5) BMF_16BPP_555
//                                      6) BMF_1BPP_3PLANES
//
//                                    Any other mismatch cause error returned.
//
//  ScanLineAlignBytes      - Total bytes needed to aligned for each scan line
//                            in the surface bitmap, it can be any unsigned
//                            8-bit number, the common ones are defined as
//
//                                  BMF_ALIGN_BYTE      ( 8-bit aligned)
//                                  BMF_ALIGN_WORD      (16-bit aligned)
//                                  BMF_ALIGN_DWORD     (32-bit aligned)
//                                  BMF_ALIGN_QWORD     (64-bit aligned)
//
//  Width                   - The width of the surface in pels.
//
//  Height                  - The height of the surface in scan lines.
//
//  MaximumQueryScanLines   - The maximum scan lines can be queried by the
//                            halftone DLL, this is used by the halftone DLL
//                            to compute the scan lines block (passed in
//                            as HTCALLBACKPARAMS data structure) durning the
//                            callback function.
//
//                            This field typically set to a zero (0) because
//                            normally both source/source mask/destination
//                            bitmaps are resides in the memory, if caller
//                            can only processed partial scan lines at one time
//                            (such as read/write disk) then this must set to
//                            the maximum scan lines it can processed at a
//                            time.   For example if the caller only have
//                            100 source scan lines memory, and 120 destination
//                            scan lines memory, then it should set 100 and 120
//                            to this field in the HTSURFACEINFO data structure
//                            for source and destination.
//
//                            NOTE: If this field value set to 0 then it
//                                  indicate that no query callback is
//                                  necessary but just use the pPlane.
//
//  BytesPerPlane           - This field only used for destination surface
//                            format is BMF_1BPP_3PLANES, it specified the
//                            offset in bytes from plane 1 to plane 2 and from
//                            plane 2 to plane 3.
//
//                            The maximum will be 0xffff (65535 bytes) per
//                            plane, if a zero is specified then BytesPerPlane
//                            is automatically computed as <SurfaceHeight *
//                            BytesPerScanLine (1 bit per pel)>.  For most of
//                            planer devices this field will be zero (0), for
//                            some special devices it may specified 64k-1 size
//                            to offset each plane.
//
//                            If this field is non-zero then caller must make
//                            sure that destination planes will not overlapped
//                            and the offset addresses are valid.
//
//  pPlane                  - Only valid if MaximumQueryScanLines is 0, it
//                            points to the begining (physical lowest memory)
//                            of the bitmap for the first plane.
//
//  pColorTriad             - Pointe to the COLORTRIAD data structure to
//                            specified the source color table, this pointer
//                            only examined by the halftone.dll for source
//                            surface.
//


#if !defined( BMF_DEVICE ) &&     \
    !defined( BMF_1BPP   ) &&     \
    !defined( BMF_4BPP   ) &&     \
    !defined( BMF_8BPP   ) &&     \
    !defined( BMF_16BPP  ) &&     \
    !defined( BMF_24BPP  ) &&     \
    !defined( BMF_32BPP  )

#define BMF_DEVICE                      0
#define BMF_1BPP                        1
#define BMF_4BPP                        2
#define BMF_8BPP                        3
#define BMF_16BPP                       4
#define BMF_24BPP                       5
#define BMF_32BPP                       6

#endif

#define BMF_1BPP_3PLANES                BMF_DEVICE
#define BMF_4BPP_VGA16                  255
#define BMF_8BPP_VGA256                 254
#define BMF_16BPP_555                   253


//
// Following are common used alignment bytes for the bitmap
//

#define BMF_ALIGN_BYTE      1                   //  8 bits = 1 byte
#define BMF_ALIGN_WORD      2                   // 16 bits = 2 bytes
#define BMF_ALIGN_DWORD     4                   // 32 bits = 4 bytes
#define BMF_ALIGN_QWORD     8                   // 64 bits = 8 bytes





#define HTSIF_SCANLINES_TOPDOWN         W_BITPOS(0)


typedef struct _HTSURFACEINFO {
    DWORD       hSurface;
    WORD        Flags;
    BYTE        SurfaceFormat;
    BYTE        ScanLineAlignBytes;
    LONG        Width;
    LONG        Height;
    WORD        MaximumQueryScanLines;
    WORD        BytesPerPlane;
    LPBYTE      pPlane;
    PCOLORTRIAD pColorTriad;
    } HTSURFACEINFO;

typedef HTSURFACEINFO   FAR *PHTSURFACEINFO;


//
// HTCALLBACKPARAMS
//
//  This is structure is used durning the bitmap halftone process to obtains
//  the source or destination bitmap surface pointers.
//
//
//  hSurface                - This is the 32-bit handle which passed to the
//                            halftone DLL, durning the HTHalftoneBitmap() call,
//                            (in HTSURFACEINFO data structure) it may be
//                            handle to source or destination depends on the
//                            nature of the callback.
//
//  CallBackMode            - Specified the nature of the callback.
//
//                              HTCALLBACK_QUERY_SRC
//
//                                  The callback is quering the source bitmap
//                                  pointer.
//
//                              HTCALLBACK_QUERY_SRC_MASK
//
//                                  The callback is quering the source mask
//                                  bitmap pointer.
//
//                              HTCALLBACK_QUERY_DEST
//
//                                  The callback is quering the destination
//                                  bitmap pointer(s).
//
//                              HTCALLBACK_SET_DEST
//
//                                  The callback is used to release halftoned
//                                  destination.   It will called in following
//                                  conditions:
//
//                                      1) Before HTCALLBACK_QUERY_DEST call
//                                         except for the very first query
//                                         destination.
//
//                                      2) After the halftone process is
//                                         completed.  This give the chance for
//                                         the caller to process the last
//                                         halftoned destination block.
//
//  SurfaceFormat           - This is the surface format specified in the
//                            original HTSURFACEINFO.
//
//  Flags                   - This is the copy of HTSURFACEINFO.Flags.
//
//  BytesPerScanLine        - This is the total bytes per scan line for the
//                            surface bitmap which computed by the halftone
//                            DLL according to the 'ScanLineAlignBytes' in the
//                            HTSURFACEINFO data structure, it can be used by
//                            the caller to calculate source/destination
//                            pointers information.
//
//  ScanStart               - Requested starting scan line number, the scan
//                            lines are number from 0 up, this number will
//                            guaranteed has following propertites:
//
//                              1) It always greater than or equal to zero.
//                              2) It will never greater than or equal to the
//                                 'height' field specified in the
//                                 HTSURFACEINFO.
//
//                                 NOTE: for banding destination surface it
//                                       will never greater than or equal to
//                                       the (rclBand.bottom - rclBand.top).
//
//                              3) The scan line number 0 always refer to the
//                                 physical lowest bitmap memory location
//                                 regardless HTSIF_SCANLINES_TOPDOWN flag set
//                                 or not, durning callback the caller only
//                                 need to compute array like bitmap buffer,
//                                 halftone DLL will compute the correct
//                                 ScanStart if the bitmap is not TOPDOWN.
//
//  ScanCount               - Total scan lines requested starting from
//                            ScanStart, this number will guaranteed has
//                            following propertites:
//
//                              1) It always greater than zero.
//                              2) Never greater then the MaximumQueryScanLines
//                                 specified for the surface (in HTSURFACEINFO
//                                 data structure).
//
//                                 NOTE: for banding destination surface it
//                                       will never greater than the
//                                       (rclBand.bottom - rclBand.top).
//
//                            NOTE: 1) ScanStart/ScanCount
//
//                                     If the flags HTSURFACEINFO data
//                                     structure HTSIF_SCANLINES_TOPDOWN is not
//                                     specified then halftone DLL automatically
//                                     calculate the correct ScanStart/ScanCount
//                                     for read/write the up-side-down bitmap.
//
//                                     For example:
//
//                                     If the surface bitmap is up-side-down
//                                     type DIB) and has 100 scan lines height
//                                     (scan line #99 is the top of the bitmap),
//                                     if halftone DLL need to get to scan line
//                                     10-14 (Start from scan line 10 and 5
//                                     lines) then halftone DLL will flip Y,
//                                     and passed ScanStart = 85 and ScanCount
//                                     = 5, but if the HTSIF_SCANLINES_TOPDOWN
//                                     flag is set (Non DIB type format) then
//                                     callback parameters will then be
//                                     ScanStart = 10 and ScanCount = 5.
//
//                                  2) The ScanStart for the callback function
//                                     always consider the lowest bitmap in the
//                                     memory as scan line 0, that is when
//                                     callback fucntion received control, it
//                                     only need to consider the ScanStart as
//                                     physical scan line location which the
//                                     Scan line #0 always starting from lowest
//                                     bitmap memory location.
//
//                                  3) The width of the destination buffer is
//                                     considered as 'Width' field specified
//                                     for the destination HTSURFACEINFO, if
//                                     destination is BANDed (horizontal or
//                                     vertical) then its width is computed as
//                                     Band.right - Band.left. and the result
//                                     always Band.left/Band.top aligned.
//
//                                  4) If caller return 'ScanCount' diff from
//                                     the one requested then caller must check
//
//                                      HTSIF_SCANLINES_TOPDOWN flag is SET
//
//                                          1. Process 'ScanStart' and
//                                             'ScanCount' fields as normal.
//
//                                          2. Set New ScanCount to passed
//                                             callback data structure.
//                                             (ie. HTCallBackParam.ScanCount)
//
//
//                                      HTSIF_SCANLINES_TOPDOWN flag is CLEAR
//
//                                          1. Re-compute 'ScanStart' before
//                                             compute pPlane as
//
//                                             ScanStart = ScanStart -
//                                                         (NewScanCount -
//                                                          RequsetedScanCount)
//
//                                          2. Process 'ScanStart' and
//                                             'ScanCount' fields as normal.
//
//                                          3. Set New ScanStart to passed
//                                             callback data structure.
//                                             (ie. HTCallBackParam.ScanStart)
//
//                                          4. Set New ScanCount to passed
//                                             callback data structure.
//                                             (ie. HTCallBackParam.ScanCount)
//
//                                      The returned new 'ScanCount' must not
//                                      greater then the 'RemainedSize' field.
//
//  MaximumQueryScanLines   - This is the copy of the MaximumQueryScanLines
//                            field from the HTSURFACEINFO data structure which
//                            passed to the the HT_HaltoneBitmap() calls.
//                            Depends on the nature of the callback, it may be
//                            source or destination.
//
//                            NOTE: for banding surface, it will be limited to
//                                  (rclBand.bottom - rclBand.top) if it is
//                                  greater than that number.
//
//  RemainedSize            - This field specified how many scan lines remained
//                            to be queried, the remainded scan lines are not
//                            include current call.
//
//  pPlane                  - pointer points to the begining of first plane of
//                            the surface.  If the callback is quering the
//                            source then this is the only pointer which need
//                            to be updated by the callback function.
//
//                              NOTE: The pPlane must points to the first byte
//                                    of the 'ScanStart' scan line number.
//
//  Field(s) returned from callback
//
//  1) HTCALLBACK_QUERY_SRC         - pPlane (Options: ScanStart/ScanCount)
//  2) HTCALLBACK_QUERY_SRC_MASK    - pPlane (Options: ScanStart/ScanCount)
//  2) HTCALLBACK_QUERY_DEST        - pPlane (Options: ScanStart/ScanCount)
//  4) HTCALLBACK_SET_DEST          - none.
//


#define HTCALLBACK_QUERY_SRC            0
#define HTCALLBACK_QUERY_SRC_MASK       1
#define HTCALLBACK_QUERY_DEST           2
#define HTCALLBACK_SET_DEST             3
#define HTCALLBACK_MODE_MAX             HTCALLBACK_SET_DEST

typedef struct _HTCALLBACKPARAMS {
    DWORD   hSurface;
    BYTE    CallBackMode;
    BYTE    SurfaceFormat;
    WORD    Flags;
    LONG    BytesPerScanLine;
    LONG    ScanStart;
    WORD    ScanCount;
    WORD    MaximumQueryScanLines;
    LONG    RemainedSize;
    LPBYTE  pPlane;
    } HTCALLBACKPARAMS;

typedef HTCALLBACKPARAMS    FAR *PHTCALLBACKPARAMS;


//
// _HTCALLBACKFUNC
//
//  The callback function is used to obtain the source and/destination bitmap
//  surface durning the halftone process, the halftone DLL will using call
//  back function is following sequences.
//
//      1) Callback to obtain block of the source bitmap, it depensds on
//         the maximum query scan lines limit by the caller. (in HTSURFACEINFO)
//
//      3) Callback to obtain block of the destination bitmap, it depends on
//         the maximum query scan lines limit by the caller. (in HTSURFACEINFO)
//
//      4) Repeat 1), 2), 3) until destination maximum queried scan lines are
//         processed then callback to the caller to release the processed
//         destination.
//
//      5) Repat 1), 2), 3) and 4) unitil all source/destination scan lines are
//         processed.
//
//  The return value of the callback is a boolean value, if false the halftone
//  processed is interrupted and an erro code is returned, if callback function
//  return true then halftone DLL assume that all queried scan lines are
//  reachable by the return pointer (in the HTCALLBACKPARAMS data structure).
//
//  NOTE: 1) If the callback function query for 100 lines and return value is
//           ture then there must all 100 scan lines can be accessable by the
//           halftone dll through the pointer(s).
//
//        2) If the caller has all the bitmap in the memory it should set the
//           maximum query scan lines count to the bitmap height to reduced the
//           callback calls.
//
//        3) If the caller do not need to released the halftoned destination
//           then it should not set the HTCBPF_NEED_SET_DEST_CALLBACK flag
//           to reduces callback calls.
//
// The callback function must be
//
//  1) Must return a 16-bit 'BOOLEAN' (TRUE/FALSE)
//  2) Must 32-bit far function
//  3) _loadds if you using your data segment at all from callback fuction
//

typedef BOOL (FAR *_HTCALLBACKFUNC)(PHTCALLBACKPARAMS pHTCallBackParams);
#define HTCALLBACKFUNCTION  BOOL FAR


//
// HALFTONEPATTERN
//
//  The HALFTONEPATTERN data structure is used to describe the halftone
//  pattern which will be used by a particular device, if the device choosed
//  to used halftone DLL's default pattern then following data structure will
//  be automatically calculated by the halftone DLL.
//
//  Flags                   - Various halftone flags for the cell, can be one
//                            of the following:
//
//                              HTPF_THRESHOLD_ARRAY
//
//                                  SET:    Specified that pToneMap field is a
//                                          pointer to BYTE array contains
//                                          threshold data, the size of the
//                                          array must be 'Width * Height' in
//                                          bytes.  The data in the array
//                                          should range from 1 to 255.
//
//                                          Halftone DLL use this thresholds
//                                          array with device X, Y, PEL
//                                          resolution and specified input/
//                                          output relationship to compute
//                                          color information.
//
//                                  CLEAR:  Specified that pToneMap points to
//                                          an WORD array contains lightness
//                                          values, the size of array must be
//                                          'sizeof(WORD) * Width * Height'.
//                                          for each data item it should in
//                                          range from 1 to MaxToneValue else
//                                          an error is generated.
//
//  Width                   - Specified the width of the pattern in pels, this
//                            field must not greater than MAX_HTPATTERN_WIDTH.
//
//  Height                  - Specified the Width of the pattern in scan line,
//                            this field only must not greater than
//                            MAX_HTPATTERN_HEIGHT.
//
//  MaxToneValue            - This field only used if HTPF_THRESHOLD_ARRAY is
//                            not set, the range of this field value must
//                            between 1 and 65535, it specified the unit of
//                            lightness values pointed by the pToneMap.
//
//  pToneMap                - Specified caller defined pattern. The data items
//                            points by this pointer must have minimum of
//                            (Width * Height) items.
//
//                            see HTPF_THRESHOLD_ARRAY flag above for the
//                            data item points by this pointer.
//


#define MAX_HTPATTERN_WIDTH         16
#define MAX_HTPATTERN_HEIGHT        16

//
// Following are the predefined halftone pattern sizes for 'HTPatternIndex'
//

#define HTPAT_SIZE_2x2              0
#define HTPAT_SIZE_2x2_M            1
#define HTPAT_SIZE_4x4              2
#define HTPAT_SIZE_4x4_M            3
#define HTPAT_SIZE_6x6              4
#define HTPAT_SIZE_6x6_M            5
#define HTPAT_SIZE_8x8              6
#define HTPAT_SIZE_8x8_M            7
#define HTPAT_SIZE_10x10            8
#define HTPAT_SIZE_10x10_M          9
#define HTPAT_SIZE_12x12            10
#define HTPAT_SIZE_12x12_M          11
#define HTPAT_SIZE_14x14            12
#define HTPAT_SIZE_14x14_M          13
#define HTPAT_SIZE_16x16            14
#define HTPAT_SIZE_16x16_M          15
#define HTPAT_SIZE_MAX_INDEX        HTPAT_SIZE_16x16_M
#define HTPAT_SIZE_DEFAULT          HTPAT_SIZE_6x6_M

#define HTPF_THRESHOLD_ARRAY        W_BITPOS(0)


typedef struct _HALFTONEPATTERN {
    WORD    Flags;
    WORD    Width;
    WORD    Height;
    WORD    MaxToneValue;
    LPVOID  pToneMap;
    } HALFTONEPATTERN, FAR *PHALFTONEPATTERN;


//
// CIECOORD
//
//  This data structure defined a C.I.E color space coordinate point, the
//  coordinate is in DECI4 format.
//
//  x   - x coordinate in C.I.E color space
//
//  y   - y coordinate in C.I.E color space.
//
//  Y   - The liminance for the color
//
//

#define CIE_x_MIN   (UDECI4)10
#define CIE_x_MAX   (UDECI4)8000
#define CIE_y_MIN   (UDECI4)10
#define CIE_y_MAX   (UDECI4)8500

typedef struct _CIECOORD {
    UDECI4  x;
    UDECI4  y;
    UDECI4  Y;
    } CIECOORD;

typedef CIECOORD FAR *PCIECOORD;


//
// CIEINFO
//
//  This data structure describe the red, green, blue, cyan, magenta, yellow
//  and alignment white coordinate in the C.I.E color space plus the Luminance
//  factor, these are used to calculate the C.I.E. transform matrix and its
//  inversion.
//
//  Red             - Red primary color in CIRCOORD format.
//
//  Green           - Green primary color in CIRCOORD format.
//
//  Blue            - Blue primary color in CIRCOORD format.
//
//  Cyan            - Cyan primary color in CIRCOORD format.
//
//  Magenta         - Magenta primary color in CIRCOORD format.
//
//  Yellow          - Yellow primary color in CIRCOORD format.
//
//  AlignmentWhite  - Alignment white in CIECOORD format.
//


typedef struct _CIEINFO {
    CIECOORD    Red;
    CIECOORD    Green;
    CIECOORD    Blue;
    CIECOORD    Cyan;
    CIECOORD    Magenta;
    CIECOORD    Yellow;
    CIECOORD    AlignmentWhite;
    } CIEINFO;

typedef CIEINFO FAR *PCIEINFO;

//
// SOLIDDYESINFO
//
//  This data structure specified device cyan, magenta and yellow dyes
//  concentration.
//
//  MagentaInCyanDye    - Mangenta component proportion in Cyan dye.
//
//  YellowInCyanDye     - Yellow component proportion in Cyan dye.
//
//  CyanInMagentaDye    - Cyan component proportion in Magenta dye.
//
//  YellowInMagentaDye  - Yellow component proportion in Magenta dye.
//
//  CyanInYellowDye     - Yellow component proportion in Cyan dye.
//
//  MagentaInYellowDye  - Magenta component proportion in Cyan dye.
//
//      NOTE: all fields in this data structure is UDECI4 number, range from
//            UDECI4_0 to UDECI_4, ie,; 0.0 to 1.0, this a inpurity proportion
//            percentge in primary dye, for example a MagentaInCyanDye=1200
//            mean a 0.12% of magenta component is in device Cyan dye.
//

typedef struct _SOLIDDYESINFO {
    UDECI4  MagentaInCyanDye;
    UDECI4  YellowInCyanDye;
    UDECI4  CyanInMagentaDye;
    UDECI4  YellowInMagentaDye;
    UDECI4  CyanInYellowDye;
    UDECI4  MagentaInYellowDye;
    } SOLIDDYESINFO, FAR *PSOLIDDYESINFO;


//
// HTCOLORADJUSTMENT
//
//  This data structure is a collection of the device color adjustments, it
//  can be changed at any calls.
//
//  Flags                       - CLRADJF_NEGATIVE
//
//                                  Produced negative picture
//
//                                CLRADJF_LOG_FILTER
//
//                                  Specified a relative logarithm should
//                                  used to calculate the final density.
//
//  IlluminantIndex             - Specified the default illuminant of the light
//                                source which the object will be view under.
//                                The predefined value has ILLUMINANT_xxxx
//                                form.
//
//  RedPowerGamma               - The n-th power applied to the red color
//                                before any other color tramsformations,
//                                this is an UDECI4 value.
//
//                                  For example if the RED = 0.8 (DECI4=8000)
//                                  and the RedPowerGammaAdjustment = 0.7823
//                                  (DECI4 = 7823) then the red is equal to
//
//                                         0.7823
//                                      0.8        = 0.8398
//
//  GreenPowerGamma             - The n-th power applied to the green color
//                                before any other color transformations, this
//                                is an UDECI4 value.
//
//  BluePowerGamma              - The n-th power applied to the blue color
//                                before any other color transformations, this
//                                is an UDECI4 value.
//
//                      NOTE: RedPowerGamma/GreenPowerGamma/BluePoweGamma are
//                            UDECI4 values and range from 100 to 65535 if any
//                            one of these values is less than 100 (0.01) then
//                            halftone dll automatically set all power gamma
//                            adjustments to selected default.
//
//  ReferenceBlack              - The black shadow reference for the colors
//                                passed to the halftone dll,  if a color's
//                                lightness is darker than the reference black
//                                then halftone dll will treated as completed
//                                blackness and render it with device maximum
//                                density.
//
//  ReferenceWhite              - The white hightlight reference for the colors
//                                passed to the halftone dll, if a color's
//                                lightness is lighter than the reference white
//                                then halftone will treated as a specular
//                                hightlight and redner with device maximum
//                                intensity.
//
//                      NOTE:   ReferenceBlack Range:  0.0000 - 0.4000
//                              ReferenceWhite Range:  0.6000 - 1.0000
//
//  Contrast                    - Primary color contrast adjustment, this is
//                                a SHORT number range from -100 to 100, this
//                                is the black to white ratio, -100 is the
//                                lowest contrast, 100 is the highest and 0
//                                indicate no adjustment.
//
//  Brightness                  - The brightness adjustment, this is a SHORT
//                                number range from -100 to 100, the brightness
//                                is adjusted by apply to change the overall
//                                saturations for the image, -100 is lowest
//                                brightness, 100 is the hightest and a zero
//                                indicate no adjustment.
//
//  Colorfulness                - The primary color are so adjusted that it
//                                will either toward or away from black/white
//                                colors, this is a SHORT number range from
//                                -100 to 100.  -100 has less colorful, 100 is
//                                most colorfull, and a zero indicate no
//                                adjustment.
//
//  RedGreenTint                - Tint adjustment between Red/Green primary
//                                color, the value is a SHORT range from -100
//                                to 100, it adjust color toward Red if number
//                                is positive, adjust toward Green if number
//                                is negative, and a zero indicate no
//                                adjustment.
//
//  NOTE: For Contrast/Brightness/Colorfulness/RedGreenTint adjustments, if its
//        value is outside of the range (-100 to 100) then halftone DLL
//        automatically set its to selected default value.
//

#if 0

// in WinGDI.H
//
// The following are predefined alignment white for 'IlluminantIndex'.
//
// If ILLUMINANT_DEFAULT is specified
//
//  1) if pDeviceCIEInfo is NULL or pDeviceCIEInfo->Red.x eqaul to 0 then
//     halftone DLL automatically choosed approx. illuminant for the output
//     device.
//
//  2) if pDeviceCIEInfo is NOT null and pDeviceCIEInf->Red.x not equal to 0
//     then the 'White' field is used as illuminant alignment white.
//
// If other ILLUMINANT_xxxx is specified it will be used as alignment white
// even pDeviceCIEInfo is not null.
//
// If the IlluminantIndex is > ILLUMINANT_MAX_INDEX then halftone DLL will
// automatically choosed approx. illuminant even pDeviceCIEInfo is not NULL.
//

#define ILLUMINANT_DEVICE_DEFAULT   0
#define ILLUMINANT_A                1
#define ILLUMINANT_B                2
#define ILLUMINANT_C                3
#define ILLUMINANT_D50              4
#define ILLUMINANT_D55              5
#define ILLUMINANT_D65              6
#define ILLUMINANT_D75              7
#define ILLUMINANT_F2               8
#define ILLUMINANT_MAX_INDEX        ILLUMINANT_F2

#define ILLUMINANT_TUNGSTEN         ILLUMINANT_A
#define ILLUMINANT_DAYLIGHT         ILLUMINANT_C
#define ILLUMINANT_FLUORESCENT      ILLUMINANT_F2
#define ILLUMINANT_NTSC             ILLUMINANT_C

#endif


#define MIN_COLOR_ADJ               COLOR_ADJ_MIN
#define MAX_COLOR_ADJ               COLOR_ADJ_MAX
#define MIN_POWER_GAMMA             RGB_GAMMA_MIN

#define NTSC_POWER_GAMMA            (UDECI4)22000

//
// The following is the default value
//


#define REFLECT_DENSITY_DEFAULT     REFLECT_DENSITY_LOG
#define ILLUMINANT_DEFAULT          0
#define POWER_GAMMA_DEFAULT         NTSC_POWER_GAMMA
#define REFERENCE_WHITE_DEFAULT     UDECI4_1
#define REFERENCE_BLACK_DEFAULT     UDECI4_0
#define CONTRAST_ADJ_DEFAULT        0
#define BRIGHTNESS_ADJ_DEFAULT      0
#define COLORFULNESS_ADJ_DEFAULT    0
#define REDGREENTINT_ADJ_DEFAULT    0


#define CLRADJF_NEGATIVE            CA_NEGATIVE
#define CLRADJF_LOG_FILTER          CA_LOG_FILTER

#define CLRADJF_FLAGS_MASK          (CLRADJF_NEGATIVE       |   \
                                     CLRADJF_LOG_FILTER)


#if 0

//
// In WinGDI.H
//

#define REFERENCE_WHITE_MIN         (UDECI4)6000
#define REFERENCE_WHITE_MAX         UDECI4_1

#define REFERENCE_BLACK_MIN         UDECI4_0
#define REFERENCE_BLACK_MAX         (UDECI4)4000

typedef struct  tagCOLORADJUSTMENT {
    WORD   caSize;
    WORD   caFlags;
    WORD   caIlluminantIndex;
    WORD   caRedGamma;
    WORD   caGreenGamma;
    WORD   caBlueGamma;
    WORD   caReferenceBlack;
    WORD   caReferenceWhite;
    SHORT  caContrast;
    SHORT  caBrightness;
    SHORT  caColorfulness;
    SHORT  caRedGreenTint;
} COLORADJUSTMENT, *PCOLORADJUSTMENT, FAR *LPCOLORADJUSTMENT;

#endif

#define HTCOLORADJUSTMENT COLORADJUSTMENT
typedef HTCOLORADJUSTMENT *PHTCOLORADJUSTMENT;

//
// HTINITINFO
//
//  This data structure is a collection of the device characteristics and
//  will used by the halftone DLL to carry out the color composition for the
//  designated device.
//
//  Version                 - Specified the version number of HTINITINFO data
//                            structure. for this version it should set to the
//                            HTINITINFO_VERSION
//
//  Flags                   - Various flag defined the initialization
//                            requirements.
//
//                              HIF_SQUARE_DEVICE_PEL
//
//                                  Specified that the device pel is square
//                                  rather then round object.  this only make
//                                  differences when the field
//                                  'PelResolutionRatio' is greater than 0.01
//                                  and it is not 1.0000.
//
//                              HIF_HAS_BLACK_DYE
//
//                                  Indicate the device has separate black dye
//                                  instead of mix cyan/magenta/yellow dyes to
//                                  procduced black, this flag will be ignored
//                                  if HIF_ADDITIVE_PRIMS is defined.
//
//                              HIF_ADDITIVE_PRIMS
//
//                                  Specified that final device primaries are
//                                  additively, that is adding device primaries
//                                  will produce lighter result. (this is true
//                                  for monitor devices and certainly false for
//                                  the reflect devices such as printers).
//
//  HTPatternIndex          - Default halftone pattern index number, the
//                            indices is predefine as HTPAT_SIZE_xxxx, this
//                            field only used if pHTalftonePattern pointer is
//                            not NULL.
//
//  DevicePowerGamma        - This field is used to adjust halftone pattern
//                            cell's gamma, the gamma applied to all the rgb
//                            colors, see gamma description in
//                            HTCOLORADJUSTMENT above.
//
//  HTCallBackFunction      - a 32-bit pointer to the caller supplied callback
//                            function which used by the halftone DLL to
//                            obtained the source/destination bitmap pointer
//                            durning the halftone process, if this pointer is
//                            NULL then halftone dll assume that caller does
//                            not need any callback and generate an error if a
//                            callback is needed.
//
//  pHalftonePattern        - pointer to HALFTONEPATTERN data structure, see
//                            descriptions above, if this pointer is NULL then
//                            halftone using HTPatternIndex field to select
//                            default halftone dll's pattern.
//
//  pInputRGBInfo           - Specified input's rgb color' coordinates within
//                            the C.I.E. color spaces.  If this pointer is NULL
//                            or pInputRGBInfo->Red.x is 0 (UDECI4_0) then it
//                            default using NTSC standard to convert the input
//                            colors.
//
//  pDeviceCIEInfo          - Specified device primary color coordinates within
//                            the C.I.E. color space, see CIEINFO data
//                            structure, if the pointer is NULL or
//                            pDeviceCIEInfo->Red.x is 0 (UDECI4_0) then
//                            halftone DLL choose the default for the output
//                            device.
//
//  pDeviceSolidDyesInfo    - Specified device solid dyes concentrations, this
//                            field will be ignored if HIF_ADDITIVE_PRIMS flag
//                            is defined, if HIF_ADDITIVE_PRIMS is not set and
//                            this pointer is NULL then halftone dll choose
//                            approximate default for the output device.
//
//  DeviceResXDPI           - Specified the device horizontal (x direction)
//                            resolution in 'dots per inch' measurement.
//
//  DeviceResYDPI           - Specified the device vertical (y direction)
//                            resolution in 'dots per inch' measurement.
//
//  DevicePelsDPI           - Specified the device pel/dot/nozzle diameter
//                            (if rounded) or width/height (if squared) in
//                            'dots per inch' measurement.
//
//                            This value is measure as if each pel only touch
//                            each other at edge of the pel.
//
//                            If this value is 0, then it assume that each
//                            device pel is rectangular shape and has
//                            DeviceResXDPI in X direction and DeviceResYDPI
//                            in Y direction.
//
//  DefHTColorAdjustment    - Specified the default color adjustment for
//                            this device.  see HTCOLORADJUSTMENT data
//                            structure above for detail.
//

#define HTINITINFO_VERSION          (DWORD)0x48546902   // 'HTi\02'


#define HIF_SQUARE_DEVICE_PEL       0x01
#define HIF_HAS_BLACK_DYE           0x02
#define HIF_ADDITIVE_PRIMS          0x04

//
// This defined the minimum acceptable device resolutions
//

#define MIN_DEVICE_DPI              12


typedef struct _HTINITINFO {
    DWORD               Version;
    WORD                Flags;
    WORD                HTPatternIndex;
    _HTCALLBACKFUNC     HTCallBackFunction;
    PHALFTONEPATTERN    pHalftonePattern;
    PCIEINFO            pInputRGBInfo;
    PCIEINFO            pDeviceCIEInfo;
    PSOLIDDYESINFO      pDeviceSolidDyesInfo;
    UDECI4              DevicePowerGamma;
    WORD                DeviceResXDPI;
    WORD                DeviceResYDPI;
    WORD                DevicePelsDPI;
    HTCOLORADJUSTMENT   DefHTColorAdjustment;
    } HTINITINFO, FAR *PHTINITINFO;


//
// BITBLTPARAMS
//
//  This data structure is used when calling the HT_HalftoneBitmap(), it
//  defined where to halftone from the source bitmap to the destination
//  bitmap.
//
//  Flags           - Various flags defined how the source, destination and
//                    source mask should be calculated.
//
//                      BBPF_HAS_DEST_CLIPRECT
//
//                          Indicate that there is a clipping
//                          rectangle for the destination and it is
//                          specified by DestClipXLeft, DestClipXRight,
//                          DestClipYTop and DestClipYBottom
//
//                      BBPF_USE_ADDITIVE_PRIMS
//
//                          Specified if the halftone result will be
//                          using Red/Green/Blue primary color or
//                          using Cyan/Magenta/Yellow primary color, depends
//                          on the destination surface format as
//
//                          BMF_1BPP:
//
//                                  Additive Prims: 0=Black, 1=White
//                              Substractive Prims: 0=White, 1=Black
//
//                          BMF_4BPP_VGA16:
//
//                              Always using RED, GREEN, BLUE primaries, and
//                              ignored this flag.
//
//                          BMF_4BPP:
//                          BMF_1BPP_3PLANE:
//
//                                  Additive Prims: RED, GREEN. BLUE
//                              Substractive Prims: CYAN, MAGENTA, YELLOW
//
//                              The order of the RGB, or CMY is specified by
//                              DestPrimaryOrder field. (see below)
//
//                          BMF_8BPP_VGA256:
//                          BMF_16BPP_555:
//
//                              Always using RED, GREEN, BLUE primaries, and
//                              ignored this flag.
//
//                      BBPF_NEGATIVE_DEST
//
//                          Invert the final destination surface, so
//                          after the halftone it just the negative
//                          result from the source.
//
//                      BBPF_INVERT_SRC_MASK
//
//                          Invert the source mask bits before using
//                          it, this in effect make mask bit 0 (off)
//                          for copy the source and mask bit 1 (on)
//                          for preserved the destination.
//
//                      BBPF_HAS_BANDRECT
//
//                          Set to specified that rclBand RECTL data structrue
//                          should be used to compute for the caller's
//                          destination bitmap buffer.
//
//                          If this flag is not set then halftone dll assumed
//                          the caller's bitmap buffer is same width/height
//                          as specified in the destination HTSURFACEINFO.
//
//                      BBPF_BW_ONLY
//
//                          Produced monochrome version of the output even the
//                          destination is the color device.
//
//  DestPrimaryOrder- Specified destination primary color order, it can be
//                    either PRIMARY_ORDER_RGB or PRIMARY_ORDER_CMY group, it
//                    depends on the surface format has following meaning.
//
//                      BMF_1BPP:
//
//                          This field is ignored.
//
//                      BMF_4BPP_VGA16:
//
//                          This field automatically set to PRIMARY_ORDER_BGR
//                          by the halftone DLL.
//
//                      BMF_4BPP:
//
//                          for each byte there are two indices entries, and
//                          for each nibble has following meaning, notice that
//                          bit 3/7 always set to 0, the index number only
//                          range from 0 to 7.
//
//                          PRIMARY_ORDER_abc
//                                        |||
//                                        ||+-- bit 0/4
//                                        ||
//                                        |+--- bit 1/5
//                                        |
//                                        +---- bit 2/7
//
//                      BMF_1BPP_3PLANES:
//
//                          PRIMARY_ORDER_abc
//                                        |||
//                                        ||+-- plane 1
//                                        ||
//                                        |+--- plane 2
//                                        |
//                                        +---- plane 3
//
//                      BMF_8BPP_VGA256:
//
//                          This field is ignored, the palette entries and its
//                          order is defined by halftone DLL at run time, the
//                          caller should get the palette for the VGA256
//                          surface through HT_Get8BPPFormatPalette() API call.
//
//                      BMF_16BPP_555:
//
//                          PRIMARY_ORDER_abc
//                                        |||
//                                        ||+-- bit 0-4   (5 bits)
//                                        ||
//                                        |+--- bit 5-9   (5 bits)
//                                        |
//                                        +---- bit 10-15 (5 bits)
//
//
//  rclSrc          - RECTL data structure defined the source rectangle area
//                    to be bitblt from, fields in this data structure are
//                    relative to the source HTSURFACEINFO's width/height.
//
//  rclDest         - RECTL data structure defined the destination rectangle
//                    area to be bitblt to, fields in this data structure are
//                    relative to the destination HTSURFACEINFO's width/height.
//
//  rclClip         - RECTL data structure defined the destination clipping
//                    rectangle area, fields in this data structure are
//                    relative to the destination HTSURFACEINFO's width/height.
//
//  rclBand         - RECTL data structure defined the device banding rectangle
//                    area, fields in this data structure are relative to the
//                    destination HTSURFACEINFO's width/height.
//
//                    This RECTL only used if BBPF_HAS_BANDRECT flag is set,
//                    when this flag is set, halftone DLL will automatically
//                    clipped the destination to this rectangle area and
//                    copied this rectangle to the output buffer with rclBand's
//                    left/top aligned to the buffer's physical origin.  The
//                    destination's buffer (bitmap) must the format specified
//                    in the destination HTSURFACEINFO.
//
//                    If rclBand rectangle is larger than the logical destination
//                    surface size (destination HTSURFACEINFO), halftone dll
//                    still move the the band's left/top location to the 0/0
//                    origin and extra width/height is remain unchanged.
//
//                    The rclBand normally is used for device which does not
//                    have enough memory to hold all the destination surface
//                    at one time, it just like to repeatly using same buffer
//                    to temporary holding the halftone results.
//
//                    The rclBand's left/top/right/bottom may not be negative
//                    numbers.
//
//  ptlSrcMask      - a POINTL data structure to specified the logical
//                    coordinate of starting point for the source mask bitmap,
//                    this field only used if a HTSURFACEINFO for the source
//                    mask is passed.
//
//                    This source mask bitmap must always monochrome and its
//                    width/height must
//
//                      Width  >= ptlSrcMask.x + source surface width.
//                      Height >= ptlSrcMask.y + source surface height;
//
//
//  NOTE:   1) all RECTL data structure are left/top inclusive and right/bottom
//             exclusive.
//
//          2) if rclSrc rectangle is not will ordered it specified the source
//             should be inverted before process for not ordered directions.
//
//          3) if rclDest rectangle is not will ordered it specified the
//             destination should be inverted after process for not ordered
//             directions.
//
//          4) if BBPF_HAS_DEST_CLIPRECT flag is set and rclClip is not well
//             ordered or its left equal to its right, or its top equal to its
//             bottom, then all destination are clipped, destination will not
//             be updated that is.
//
//          5) if BBPF_HAS_BANDRECT flag is set and rclBand is not well orderd
//             or it left eqaul to its right, or its top eqaul to its bottom,
//             then a HTERR_INVALID_BANDRECT is returned.
//
//

#define BBPF_HAS_DEST_CLIPRECT      W_BITPOS(0)
#define BBPF_USE_ADDITIVE_PRIMS     W_BITPOS(1)
#define BBPF_NEGATIVE_DEST          W_BITPOS(2)
#define BBPF_INVERT_SRC_MASK        W_BITPOS(3)
#define BBPF_HAS_BANDRECT           W_BITPOS(4)
#define BBPF_BW_ONLY                W_BITPOS(5)
#define BBPF_FILL_MODE              W_BITPOS(6)



#define BBP_FILL_NTSC_COLOR_BAR     0
#define BBP_FILL_MEMORY_COLORS      1
#define BBP_FILL_VGA_16_PALETTE     2
#define BBP_FILL_COLOR_TABLE        3
#define BBP_FILL_MAX_INDEX          3


typedef struct _BITBLTPARAMS {
    WORD    Flags;
    BYTE    FillModeIndex;
    BYTE    DestPrimaryOrder;
    LPWORD  pAbort;
    RECTL   rclSrc;
    RECTL   rclDest;
    RECTL   rclClip;
    RECTL   rclBand;
    POINTL  ptlBrushOrg;
    POINTL  ptlSrcMask;
    } BITBLTPARAMS, FAR *PBITBLTPARAMS;



//
// DEVICEHALFTONEINFO
//
//  This data structure is passed for every HT_xxx api calls except the
//  HT_CreateDeviceHalftoneInfo() which return the pointer to this data
//  structure.  It is used to identify the device color characteristics
//  durning the halftone process.
//
//  DeviceOwnData       - this field will initially set to NULL, and will be
//                        used by the caller to stored useful information
//                        such as handle/pointer.
//
//  cxPattern           - width of the halftone pattern in pels
//
//  cyPattern           - height of the halftone pattern in pels.
//
//  HTColorAdjustment   - Current default color adjustment, if an halftone
//                        APIs required a PHTCOLORADJUSTMENT parameter and its
//                        passed as NULL pointer then default color adjustment
//                        is taken from here, the caller can change the
//                        HTCOLORADJUSTMENT data structure to affect all the
//                        color adjustment on this device.
//

typedef struct _DEVICEHALFTONEINFO {
    DWORD               DeviceOwnData;
    WORD                cxPattern;
    WORD                cyPattern;
    HTCOLORADJUSTMENT   HTColorAdjustment;
    } DEVICEHALFTONEINFO;

typedef DEVICEHALFTONEINFO  FAR *PDEVICEHALFTONEINFO;
typedef PDEVICEHALFTONEINFO FAR *PPDEVICEHALFTONEINFO;


//
// CHBINFO
//
//  This data structure is one of the parameter passed to the halftone entry
//  point HT_CreateHalftoneBrush();
//
//  Flags                   - one or more following flags can be defined
//
//                              CHBF_BW_ONLY
//
//                                  Create only black/white even the device is
//                                  color.
//
//                              CHBF_USE_ADDITIVE_PRIMS
//
//
//                                  Specified if the halftone result will be
//                                  using Red/Green/Blue primary color or using
//                                  Cyan/Magenta/Yellow primary color, depends
//                                  on the destination surface format as
//
//                                  BMF_1BPP:
//
//                                          Additive Prims: 0=Black, 1=White
//                                      Substractive Prims: 0=White, 1=Black
//
//                                  BMF_4BPP_VGA16:
//                                  BMF_8BPP_VGA256:
//                                  BMF_16BPP_555:
//
//                                      Always using RED, GREEN, BLUE primaries
//                                      and this flag is ignored.
//
//                                  BMF_4BPP:
//                                  BMF_1BPP_3PLANE:
//
//                                      Additive Prims: RED, GREEN. BLUE
//                                  Substractive Prims: CYAN, MAGENTA, YELLOW
//
//                                  The order of the RGB, or CMY is specified
//                                  by DestPrimaryOrder field. (see below)
//
//                              CHBF_NEGATIVE_BRUSH
//
//                                  Create the negative version of the brush.
//
//
//  DestSurfaceFormat       - One of the following can be specified,
//
//                              BMF_1BPP, BMF_4BPP, BMF_4BPP_VGA16,
//                              BMF_8BPP_VGA256, BMF_1BPP_3PLANES and
//                              BMF_16BPP_555.
//
//                              for VGA16, VGA256, 16BPP_555  surface format
//                              see HTSURFACEINFO for descriptions.
//
//  DestScanLineAlignBytes  - Alignment bytes needed for each output pattern
//                            scan line, some common ones are
//
//                              BMF_ALIGN_BYTE      ( 8-bit aligned)
//                              BMF_ALIGN_WORD      (16-bit aligned)
//                              BMF_ALIGN_DWORD     (32-bit aligned)
//                              BMF_ALIGN_QWORD     (64-bit aligned)
//
//  DestPrimaryOrder        - Specified destination primary color order, it can
//                            be either PRIMARY_ORDER_RGB or PRIMARY_ORDER_CMY
//                            group, it depends on the surface format has
//                            following meaning.
//
//                              BMF_1BPP:
//
//                                  This field is ignored.
//
//
//                              BMF_4BPP_VGA16:
//
//                                  This field automatically set to
//                                  PRIMARY_ORDER_BGR by the halftone DLL.
//
//                              BMF_4BPP:
//
//                                  for each byte there are two indices
//                                  entries, and for each nibble has following
//                                  meaning, notice that bit 3/7 always set to
//                                  0, the index number only range from 0 to 7.
//
//                                      PRIMARY_ORDER_abc
//                                                    |||
//                                                    ||+-- bit 0/4
//                                                    ||
//                                                    |+--- bit 1/5
//                                                    |
//                                                    +---- bit 2/7
//
//
//                              BMF_1BPP_3PLANES:
//
//                                      PRIMARY_ORDER_abc
//                                                    |||
//                                                    ||+-- plane 1
//                                                    ||
//                                                    |+--- plane 2
//                                                    |
//                                                    +---- plane 3
//
//                              BMF_8BPP_VGA256:
//
//                                  This field is ignored, the palette entries
//                                  and its order is defined by halftone DLL
//                                  at run time, the caller should get the
//                                  palette for the VGA256 surface through
//                                  HT_Get8BPPFormatPalette() API call.
//
//                              BMF_16BPP_555:
//
//                                  PRIMARY_ORDER_abc
//                                                |||
//                                                ||+-- bit 0-4   (5 bits)
//                                                ||
//                                                |+--- bit 5-9   (5 bits)
//                                                |
//                                                +---- bit 10-15 (5 bits)
//
//
//


#define CHBF_BW_ONLY                    0x01
#define CHBF_USE_ADDITIVE_PRIMS         0x02
#define CHBF_NEGATIVE_BRUSH             0x04
#define CHBF_BOTTOMUP_BRUSH             0x08

typedef struct _CHBINFO {
    BYTE        Flags;
    BYTE        DestSurfaceFormat;
    BYTE        DestScanLineAlignBytes;
    BYTE        DestPrimaryOrder;
    } CHBINFO;



//
// STDMONOPATTERN
//
//  This data structure is used when calling the HT_CreateStdMonoPattern().
//
//  Flags               - One or more following flags may be defined
//
//                          SMP_TOPDOWN
//
//                              Specified that first scan line of the pattern
//                              bitmap will be the viewing top, if this flag
//                              is not defined then the last scan line is the
//                              viewing top.
//
//                          SMP_0_IS_BLACK
//
//                              specified that the in the bitmap a bit value
//                              0 = black and bit value 1=white, if this flag
//                              is not defined then bit value 0=white and
//                              bit value 1=black.
//
//  ScanLineAlignBytes  - Alignment bytes needed for each output pattern scan
//                        line, some common ones are
//
//                          BMF_ALIGN_BYTE      ( 8-bit aligned)
//                          BMF_ALIGN_WORD      (16-bit aligned)
//                          BMF_ALIGN_DWORD     (32-bit aligned)
//                          BMF_ALIGN_QWORD     (64-bit aligned)
//
//  PatternIndex        - Specified the pattern index number, this has the
//                        predefined value as HT_STDMONOPAT_xxxx.  If a invalid
//                        index number is passed then it return an error
//                        HTERR_INVALID_STDMONOPAT_INDEX is returned.
//
//  LineWidth           - This field only applied to the pattern which has
//                        lines in them, the value range from 0-255 (byte) and
//                        it repesent as LineWidth/1000 of inch, for example
//                        a 3 indicate 3/1000 = 0.003 inch width, if this value
//                        is less than the device minimum pel size, it will
//                        default to the 1 pel, the maximum is 255/1000 = 0.255
//                        inch width.
//
//                        If a zero is specified then it halftone will using
//                        default line width settting.
//
//  LinesPerInch        - This field only applied to the pattern wich has lines
//                        in them, the value range from 0 to 255 (byte). The
//                        LinesPerInch is calculated in the perpendicular
//                        direction of two parallel lines, the distances
//                        between two parallel lines that is.
//
//                        If a zero is specified then it halftone will using
//                        default line per inch setting.
//
//  BytesPerScanLine    - If will be filled by halftone dll to specified the
//                        size in bytes for each scan line in the pattern.
//
//  cxPels              - It will be filled by halftone dll of the final
//                        pattern's width in pel.
//
//  cyPels              - It will be filled by halftone dll of the final
//                        pattern's height in scan line.
//
//  pPattern            - Specified the memory location where the pattern will
//                        be stored, if this field is NULL then it will fill in
//                        the width/height fields.
//
//




#define HT_SMP_HORZ_LINE                0
#define HT_SMP_VERT_LINE                1
#define HT_SMP_HORZ_VERT_CROSS          2
#define HT_SMP_DIAG_15_LINE_UP          3
#define HT_SMP_DIAG_15_LINE_DOWN        4
#define HT_SMP_DIAG_15_CROSS            5
#define HT_SMP_DIAG_30_LINE_UP          6
#define HT_SMP_DIAG_30_LINE_DOWN        7
#define HT_SMP_DIAG_30_CROSS            8
#define HT_SMP_DIAG_45_LINE_UP          9
#define HT_SMP_DIAG_45_LINE_DOWN        10
#define HT_SMP_DIAG_45_CROSS            11
#define HT_SMP_DIAG_60_LINE_UP          12
#define HT_SMP_DIAG_60_LINE_DOWN        13
#define HT_SMP_DIAG_60_CROSS            14
#define HT_SMP_DIAG_75_LINE_UP          15
#define HT_SMP_DIAG_75_LINE_DOWN        16
#define HT_SMP_DIAG_75_CROSS            17

#define HT_SMP_PERCENT_SCREEN_START     (HT_SMP_DIAG_75_CROSS + 1)
#define HT_SMP_PERCENT_SCREEN(x)        (x + HT_SMP_PERCENT_SCREEN_START)

#define HT_SMP_MAX_INDEX                HT_SMP_PERCENT_SCREEN(100)


#define SMP_TOPDOWN             W_BITPOS(0)
#define SMP_0_IS_BLACK          W_BITPOS(1)


typedef struct _STDMONOPATTERN {
    WORD    Flags;
    BYTE    ScanLineAlignBytes;
    BYTE    PatternIndex;
    BYTE    LineWidth;
    BYTE    LinesPerInch;
    WORD    BytesPerScanLine;
    WORD    cxPels;
    WORD    cyPels;
    LPBYTE  pPattern;
    } STDMONOPATTERN;

typedef STDMONOPATTERN FAR *PSTDMONOPATTERN;


//
// Following is used in ConvertColorTable
//


#define CCTF_BW_ONLY        0x0001
#define CCTF_NEGATIVE       0x0002


//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//                                                                          //
// Exposed Halftone DLL APIs                                                //
//                                                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#ifndef _HTAPI_ENTRY_

LONG
APIENTRY
HT_CreateDeviceHalftoneInfo(
    PHTINITINFO             pHTInitInfo,
    PPDEVICEHALFTONEINFO    ppDeviceHalftoneInfo
    );

/*++

Routine Description:

    This function initialize a device to the halftone dll, it calculate all
    the necessary parameters for the device and return a pointer points to
    the DEVICEHALFTONEINFO data structure back to the caller.

    NOTE: return pointer will not be particulary anchor to a single physucal
          device, but rather to a group of physical devices, that is if the
          caller has simillar devices which share the same characteristics
          then it may use the same pointer to halftone the bitmap.

Arguments:

    pHTInitInfo - Pointer to the HTINITINFO data structure which describe
                  the device characteristics and other initialzation requests.


Return Value:

    The return value will be greater than 0L if the function sucessfully, and
    it will be an error code (less than or equal to 0) if function failed.

    If the function sucessful the the pointer location points by the
    ppDeviceHalftoneInfo will be updated to stored the pointer which points
    to the DEVICEHALFTONEINFO data structure for later any HT_xxxx() api calls,
    if the function failed, the storage points by the ppDeviceHalftoneInfo is
    undefined.

    This function may return following error codes.

    HTERR_INSUFFICIENT_MEMORY       - Not enough memory for halftone process.

    HTERR_HTPATTERN_SIZE_TOO_BIG    - Caller defined halftone pattern's width
                                      or height is excessed limit.

    HTERR_INVALID_HALFTONEPATTERN   - One or more HALFTONEPATTERN data
                                      structure field specified invalid values.


    Note: The first field in the DEVICEHALFTONEINFO (DeviceOwnData) is a 32-bit
          area which will be set to 0L upon sucessful returned, the caller can
          put any data in this field.

Author:

    05-Feb-1991 Tue 10:54:32 created  -by-  Daniel Chou (danielc)


Revision History:

    05-Jun-1991 Wed 10:22:07 updated  -by-  Daniel Chou (danielc)

        Fixed the typing errors for halftone pattern default


--*/



BOOL
APIENTRY
HT_DestroyDeviceHalftoneInfo(
    PDEVICEHALFTONEINFO     pDeviceHalftoneInfo
    );

/*++

Routine Description:

    This function destroy the handle which returned from halftone initialize
    function HT_CreateDeviceHalftoneInfo()

Arguments:

    pDeviceHalftoneInfo - Pointer to the DEVICEHALFTONEINFO data structure
                          which returned from the HT_CreateDeviceHalftoneInfo.

Return Value:

    TRUE    - if function sucessed.
    FALSE   - if function failed.

Author:

    05-Feb-1991 Tue 14:18:20 created  -by-  Daniel Chou (danielc)


Revision History:


--*/



LONG
APIENTRY
HT_CreateHalftoneBrush(
    PDEVICEHALFTONEINFO pDeviceHalftoneInfo,
    PHTCOLORADJUSTMENT  pHTColorAdjustment,
    PCOLORTRIAD         pColorTriad,
    CHBINFO             CHBInfo,
    LPVOID              pOutputBuffer
    );

/*++

Routine Description:

    This function create halftone mask for the requested solid color.

Arguments:

    pDeviceHalftoneInfo - Pointer to the DEVICEHALFTONEINFO data structure
                          which returned from the HT_CreateDeviceHalftoneInfo.

    pHTColorAdjustment  - Pointer to the HTCOLORADJUSTMENT data structure to
                          specified the input/output color adjustment/transform,
                          if this pointer is NULL then a default color
                          adjustments is applied.

    pColorTriad         - Pointer to the COLORTRIAD data structure to describe
                          the brush colors.

    CHBInfo             - CHBINFO data structure, see CHBINFO data structure
                          for descriptions.

    pOutputBuffer       - Pointer to the buffer area to received indices/mask.
                          in bytes needed to stored the halftone pattern.


Return Value:

    if the return value is negative or zero then an error was encountered,
    possible error codes are

        HTERR_INVALID_DHI_POINTER           - Invalid pDevideHalftoneInfo is
                                              passed.

        HTERR_INVALID_DEST_FORMAT           - the Format of the destination
                                              surface is not one of the defined
                                              HSC_FORMAT_xxxx

        HTERR_CHB_INV_COLORTABLE_SIZE       - Color table size is not 1

    otherwise

        If pSurface is NULL, it return the bytes count which need to stored
        the pattern, otherwise it return the size in byte copied to the output
        buffer.

        form BMF_1BPP_3PLANES, the return value is the size for one plane, the
        caller should multiply it by 3 to get the correct buffer size.

Author:

    05-Feb-1991 Tue 14:28:23 created  -by-  Daniel Chou (danielc)


Revision History:


--*/


LONG
APIENTRY
HT_ConvertColorTable(
    PDEVICEHALFTONEINFO pDeviceHalftoneInfo,
    PHTCOLORADJUSTMENT  pHTColorAdjustment,
    PCOLORTRIAD         pColorTriad,
    DWORD               Flags
    );


/*++

Routine Description:

    This function modified input color table entries base on the
    pHTColorAdjustment data structure specification.

Arguments:

    pDeviceHalftoneInfo - Pointer to the DEVICEHALFTONEINFO data structure
                          which returned from the HT_CreateDeviceHalftoneInfo.

    pHTColorAdjustment  - Pointer to the HTCOLORADJUSTMENT data structure to
                          specified the input/output color adjustment/transform,
                          if this pointer is NULL then a default color
                          adjustments is applied.

    pColorTriad         - Specified the source color table format and location.

    Flags               - One of the following may be specified

                            CCTF_BW_ONLY

                                Create grayscale of the color table.

                            CCTF_NEGATIVE

                                Create negative version of the original color
                                table.


Return Value:

    if the return value is negative or zero then an error was encountered,
    possible error codes are

        HTERR_INVALID_COLOR_TABLE   - The ColorTableEntries field is = 0 or
                                      CCTInfo.SizePerColorTableEntry is not
                                      between 3 to 255, or if the
                                      CCTInfo.FirstColorIndex in CCTInfo is
                                      not in the range of 0 to
                                      (SizePerColorTableEntry - 3).

        HTERR_INVALID_DHI_POINTER   - Invalid pDevideHalftoneInfo is passed.

    otherwise

        Total entries of the converted color table is returned.


Author:

    14-Aug-1991 Wed 12:43:29 updated  -by-  Daniel Chou (danielc)


Revision History:


--*/


//
// HT_ComputeRGBGammaTable() and HT_Get8BPPFormatPalette() prototypes are moved
// to the winddi.h for windows
//

#if 0   // To WinDDI.h


LONG
APIENTRY
HT_ComputeRGBGammaTable(
    WORD    GammaTableEntries,
    WORD    GammaTableType,
    UDECI4  RedGamma,
    UDECI4  GreenGamma,
    UDECI4  BlueGamma,
    LPBYTE  pGammaTable
    );

/*++

Routine Description:

    This function compute device gamma correction table.

                                                      (1/RedGamma)
        RedGamma[N]   = INT((N / (GammaTableEntries-1))              * 255)

                                                      (1/GreenGamma)
        GreenGamma[N] = INT((N / (GammaTableEntries-1))              * 255)

                                                      (1/BlueGamma)
        BlueGamma[N]  = INT((N / (GammaTableEntries-1))              * 255)


        1. INT() is a integer function which round up to next integer number if
           resulting fraction is 0.5 or higher, the final result always limit
           to have range between 0 and 255.

        2. N is a integer step number and range from 0 to (GammaTableEntries-1)
           in one (1) increment.


Arguments:

    GammaTableEntries       - Total gamma table entries for each of red, green
                              and blue gamma table, halftone dll normalized
                              the gamma table with step value computed as
                              1/GammaTableEntries.

                              This value must range from 3 to 255 else a 0
                              is returned and no table is updated.

    GammaTableType          - red, green and blue gamma table organizations

                                0 - The gamma table is Red, Green, Blue 3 bytes
                                    for each gamma step entries and total of
                                    GammaTableEntries entries.

                                1 - The gamma table is Red Gamma tables follow
                                    by green gamma table then follow by blue
                                    gamma table, each table has total of
                                    GammaTableEntries bytes.

                                Other value default to 0.

    RedGamma                - Red gamma number in UDECI4 format

    GreenGamma              - Green gamma number in UDECI4 format

    BlueGamma               - Blue gamma number in UDECI4 format

    pGammaTable             - pointer to the gamma table byte array.
                              each output gamma number is range from 0 to 255.


Return Value:

    Return value is the total table entries updated.

Author:

    17-Jul-1992 Fri 19:04:31 created    -by-  Daniel Chou (danielc)


Revision History:


--*/


LONG
APIENTRY
HT_Get8BPPFormatPalette(
    LPPALETTEENTRY  pPaletteEntry,
    UDECI4          RedGamma,
    UDECI4          GreenGamma,
    UDECI4          BlueGamma
    );

/*++

Routine Description:

    This functions retrieve a halftone's VGA256 color table definitions

Arguments:

    pPaletteEntry   - Pointer to PALETTEENTRY data structure array,

    RedGamma        - The monitor's red gamma value in UDECI4 format

    GreenGamma      - The monitor's green gamma value in UDECI4 format

    BlueGamma       - The monitor's blue gamma value in UDECI4 format


Return Value:

    if pPaletteEntry is NULL then it return the PALETTEENTRY count needed for
    VGA256 halftone process, if it is not NULL then it return the total
    paletteEntry updated.

    If the pPaletteEntry is not NULL then halftone.dll assume it has enough
    space for the size returned when this pointer is NULL.

Author:

    14-Apr-1992 Tue 13:03:21 created  -by-  Daniel Chou (danielc)


Revision History:


--*/


#endif  // To WinDDI.h



LONG
APIENTRY
HT_CreateStandardMonoPattern(
    PDEVICEHALFTONEINFO pDeviceHalftoneInfo,
    PSTDMONOPATTERN     pStdMonoPattern
    );

/*++

Routine Description:

    This function create standard predefined monochrome pattern for the device.

Arguments:

    pDeviceHalftoneInfo - Pointer to the DEVICEHALFTONEINFO data structure
                          which returned from the HT_CreateDeviceHalftoneInfo.

    pStdMonoPattern     - Pointer to the STDMONOPATTERN data structure, the
                          pPattern in this data structure is optional.

Return Value:

    if the return value is negative or zero then an error was encountered,
    possible error codes are

        HTERR_INVALID_DHI_POINTER           - Invalid pDevideHalftoneInfo is
                                              passed.

        HTERR_INVALID_STDMONOPAT_INDEX      - The PatternIndex field in
                                              STDMONOPATTERN data structure is
                                              invalid.
    otherwise

        If pPattern field in STDMONOPATTERN data structure Surface is NULL, it
        return the bytes count which need to stored the pattern, otherwise it
        return the size in byte copied to the pattern buffer.

Author:

    05-Feb-1991 Tue 14:28:23 created  -by-  Daniel Chou (danielc)


Revision History:

    05-Jun-1991 Wed 10:22:41 updated  -by-  Daniel Chou (danielc)

        Fixed the bugs when the pStdMonoPattern is NULL, it was used without
        checking it.


--*/



LONG
APIENTRY
HT_HalftoneBitmap(
    PDEVICEHALFTONEINFO pDeviceHalftoneInfo,
    PHTCOLORADJUSTMENT  pHTColorAdjustment,
    PHTSURFACEINFO      pSourceHTSurfaceInfo,
    PHTSURFACEINFO      pSourceMaskHTSurfaceInfo,
    PHTSURFACEINFO      pDestinationHTSurfaceInfo,
    PBITBLTPARAMS       pBitbltParams
    );

/*++

Routine Description:

    This function halftone the source bitmap and output to the destination
    surface depends on the surface type and bitblt parameters

    The source surface type must one of the following:

        1-bit per pel. (BMF_1BPP)
        4-bit per pel. (BMF_4BPP)
        8-bit per pel. (BMF_8BPP)
       16-bit per pel. (BMF_16BPP)
       24-bit per pel. (BMF_24BPP)
       32-bit per pel. (BMF_32BPP)

    The destination surface type must one of the following:

        1-bit per pel.                  (BMF_1BPP)
        4-bit per pel.                  (BMF_4BPP)
        3 plane and 1 bit per pel.      (BMF_1BPP_3PLANES)

Arguments:

    pDeviceHalftoneInfo         - pointer to the DEVICEHALFTONEINFO data
                                  structure

    pHTColorAdjustment          - Pointer to the HTCOLORADJUSTMENT data
                                  structure to specified the input/output color
                                  adjustment/transform, if this pointer is NULL
                                  then a default color adjustments is applied.

    pSourceHTSurfaceInfo        - pointer to the source surface infomation.

    pSourceMaskHTSurfaceInfo    - pointer to the source mask surface infomation,
                                  if this pointer is NULL then there is no
                                  source mask for the halftoning.

    pDestinationHTSurfaceInfo   - pointer to the destination surface infomation.

    pBitbltParams               - pointer to the BITBLTPARAMS data structure to
                                  specified the source, destination, source
                                  mask and clipping rectangle information, the
                                  content of this data structure will not be
                                  modified by this function.


Return Value:

    if the return value is less than zero then an error has occurred,
    the error code is one of the following #define which start with HTERR_.

    HTERR_INSUFFICIENT_MEMORY           - not enough memory to do the halftone
                                          process.

    HTERR_COLORTABLE_TOO_BIG            - can not create the color table to map
                                          the colors to the dyes' densities.

    HTERR_QUERY_SRC_BITMAP_FAILED       - callback function return FALSE when
                                          query the source bitmap pointer.

    HTERR_QUERY_DEST_BITMAP_FAILED      - callback function return FALSE when
                                          query the destination bitmap pointers.

    HTERR_INVALID_SRC_FORMAT            - Invalid source surface format.

    HTERR_INVALID_DEST_FORMAT           - Invalid destination surface type,
                                          this function only recongnized 1/4/
                                          bits per pel source surfaces or 1 bit
                                          per pel 3 planes.

    HTERR_INVALID_DHI_POINTER           - Invalid pDevideHalftoneInfo is passed.

    HTERR_SRC_MASK_BITS_TOO_SMALL       - If the source mask bitmap is too
                                          small to cover the visible region of
                                          the source bitmap.

    HTERR_INVALID_MAX_QUERYLINES        - One or more of Source/Destination
                                          SourceMasks' maximum query scan line
                                          is < 0

    HTERR_INTERNAL_ERRORS_START         - any other negative numbers indicate
                                          a halftone internal failue.

   else                                - the total destination scan lines
                                          halftoned.


Author:

    05-Feb-1991 Tue 15:23:07 created  -by-  Daniel Chou (danielc)


Revision History:


--*/

#if defined(__cplusplus)
};
#endif

#endif  // _HTAPI_ENTRY_
#endif  // _HT_
