//==========================================================================;
//
//  acmthunk.c
//
//  Copyright (c) 1991-1993 Microsoft Corporation.  All Rights Reserved.
//
//  Description:
//      This module contains routines for assisting thunking of the ACM
//      APIs from 16-bit Windows to 32-bit WOW.
//
//  History:
//
//==========================================================================;

/*

    WOW Thunking design:

        Thunks are generated as follows :

        16-bit :
           acmBootDrivers->acmInitThunks :

               Generate calls to 32-bit drivers if we're in WOW call across
               to KERNEL to find thunking entry points.

               If we're thunking 'load' all the 32-bit ACM drivers as well as
               the 16-bit ones.

               Priority is always to find a 32-bit driver first but this is
               done via searching for one on open.

               The internal flag ACM_DRIVERADDF_32BIT is specified when
               calling IDriverAdd and this flag is stored in the ACMDRIVERID
               structure.

           IDriverAdd->IDriverLoad->IDriverLoad32

               The 16-bit side calls the 32-bit side passing in the driver
               alias which is used to compare against the aliases on the 32
               bit side and the 32-bit HACMDRIVERID is passed back for the
               relevant driver and stored in the hdrvr field of the
               ACMDRIVERID structure.

           IDriverOpen->IDriverOpen32

               The parameters are passed to the 32-bit side using the hdrvr
               field deduced from the HACMDRIVERID as the 32-bit HACMDRIVERID.

           IDriverMessageId->IDriverMessageId32 :

               If the driver is 32-bit (as identified in the ACMDRIVERID
               structure) then call IDriverMessageId32.  The hadid for
               the 32-bit driver is stored in the hdrvr field of ACMDRIVERID
               on the 16-bit side.

           IDriverMessage->IDriverMessage32

               If the driver is 32-bit (as identified in the ACMDRIVERID
               structure pointed to by the ACMDRIVER structure) then call
               IDriverMessage32.  The had for the 32-bit driver is stored
               in the hdrvr field of ACMDRIVER on the 16-bit side.

           Stream headers

               These must be persistent on the 32-bit side too and kept
               in synch.

               They are allocated on the 32-bit side for ACMDM_STREAM_PREPARE
               and freed on ACMDM_STREAM_UNPREPARE.  While in existence
               the 32-bit stream header is stored in the dwDriver field in

*/

/*
    Additional Chicago implementation notes:

	PUT SOMETHING HERE FRANK!!!

*/
    

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <mmddk.h>
#include <mmreg.h>
#include <memory.h>

#ifdef WIN32
#ifndef WIN4
#include <wownt32.h>
#endif
#endif // WIN32

#include "msacm.h"
#include "msacmdrv.h"
#include "acmi.h"
#include "pcm.h"
#include "chooseri.h"
#include "uchelp.h"
#include "acmthunk.h"
#include "debug.h"

#ifdef WIN32


/* -------------------------------------------------------------------------
** Handle and memory mapping functions.
** -------------------------------------------------------------------------
*/

//
//  16-bit structures
//

typedef struct {
    DWORD   dwDCISize;
    LPCSTR  lpszDCISectionName;
    LPCSTR  lpszDCIAliasName;
} DRVCONFIGINFO16;



#ifdef WIN4
PVOID FNLOCAL ptrFixMap16To32(const VOID * pv)
{
    return MapSLFix(pv);
}

VOID FNLOCAL ptrUnFix16(const VOID * pv)
{
    UnMapSLFixArray(1, &pv);
}

#else
LPWOWHANDLE32          lpWOWHandle32;
LPWOWHANDLE16          lpWOWHandle16;
LPGETVDMPOINTER        GetVdmPointer;
int                    ThunksInitialized;

PVOID FNLOCAL ptrFixMap16To32(const VOID * pv)
{
    return WOW32ResolveMemory(pv);
}

VOID FNLOCAL ptrUnFix16(const VOID * pv)
{
    return;
}

#endif

//
//  Useful functions
//

#define WaveFormatSize(pv)                                            \
    (((WAVEFORMATEX UNALIGNED *)(pv))->wFormatTag == WAVE_FORMAT_PCM ?\
        sizeof(PCMWAVEFORMAT) :                                       \
        ((WAVEFORMATEX UNALIGNED *)pv)->cbSize + sizeof(WAVEFORMATEX))

PVOID CopyAlloc(
    PVOID   pvSrc,
    UINT    uSize
)
{
    PVOID   pvDest;

    pvDest = (PVOID)LocalAlloc(LMEM_FIXED, uSize);

    if (pvDest != NULL) {
        CopyMemory(pvDest, pvSrc, uSize);
    }

    return pvDest;
}

//
//  Thunking callbacks to WOW32 (or wherever)
//


MMRESULT IThunkFilterDetails
(
    HACMDRIVERID                 hadid,
    ACMFILTERDETAILSA UNALIGNED *pafd16,
    DWORD                        fdwDetails
)
{
    ACMFILTERDETAILSA UNALIGNED *pafd;
    ACMFILTERDETAILSW afdw;
    PWAVEFILTER       pwfl;
    UINT              uRet;

    //
    //  Map pointers to 32-bit
    //

    pafd = ptrFixMap16To32((PVOID)pafd16);
    pwfl = ptrFixMap16To32((PVOID)pafd->pwfltr);
    
    //
    //  Thunk the format details structure
    //  The validation on the 16-bit side ensures that the 16-bit
    //  structure contains all the necessary fields.
    //
    afdw.cbStruct       = sizeof(afdw);
    afdw.dwFilterIndex  = pafd->dwFilterIndex;
    afdw.dwFilterTag    = pafd->dwFilterTag;
    afdw.fdwSupport     = pafd->fdwSupport;
    afdw.pwfltr         = (PWAVEFILTER)CopyAlloc(pwfl, pafd->cbwfltr);


    if (afdw.pwfltr == NULL) {
    	ptrUnFix16((PVOID)pafd->pwfltr);
	    ptrUnFix16((PVOID)pafd16);
        return MMSYSERR_NOMEM;
    }

    afdw.cbwfltr        = pafd->cbwfltr;

    //
    //  Copy the string if it's used
    //

    Imbstowcs(afdw.szFilter, (LPSTR)pafd->szFilter, sizeof(pafd->szFilter));

    //
    //  Call the driver
    //

    uRet =
        ((PACMDRIVER)hadid)->uHandleType == TYPE_HACMDRIVERID ?
         IDriverMessageId(
            hadid,
            ACMDM_FILTER_DETAILS,
            (DWORD)&afdw,
            fdwDetails) :
         IDriverMessage(
            (HACMDRIVER)hadid,
            ACMDM_FILTER_DETAILS,
            (DWORD)&afdw,
            fdwDetails);

    //
    //  If successful copy back the format info
    //

    if (uRet == MMSYSERR_NOERROR) {
        pafd->dwFilterTag    = afdw.dwFilterTag;
        pafd->fdwSupport     = afdw.fdwSupport;
        CopyMemory((PVOID)pwfl, (PVOID)afdw.pwfltr, afdw.cbwfltr);
        Iwcstombs((LPSTR)pafd->szFilter, afdw.szFilter, sizeof(pafd->szFilter));

    }

    LocalFree((HLOCAL)afdw.pwfltr);

    ptrUnFix16((PVOID)pafd->pwfltr);
    ptrUnFix16((PVOID)pafd16);

    return uRet;
}

MMRESULT IThunkFormatDetails
(
    HACMDRIVERID                 hadid,
    ACMFORMATDETAILSA UNALIGNED *pafd16,
    DWORD                        fdwDetails
)
{
    ACMFORMATDETAILSA UNALIGNED *pafd;
    ACMFORMATDETAILSW afdw;
    PWAVEFORMATEX     pwfx;
    UINT              uRet;

    //
    //  Map pointers to 32-bit
    //

    pafd = ptrFixMap16To32((PVOID)pafd16);
    pwfx = ptrFixMap16To32((PVOID)pafd->pwfx);


    //
    //  Thunk the format details structure
    //  The validation on the 16-bit side ensures that the 16-bit
    //  structure contains all the necessary fields.
    //
    afdw.cbStruct       = sizeof(afdw);
    afdw.dwFormatIndex  = pafd->dwFormatIndex;
    afdw.dwFormatTag    = pafd->dwFormatTag;
    afdw.fdwSupport     = pafd->fdwSupport;
    afdw.pwfx           = (PWAVEFORMATEX)CopyAlloc(pwfx, pafd->cbwfx);

    if (afdw.pwfx == NULL) {
    	ptrUnFix16((PVOID)pafd->pwfx);
	    ptrUnFix16((PVOID)pafd16);
        return MMSYSERR_NOMEM;
    }

    afdw.cbwfx          = pafd->cbwfx;

    //
    //  Copy the string if it's used
    //

    Imbstowcs(afdw.szFormat, (LPSTR)pafd->szFormat, sizeof(pafd->szFormat));

    //
    //  Call the driver
    //

    uRet =
        ((PACMDRIVER)hadid)->uHandleType == TYPE_HACMDRIVERID ?
         IDriverMessageId(
            hadid,
            ACMDM_FORMAT_DETAILS,
            (DWORD)&afdw,
            fdwDetails) :
         IDriverMessage(
            (HACMDRIVER)hadid,
            ACMDM_FORMAT_DETAILS,
            (DWORD)&afdw,
            fdwDetails);

    //
    //  If successful copy back the format info
    //

    if (uRet == MMSYSERR_NOERROR) {

        //
        //  Someone should be shot for designing interfaces with
        //  inputs and outputs in the same structure!!
        //
        pafd->dwFormatTag    = afdw.dwFormatTag;
        pafd->fdwSupport     = afdw.fdwSupport;
        CopyMemory((PVOID)pwfx, (PVOID)afdw.pwfx, afdw.cbwfx);
        Iwcstombs((LPSTR)pafd->szFormat, afdw.szFormat, sizeof(pafd->szFormat));
    }
    
    LocalFree((HLOCAL)afdw.pwfx);
    ptrUnFix16((PVOID)pafd->pwfx);
    ptrUnFix16((PVOID)pafd16);
    
    return uRet;
}
MMRESULT IThunkFormatSuggest
(
    HACMDRIVERID                    hadid,
    ACMDRVFORMATSUGGEST UNALIGNED  *pafs16
)
{
    ACMDRVFORMATSUGGEST UNALIGNED  *pafs;
    ACMDRVFORMATSUGGEST afs;
    PWAVEFORMATEX       pwfxSrc;
    PWAVEFORMATEX       pwfxDst;
    UINT                uRet;

    //
    //  Map pointers to 32-bit
    //

    pafs    = ptrFixMap16To32((PVOID)pafs16);
    pwfxSrc = ptrFixMap16To32((PVOID)pafs->pwfxSrc);
    pwfxDst = ptrFixMap16To32((PVOID)pafs->pwfxDst);

    //
    //  Thunk the format details structure
    //  The validation on the 16-bit side ensures that the 16-bit
    //  structure contains all the necessary fields.
    //
    CopyMemory((PVOID)&afs, (PVOID)pafs, sizeof(afs));

    //
    //  Deal with the wave format pointers
    //
    afs.pwfxSrc        =
        (PWAVEFORMATEX)CopyAlloc((PVOID)pwfxSrc, pafs->cbwfxSrc);

    if (afs.pwfxSrc == NULL) {
    	ptrUnFix16((PVOID)pafs->pwfxDst);
	    ptrUnFix16((PVOID)pafs->pwfxSrc);
    	ptrUnFix16((PVOID)pafs16);
        return MMSYSERR_NOMEM;
    }

    afs.pwfxDst        =
        (PWAVEFORMATEX)CopyAlloc((PVOID)pwfxDst, pafs->cbwfxDst);

    if (afs.pwfxDst == NULL) {
        LocalFree((HLOCAL)afs.pwfxSrc);
	    ptrUnFix16((PVOID)pafs->pwfxDst);
    	ptrUnFix16((PVOID)pafs->pwfxSrc);
	    ptrUnFix16((PVOID)pafs16);
        return MMSYSERR_NOMEM;
    }

    //
    //  Call the driver
    //

    uRet =
        ((PACMDRIVER)hadid)->uHandleType == TYPE_HACMDRIVERID ?
         IDriverMessageId(
            hadid,
            ACMDM_FORMAT_SUGGEST,
            (DWORD)&afs,
            0L) :
         IDriverMessage(
            (HACMDRIVER)hadid,
            ACMDM_FORMAT_SUGGEST,
            (DWORD)&afs,
            0L);

     //
     //  If successful copy back the format info
     //

     if (uRet == MMSYSERR_NOERROR) {
         CopyMemory((PVOID)pwfxDst, (PVOID)afs.pwfxDst, afs.cbwfxDst);
     }

     LocalFree((HLOCAL)afs.pwfxSrc);
     LocalFree((HLOCAL)afs.pwfxDst);
     ptrUnFix16((PVOID)pafs->pwfxDst);
     ptrUnFix16((PVOID)pafs->pwfxSrc);
     ptrUnFix16((PVOID)pafs16);

     return uRet;
}

LRESULT IThunkConfigure
(
    HACMDRIVERID              hadid,
    HWND                      hwnd,
    DRVCONFIGINFO16 UNALIGNED * pdrvConfigInfo1616
)
{
    DRVCONFIGINFO16 UNALIGNED * pdrvConfigInfo16;
    DRVCONFIGINFO drvConfigInfo;
    LRESULT       lResult;
    LPSTR	  lpszDCISectionNameA;
    LPSTR	  lpszDCIAliasNameA;
    LPWSTR        lpszDCISectionNameW;
    LPWSTR        lpszDCIAliasNameW;

    //
    //  Thunk the hwnd if necessary
    //

    if (hwnd != NULL && hwnd != (HWND)-1L) {
#ifdef WIN4
	//  ??? Don't think I need to do anything for Win4 ???
#else
	hwnd = (HWND)(*lpWOWHandle32)( (WORD)hwnd, WOW_TYPE_HWND);
#endif
    }

    drvConfigInfo.dwDCISize = sizeof(drvConfigInfo);

    //
    //  Thunk the config info if necessary
    //

    if (pdrvConfigInfo1616 != NULL) {
	//
	//  Map all the pointers
	//
        pdrvConfigInfo16    = ptrFixMap16To32((PVOID)pdrvConfigInfo1616);
    	lpszDCISectionNameA = ptrFixMap16To32((PVOID)pdrvConfigInfo16->lpszDCISectionName);
    	lpszDCIAliasNameA   = ptrFixMap16To32((PVOID)pdrvConfigInfo16->lpszDCIAliasName);
	
        drvConfigInfo.dwDCISize = sizeof(drvConfigInfo);
        lpszDCISectionNameW =
            (LPWSTR)
            LocalAlloc(LPTR,
                       (lstrlenA(lpszDCISectionNameA) + 1) * sizeof(WCHAR));

        if (lpszDCISectionNameW == NULL) {
	    ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCISectionName);
	    ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCIAliasName);
	    ptrUnFix16((PVOID)pdrvConfigInfo1616);
	    return MMSYSERR_NOMEM;
        }

        lpszDCIAliasNameW =
            (LPWSTR)
            LocalAlloc(LPTR,
                       (lstrlenA(lpszDCIAliasNameA) + 1) * sizeof(WCHAR));

        if (lpszDCIAliasNameW == NULL) {
            LocalFree((HLOCAL)lpszDCISectionNameW);
	    ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCISectionName);
	    ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCIAliasName);
	    ptrUnFix16((PVOID)pdrvConfigInfo1616);
	    return MMSYSERR_NOMEM;
        }

        Imbstowcs(lpszDCISectionNameW,
		  lpszDCISectionNameA,
		  lstrlenA(lpszDCISectionNameA) + 1);

        Imbstowcs(lpszDCIAliasNameW,
		  lpszDCIAliasNameA,
		  lstrlenA(lpszDCIAliasNameA) + 1);

        drvConfigInfo.lpszDCISectionName = lpszDCISectionNameW;
        drvConfigInfo.lpszDCIAliasName   = lpszDCIAliasNameW;
    }

    //
    //  Make the call
    //

    lResult =
        ((PACMDRIVER)hadid)->uHandleType == TYPE_HACMDRIVERID ?
         IDriverMessageId(
            hadid,
            DRV_CONFIGURE,
            (LPARAM)hwnd,
            (LPARAM)(pdrvConfigInfo16 == NULL ? NULL : &drvConfigInfo)) :
         IDriverMessage(
            (HACMDRIVER)hadid,
            DRV_CONFIGURE,
            (LPARAM)hwnd,
            (LPARAM)(pdrvConfigInfo16 == NULL ? NULL : &drvConfigInfo));

    if (pdrvConfigInfo16 != NULL) {
        LocalFree((HLOCAL)drvConfigInfo.lpszDCISectionName);
        LocalFree((HLOCAL)drvConfigInfo.lpszDCIAliasName);
	ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCISectionName);
	ptrUnFix16((PVOID)pdrvConfigInfo16->lpszDCIAliasName);
	ptrUnFix16((PVOID)pdrvConfigInfo1616);
    }

    return lResult;
}

BOOL IThunkStreamInstance
(
    ACMDRVSTREAMINSTANCE UNALIGNED *padsi16,
    PACMDRVSTREAMINSTANCE          padsi32
)
{
    PWAVEFORMATEX pwfxSrc;
    PWAVEFORMATEX pwfxDst;
    PWAVEFILTER   pwfltr16;

    pwfxSrc  = (PWAVEFORMATEX)ptrFixMap16To32((PVOID)padsi16->pwfxSrc);
    pwfxDst  = (PWAVEFORMATEX)ptrFixMap16To32((PVOID)padsi16->pwfxDst);
    pwfltr16 = (PWAVEFILTER)  ptrFixMap16To32((PVOID)padsi16->pwfltr);

    //
    //  The 16-bit side has 2 fewer bytes in the stream instance data
    //  because the handle is only 2 bytes
    //

    padsi32->has = NULL;
    CopyMemory((PVOID)padsi32, (PVOID)padsi16, sizeof(*padsi32) - 2);

    //
    //  Fix up the pointers
    //

    if (pwfxSrc != NULL) {
        padsi32->pwfxSrc = CopyAlloc((PVOID)pwfxSrc, WaveFormatSize(pwfxSrc));
        if (padsi32->pwfxSrc == NULL) {
	    ptrUnFix16((PVOID)padsi16->pwfltr);
	    ptrUnFix16((PVOID)padsi16->pwfxDst);
	    ptrUnFix16((PVOID)padsi16->pwfxSrc);
            return FALSE;
        }
    } else {
        padsi32->pwfxSrc = NULL;
    }


    if (pwfxDst != NULL) {
        padsi32->pwfxDst = CopyAlloc((PVOID)pwfxDst, WaveFormatSize(pwfxDst));
        if (padsi32->pwfxDst == NULL) {
            if (padsi32->pwfxSrc != NULL) {
                LocalFree((HLOCAL)padsi32->pwfxSrc);
            }
	    ptrUnFix16((PVOID)padsi16->pwfltr);
	    ptrUnFix16((PVOID)padsi16->pwfxDst);
	    ptrUnFix16((PVOID)padsi16->pwfxSrc);
	    return FALSE;
        }
    } else {
        padsi32->pwfxDst = NULL;
    }


    if (padsi16->pwfltr != NULL) {
        padsi32->pwfltr = CopyAlloc(pwfltr16, pwfltr16->cbStruct);

        if (padsi32->pwfltr == NULL) {
            if (padsi32->pwfxSrc != NULL) {
                LocalFree((HLOCAL)padsi32->pwfxSrc);
            }
            if (padsi32->pwfxDst != NULL) {
                LocalFree((HLOCAL)padsi32->pwfxDst);
            }
	        ptrUnFix16((PVOID)padsi16->pwfltr);
	        ptrUnFix16((PVOID)padsi16->pwfxDst);
	        ptrUnFix16((PVOID)padsi16->pwfxSrc);
            return FALSE;
        }
    } else {
        padsi32->pwfltr = NULL;
    }

    ptrUnFix16((PVOID)padsi16->pwfltr);
    ptrUnFix16((PVOID)padsi16->pwfxDst);
    ptrUnFix16((PVOID)padsi16->pwfxSrc);
    return TRUE;
}

VOID IUnThunkStreamInstance
(
    PACMDRVSTREAMINSTANCE  padsi
)
{
    if (padsi->pwfxSrc != NULL) {
        LocalFree((HLOCAL)padsi->pwfxSrc);
    }
    if (padsi->pwfxDst != NULL) {
        LocalFree((HLOCAL)padsi->pwfxDst);
    }
    if (padsi->pwfltr != NULL) {
        LocalFree((HLOCAL)padsi->pwfltr);
    }

}

//--------------------------------------------------------------------------;
//
//  LRESULT IOpenDriver32
//
//  Description:
//
//      Open a 32-bit driver
//
//  Arguments:
//      HACMDRIVERID hadid:
//
//      UINT uMsg:
//
//      LPARAM lParam1:
//
//      LPARAM lParam2:
//
//  Return (LRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;
MMRESULT IDriverOpen32
(
    HACMDRIVER UNALIGNED * phad16,
    HACMDRIVERID           hadid,
    DWORD                  fdwOpen
)
{
    HACMDRIVER UNALIGNED * phad;
    HACMDRIVER      had;
    MMRESULT        mmr;

    mmr = IDriverOpen(&had, hadid, fdwOpen);

    if (mmr == MMSYSERR_NOERROR) {
        phad = (HACMDRIVER*)ptrFixMap16To32((PVOID)phad16);
        *phad = had;
	ptrUnFix16((PVOID)phad16);
    }

    return mmr;
}

//--------------------------------------------------------------------------;
//
//  LRESULT IDriverMessageId32
//
//  Description:
//
//
//  Arguments:
//      HACMDRIVERID hadid:
//
//      UINT uMsg:
//
//      LPARAM lParam1:
//
//      LPARAM lParam2:
//
//  Return (LRESULT):
//
//  History:
//      09/05/93    cjp     [curtisp]
//
//--------------------------------------------------------------------------;
LRESULT FNLOCAL IDriverMessageId32
(
    HACMDRIVERID        hadid,
    UINT                uMsg,
    LPARAM              lParam1,
    LPARAM              lParam2
)
{

    switch (uMsg) {

    //
    //  Common with IDriverMessage32
    //
    case DRV_CONFIGURE:
        return IThunkConfigure(hadid,
                               (HWND)lParam1,
                               (DRVCONFIGINFO16 UNALIGNED *)lParam2);

    case ACMDM_FILTER_DETAILS:
        //
        //
        //
        return IThunkFilterDetails(hadid,
                                   (ACMFILTERDETAILSA UNALIGNED *)lParam1,
                                   (DWORD)lParam2);
        break;

    case ACMDM_FORMAT_DETAILS:
        return IThunkFormatDetails(hadid,
                                   (ACMFORMATDETAILSA UNALIGNED *)lParam1,
                                   (DWORD)lParam2);

    case ACMDM_FORMAT_SUGGEST:
        return IThunkFormatSuggest(hadid,
                                   (PACMDRVFORMATSUGGEST)lParam1);



    //
    //
    //
    case DRV_QUERYCONFIGURE:
        //
        //  Just pass the message on
        //
        return IDriverMessageId(hadid, uMsg, lParam1, lParam2);

    case ACMDM_DRIVER_DETAILS:
        {
            ACMDRIVERDETAILSA  acmd;
            MMRESULT uRet;

            acmd.cbStruct = sizeof(acmd);
            uRet = acmDriverDetailsA(hadid, &acmd, 0L);

            if (uRet == MMSYSERR_NOERROR) {
                PVOID pvStart;
                WORD  wicon;

                /*
                **  No async support - we don't want to support callbacks
                */
                acmd.fdwSupport &= ~ACMDRIVERDETAILS_SUPPORTF_ASYNC;

		//
		//  Map pointer from 16- to 32-bits
		//
		pvStart = ptrFixMap16To32((PVOID)lParam1);

                /*
                **  Copy it all back but remember HICON is 16-bit
                **  on the 16-bit side
                */
		
                CopyMemory(pvStart,
                           (PVOID)&acmd,
                           FIELD_OFFSET(ACMDRIVERDETAILSA, hicon) );

                /*
                ** map and copy the icon handle
                **
                ** BugBug::There is not a WOW_TYPE_ICON in the WOW_HANDLE_TYPE
                ** enumeration.
                */
#ifdef WIN4
		wicon = (WORD)acmd.hicon;
#else
                wicon = (*lpWOWHandle16)( acmd.hicon, WOW_TYPE_HWND);
#endif

                CopyMemory((PVOID)((PBYTE)pvStart +
			   FIELD_OFFSET(ACMDRIVERDETAILSA, hicon)),
			   &wicon,
                           sizeof(WORD) );

                CopyMemory((PVOID)((PBYTE)pvStart +
                               FIELD_OFFSET(ACMDRIVERDETAILSA, hicon) +
                               sizeof(WORD)),
                           (PVOID)acmd.szShortName,
                           sizeof(acmd) -
                               FIELD_OFFSET(ACMDRIVERDETAILSA, szShortName[0]));

		//
		//  Unmap pointer
		//
		ptrUnFix16((PVOID)lParam1);
            }
            return uRet;
        }

    case ACMDM_FORMATTAG_DETAILS:
        {
            ACMFORMATTAGDETAILS             acmf;
            ACMFORMATTAGDETAILSA UNALIGNED *pvacmf;
            MMRESULT                        uRet;

            pvacmf = (ACMFORMATTAGDETAILSA UNALIGNED *)
                          ptrFixMap16To32((PVOID)lParam1);

#ifdef TRUE	// UNICODE
            CopyMemory((PVOID)&acmf, (PVOID)pvacmf,
                       FIELD_OFFSET(ACMFORMATTAGDETAILS, szFormatTag[0]));

            acmf.cbStruct = sizeof(acmf);


	    Imbstowcs(acmf.szFormatTag,
		      (LPSTR)pvacmf->szFormatTag,
		      sizeof(pvacmf->szFormatTag));
#else
	    CopyMemory((PVOID)&acmf, (PVOID)pvacmf, sizeof(acmf));
	    acmf.cbStruct = sizeof(acmf);
#endif

            uRet = IDriverMessageId(hadid,
                                    uMsg,
                                    (LPARAM)&acmf,
                                    lParam2);

            if (uRet == MMSYSERR_NOERROR) {
#ifdef TRUE	// UNICODE
                CopyMemory((PVOID)pvacmf, (PVOID)&acmf,
                           FIELD_OFFSET(ACMFORMATTAGDETAILS, szFormatTag[0]));
                Iwcstombs((LPSTR)pvacmf->szFormatTag,
                         acmf.szFormatTag,
                         sizeof(pvacmf->szFormatTag));
#else
		CopyMemory((PVOID)pvacmf, (PVOID)&acmf, sizeof(acmf));
#endif
            }
	    ptrUnFix16((PVOID)lParam1);
            return uRet;
        }
        break;

    case ACMDM_FILTERTAG_DETAILS:
        {
            ACMFILTERTAGDETAILS             acmf;
            ACMFILTERTAGDETAILSA UNALIGNED *pvacmf;
            MMRESULT                        uRet;

            pvacmf = (ACMFILTERTAGDETAILSA UNALIGNED *)
			 ptrFixMap16To32((PVOID)lParam1);

#ifdef TRUE	// UNICODE
            CopyMemory((PVOID)&acmf, (PVOID)pvacmf,
                       FIELD_OFFSET(ACMFILTERTAGDETAILS, szFilterTag[0]));

            acmf.cbStruct = sizeof(acmf);

            Imbstowcs(acmf.szFilterTag,
		      (LPSTR)pvacmf->szFilterTag,
		      sizeof(pvacmf->szFilterTag));
#else
	    CopyMemory((PVOID)&acmf, (PVOID)pvacmf, sizeof(acmf));
	    acmf.cbStruct = sizeof(acmf);
#endif

            uRet = IDriverMessageId(hadid,
                                    uMsg,
                                    (LPARAM)&acmf,
                                    lParam2);


            if (uRet == MMSYSERR_NOERROR) {
#ifdef TRUE	// UNICODE
                CopyMemory((PVOID)pvacmf, (PVOID)&acmf,
                           FIELD_OFFSET(ACMFILTERTAGDETAILS, szFilterTag[0]));
                Iwcstombs((LPSTR)pvacmf->szFilterTag,
                         acmf.szFilterTag,
                         sizeof(pvacmf->szFilterTag));
#else
		CopyMemory((PVOID)pvacmf, (PVOID)&acmf, sizeof(acmf));
#endif
            }
	    ptrUnFix16((PVOID)lParam1);
            return uRet;
        }
        break;

    case ACMDM_HARDWARE_WAVE_CAPS_INPUT:
        {
            //
            //  wave input
            //
            WAVEINCAPSA  wica;
            WAVEINCAPSW  wicw;
            MMRESULT     uRet;

            uRet = IDriverMessageId(hadid,
                                    uMsg,
                                    (LPARAM)&wicw,
                                    sizeof(wicw));

            if (uRet == MMSYSERR_NOERROR) {
                CopyMemory((PVOID)&wica, (PVOID)&wicw,
                           FIELD_OFFSET(WAVEINCAPS, szPname[0]));

                Iwcstombs(wica.szPname, wicw.szPname, sizeof(wica.szPname));

                CopyMemory(ptrFixMap16To32((PVOID)lParam1),
                           (PVOID)&wica,
                           lParam2);
		ptrUnFix16((PVOID)lParam1);
            }

            return uRet;
        }

    case ACMDM_HARDWARE_WAVE_CAPS_OUTPUT:
        {
            //
            //  wave output
            //
            WAVEOUTCAPSA  woca;
            WAVEOUTCAPSW  wocw;
            MMRESULT uRet;

            uRet = IDriverMessageId(hadid,
                                    uMsg,
                                    (LPARAM)&wocw,
                                    sizeof(wocw));

            if (uRet == MMSYSERR_NOERROR) {
                CopyMemory((PVOID)&woca, (PVOID)&wocw,
                           FIELD_OFFSET(WAVEOUTCAPS, szPname[0]));

                Iwcstombs(woca.szPname, wocw.szPname, sizeof(woca.szPname));

                CopyMemory(ptrFixMap16To32((PVOID)lParam1),
                           (PVOID)&woca,
                           lParam2);
		ptrUnFix16((PVOID)lParam1);
            }

            return uRet;
        }
        break;

    case ACMDM_DRIVER_ABOUT:

        //
        //  Map the window handle
        //
#ifndef WIN4
        lParam1 = (LPARAM)(*lpWOWHandle32)( (WORD)lParam1, WOW_TYPE_HWND);
#endif

        return IDriverMessageId(hadid, uMsg, lParam1, lParam2);

    case ACMDM_DRIVER_NOTIFY:
        return MMSYSERR_NOTSUPPORTED;

    }
    
}

//--------------------------------------------------------------------------;
//
//  LRESULT IDriverMessage32
//
//  Description:
//
//
//  Arguments:
//      HACMDRIVERID hadid:
//
//      UINT uMsg:
//
//      LPARAM lParam1:
//
//      LPARAM lParam2:
//
//  Return (LRESULT):
//
//  History:
//      09/05/93    cjp     [curtisp]
//
//--------------------------------------------------------------------------;
LRESULT FNLOCAL IDriverMessage32
(
    HACMDRIVER          had,
    UINT                uMsg,
    LPARAM              lParam1,
    LPARAM              lParam2
)
{
    DWORD		dwSave;
    
    switch (uMsg) {
    //
    //  Common with IDriverMessageId32
    //
    case DRV_CONFIGURE:
        //
        //  16-bit apps can configure 32-bit drivers
        //
        return IThunkConfigure((HACMDRIVERID)had,
                               (HWND)lParam1,
                               (DRVCONFIGINFO16 UNALIGNED *)lParam2);

    case ACMDM_FILTER_DETAILS:
        //
        //
        //
        return IThunkFilterDetails((HACMDRIVERID)had,
                                   (ACMFILTERDETAILSA UNALIGNED *)lParam1,
                                   (DWORD)lParam2);
        break;

    case ACMDM_FORMAT_DETAILS:
        return IThunkFormatDetails((HACMDRIVERID)had,
                                   (ACMFORMATDETAILSA UNALIGNED *)lParam1,
                                   (DWORD)lParam2);

    case ACMDM_FORMAT_SUGGEST:
        return IThunkFormatSuggest((HACMDRIVERID)had,
                                   (ACMDRVFORMATSUGGEST UNALIGNED *)lParam1);


    //
    //
    //
    case ACMDM_STREAM_OPEN:
    case ACMDM_STREAM_CLOSE:
    case ACMDM_STREAM_RESET:
    case ACMDM_STREAM_SIZE:
        //
        //  Passes in PACMDRVSTREAMINSTANCE in lPararm1
        //
        {
            ACMDRVSTREAMINSTANCE  adsi;
            ACMDRVSTREAMINSTANCE UNALIGNED *padsi;    // unaligned 16-bit version
            MMRESULT              uRet;
            ACMDRVSTREAMSIZE      adss;
            ACMDRVSTREAMSIZE UNALIGNED *lpadss16;     // unaligned 16-bit version

            padsi = (ACMDRVSTREAMINSTANCE*)ptrFixMap16To32((PVOID)lParam1);

            if (!IThunkStreamInstance(padsi, &adsi)) {
		ptrUnFix16((PVOID)lParam1);
                return MMSYSERR_NOMEM;
            }

            if (uMsg == ACMDM_STREAM_SIZE) {
                lpadss16 = (LPACMDRVSTREAMSIZE)ptrFixMap16To32((PVOID)lParam2);
                CopyMemory( (PVOID)&adss, (PVOID)lpadss16, sizeof(adss));
            }

            //
            //  Call the driver
            //

            uRet = IDriverMessage(had,
                                  uMsg,
                                  (LPARAM)&adsi,
                                  uMsg == ACMDM_STREAM_SIZE ?
                                      (LPARAM)&adss : lParam2);

            IUnThunkStreamInstance(&adsi);

            if (uRet == MMSYSERR_NOERROR) {

                //
                //  Don't lose data the driver may have set up
                //
                padsi->fdwDriver = adsi.fdwDriver;
                padsi->dwDriver  = adsi.dwDriver;

                //
                //  Return the size stuff if requested
                //

                if (uMsg == ACMDM_STREAM_SIZE) {
                    CopyMemory( (PVOID)lpadss16, (PVOID)&adss, sizeof(adss) );
                }
            }

	    ptrUnFix16((PVOID)lParam2);
	    ptrUnFix16((PVOID)lParam1);
            return uRet;
        }

    case ACMDM_STREAM_PREPARE:
    case ACMDM_STREAM_UNPREPARE:
    case ACMDM_STREAM_CONVERT:
        //
        //  Passes in PACMDRVSTREAMINSTANCE in lPararm1
        //
        {
            ACMDRVSTREAMINSTANCE  adsi;
            ACMDRVSTREAMINSTANCE UNALIGNED *padsi;    // unaligned 16-bit version
            MMRESULT              uRet;
            ACMDRVSTREAMHEADER UNALIGNED *padsh;
            PACMDRVSTREAMHEADER   padsh32;

            padsi = ptrFixMap16To32((PVOID)lParam1);
            padsh = ptrFixMap16To32((PVOID)lParam2);

            if (!IThunkStreamInstance(padsi, &adsi)) {
		ptrUnFix16((PVOID)lParam2);
		ptrUnFix16((PVOID)lParam1);
                return MMSYSERR_NOMEM;
            }

            //
            //  If this not prepare we already have a 32-bit
            //  stream header.
            //

            if (uMsg == ACMDM_STREAM_PREPARE) {
                padsh->dwDriver = (DWORD)LocalAlloc(LMEM_FIXED, sizeof(*padsh));
            }
            padsh32 = (PACMDRVSTREAMHEADER)padsh->dwDriver;

            if (padsh32 != NULL) {

                //  Thunk the stream header
                //
                //  NOTE - NO ATTEMPT is made to align the byte fields,
                //  this is up to the drivers.
                //
		
		dwSave = padsh32->dwDriver;
                CopyMemory((PVOID)padsh32, (PVOID)padsh, sizeof(*padsh));
		padsh32->dwDriver = dwSave;

                padsh32->pbSrc  = (PBYTE)ptrFixMap16To32((PVOID)padsh32->pbSrc);
                padsh32->pbDst  = (PBYTE)ptrFixMap16To32((PVOID)padsh32->pbDst);

                //
                //  Call the driver
                //

                uRet = IDriverMessage(had,
                                      uMsg,
                                      (LPARAM)&adsi,
                                      (LPARAM)padsh32);
            } else {
                uRet = MMSYSERR_NOMEM;
            }

            IUnThunkStreamInstance(&adsi);

            if (uRet == MMSYSERR_NOERROR) {

                //
                //  Don't lose data the driver may have set up
                //
                padsi->fdwDriver = adsi.fdwDriver;
                padsi->dwDriver  = adsi.dwDriver;

                //
                //  Copy back the stream header (don't mess up the pointers
                //  or driver instance data though!).
                //

                padsh32->pbSrc    = padsh->pbSrc;
                padsh32->pbDst    = padsh->pbDst;
		dwSave = padsh32->dwDriver;
                padsh32->dwDriver = padsh->dwDriver;
                CopyMemory((PVOID)padsh, (PVOID)padsh32, sizeof(*padsh));
		padsh32->dwDriver = dwSave;

            }

            //
            //  Free if this is unprepare (note that this must be done
            //  whether the driver succeeds of not since the driver may not
            //  support unprepare.
            //

            if (uMsg == ACMDM_STREAM_UNPREPARE) {
                LocalFree((HLOCAL)padsh->dwDriver);
                padsh->dwDriver = 0;
            }

	    if (NULL != padsh32)
	    {
		    ptrUnFix16((PVOID)padsh->pbDst);
		    ptrUnFix16((PVOID)padsh->pbSrc);
	    }
	        ptrUnFix16((PVOID)lParam2);
	        ptrUnFix16((PVOID)lParam1);
            return uRet;
        }

    }
}

//--------------------------------------------------------------------------;
//
//  DWORD acmMessage32
//
//  Description:
//
//      32-bit function dispatcher for thunks.
//
//  Arguments:
//      DWORD dwThunkId:
//
//      DWORD dw1:
//
//      DWORD dw2:
//
//      DWORD dw3:
//
//      DWORD dw4:
//
//  Return (DWORD):
//
//  History:
//
//--------------------------------------------------------------------------;

#ifdef WIN4
DWORD WINAPI acmMessage32
#else
DWORD acmMessage32
#endif
(
    DWORD dwThunkId,
    DWORD dw1,
    DWORD dw2,
    DWORD dw3,
    DWORD dw4
)
{
    // DPF(4,"acmMessage32(dwThunkId=%08lxh, dw1=%08lxh, dw2=%08lxh, dw3=%08lxh, dw4=%08lxh);", dwThunkId, dw1, dw2, dw3, dw4);
#ifndef WIN4
    //
    //  Make sure we've got thunking functionality
    //

    if (ThunksInitialized <= 0) {

        HMODULE hMod;

        if (ThunksInitialized == -1) {
            return MMSYSERR_ERROR;
        }

        hMod = GetModuleHandle(GET_MAPPING_MODULE_NAME);
        if (hMod != NULL) {

            GetVdmPointer =
                (LPGETVDMPOINTER)GetProcAddress(hMod, GET_VDM_POINTER_NAME);
            lpWOWHandle32 =
                (LPWOWHANDLE32)GetProcAddress(hMod, GET_HANDLE_MAPPER32 );
            lpWOWHandle16 =
                (LPWOWHANDLE16)GetProcAddress(hMod, GET_HANDLE_MAPPER16 );
        }

        if ( GetVdmPointer == NULL
          || lpWOWHandle16 == NULL
          || lpWOWHandle32 == NULL ) {

            ThunksInitialized = -1;
            return MMSYSERR_ERROR;

        } else {
            ThunksInitialized = 1;
        }
    }
#endif


    //
    //  Perform the requested function
    //

    switch (dwThunkId) {

        case acmThunkDriverMessageId32:
            return (DWORD)IDriverMessageId32(
                              (HACMDRIVERID)dw1,
                              (UINT)dw2,
                              (LPARAM)dw3,
                              (LPARAM)dw4);

        case acmThunkDriverMessage32:
            return (DWORD)IDriverMessage32(
                              (HACMDRIVER)dw1,
                              (UINT)dw2,
                              (LPARAM)dw3,
                              (LPARAM)dw4);

        case acmThunkDriverLoad32:
        {
	    PACMGARB	    pag;
            LPCSTR	    lpstrAlias;
            PACMDRIVERID    padid;
	    WCHAR	    strAlias[MAX_DRIVER_NAME_CHARS];

            if ((dw2 & ACM_DRIVERADDF_TYPEMASK) != ACM_DRIVERADDF_FUNCTION) {
                lpstrAlias = ptrFixMap16To32((PVOID)dw1);
                Imbstowcs(strAlias, lpstrAlias, MAX_DRIVER_NAME_CHARS);
		ptrUnFix16((PVOID)dw1);
            }


            //
            //
            //
            //

            // Enter ???
	    pag = pagFind();
	    if (NULL == pag)
	    {
		DPF(1, "acmThunkDriverLoad32: NULL pag!!!");
		return (DWORD)NULL;
	    }
	    
            for (padid = pag->padidFirst;
                 padid != NULL;
                 padid = padid->padidNext)
            {
                if ((dw2 & ACM_DRIVERADDF_TYPEMASK) == ACM_DRIVERADDF_FUNCTION) {
                    //
                    //  Look for the internal converter
                    //

                    if (padid->fnDriverProc == pcmDriverProc) {
                        return (DWORD)(HACMDRIVERID)padid;
                    }
                } else {
		    if (lstrcmpW(strAlias, padid->szAlias) == 0) {
                        // Leave ???
                        return (DWORD)(HACMDRIVERID)padid;
                    }
                }
            }
            // Leave ???

            return (DWORD)NULL;
        }
        case acmThunkDriverOpen32:
            return (DWORD)IDriverOpen32(
                              (HACMDRIVER UNALIGNED *)dw1,
                              (HACMDRIVERID)dw2,
                              (DWORD)dw3);

        case acmThunkDriverClose32:

            //
            //  Call close directly
            //
            return (DWORD)IDriverClose((HACMDRIVER)dw1, dw2);

	case acmThunkBootDrivers32:
	{
	    //
	    //
	    //
	    if (NULL == pagFindAndBoot())
	    {
		DPF(1, "acmThunkBootDrivers32: NULL pag!!!");
		return (DWORD)(MMSYSERR_ERROR);
	    }
	    return (DWORD)(MMSYSERR_NOERROR);
	}

    }
}

#else // !WIN32

//--------------------------------------------------------------------------;
//
//
//
//--------------------------------------------------------------------------;

//--------------------------------------------------------------------------;
//
//  MMRESULT acmBootDrivers32
//
//  Description:
//
//  Arguments:
//	PACMGARB pag:
//
//  Return (MMRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;
MMRESULT FNGLOBAL acmBootDrivers32
(
    PACMGARB pag
)
{
    MMRESULT mmr;

    DPF(5,"acmBootDrivers32();");

#ifdef WIN4
    mmr = (MMRESULT) acmMessage32(acmThunkBootDrivers32,
				  (DWORD)(UINT)pag,
				  (DWORD)0,
				  (DWORD)0,
				  (DWORD)0 );

#else
    mmr = (MMRESULT)(*pag->lpfnCallproc32W)(acmThunkBootDrivers32,
					     (DWORD)(UINT)pag,
					     (DWORD)0,
					     (DWORD)0,
					     (DWORD)0,
					     pag->lpvThunkEntry,
					     0L,    // Don't map pointers
					     5L);
#endif

    return mmr;
}

//--------------------------------------------------------------------------;
//
//  LRESULT IDriverMessageId32
//
//  Description:
//
//      Pass a message to a 32-bit driver using the driver id.
//
//  Arguments:
//      HACMDRIVERID hadid:
//
//      UINT uMsg:
//
//      LPARAM lParam1:
//
//      LPARAM lParam2:
//
//  Return (LRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;

LRESULT FNGLOBAL IDriverMessageId32
(
    DWORD               hadid,
    UINT                uMsg,
    LPARAM              lParam1,
    LPARAM              lParam2
)
{
    LRESULT lr;

    DPF(5,"IDriverMessageId32();");

#ifdef WIN4
    lr = acmMessage32(acmThunkDriverMessageId32,
		      (DWORD)hadid,
		      (DWORD)uMsg,
		      (DWORD)lParam1,
		      (DWORD)lParam2);

#else
    {
        PACMGARB pag;

        pag = pagFind();
        lr = (LRESULT)(*pag->lpfnCallproc32W)(acmThunkDriverMessageId32,
					   (DWORD)hadid,
					   (DWORD)uMsg,
					   (DWORD)lParam1,
					   (DWORD)lParam2,
					   pag->lpvThunkEntry,
					   0L,    // Don't map pointers
					   5L);
    }
#endif

    return lr;
}


//--------------------------------------------------------------------------;
//
//  LRESULT IDriverMessage32
//
//  Description:
//
//      Pass a message to a 32-bit driver using the instance handle.
//
//  Arguments:
//      HACMDRIVERID hadid:
//
//      UINT uMsg:
//
//      LPARAM lParam1:
//
//      LPARAM lParam2:
//
//  Return (LRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;

LRESULT FNGLOBAL IDriverMessage32
(
    DWORD               hadid,
    UINT                uMsg,
    LPARAM              lParam1,
    LPARAM              lParam2
)
{
    LRESULT lr;

    DPF(5,"IDriverMessage32();");
    
#ifdef WIN4
    lr = acmMessage32(acmThunkDriverMessage32,
		      (DWORD)hadid,
		      (DWORD)uMsg,
		      (DWORD)lParam1,
		      (DWORD)lParam2);

#else
    {
        PACMGARB pag;

        pag = pagFind();
        lr = (LRESULT)(*pag->lpfnCallproc32W)(acmThunkDriverMessage32,
					   (DWORD)hadid,
					   (DWORD)uMsg,
					   (DWORD)lParam1,
					   (DWORD)lParam2,
					   pag->lpvThunkEntry,
					   0L,    // Don't map pointers
					   5L);
    }
#endif

    return (lr);
    
}

//--------------------------------------------------------------------------;
//
//  DWORD IDriverLoad32
//
//  Description:
//
//      Load a 32-bit ACM driver (actually just find its hadid)
//
//  Arguments:
//      LPCSTR lpszAlias:
//      DWORD  fdwFlags
//
//  Return (HDRVR):
//
//  History:
//
//--------------------------------------------------------------------------;
DWORD FNGLOBAL IDriverLoad32
(
    LPCSTR  lpszAlias,
    DWORD   fdwFlags
)
{
#ifdef WIN4
    DWORD dw;
    
    DPF(5,"IDriverLoad(%s,%08lxh);", lpszAlias, fdwFlags);
    
    dw = acmMessage32(acmThunkDriverLoad32,
		      (DWORD)lpszAlias,
		      (DWORD)fdwFlags,
		      (DWORD)0L,
		      (DWORD)0L);
    
#else
    DWORD dw;
    PACMGARB pag;
    
    DPF(5,"IDriverLoad(%s,%08lxh);", lpszAlias, fdwFlags);

    pag = pagFind();
    dw = (DWORD)(*pag->lpfnCallproc32W)(acmThunkDriverLoad32,
					 (DWORD)lpszAlias,
					 (DWORD)fdwFlags,
					 (DWORD)0L,
					 (DWORD)0L,
					 pag->lpvThunkEntry,
					 0L,    // Don't map pointers
					 5L);

#endif

    return (dw);
}

//--------------------------------------------------------------------------;
//
//  MMERESULT IDriverOpen32
//
//  Description:
//
//      Open a 32-bit ACM driver
//
//  Arguments:
//      LPHACMDRIVER lphadNew:
//
//      HACMDRIVERID hadid:
//
//      DWORD fdwOpen:
//
//  Return (MMRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;

MMRESULT FNGLOBAL IDriverOpen32
(
    LPDWORD             lphadNew,
    DWORD               hadid,
    DWORD               fdwOpen
)
{
    MMRESULT mmr;

    DPF(5,"IDriverOpen32();");

#ifdef WIN4
    mmr = (MMRESULT)acmMessage32(acmThunkDriverOpen32,
		       (DWORD)lphadNew,
		       (DWORD)hadid,
		       (DWORD)fdwOpen,
		       (DWORD)0L);

#else
    {
        PACMGARB pag;

        pag = pagFind();
        mmr = (MMRESULT)(*pag->lpfnCallproc32W)(acmThunkDriverOpen32,
					     (DWORD)lphadNew,
					     (DWORD)hadid,
					     (DWORD)fdwOpen,
					     (DWORD)0L,
					     pag->lpvThunkEntry,
					     0L,    // Don't map pointers
					     5L);
    }
#endif

    return (mmr);
    
}

//--------------------------------------------------------------------------;
//
//  MMRESULT IDriverClose32
//
//  Description:
//
//      Cloase a 32-bit ACM driver
//
//  Arguments:
//      HDRVR hdrvr:
//
//      DWORD fdwClose:
//
//  Return (MMRESULT):
//
//  History:
//
//--------------------------------------------------------------------------;

LRESULT FNGLOBAL IDriverClose32
(
    DWORD               hdrvr,
    DWORD               fdwClose
)
{
#ifdef WIN4
    LRESULT lr;

    DPF(5,"IDriverClose32();");

    lr = acmMessage32(acmThunkDriverClose32,
		      (DWORD)hdrvr,
		      (DWORD)fdwClose,
		      (DWORD)0L,
		      (DWORD)0L);

#else
    LRESULT lr;
    PACMGARB pag;

    DPF(5,"IDriverClose32();");

    pag = pagFind();
    lr = (LRESULT)(*pag->lpfnCallproc32W)(acmThunkDriverClose32,
					   (DWORD)hdrvr,
					   (DWORD)fdwClose,
					   (DWORD)0L,
					   (DWORD)0L,
					   pag->lpvThunkEntry,
					   0L,    // Don't map pointers
					   5L);

#endif

    return (lr);
    
}


#endif // !WIN32
