/*++

    Intel Corporation Proprietary Information
    Copyright (c) 1995 Intel Corporation

    This listing is supplied under the terms of a license agreement with
    Intel Corporation and may not be used, copied, nor disclosed except in
    accordance with the terms of that agreeement.

Module Name:

    nspinstl.cpp

Abstract:

    This module contains the entry points for name space provider
    installation/configuration.

Author:

    Dirk Brandewie (dirk@mink.intel.com) 04-December-1995

Notes:

    $Revision:   1.11  $

    $Modtime:   08 Mar 1996 15:46:16  $


Revision History:

    most-recent-revision-date email-name
        description

    04-Dec-1995 dirk@mink.intel.com
        Initial revision

--*/

#include "precomp.h"

// Structure used as context value for UnInstallEnumProc().
typedef struct
{
    LPGUID     ProviderId;
    PNSCATALOG Catalog;
    LPINT      ReturnCode;
} NS_UNINSTALL_CONTEXT, *PNS_UNINSTALL_CONTEXT;

// Structure used as context value for EnableEnumProc().
typedef struct
{
    LPGUID     ProviderId;
    BOOL       EnableValue;
    LPINT      ReturnCode;
} NS_ENABLE_CONTEXT, *PNS_ENABLE_CONTEXT;



INT WSAAPI
WSCInstallNameSpace (
    IN  LPWSTR  lpszIdentifier,
    IN  LPWSTR  lpszPathName,
    IN  DWORD   dwNameSpace,
    IN  DWORD   dwVersion,
    IN  LPGUID  lpProviderId
    )
/*++

Routine Description:

    WSCInstallNameSpace() is used to install a name space provider. For
    providers that are able to support multiple names spaces, this function
    must be called once for every name space supported, and a unique provider
    ID must be supplied each time.

Arguments:

    lpszIdentifier - Display string for the provider.

    lpszPathname - Points to a path to the providers DLL image which
                   follows  the usual rules for path resolution. This path may
                   contain embedded environment strings (such as
                   %SystemRoot%). Such environment strings are expanded
                   whenever the WinSock 2 DLL needs to subsequently load
                   theprovider DLL on behalf of an application. After any
                   embedded environment strings are expanded, the WinSock 2 DLL
                   passes the resulting string into the LoadLibrary() function
                   to load the provider into memory.

    dwNameSpace - Specifies the name space supported by this provider.

    dwVersion - Specifies the version number of the provider.

    bStoresAllServiceClassInfo - Specifies that this provider supports the
                                 storage  of service class schema information
                                 for all service classes across all namespaces.
                                 The Winsock DLL will then use this provider to
                                 get all of its classinfo information rather
                                 than asking each individual provider.

    lpProviderId - A unique identifier for this provider.  This GUID should be
                   generated by UUIDGEN.EXE.

Return Value:

    The function should return NO_ERROR (0) if the routine succeeds.  It should
    return SOCKET_ERROR (-1) if the routine fails and it must set the
    appropriate error code using SetLastError().
--*/
{
    INT             ReturnCode;
    PNSCATALOG      Catalog =NULL;
    PNSCATALOGENTRY Item =NULL;
    HKEY            registry_root;
    BOOL            lock_owned = FALSE;

    registry_root = OpenWinSockRegistryRoot();
    if (NULL == registry_root) {
        DEBUGF(
            DBG_ERR,
            ("Opening registry root\n"));
        SetLastError(WSASYSCALLFAILURE);
        return(SOCKET_ERROR);
    }

    //
    // Check the current protocol catalog key. If it doesn't match
    // the expected value, blow away the old key and update the
    // stored value.
    //

    ValidateCurrentCatalogName(
        registry_root,
        WINSOCK_CURRENT_NAMESPACE_CATALOG_NAME,
        NSCATALOG::GetCurrentCatalogName()
        );

    TRY_START(guard_memalloc){
        Catalog = new NSCATALOG;
        if (NULL == Catalog){
            ReturnCode = WSA_NOT_ENOUGH_MEMORY;
            TRY_THROW(guard_memalloc);
        } //item

        Item = new NSCATALOGENTRY;
        if (NULL == Item){
            ReturnCode = WSA_NOT_ENOUGH_MEMORY;
            TRY_THROW(guard_memalloc);
        } //if

        ReturnCode = Item->InitializeFromValues(
            lpszPathName,
            lpszIdentifier,
            lpProviderId,
            dwNameSpace,
            dwVersion
            );
        if (ERROR_SUCCESS != ReturnCode){
            TRY_THROW(guard_memalloc);
        } //if

        Catalog->AcquireCatalogLock();
        lock_owned = TRUE;

        ReturnCode = Catalog->InitializeFromRegistry(
            registry_root);
        if (ERROR_SUCCESS != ReturnCode){
            TRY_THROW(guard_memalloc);
        } //if

        // Sift through the catalog and see if this ProviderId already exsits.
        // We can't use the enumeration procedure for the catalog since we
        // can't walk an empty list

        PLIST_ENTRY list_entry;
        PNSCATALOGENTRY list_item;
        LPGUID list_item_provider_id;

        list_entry = Catalog->m_namespace_list.Flink;

        while (list_entry != &Catalog->m_namespace_list){
            list_item = CONTAINING_RECORD(
                list_entry,
                NSCATALOGENTRY,
                m_CatalogLinkage);
            list_item_provider_id = list_item->GetProviderId();
            if (*list_item_provider_id == *lpProviderId){
                ReturnCode = WSAEINVAL;
                TRY_THROW(guard_memalloc);
           } //if
            list_entry = list_entry->Flink;
        } //while
        Catalog->AppendCatalogItem(
            Item);
        Item = NULL;  // item deletion is now covered by catalog

        Catalog->WriteToRegistry(
            registry_root
            );

        Catalog->ReleaseCatalogLock();
        lock_owned = FALSE;
        delete Item;
        delete Catalog;
    } TRY_CATCH(guard_memalloc){

        if (Item){
            delete Item;
        } //if

        if( lock_owned ) {
            assert( Catalog != NULL );
            Catalog->ReleaseCatalogLock();
        }

        if (Catalog){
            delete Catalog;
        } //if
    } TRY_END(guard_memalloc);

    if (ReturnCode != ERROR_SUCCESS) {
        SetLastError(ReturnCode);
        ReturnCode = SOCKET_ERROR;
    }

    CloseWinSockRegistryRoot(registry_root);

    return(ReturnCode);
}




BOOL
UnInstallEnumProc(
    IN DWORD             PassBack,
    IN PNSCATALOGENTRY   CatalogItem
    )
/*++

Routine Description:

    Enumeration proceedure for WSCUnInstallNameSpace.

Arguments:

    PassBack - A context value passed to EunerateCatalogItems. It is really a
               pointer to a NS_UNINSTALL_CONTEXT struct.

    CatalogItem - A pointer to a catalog item to be inspected.

Return Value:

    TRUE if the Enumeration should continue else FALSE.

--*/
{
    LPGUID                ItemProviderId;
    PNS_UNINSTALL_CONTEXT Context;
    LPGUID                ContextProviderId;
    BOOL                  ContinueEnumeration =TRUE;

    Context = (PNS_UNINSTALL_CONTEXT) PassBack;
    ContextProviderId = Context->ProviderId;

    ItemProviderId = CatalogItem->GetProviderId();

    if (*ContextProviderId == *ItemProviderId){
        Context->Catalog->RemoveCatalogItem(CatalogItem);
        *(Context->ReturnCode) = ERROR_SUCCESS;
        ContinueEnumeration = FALSE;
    } //if
    return(ContinueEnumeration);
}




INT
WSAAPI
WSCUnInstallNameSpace (
    IN  LPGUID  lpProviderId
    )
/*++

Routine Description:

    WSCUnInstallNameSpace() is used to uninstall the indicated name space
    provider.

Arguments:

    lpProviderId - The unique identifier for a provider.

Return Value:

    The function should return NO_ERROR (0) if the routine succeeds.  It should
    return SOCKET_ERROR (-1) if the routine fails and it must set the
    appropriate error code using SetLastError().
--*/
{
    INT                  ReturnCode;
    PNSCATALOG           Catalog =NULL;
    HKEY                 registry_root;
    NS_UNINSTALL_CONTEXT EnumContext;
    BOOL                 lock_owned = FALSE;

    //Open and Initialize a name space jprovider catalog
    registry_root = OpenWinSockRegistryRoot();
    if (NULL == registry_root) {
        DEBUGF(
            DBG_ERR,
            ("Opening registry root\n"));
        SetLastError(WSASYSCALLFAILURE);
        return(SOCKET_ERROR);
    }

    TRY_START(guard_memalloc){
        Catalog = new NSCATALOG;
        if (NULL == Catalog){
            ReturnCode = WSA_NOT_ENOUGH_MEMORY;
            TRY_THROW(guard_memalloc);
        } //item

        Catalog->AcquireCatalogLock();
        lock_owned = TRUE;

        ReturnCode = Catalog->InitializeFromRegistry(
            registry_root);
        if (ERROR_SUCCESS != ReturnCode){
            TRY_THROW(guard_memalloc);
        } //if

        // Iterate through the catalog and remove the provider entry if it is
        // present in the catalog.
        ReturnCode = WSAEINVAL;
        EnumContext.ProviderId = lpProviderId;
        EnumContext.Catalog = Catalog;
        EnumContext.ReturnCode = &ReturnCode;
        Catalog->EnumerateCatalogItems(
            UnInstallEnumProc,
            (DWORD) &EnumContext);

        Catalog->WriteToRegistry(
            registry_root
            );

        if (ERROR_SUCCESS != ReturnCode){
            SetLastError(ReturnCode);
            ReturnCode = SOCKET_ERROR;
        } //if

        Catalog->ReleaseCatalogLock();
        lock_owned = FALSE;
        delete Catalog;
    } TRY_CATCH(guard_memalloc){
        if( lock_owned ) {
            assert( Catalog != NULL );
            Catalog->ReleaseCatalogLock();
        }

        if (Catalog){
            delete Catalog;
        } //if
    } TRY_END(guard_memalloc);

    if (ReturnCode != ERROR_SUCCESS) {
        SetLastError(ReturnCode);
        ReturnCode = SOCKET_ERROR;
    }
    return(ReturnCode);
}




BOOL
EnableEnumProc(
    IN DWORD             PassBack,
    IN PNSCATALOGENTRY   CatalogItem
    )
/*++

Routine Description:

    Enumeration proceedure for WSCEnableNSProvider.

Arguments:

    PassBack - A context value passed to EunerateCatalogItems. It is really a
               pointer to a NS_ENABLE_CONTEXT struct.

    CatalogItem - A pointer to a catalog item to be inspected.

Return Value:

    TRUE if the Enumeration should continue else FALSE.

--*/
{
    BOOL               ContinueEnumeration = TRUE;
    LPGUID             ItemProviderId;
    PNS_ENABLE_CONTEXT Context;
    LPGUID             ContextProviderId;

    Context = (PNS_ENABLE_CONTEXT) PassBack;
    ContextProviderId = Context->ProviderId;

    ItemProviderId = CatalogItem->GetProviderId();

    if (*ContextProviderId == *ItemProviderId){
        CatalogItem->Enable(Context->EnableValue);
        *(Context->ReturnCode) = ERROR_SUCCESS;
        ContinueEnumeration = FALSE; // Stop Enumeration
    } //if
    return(ContinueEnumeration);

}




INT WSAAPI
WSCEnableNSProvider (
    IN  LPGUID  lpProviderId,
    IN  BOOL    fEnable
    )
/*++

Routine Description:

    WSCEnableNSProvider() is used to change the state of a given name space
    provider.  This function is intended to be used by the control panel applet
    to change the state of the providers.  An ISV should not just blindly
    de-activate another ISV's provider in order to activate their own.  This
    choice should be left up to the user.description-of-function

Arguments:

    lpProviderId - The unique identifier for this provider.

    fEnable - If TRUE, the provider is set to the active state.  If FALSE, the
              provider is disabled and will not be available for query
              operations or service registration.

Return Value:

    The function should return NO_ERROR (0) if the routine succeeds.  It should
    return SOCKET_ERROR (-1) if the routine fails and it must set the
    appropriate error code using SetLastError().

--*/
{
    INT                  ReturnCode;
    PNSCATALOG           Catalog =NULL;
    HKEY                 registry_root;
    NS_ENABLE_CONTEXT    EnumContext;
    BOOL                 lock_owned = FALSE;

    registry_root = OpenWinSockRegistryRoot();
    if (NULL == registry_root) {
        DEBUGF(
            DBG_ERR,
            ("Opening registry root\n"));
        SetLastError(WSASYSCALLFAILURE);
        return(SOCKET_ERROR);
    }

    TRY_START(guard_memalloc){
        Catalog = new NSCATALOG;
        if (NULL == Catalog){
            ReturnCode = WSA_NOT_ENOUGH_MEMORY;
            TRY_THROW(guard_memalloc);
        } //item

        Catalog->AcquireCatalogLock();
        lock_owned = TRUE;

        ReturnCode = Catalog->InitializeFromRegistry(
            registry_root);
        if (ERROR_SUCCESS != ReturnCode){
            TRY_THROW(guard_memalloc);
        } //if

        ReturnCode = WSAEINVAL;
        EnumContext.ProviderId = lpProviderId;
        EnumContext.ReturnCode = &ReturnCode;
        EnumContext.EnableValue = fEnable;

        Catalog->EnumerateCatalogItems(
            EnableEnumProc,
            (DWORD) &EnumContext);

        Catalog->WriteToRegistry(
            registry_root
            );

        Catalog->ReleaseCatalogLock();
        lock_owned = FALSE;
        delete Catalog;
    } TRY_CATCH(guard_memalloc){
        if( lock_owned ) {
            assert( Catalog != NULL );
            Catalog->ReleaseCatalogLock();
        }

        if (Catalog){
            delete Catalog;
        } //if
    } TRY_END(guard_memalloc);

    if (ERROR_SUCCESS != ReturnCode){
        SetLastError(ReturnCode);
        ReturnCode = SOCKET_ERROR;
    } //if
    return(ReturnCode);
}




