/*++

Copyright (c) 1993  Digital Equipment Corporation

Module Name:

    gamma.h

Abstract:

    This file defines the structures and definitions common to all
    sable-based platforms.

Author:

    Joe Notarangelo  26-Oct-1993
    Steve Jenness    26-Oct-1993

Environment:

    Kernel mode

Revision History:

    28-Dec 1994     Steve Brooks
                    t2.h and rattler.h extracted from this file.

--*/

#ifndef _GAMMAH_
#define _GAMMAH_

#include "sableref.h"               // Sable reference IO structure
#include "lynxref.h"                // Lynx interrupt structure
#include "xioref.h"                 // XIO interrupt structure
#if !defined(_LANGUAGE_ASSEMBLY)
#include "errframe.h"
#endif


//
// Constants used by dense space I/O routines
//

#define GAMMA_PCI0_DENSE_BASE_PHYSICAL_SUPERPAGE   0xfffffc83c0000000
#define GAMMA_PCI1_DENSE_BASE_PHYSICAL_SUPERPAGE   0xfffffc8180000000

#define PCI0_DENSE_BASE_PHYSICAL_SUPERPAGE \
    (GAMMA_PCI0_DENSE_BASE_PHYSICAL_SUPERPAGE - SABLE_PCI0_DENSE_MEMORY_QVA)

#define PCI1_DENSE_BASE_PHYSICAL_SUPERPAGE \
    (GAMMA_PCI1_DENSE_BASE_PHYSICAL_SUPERPAGE - SABLE_PCI1_DENSE_MEMORY_QVA)


#if !defined(_LANGUAGE_ASSEMBLY)

#include "rattler.h"                // Rattler chipset definitions
#include "t2.h"                     // T2 chipset definitions
#include "icic.h"                   // ICIC definitions

#define GAMMA_QVA_PHYSICAL_BASE  ((ULONGLONG)0x8000000000)

//
// QVA
// HAL_MAKE_QVA(
//     ULONGLONG PhysicalAddress
//     )
//
// Routine Description:
//
//    This macro returns the Qva for a physical address in system space.
//
// Arguments:
//
//    PhysicalAddress - Supplies a 64-bit physical address.
//
// Return Value:
//
//    The Qva associated with the physical address.
//

#define HAL_MAKE_QVA(PA)                                            \
    ( (PVOID)( QVA_ENABLE |                                         \
      (ULONG)( ((PA)-GAMMA_QVA_PHYSICAL_BASE) >> IO_BIT_SHIFT) ) ) 


//
// Define physical address spaces for GAMMA.
//
//    PCI0 - 32bit PCI bus
//    PCI1 - 64bit PCI bus
//

#define GAMMA_PCI1_DENSE_MEMORY_PHYSICAL  ((ULONGLONG)0x8180000000)
#define GAMMA_PCI1_SPARSE_IO_PHYSICAL     ((ULONGLONG)0x81C0000000)
#define GAMMA_PCI0_SPARSE_MEMORY_PHYSICAL ((ULONGLONG)0x8200000000)
#define GAMMA_PCI1_SPARSE_MEMORY_PHYSICAL ((ULONGLONG)0x8300000000)

#define GAMMA_PCI1_SPARSE_ISA_LEGACY_MEMORY_PHYSICAL ((ULONGLONG)0x8302000000)
#define GAMMA_PCI1_SPARSE_ISA_LEGACY_IO_PHYSICAL ((ULONGLONG)0x8200000000)

#define GAMMA_CBUS_CSRS_PHYSICAL          ((ULONGLONG)0x8380000000)
#define GAMMA_CPU0_CSRS_PHYSICAL          ((ULONGLONG)0x8380000000)
#define GAMMA_CPU1_CSRS_PHYSICAL          ((ULONGLONG)0x8381000000)
#define GAMMA_CPU2_CSRS_PHYSICAL          ((ULONGLONG)0x8382000000)
#define GAMMA_CPU3_CSRS_PHYSICAL          ((ULONGLONG)0x8383000000)
#define GAMMA_CPU0_SICR_PHYSICAL          ((ULONGLONG)0x8380000320)
#define GAMMA_CPU1_SICR_PHYSICAL          ((ULONGLONG)0x8381000320)
#define GAMMA_CPU2_SICR_PHYSICAL          ((ULONGLONG)0x8382000320)
#define GAMMA_CPU3_SICR_PHYSICAL          ((ULONGLONG)0x8383000320)
#define GAMMA_MEM0_CSRS_PHYSICAL          ((ULONGLONG)0x8388000000)
#define GAMMA_MEM1_CSRS_PHYSICAL          ((ULONGLONG)0x8389000000)
#define GAMMA_MEM2_CSRS_PHYSICAL          ((ULONGLONG)0x838A000000)
#define GAMMA_MEM3_CSRS_PHYSICAL          ((ULONGLONG)0x838B000000)
#define GAMMA_T2_CSRS_PHYSICAL            ((ULONGLONG)0x838E000000)
#define GAMMA_T4_CSRS_PHYSICAL            ((ULONGLONG)0x838F000000)

#define T2_CSRS_QVA                       (HAL_MAKE_QVA(GAMMA_T2_CSRS_PHYSICAL))
#define T4_CSRS_QVA                       (HAL_MAKE_QVA(GAMMA_T4_CSRS_PHYSICAL))


#define GAMMA_PCI0_CONFIGURATION_PHYSICAL ((ULONGLONG)0x8390000000)
#define GAMMA_PCI1_CONFIGURATION_PHYSICAL ((ULONGLONG)0x8398000000)
#define GAMMA_PCI0_SPARSE_IO_PHYSICAL     ((ULONGLONG)0x83A0000000)
#define GAMMA_PCI0_DENSE_MEMORY_PHYSICAL  ((ULONGLONG)0x83C0000000)

//
//  Define the limits of User mode Sparse and dense space:
//  A special hack is applied to these values for Gamma, since these addresses
//  are 40 bits in length, and are therefore beyond the range of QVAs. Bit 36
//  is set in these constants, which will get shifted to bit 31 when a QVA is
//  formed. When the physical address is then generated by the access macros,
//  the high order bits will get set, forcing bit 39 to be set, causing a
//  noncached access to occur
//
#define GAMMA_USER_PCI1_DENSE_MEMORY_PHYSICAL      ((ULONGLONG)0x8980000000)
#define GAMMA_USER_PCI1_SPARSE_IO_PHYSICAL         ((ULONGLONG)0x89c0000000)
#define GAMMA_USER_PCI0_SPARSE_MEMORY_PHYSICAL     ((ULONGLONG)0x8A00000000)
#define GAMMA_USER_PCI0_SPARSE_MEMORY_END_PHYSICAL ((ULONGLONG)0x8b00000000)
#define GAMMA_USER_PCI1_SPARSE_MEMORY_PHYSICAL     ((ULONGLONG)0x8b00000000)
#define GAMMA_USER_PCI1_SPARSE_MEMORY_END_PHYSICAL ((ULONGLONG)0x8b80000000)
#define GAMMA_USER_PCI0_SPARSE_IO_PHYSICAL         ((ULONGLONG)0x8ba0000000)
#define GAMMA_USER_PCI0_SPARSE_IO_END_PHYSICAL     ((ULONGLONG)0x8bc0000000)
#define GAMMA_USER_PCI0_DENSE_MEMORY_PHYSICAL      ((ULONGLONG)0x8bc0000000)

#define GAMMA_EDGE_LEVEL_CSRS_PHYSICAL	  ((ULONGLONG)0x83A00004C0)
#define GAMMA_INTERRUPT_CSRS_PHYSICAL	  ((ULONGLONG)0x83A000A640)
#define XIO_INTERRUPT_CSRS_PHYSICAL	      ((ULONGLONG)0x81C000A640)

//
//  For compatability, define the SABLE_* constants to refer to the GAMMA_*
//  constants. This allows gamma to share sources from the sable tree.
//

#define SABLE_PCI1_DENSE_MEMORY_PHYSICAL    GAMMA_PCI1_DENSE_MEMORY_PHYSICAL
#define SABLE_PCI1_SPARSE_IO_PHYSICAL       GAMMA_PCI1_SPARSE_IO_PHYSICAL
#define SABLE_PCI0_SPARSE_MEMORY_PHYSICAL   GAMMA_PCI0_SPARSE_MEMORY_PHYSICAL
#define SABLE_PCI1_SPARSE_MEMORY_PHYSICAL   GAMMA_PCI1_SPARSE_MEMORY_PHYSICAL

#define SABLE_USER_PCI1_DENSE_MEMORY_PHYSICAL      GAMMA_USER_PCI1_DENSE_MEMORY_PHYSICAL
#define SABLE_USER_PCI1_SPARSE_IO_PHYSICAL         GAMMA_USER_PCI1_SPARSE_IO_PHYSICAL
#define SABLE_USER_PCI0_SPARSE_MEMORY_PHYSICAL     GAMMA_USER_PCI0_SPARSE_MEMORY_PHYSICAL
#define SABLE_USER_PCI0_SPARSE_MEMORY_END_PHYSICAL GAMMA_USER_PCI0_SPARSE_MEMORY_END_PHYSICAL
#define SABLE_USER_PCI1_SPARSE_MEMORY_PHYSICAL     GAMMA_USER_PCI1_SPARSE_MEMORY_PHYSICAL
#define SABLE_USER_PCI1_SPARSE_MEMORY_END_PHYSICAL GAMMA_USER_PCI1_SPARSE_MEMORY_END_PHYSICAL
#define SABLE_USER_PCI0_SPARSE_IO_PHYSICAL         GAMMA_USER_PCI0_SPARSE_IO_PHYSICAL
#define SABLE_USER_PCI0_SPARSE_IO_END_PHYSICAL     GAMMA_USER_PCI0_SPARSE_IO_END_PHYSICAL
#define SABLE_USER_PCI0_DENSE_MEMORY_PHYSICAL      GAMMA_USER_PCI0_DENSE_MEMORY_PHYSICAL

#define SABLE_CBUS_CSRS_PHYSICAL            GAMMA_CBUS_CSRS_PHYSICAL
#define SABLE_CPU0_CSRS_PHYSICAL            GAMMA_CPU0_CSRS_PHYSICAL
#define SABLE_CPU1_CSRS_PHYSICAL            GAMMA_CPU1_CSRS_PHYSICAL
#define SABLE_CPU2_CSRS_PHYSICAL            GAMMA_CPU2_CSRS_PHYSICAL
#define SABLE_CPU3_CSRS_PHYSICAL            GAMMA_CPU3_CSRS_PHYSICAL
#define SABLE_CPU0_IPIR_PHYSICAL            GAMMA_CPU0_SICR_PHYSICAL
#define SABLE_CPU1_IPIR_PHYSICAL            GAMMA_CPU1_SICR_PHYSICAL
#define SABLE_CPU2_IPIR_PHYSICAL            GAMMA_CPU2_SICR_PHYSICAL
#define SABLE_CPU3_IPIR_PHYSICAL            GAMMA_CPU3_SICR_PHYSICAL

#define SABLE_PCI0_CONFIGURATION_PHYSICAL   GAMMA_PCI0_CONFIGURATION_PHYSICAL
#define SABLE_PCI1_CONFIGURATION_PHYSICAL   GAMMA_PCI1_CONFIGURATION_PHYSICAL
#define SABLE_PCI0_SPARSE_IO_PHYSICAL       GAMMA_PCI0_SPARSE_IO_PHYSICAL
#define SABLE_PCI0_DENSE_MEMORY_PHYSICAL    GAMMA_PCI0_DENSE_MEMORY_PHYSICAL

//
// Define Interrupt Controller CSRs.
//

#define SABLE_EDGE_LEVEL_CSRS_QVA (HAL_MAKE_QVA(GAMMA_EDGE_LEVEL_CSRS_PHYSICAL))
#define SABLE_INTERRUPT_CSRS_QVA (HAL_MAKE_QVA(GAMMA_INTERRUPT_CSRS_PHYSICAL))

//
// Define XIO interrupt controller CSRs.
//

#define XIO_INTERRUPT_CSRS_QVA (HAL_MAKE_QVA(XIO_INTERRUPT_CSRS_PHYSICAL))

#define SABLE_PCI_CONFIG_BASE_QVA (HAL_MAKE_QVA(GAMMA_PCI0_CONFIGURATION_PHYSICAL))

//
//  Gamma uses the Rattler chipset for the CBUS bridge. Define the Sable
//  CPU CSRS to be Rattler CPU Csrs.
//
#define SABLE_CPU_CSRS      RATTLER_CPU_CSRS
#define PSABLE_CPU_CSRS     PRATTLER_CPU_CSRS
	

//
// Define CPU CSRs and masks.
//

#define SABLE_CPU0_CSRS_QVA (HAL_MAKE_QVA(GAMMA_CPU0_CSRS_PHYSICAL))
#define SABLE_CPU1_CSRS_QVA (HAL_MAKE_QVA(GAMMA_CPU1_CSRS_PHYSICAL))
#define SABLE_CPU2_CSRS_QVA (HAL_MAKE_QVA(GAMMA_CPU2_CSRS_PHYSICAL))
#define SABLE_CPU3_CSRS_QVA (HAL_MAKE_QVA(GAMMA_CPU3_CSRS_PHYSICAL))
#define GAMMA_MEM0_CSRS_QVA (HAL_MAKE_QVA(GAMMA_MEM0_CSRS_PHYSICAL))
#define GAMMA_MEM1_CSRS_QVA (HAL_MAKE_QVA(GAMMA_MEM1_CSRS_PHYSICAL))
#define GAMMA_MEM2_CSRS_QVA (HAL_MAKE_QVA(GAMMA_MEM2_CSRS_PHYSICAL))
#define GAMMA_MEM3_CSRS_QVA (HAL_MAKE_QVA(GAMMA_MEM3_CSRS_PHYSICAL))

#define GAMMA_PRIMARY_PROCESSOR         ((ULONG)0x0)
#define GAMMA_SECONDARY_PROCESSOR       ((ULONG)0x1)
#define GAMMA_MAXIMUM_PROCESSOR         ((ULONG)0x3)
#define HAL_PRIMARY_PROCESSOR           (GAMMA_PRIMARY_PROCESSOR)
#define HAL_MAXIMUM_PROCESSOR           (GAMMA_MAXIMUM_PROCESSOR)

//
// Define the default processor frequency to be used before the actual
// frequency can be determined.
//

#define DEFAULT_PROCESSOR_FREQUENCY_MHZ (275)

enum {
    NoError,
    UncorrectableError,
    CorrectableError
} ErrorType;

//
// Define the memory module CSRs (from chapter 10 of the GAMMA CPU spec)
//

//
// Error Summary register
//

typedef struct _GAMMA_ESREG_CSR1 {
    union {
        ULONG EVBCorrecectableError0: 3;                // 0-2
        ULONG Resrved1: 1;                              // 3
        ULONG EVBFatalError0: 4;                        // 4-7
        ULONG DTError0: 2;                              // 8-9
        ULONG DTSummary0: 1;                            // 10
        ULONG Reserved2: 1;                             // 11
        ULONG IBParError0: 1;                           // 12
        ULONG IBErrorInfo0: 2;                          // 13-14
        ULONG IBSummary0: 1;                            // 15
        ULONG CBError0: 8;                              // 16-23
        ULONG CBSummary0: 1;                            // 24
        ULONG CBCommand0: 1;                            // 25
        ULONG Reserved3: 2;                             // 26-27
        ULONG EVNoResponse0: 1;                         // 28
        ULONG Reserved4: 3;                             // 29-31

        ULONG EVBCorrecectableError1: 3;                // 32-34
        ULONG Resrved5: 1;                              // 35
        ULONG EVBFatalError1: 4;                        // 36-39
        ULONG DTError1: 2;                              // 40-41
        ULONG DTSummary1: 1;                            // 42
        ULONG Reserved6: 1;                             // 43
        ULONG IBParError1: 1;                           // 44
        ULONG IBErrorInfo1: 2;                          // 45-46
        ULONG IBSummary1: 1;                            // 47
        ULONG CBError1: 4;                              // 48-51
        ULONG Reserved7: 4;                             // 52-55
        ULONG CBSummary1: 1;                            // 56
        ULONG CBCommand1: 1;                            // 57
        ULONG Reserved8: 2;                             // 58-59
        ULONG EVNoResponse1: 1;                         // 60
        ULONG Reserved9: 3;                             // 61-63
    };
    ULONGLONG all;

} GAMMA_ESREG_CSR1, *PGAMMA_ESREG_CSR1;

typedef struct _GAMMA_EVBCR_CSR2 {
    union {
        ULONG EnableAddressCommandBusParityCheck0: 1;   // 0
        ULONG Reserved1: 3;                             // 1-3
        ULONG EnableCorrectionErrorInterrupt0: 1;       // 4
        ULONG EnableECCCorrection0: 1;                  // 5
        ULONG EnableRattlerECCCheck0: 1;                // 6
        ULONG Reserved2: 20;                            // 7-26
        ULONG ForceFilledShared: 1;                     // 27
        ULONG RmmStxcFillShared: 1;                     // 28
        ULONG Reserved3: 3;                             // 29-31
        ULONG EnableAddressCommandBusParityCheck1: 1;   // 32
        ULONG Reserved4: 4;                             // 33-35
        ULONG EnableCorrectionErrorInterrupt1: 1;       // 36
        ULONG EnableECCCorrection1: 1;                  // 37
        ULONG EnableRattlerECCCheck1: 1;                // 38
        ULONG DisableEV5ECCChecking1: 1;                // 39
        ULONG Reserved5: 24;                            // 40-63
    };
    ULONGLONG all;

} GAMMA_EVBCR_CSR2, *PGAMMA_EVBCR_CSR2;


typedef struct _GAMMA_EVBVEAR_CSR3 {
    union {
        ULONG EVBVictimErrorAddress0: 30;               // 0-29
        ULONG Reserved1: 2;                             // 30-31
        ULONG EVBVictimErrorAddress1: 30;               // 32-61
        ULONG Reserved2: 2;                             // 62-63
    };

} GAMMA_EVBVEAR_CSR3, *PGAMMA_EVBVEAR_CSR3;

typedef struct _GAMMA_EVBCER_CSR4 {
    union {
        ULONG CorrectableError0: 2;                     // 0-1
        ULONG ReadDirty0: 1;                            // 2
        ULONG MissedCorrectable0: 1;                    // 3
        ULONG Reserved1: 4;                             // 4-7
        ULONG ECCSyndrome0: 8;                          // 8-15
        ULONG ECCSyndrome2: 8;                          // 16-23
        ULONG Reserved2: 8;                             // 24-31

        ULONG CorrectableError1: 2;                     // 32-33
        ULONG ReadDirty1: 1;                            // 34
        ULONG MissedCorrectable1: 1;                    // 35
        ULONG Reserved3: 4;                             // 36-39
        ULONG ECCSyndrome1: 8;                          // 40-47
        ULONG ECCSyndrome3: 8;                          // 48-55
        ULONG Reserved4: 8;                             // 56-63
    };
    ULONGLONG all;

} GAMMA_EVBCER_CSR4, *PGAMMA_EVBCER_CSR4;

typedef struct _GAMMA_EVBCEAR_CSR5 {
    union {
        ULONG EVBCorrectableErrorAddress0: 32;          // 0-31
        ULONG EVBCorrectableErrorAddress1: 32;          // 32-63
    };
    ULONGLONG all;

} GAMMA_EVBCEAR_CSR5, *PGAMMA_EVBCEAR_CSR5;

typedef struct _GAMMA_EVBUER_CSR6 {
    union {
        ULONG UncorrectableError0: 2;                   // 0-1
        ULONG ReadDirty0: 1;                            // 2
        ULONG Reserved1: 1;                             // 3
        ULONG ParityErrorOnAddressCommand0: 1;          // 4
        ULONG ParityErrorOnVictim0: 1;                  // 5
        ULONG Reserved2: 2;                             // 6-7
        ULONG ECCSyndrome0: 8;                          // 8-15
        ULONG ECCSyndrome2: 8;                          // 16-23
        ULONG Reserved3: 4;                             // 24-27
        ULONG EVBCommand0: 4;                           // 28-31

        ULONG UncorrectableError1: 2;                   // 32-33
        ULONG ReadDirty1: 1;                            // 34
        ULONG Reserved4: 1;                             // 35
        ULONG ParityErrorOnAddressCommand1: 1;          // 36
        ULONG ParityErrorOnVictim1: 1;                  // 37
        ULONG Reserved5: 2;                             // 38-39
        ULONG ECCSyndrome1: 8;                          // 40-47
        ULONG ECCSyndrome3: 8;                          // 48-55
        ULONG Reserved6: 4;                             // 56-59
        ULONG EVBCommand1: 4;                           // 60-63
    };
    ULONGLONG all;

} GAMMA_EVBUER_CSR6, *PGAMMA_EVBUER_CSR6;

typedef struct _GAMMA_EVBUEAR_CSR7 {
    union {
        ULONG EVBUncorrectableErrorAddress0: 32;        // 0-31
        ULONG EVBUncorrectableErrorAddress1: 32;        // 32-63
    };
    ULONGLONG all;

} GAMMA_EVBEUAR_CSR7, *PGAMMA_EVBUEAR_CSR7;


typedef struct _GAMMA_CBER_CSR18 {
    union {
        ULONG UncorrectableReadError0: 1;               // 0
        ULONG Reserved1: 3;                             // 1-3
        ULONG CAParity0ErrorL: 1;                       // 4
        ULONG CAParity0ErrorH: 1;                       // 5
        ULONG Reserved2: 2;                             // 6-7
        ULONG ParityErrorWriteDataLW0: 1;               // 8
        ULONG ParityErrorWriteDataLW1: 1;               // 9
        ULONG ParityErrorWriteDataLW4: 1;               // 10
        ULONG ParityErrorWriteDataLW5: 1;               // 11
        ULONG Reserved3: 4;                             // 12-15
        ULONG ParityErrorReadDataLW0: 1;                // 16
        ULONG ParityErrorReadDataLW1: 1;                // 17
        ULONG ParityErrorReadDataLW4: 1;                // 18
        ULONG ParityErrorReadDataLW5: 1;                // 19
        ULONG UnexpectedSharedResponse: 1;              // 20
        ULONG Reserved4: 3;                             // 21-23
        ULONG CANotAcked: 1;                            // 24
        ULONG Reserved5: 3;                             // 25-27
        ULONG Data0NotAcked: 1;                         // 28
        ULONG Data1NotAcked: 1;                         // 29
        ULONG Reserved6: 2;                             // 30-31
        
        ULONG UncorrectableReadError1: 1;               // 32
        ULONG Reserved7: 3;                             // 33-35
        ULONG CAParity1ErrorL: 1;                       // 36
        ULONG CAParity1ErrorH: 1;                       // 37
        ULONG Reserved8: 2;                             // 38-39
        ULONG ParityErrorWriteDataLW2: 1;               // 40
        ULONG ParityErrorWriteDataLW3: 1;               // 41
        ULONG ParityErrorWriteDataLW6: 1;               // 42
        ULONG ParityErrorWriteDataLW7: 1;               // 43
        ULONG Reserved9: 4;                             // 44-47
        ULONG ParityErrorReadDataLW2: 1;                // 48
        ULONG ParityErrorReadDataLW3: 1;                // 49
        ULONG ParityErrorReadDataLW6: 1;                // 50
        ULONG ParityErrorReadDataLW7: 1;                // 51
        ULONG Reserved10: 12;                           // 52-63
    };
    ULONGLONG all;

} GAMMA_CBER_CSR18, *PGAMMA_CBER_CSR18;

typedef struct _GAMMA_CBEALR_CSR19 {
    union {
        ULONG CBusErrorLowAddress0: 32;                 // 0-31
        ULONG CBusErrorLowAddress1: 32;                 // 32-63
    };
    ULONGLONG all;

} GAMMA_CBEALR_CSR19, *PGAMMA_CBEALR_CSR19;

typedef struct _GAMMA_CBEAHR_CSR20 {
    union {
        ULONG CBusErrorHighAddress0: 32;                 // 0-31
        ULONG CBusErrorHighAddress1: 32;                 // 32-63
    };
    ULONGLONG all;

} GAMMA_CBEAHR_CSR20, *PGAMMA_CBEAHR_CSR20;


typedef struct _SGL_MEM_CSR0 {
    union {
        ULONG ErrorSummary1: 1;                     // 0
        ULONG SyncError1: 1;                        // 1
        ULONG CAParityError1: 1;                    // 2
        ULONG CAMissedParityError1: 1;              // 3
        ULONG WriteParityError1: 1;                 // 4
        ULONG MissedWriteParityError1: 1;           // 5
        ULONG Reserved1: 2;                         // 6-7

        ULONG CAParityErrorLW0: 1;                  // 8
        ULONG CAParityErrorLW2: 1;                  // 9
        ULONG ParityErrorLW0: 1;                    // 10
        ULONG ParityErrorLW2: 1;                    // 11
        ULONG ParityErrorLW4: 1;                    // 12
        ULONG ParityErrorLW6: 1;                    // 13
        ULONG Reserved2: 2;                         // 14-15

        ULONG EDCUncorrectable1: 1;                 // 16
        ULONG EDCMissedUncorrectable1: 1;           // 17
        ULONG EDCCorrectable1: 1;                   // 18
        ULONG EDCMissdedCorrectable1: 1;            // 19
        ULONG Reserved3: 12;                        // 20-31

        ULONG ErrorSummary2: 1;                     // 32
        ULONG SyncError2: 1;                        // 33
        ULONG CAParityError2: 1;                    // 34
        ULONG CAMissedParityError2: 1;              // 35
        ULONG WriteParityError2: 1;                 // 36
        ULONG MissedWriteParityError2: 1;           // 37
        ULONG Reserved4: 2;                         // 38-39

        ULONG CAParityErrorLW1: 1;                  // 40
        ULONG CAParityErrorLW3: 1;                  // 41
        ULONG ParityErrorLW1: 1;                    // 42
        ULONG ParityErrorLW3: 1;                    // 43
        ULONG ParityErrorLW5: 1;                    // 44
        ULONG ParityErrorLW7: 1;                    // 45
        ULONG Reserved5: 2;                         // 46-47

        ULONG EDCUncorrectable2: 1;                 // 48
        ULONG EDCMissedUncorrectable2: 1;           // 49
        ULONG EDCCorrectable2: 1;                   // 50
        ULONG EDCMissdedCorrectable2: 1;            // 51
        ULONG Reserved6: 12;                        // 52-63
    };
    ULONGLONG all;

} SGL_MEM_CSR0, *PSGL_MEM_CSR0;

//
// Define the per-processor data structures allocated in the PCR
// for each Gamma processor.
//
// NOTE: If the IpirSva field is moved, the change must be reflected in
//       the routine HalpGammaIpiInterrupt
//

typedef struct _GAMMA_PCR{
    ULONGLONG       HalpCycleCount; 	// 64-bit per-processor cycle count
    ULONGLONG       IpirSva;           // Superpage Va of per-processor IPIR CSR
    PVOID           CpuCsrsQva;		    // Qva of per-cpu csrs
    EV5ProfileCount ProfileCount;       // Profile counter state
} GAMMA_PCR, *PGAMMA_PCR;

#define HAL_PCR ( (PGAMMA_PCR)(&(PCR->HalReserved)) )


//
// Define Miscellaneous Gamma routines.
//

VOID
WRITE_CPU_REGISTER(
    PVOID,
    ULONGLONG
    );

ULONGLONG
READ_CPU_REGISTER(
    PVOID
    );

ULONGLONG
READ_MEM_REGISTER(
    PVOID
    );

BOOLEAN
HalpGammaDispatch(
    VOID
    );

VOID
HalpGammaIpiInterrupt(
    VOID
    );

#endif //!_LANGUAGE_ASSEMBLY

#endif //_GAMMAH_
