/*---------------------------------------------------------------------------

 File: CAccCntrl.h

 Copyright (c) 1996-1996, Microsoft Corp. All rights reserved.

 Description: This file contains the class definitions of COAccessControl,
              CImpAccessControl, and CFAccessControl.
              Note that the class and structure declarations contained in
              are not meant to be used and seen directly by developers who
              only uses the IAccessControl interface.

 Classes:  COAccessControl - This is the principle class that implements
                             the DCOM IAccessControl component object.
                             Except the nondelegating IUnknown interface, the
                             COAccessControl class supports the IPersist,
                             IPersistStream and IAccessControl interfaces by
                             exposing its inner CImpAccessCOntrol pointer.
                             Through this arrangement, the COAccessControl
                             class is able to support aggregation by
                             controlling the object to which the IUnknown
                             calls of the inner CImpAccess control are
                             delegated at object contruction.

           CImpAccessControl - This is the class that nested inside
                               COAccessControl. CImpAccessControl
                               implements the IPersist, IPersistStream,
                               IAccessControl  interfaces and the IUnknown
                               interface which always delegates the call to
                               the IUnknown methods of the controlling object.
                               When COAccessControl is not part of an
                               aggregate CImpAccessControl IUnknown calls
                               should be delegated to its outer
                               COAccessControl, otherwise the IUnknown calls
                               should be delegated to the object controlling
                               the outer COAccessControl object.

           CFAccessControl - Class factory for manufacturing COAccessControl
                             objects.

 Notes: The definition of IAccessControl interface can be found in
        oleext.h in ...sdk\inc and the data types that are defined to use with
        IAccessControl can be found in sdk\inc\accctrl.h.

--------------------------------------------------------------------------*/

#ifndef _CACCCNTRL_H_
#define _CACCCNTRL_H_

//////////////////////////////////////////////////////////////////////////////
// Internal data types
//////////////////////////////////////////////////////////////////////////////

/*////////////////////////////////////////////////////////////////////////////
 PCB- Pickle Control Block.
      Originally intended to be a structure for maintaining the information
      about the pickling buffer, the PCB has become something that transcends
      its intended purposes. Besides data related to the pickling buffer, the
      structure also contains a copy of the access control object's ACL in a
      format that can readily be serialized into a buffer by one of the type
      encoding function generated by the midl compiler, see acpickl.idl for
      details. The bDirtyHandle and the bPickled fields in the PCB structure
      are control flags which allows better coordination between different
      methods in CImpAccessControl. The purpose of the bDirtyHandle flag is
      to minimize the number of times the encoding handle has to be reset
      and the purpose of the bPickled flag is to minimize the number of times
      the ACL has to be serialized into a buffer.

////////////////////////////////////////////////////////////////////////////*/
typedef struct tagPCB
{
    char       *pPicklingBuff;      // This pointer is always aligned on the 8-byte
                                    // boundary
    char       *pTruePicklingBuff;  // This is the true pickling buffer pointer
    ULONG      ulPicklingBuffSize;  // Size of the pickling buffer after the 8-byte alignment
    ULONG      ulBytesUsed;         // This field indicates the number of bytes requires
                                    // to serialize the interanl ACL
    STREAM_ACL StreamACL;           // The stream format ACL
    handle_t   PickleHandle;        // Handle for encoding and decoding
    BOOL       bDirtyHandle;        // This flag indicates whether the handle needs to be reset
    BOOL       bPickled;            // This flag indicates whether the current stream ACL has been encoded
                                    // into the pickling buffer.
    ULONG      ulMaxNumOfStreamACEs;// The maximum number of stream ACEs that the StreamACL structure has been allocated for
    ULONG      ulNumOfStreamACEs;   // The number of stream ACEs that the StreamACL structure is holding

} PCB;

#ifdef _CHICAGO_

///////////////////////////////////////////////////////////////////////////////
// ACL_IMAGE - This structure is used on Chicago only
//  This structure can only be considered a partial representation of the
//  ACL using the native structure available on the Chicago plaform. Owing
//  to fact that the LAN Manager APIs available on the Chicago platform
//  can distinguish between GRANT_ACCESS mode ACEs and DENY_MODE ACEs, two
//  LAN Manager ACLs are used to create the illusion of having two different
//  types of ACEs can appear simultaneously in the ACL.
///////////////////////////////////////////////////////////////////////////////

typedef struct tagACL_IMAGE
{
    char        *pACL;         // Pointer to a LAN Manager ACL.
                               // Format: |access_info_1|array of access_list|
    SHORT       *psNumOfACEs;  // The number of ACE
    SHORT       sMaxNumOfACEs; // The maximum number of ACEs that the
                               // structure can hold
    BOOL        bDirtyACL;     // This flag indicatess whether the ACL has
                               // changed since the last time it was mapped to
                               // the system registry use the NetAccess* functions
} ACL_IMAGE;

///////////////////////////////////////////////////////////////////////////////
// ACL_DESC - ACL Descriptor
//  The ACL descriptor is a structure that describes how access control
//  native to specific platform can be used to emulate the ACL as perceived
//  through the IAccessControl interface.
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// The following structure is the ACL descriptor on the Chicago platform.
// This structure is divided into two symmetric portions, one portion holds
// the DENY_ACCESS mode ACEs while the other holds GRANT_ACCESS mode ACEs.
///////////////////////////////////////////////////////////////////////////////
typedef struct tagACLDescriptor
{
    ACL_IMAGE DenyACL;
    ACL_IMAGE GrantACL;
} ACL_DESCRIPTOR;

#else

///////////////////////////////////////////////////////////////////////////////
// The following is supposed to be the ACL descriptor on the Windows NT
// platform. Since the Windows NT version of DCOM IAccessControl implementation
// is still under developement, the content of the following structure may
// change in the future.
//////////////////////////////////////////////////////////////////////////////
typedef struct tagACL_DESCRIPTOR
{
    void                *pACLBuffer;     // Pointer to the NT ACL buffer
    ULONG               ulACLBufferSize; // Size of the ACL buffer
    ULONG               ulSIDSize;       // Exacted size of all the SIDs in the NT ACL
    BOOL                bDirtyACL;       // This flag indicates whether the internal
                                         // ACL has been chcanged since the last time
                                         // it was mapped to an NT ACL.
    SECURITY_DESCRIPTOR SecDesc;         // We need the security descriptor to call
                                         // AccessCheck

} ACL_DESCRIPTOR;



#endif

//////////////////////////////////////////////////////////////////////////////
// CFAccessControl - COAccessControl class factory.
//////////////////////////////////////////////////////////////////////////////

class CFAccessControl : public IClassFactory
{
private:

    // Private variables.
    LONG               m_cRefs; // Object reference count

public:

    // IUnknown methods

    STDMETHODIMP_(HRESULT) QueryInterface
    (
    REFIID iid,
    void   **ppv
    );

    STDMETHODIMP_(ULONG) AddRef(void);

    // If the object's reference count reaches zero, the RElease method will
    // decrement the global object count named g_cObjects in acsrv.cxx.
    STDMETHODIMP_(ULONG) Release(void);


    // IClassFactory methods
    // Upon successful creation of a new COAccessControl object,
    // the following function will increment the global object
    // count named g_cObjects by one.
    STDMETHODIMP_(HRESULT) CreateInstance
    (
    IUnknown *pUnkOuter,
    REFIID   riid,
    void     **ppv
    );

    // The following method relies on a global lock count named g_cServer
    // which is maintained inside acsrv.cxx.
    STDMETHODIMP_(HRESULT) LockServer
    (
    BOOL fLock
    );

    // Constructor
    CFAccessControl(void);

    // Destructor
    ~CFAccessControl(void);

}; // CFAccessControl

//////////////////////////////////////////////////////////////////////////////
// COAccessControl - The DCOM IAccessControl implementation component. The
//                   COAccessControl componnet is implemented as a nested
//                   class to support aggregation.
//////////////////////////////////////////////////////////////////////////////
class COAccessControl : public IUnknown
{
public:

    // Main object IUnknown Methods - These IUnknown methods are non-delegating
    STDMETHODIMP_(HRESULT) QueryInterface
    (
    REFIID riid,
    void   **ppv
    );

    STDMETHODIMP_(ULONG) AddRef(void);

    // The following method relies on a global lock count named g_cServer
    // which is maintained inside acsrv.cxx.
    STDMETHODIMP_(ULONG) Release(void);

    // Constructor
    COAccessControl(void);
    STDMETHODIMP_(HRESULT) Init(IUnknown *pOuter);

    // Destructor
    ~COAccessControl(void);

    //////////////////////////////////////////////////////////////////////
    // CImpAccessControl - This class is nested inside COAccessControl.
    //                     The CImpAccessControl class implements
    //                     the IPersistStream interface, IPersist interface
    //                     , and the IAccessControl interface
    //////////////////////////////////////////////////////////////////////
    class CImpAccessControl : public IPersistStream, public IAccessControl
    {
    public:

        // IUnknown methods, all calls are delegated to the controlling object.
        STDMETHODIMP_(HRESULT) QueryInterface
        (
        REFIID riid,
        void   **ppv
        );

        STDMETHODIMP_(ULONG) AddRef(void);

        STDMETHODIMP_(ULONG) Release(void);

        // IPersist method

        STDMETHODIMP_(HRESULT) GetClassID
        (
        CLSID *pClassID
        );

        // IPersistStream Methods
        STDMETHODIMP_(HRESULT) IsDirty
        (
        void
        );

        // Object initialization method. This method must be called
        // before any non-IUnknown methods.
        STDMETHODIMP_(HRESULT)Load
        (
        IStream *pStm
        );

        STDMETHODIMP_(HRESULT)Save
        (
        IStream *pStm,
        BOOL    fClearDirty
        );

        STDMETHODIMP_(HRESULT) GetSizeMax
        (
        ULARGE_INTEGER *pcdSize
        );

        // IAccessControl Methods
        STDMETHODIMP_(HRESULT) GrantAccessRights
        (
        PACTRL_ACCESSW pAccessList
        );

        // This function is not implemented.
        STDMETHODIMP_(HRESULT) SetAccessRights
        (
        PACTRL_ACCESSW pAccessList
        );

        // This function is not implemented.
        STDMETHODIMP_(HRESULT) SetOwner
        (
        PTRUSTEEW pOwner,
        PTRUSTEEW pGroup
        );

        STDMETHODIMP_(HRESULT) RevokeAccessRights
        (
        LPWSTR         lpProperty,
        ULONG          cCount,
        TRUSTEEW       pTrustee[]
        );

        STDMETHODIMP_(HRESULT) GetAllAccessRights
        (
        LPWSTR               lpProperty,
        PACTRL_ACCESSW      *ppAccessList,
        PTRUSTEEW           *ppOwner,
        PTRUSTEEW           *ppGroup
        );

        STDMETHODIMP_(HRESULT) IsAccessAllowed
        (
        PTRUSTEEW            pTrustee,
        LPWSTR               lpProperty,
        ACCESS_RIGHTS        AccessRights,
        BOOL                *pfAccessAllowed
        );

        // Constructor
        CImpAccessControl
        (
        IUnknown *pBackPtr,
        IUnknown *pUnkOuter
        );

        // Destructor
        ~CImpAccessControl(void);


    private:

        STDMETHODIMP_(void) CleanupAccessList
        (
        BOOL           fReleaseAll,
        STREAM_ACE    *pStreamACEReqs,
        void          *pACEReqs,
        ULONG          cGrant,
        ULONG          cDeny
        );

        STDMETHODIMP_(HRESULT) AddAccessList
        (
        STREAM_ACE    *pStreamACEReqs,
        void          *pACEReqs,
        ULONG          ulEstPickledSize,
        ULONG          cGrant,
        ULONG          cDeny
        );

#ifndef _CHICAGO_
        STDMETHODIMP_(HRESULT) GetEffAccRightsUsingSID
        (
        PSID    pSID,
        DWORD   *pdwRights
        );

        STDMETHODIMP_(HRESULT) GetEffAccRightsUsingName
        (
        LPWSTR  pTrusteeName,
        DWORD   *pdwRights
        );
#endif
        // Static data members
        BOOL                 m_bInitialized; // Object initialization flag.
        BOOL                 m_bDirty;       // This flag is set to TRUE if the
                                             // object has been changed since the
                                             // last save.
        IUnknown             *m_pUnkOuter;   // Pointer to the controlling object's
                                             // IUnkown implementation.
        CRITICAL_SECTION     m_ACLLock;      // Critical section object for
                                             // protecting the ACL from concurrent
                                             // access.
#ifdef _CHICAGO_
        CEffectivePermsCache m_Cache;        // This cache stroes the results of
                                             // previous access checking.
#else
        CEffPermsCacheString m_CacheString;  // Access check results cache indexed by Unicode string
        CEffPermsCacheSID    m_CacheSID;     // Access check results cache indexed by SID.
#endif
        ACL_DESCRIPTOR       m_ACLDesc;      // Platform dependent  representation
                                             // of the ACL
        PCB                  m_pcb;          // Pickle control block

    }; // COAccessControl::CImpAccessControl

    // Private variables

    LONG              m_cRefs;       // Object's reference count
    CImpAccessControl *m_ImpObj;     // Pointer to the inner CImpAccessControl object

}; // COAccessControl


#endif // #ifndef _CACCCNTRL_H_
